'''
A class for handling consensus FASTA files generated by the ARTIC nCoV
analysis pipeline.
'''

import os
import sys
import re
import collections
import pysam

class Consensus():
    '''
    A class to handle the consensus FASTA file.
    '''

    def __init__(self, file):
        '''
        Initialize the Consensus class.
        '''
        if os.path.exists(file):
            self.file = file
        else:
            sys.exit("Invalid or missing file.")
        try:
            fasta = pysam.FastxFile(file)
            for record in fasta:
                samplename = re.sub('.primertrimmed.consensus_threshold_0.75_quality_20$',
                                    '',
                                    record.name)
                samplename = re.sub('^Consensus_', '', samplename)
                self.sequence = record.sequence.upper()
                self.name = samplename
        except:
            print("File does not exist or cannot read file.")


    def count_iupac_in_fasta(self):
        '''
        Count the number of IUPAC occurrences in the consensus
        FASTA file.  N is counted as a seperate value.

        Return Value:
            The function returns an integer representing the number of IUPAC code
            occurrences in the FASTA file.  Note that Ns were not considered.
        '''
        iupac_codes = ['R', 'Y', 'S', 'W', 'K', 'M', 'B', 'D', 'H', 'V']
        base_counter = collections.Counter()
        base_counter.update(self.sequence)
        total = 0
        for base, count in base_counter.items():
            if base in iupac_codes:
                total += count
        return {'num_consensus_iupac' : total,
                'num_consensus_n' : base_counter['N']}


    def get_genome_completeness(self):
        '''
        Get the proportion of Ns in the genome.
        '''
        base_counter = collections.Counter()
        for base in self.sequence:
            base_counter.update(base.upper())

        total = 0
        for base, count in base_counter.items():
            total += count
        completeness = 0
        if total > 0:
            completeness = 1 - (float(base_counter['N']) / total)
            completeness = round(completeness, 4)
        return {'genome_completeness' : completeness}
