"""Validify - rule-based validator for assessing the structure of an xml tree."""

from lxml import etree
from loguru import logger
import collections
import re

from validify.helpers.cleanup_compare_strings import get_compare_value
from validify.helpers import normalize_space
from validify.helpers import messages
from validify.tests.prepare_testdata import compile_test_rules


def log_message(message: str, level: str, log_to_console: bool, log_debug: bool = False):
    """Deliver log message to logger if console logging is enabled."""

    if log_to_console:
        if level == "debug":
            if log_debug:
                logger.debug(message)
        elif level == "info":
            logger.info(message)
        elif level == "warn":
            logger.warning(message)
        elif level == "error":
            logger.error(message)

def assess_root_element(root_element: etree.Element, validation_rules: dict, validation_messages: list, validation_results: list, message_lang: str):
    """Evaluate the document's root element and check if it is the correct element in the expected namespace."""

    if "$root_element" in validation_rules:
        for validation_rules_set in validation_rules["$root_element"]:
            root_element_name = root_element.tag
            root_element_local_name = etree.QName(root_element).localname
            root_element_sourceline = root_element.sourceline

            expected_root_element_name = validation_rules_set["element_name"]
            expected_root_element_local_name = validation_rules_set["element_local_name"]

            if root_element_name != expected_root_element_name:
                if root_element_local_name == expected_root_element_local_name:
                    message_id = "s0002"
                    message_text = messages.get_message_by_id(message_id, message_lang).format(root_element_name,
                                                                                               expected_root_element_name)
                    validation_messages.append(message_text)
                    validation_results.append(
                        {"message_id": message_id, "message_text": message_text, "element_name": root_element_name,
                         "element_local_name": root_element_local_name, "element_sourceline": root_element_sourceline,
                         "element_path": "/{}".format(root_element_name)})
                else:
                    message_id = "s0001"
                    message_text = messages.get_message_by_id(message_id, message_lang).format(root_element_name, expected_root_element_name)
                    validation_messages.append(message_text)
                    validation_results.append({"message_id": message_id, "message_text": message_text, "element_name": root_element_name,
                                               "element_local_name": root_element_local_name, "element_sourceline": root_element_sourceline,
                                               "element_path": "/{}".format(root_element_name)})


def get_element_path(element: etree.Element, local_name=True) -> str:
    """Get the full path of an xml element by iterating through the ancestor elements.

    If parameter local_name=True, local element names (without namespaces) will be used in the result string.
    """

    element_name = element.tag
    element_ancestors = list(element.iterancestors())
    if local_name:
        element_name = etree.QName(element).localname
        element_ancestor_tags = list(reversed([etree.QName(element).localname for element in element_ancestors]))
    else:
        element_ancestor_tags = list(reversed([element.tag for element in element_ancestors]))
    element_path_string = "/".join(element_ancestor_tags)
    if len(element_ancestor_tags) > 0:
        element_path_string = "/{}/{}".format(element_path_string, element_name)
    else:
        element_path_string = "/{}".format(element_name)

    return element_path_string

def assess_element_structure(element: etree.Element, element_sourceline: int, xmlns_def: dict, validation_rules: dict, validation_messages: list, validation_results: list, message_lang: str, log_to_console: bool, log_debug: bool) -> list:
    """Asssess the structure of an xml element, according to the provided validation rules.

    Takes an etree.Element, namespace definition and validation rules.
    If a rule's condition is not satisfied, a validation message is created as a dict, logged and added to a validation results list.

    Example output: [{'message_id': '0001', 'message_text ': 'Element example_element does not contain any subelements, although one or more subelements are expected.', 'element_name': '{namespace}example_element', 'local_name': 'example_element' 'element_sourceline': '23'}]
    """

    element_name = element.tag
    element_local_name = etree.QName(element).localname
    element_path = get_element_path(element)
    element_attributes = element.attrib
    element_subelements = [subelement.tag for subelement in element]


    if element_name in validation_rules:

        for validation_rules_set in validation_rules[element_name]:
            # Check if conditions for applying the validiation rule set are satisfied (validation_rules_set["rule_conditions"])
            condition_satisfied = {"text_values": True, "attribute_values": True, "reference_elements": True}

            for validation_rule_condition in validation_rules_set["rule_conditions"]:
                if len(validation_rule_condition["text_values"]) > 0:
                    if element.text not in validation_rule_condition["text_values"]:
                        condition_satisfied["text_values"] = False
                if len(validation_rule_condition["attribute_def"]) > 0:
                    for attribute in validation_rule_condition["attribute_def"]:
                        if attribute["attribute_name"] in element.attrib:
                            if element.attrib[attribute["attribute_name"]] not in attribute["allowed_values"]:
                                condition_satisfied["attribute_values"] = False
                        else:
                            condition_satisfied["attribute_values"] = False
                if len(validation_rule_condition["reference_elements"]) > 0:
                    for reference in validation_rule_condition["reference_elements"]:
                        reference_element = element
                        for step in range(reference["preceding_elements"]):
                            if reference_element.getparent() is not None:
                                reference_element = reference_element.getparent()

                        if reference_element.tag != reference["element_name"]:
                            condition_satisfied["reference_elements"] = False
                        for attribute in reference["attribute_def"]:
                            if attribute["attribute_name"] in reference_element.attrib:
                                if reference_element.attrib[attribute["attribute_name"]] not in attribute["allowed_values"]:
                                    condition_satisfied["reference_elements"] = False
                            else:
                                condition_satisfied["reference_elements"] = False


            not_satisfied = [condition_key for condition_key, condition_value in condition_satisfied.items() if condition_value is False]
            if len(not_satisfied) > 0:
                log_message("Validation ruleset for element {} not applied because the following rule conditions are not satisfied: {}.".format(element_name, ", ".join(not_satisfied)), "debug", log_to_console, log_debug)
                continue


            # element children optional
            if not validation_rules_set["element_children_optional"]:
                if len(element) == 0:
                    message_id = "0001"
                    message_text = messages.get_message_by_id(message_id, message_lang).format(element_name)
                    validation_messages.append(message_text)
                    validation_results.append({"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

            # element content optional
            if not validation_rules_set["element_content_optional"]:
                if get_compare_value(element) == "":
                    message_id = "0002"
                    message_text = messages.get_message_by_id(message_id, message_lang).format(element_name)
                    validation_messages.append(message_text)
                    validation_results.append(
                        {"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

            # optional attributes
            for element_attribute in element_attributes:
                if (element_attribute not in validation_rules_set["optional_attributes"]) and (element_attribute not in validation_rules_set["obligatory_attributes"]):
                    message_id = "0003"
                    message_text = messages.get_message_by_id(message_id, message_lang).format(element_name, element_attribute)
                    validation_messages.append(message_text)
                    validation_results.append(
                        {"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

            # obligatory attributes
            for obligatory_attribute in validation_rules_set["obligatory_attributes"]:
                if obligatory_attribute not in element_attributes:
                    message_id = "0004"
                    message_text = messages.get_message_by_id(message_id, message_lang).format(element_name, obligatory_attribute)
                    validation_messages.append(message_text)
                    validation_results.append(
                        {"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

            # optional subelements
            for element_subelement in element_subelements:
                if (element_subelement not in validation_rules_set["optional_subelements"]) and (element_subelement not in validation_rules_set["obligatory_subelements"]):
                    if str(element_subelement).startswith("<cyfunction Comment"):
                        log_message("Subelement validation: Ignoring comment function under parent element {}.".format(element_name), "debug", log_to_console, log_debug)
                        continue
                    message_id = "0005"
                    message_text = messages.get_message_by_id(message_id, message_lang).format(element_name, element_subelement)
                    validation_messages.append(message_text)
                    validation_results.append(
                        {"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

            # obligatory subelements
            for obligatory_subelement in validation_rules_set["obligatory_subelements"]:
                if obligatory_subelement not in element_subelements:
                    message_id = "0006"
                    message_text = messages.get_message_by_id(message_id, message_lang).format(element_name, obligatory_subelement)
                    validation_messages.append(message_text)
                    validation_results.append(
                        {"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

            # max occurence
            if validation_rules_set["max_occurence"] is not None:  # max occurence nur prüfen, wenn das Element nicht unbegrenzt auftreten kann
                if element.getparent() is not None:
                    element_siblings = element.getparent().findall(element_name, namespaces=xmlns_def)
                    if len(element_siblings) > validation_rules_set["max_occurence"]:
                        message_id = "0007"
                        message_text = messages.get_message_by_id(message_id, message_lang).format(element_name, len(element_siblings), validation_rules_set["max_occurence"])
                        validation_messages.append(message_text)
                        validation_results.append(
                            {"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

            # character content allowed
            if validation_rules_set["text_character_content_allowed"] is False:
                if element.text is not None:
                    if element.text != "":
                        message_id = "0008"
                        message_text = messages.get_message_by_id(message_id, message_lang).format(element_name)
                        validation_messages.append(message_text)
                        validation_results.append(
                            {"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})
            if validation_rules_set["tail_character_content_allowed"] is False:
                if element.tail is not None:
                    if element.tail != "":
                        message_id = "0009"
                        message_text = messages.get_message_by_id(message_id, message_lang).format(element_name)
                        validation_messages.append(message_text)
                        validation_results.append(
                            {"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

            # allowed values (xs:enumeration)
            if len(validation_rules_set["allowed_values"]) > 0:
                if element.text is not None:
                    if element.text not in validation_rules_set["allowed_values"]:
                        message_id = "0010"
                        message_text = messages.get_message_by_id(message_id, message_lang).format(element_name, element.text, ", ".join(validation_rules_set["allowed_values"]))
                        validation_messages.append(message_text)
                        validation_results.append({"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

            # allowed patterns (xs:pattern)
            if len(validation_rules_set["allowed_patterns"]) > 0:
                valid_value = False
                if element.text is not None:
                    for pattern in validation_rules_set["allowed_patterns"]:
                        match_pattern = re.compile(pattern)
                        pattern_matches = match_pattern.match(element.text)
                        if pattern_matches:
                            valid_value = True
                            break
                if not valid_value:
                    message_id = "0011"
                    message_text = messages.get_message_by_id(message_id, message_lang).format(element_name, element.text, ", ".join(validation_rules_set["allowed_patterns"]))
                    validation_messages.append(message_text)
                    validation_results.append({"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

            # Attribute definition
            for attribute_definition in validation_rules_set["attribute_def"]:
                if attribute_definition["attribute_name"] in element.attrib:
                    # allowed values (xs:enumeration)
                    if len(attribute_definition["allowed_values"]) > 0:
                        if element.attrib[attribute_definition["attribute_name"]] not in attribute_definition["allowed_values"]:
                            message_id = "0012"
                            message_text = messages.get_message_by_id(message_id, message_lang).format(element_name, attribute_definition["attribute_name"], element.attrib[attribute_definition["attribute_name"]], ", ".join(attribute_definition["allowed_values"]))
                            validation_messages.append(message_text)
                            validation_results.append({"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})

                    # allowed patterns (xs:pattern)
                    if len(attribute_definition["allowed_patterns"]) > 0:
                        valid_value = False
                        for pattern in attribute_definition["allowed_patterns"]:
                            match_pattern = re.compile(pattern)
                            pattern_matches = match_pattern.match(element.attrib[attribute_definition["attribute_name"]])
                            if pattern_matches:
                                valid_value = True
                                break
                        if not valid_value:
                            message_id = "0013"
                            message_text = messages.get_message_by_id(message_id, message_lang).format(element_name, attribute_definition["attribute_name"], element.attrib[attribute_definition["attribute_name"]], ", ".join(attribute_definition["allowed_patterns"]))
                            validation_messages.append(message_text)
                            validation_results.append({"message_id": message_id, "message_text": message_text, "element_name": element_name, "element_local_name": element_local_name, "element_sourceline": element_sourceline, "element_path": element_path})




    return validation_results



def validate(input_file=None, input_elementtree=None, xmlns_def=None, validation_rules=None, message_lang=None, log_to_console=True, log_debug=False):
    """Validate a given xml file according to the supplied validation rules and return validation messages as a dictionary.

    All elements in the xml file will be processed.
    Since version 0.0.11, an etree.ElementTree object (parameter 'input_elementtree') can also be passed, instead of an input file string.
    For a documentation of parameters, see https://github.com/olivergoetze/validify.
    """

    if xmlns_def is None:
        xmlns_def = {}
    if validation_rules is None:
        validation_rules = compile_test_rules()
        log_message("No validation rules defined; using example rules for validation.", "warn", log_to_console)
    message_lang_options = ["de", "en"]
    if message_lang is None:
        message_lang = "de"
        log_message("No message language option delivered; using standard value 'de'.", "info", log_to_console)
    elif message_lang not in message_lang_options:
        message_lang = "de"
        log_message("Non-valid message language option delivered; using standard value 'de'.", "warn", log_to_console)

    validation_messages = []
    validation_results = []

    if input_file is not None or input_elementtree is not None:
        try:
            if input_file is not None:
                xml_in = etree.parse(input_file)
            else:
                xml_in = input_elementtree
            xml_root_element = xml_in.getroot()
            assess_root_element(xml_root_element, validation_rules, validation_messages, validation_results, message_lang)

            xml_elements = xml_in.findall("//{*}*")
            xml_elements = [xml_root_element] + xml_elements
            for xml_element in xml_elements:
                xml_element_sourceline = xml_element.sourceline  # get original source line before applying normalize-space
                normalize_space.parse_xml_content(xml_element)  # apply normalize-space so only actual character content is found
                validation_results = assess_element_structure(xml_element, xml_element_sourceline, xmlns_def, validation_rules, validation_messages, validation_results, message_lang, log_to_console, log_debug)
        except etree.XMLSyntaxError as e:
            message_id = "e0001"
            message_text = messages.get_message_by_id(message_id, message_lang).format(input_file, e)
            validation_messages.append(message_text)
            validation_results.append({"message_id": message_id, "message_text": message_text, "element_name": "",
                                       "element_local_name": "", "element_sourceline": "",
                                       "element_path": ""})
        except AttributeError as e:
            message_id = "e0002"
            message_text = messages.get_message_by_id(message_id, message_lang).format(e)
            validation_messages.append(message_text)
            validation_results.append({"message_id": message_id, "message_text": message_text, "element_name": "",
                                       "element_local_name": "", "element_sourceline": "",
                                       "element_path": ""})


    # Aggregate and output validation messages
    if len(validation_messages) > 0:
        if input_file is not None:  # log only when the input is a file and not an ElementTree.
            log_message("Validation results for file '{}':".format(input_file), "info", log_to_console)
        aggregated_validation_messages = collections.Counter(validation_messages)
        for validation_message in aggregated_validation_messages:
            log_message("{} ({} occurences)".format(validation_message, aggregated_validation_messages[validation_message]), "info", log_to_console)

    return validation_results
