from dataclasses import dataclass
from typing import Set, Callable, Dict, Type, Optional

from pybeandi.model import UserGeneralBeanRef, id_ref, BeanRef, BeanId
from pybeandi.util import camel_case_to_snake_case


@dataclass
class BeanMeta:
    """
    Bean meta. Usually attached to it's class or factory function
    """
    bean_id: BeanId
    profile_func: Callable[[Set[str]], bool]
    cls: Type
    depends_on: Dict[str, BeanRef]


def bean(bean_id: Optional[BeanId] = None,
         profiles: Set[str] = None,
         profile_func: Callable[[Set[str]], bool] = lambda profs: True,
         **depends_on: Dict[str, UserGeneralBeanRef]):
    """
    Annotate class or factory function (they are same for lib)
    as importable as bean

    @param bean_id: bean id of bean.
    If None then autogenerated from factory func's name
    @param profiles: set of profiles which must be active
    @param profile_func: function that indicate been
    should be created by active profiles. Overrides 'profiles'
    @param depends_on: dictionary of func argument name to bean definition
    @return: same factory func
    """
    def wrapper(cls):
        nonlocal bean_id, depends_on, profile_func
        depends_on = {
            arg_name: (
                id_ref(arg_ref)
                if isinstance(arg_ref, BeanId)
                else arg_ref
            )
            for (arg_name, arg_ref) in depends_on.items()
        }
        bean_id = bean_id \
            if bean_id is not None \
            else camel_case_to_snake_case(cls.__name__)
        profile_func = profile_func \
            if profiles is None \
            else lambda profs: profs >= profiles

        cls._bean_meta = BeanMeta(bean_id, profile_func, cls, depends_on)

        return cls

    return wrapper


@dataclass
class AfterInitMeta:
    """
    After init bean method meta. Usually attached to it's method
    """
    depends_on: Dict[str, BeanRef]


def after_init(**depends_on: Dict[str, UserGeneralBeanRef]):
    """
    Annotate method of bean to be called after init all beans.
    Useful for recursive dependencies (it's not good but can happen)

    @param depends_on: dictionary of func argument name to bean definition
    @return: same method
    """
    def wrapper(func):
        nonlocal depends_on
        depends_on = {
            arg_name: (
                id_ref(arg_ref)
                if isinstance(arg_ref, BeanId)
                else arg_ref
            )
            for (arg_name, arg_ref) in depends_on.items()}

        func._bean_meta = AfterInitMeta(depends_on)

        return func

    return wrapper
