# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['task_assembly', 'task_assembly.example']

package_data = \
{'': ['*']}

install_requires = \
['api-client>=1.3.1,<2.0.0',
 'boto3>=1.26.0,<2.0.0',
 'larry>=0.2.12,<0.3.0',
 'pyyaml>=6.0,<7.0',
 'tabulate>=0.9.0,<0.10.0',
 'toml>=0.10.2,<0.11.0']

entry_points = \
{'console_scripts': ['task-assembly = task_assembly.cli:main']}

setup_kwargs = {
    'name': 'task-assembly',
    'version': '0.1.14',
    'description': 'SDK and CLI for using the Task Assembly crowdwork service',
    'long_description': '# Task Assembly Client\nTools for working with the Task Assembly service for managing crowdwork projects.\n\n## Initial Setup\nStart by installing the task-assembly client by running the following command.\n\n```shell\npip install task-assembly\n```\n\nNote that the task-assembly client uses some AWS resources. If you haven\'t done so already, you will need to install\nthe AWS CLI and configure it with your credentials (`aws configure`) before using the client. \n\nNext, run the following command to configure your Task Assembly API key:\n\n```shell\ntask-assembly configure --key <api-key>\n```\n\nAt any point, you can run the following command to check that your account has been correctly configured:\n\n```shell\ntask-assembly configure --validate\n```\n\n## Example project\nIf you are new to Task Assembly, you can start with a small sample project by running `task-assembly example` to \ncopy a set of files to your working directory. These can be used with the instructions below to set up your first\nproject and help you build familiarity with how Task Assembly works.\n\n## Project setup\n\nFirst, we\'ll need to create a task-type for this project. This is just a placeholder and will be removed in\nfuture updates to Task Assembly. Simply run the following command and capture the id that is generated.\nWhen building the example application, use `NumbersExample` as the type name.\n\n```shell\ntask-assembly create_task_type <type_name>\n```\n\nNow, we\'ll want to create a new Task Definition for our task using the Task Type created in the previous step.\nWhen building the example application, use `Numbers` as the definition name.\n\n```shell\ntask-assembly create_task_definition <definition_name> <task_type_id>\n```\n\nThis will generate a new definition for our project and write a definition.json file that we will use to capture\nour task attributes.\n\n## Build the Task Definition\nNow we can update the `definition.json` file with the appropriate attributes for our task. We\'ve included\nthe core attributes for the example task below:\n\n```yaml\nDefinitionId: <your_definition_id>\nTaskType: <your_task_type_id>\nTitle: Convert number to text\nDescription: Write out a number as text\nRewardCents: 10\nLifetime: 3600\nAssignmentDuration: 300\nDefaultAssignments: 2\nMaxAssignments: 5\nAutoApprovalDelay: 0\n```\n\nYou can now run the following command to update the definition in Task Assembly:\n\n```shell\ntask-assembly update_task_definition\n```\n\nThe task interface for our project is in the `template.html` file, and you will note that it uses a very similar \ntemplating language as Amazon SageMaker Ground Truth and the Amazon MTurk website. In most cases you can simply\ncopy/paste existing task templates from those tools and update the variable names within the `{{ name }}` values.\nTo apply this new task template to our definition, we can simply add a reference to it in our `definition.json` file:\n\n```yaml\nTemplateFile: template.html\n```\nThen run the `update_task_defintion` command again.\n```shell\ntask-assembly update_task_definition\n```\n\nThe `update_task_definition` command can be used to submit any future changes you make to your definition.json.\n\n## Create a task in the sandbox\nNow that we\'ve completed the setup for our task, we should start by creating a test task in the *Sandbox* environment.\nThe Sandbox is a mirror of the Production environment but no money changes hands and work isn\'t generally completed\nunless you do it yourself. This is a great way to validate that your task is set up correctly before putting it in the\nhands of *real* Workers.\n\nNote: The MTurk Sandbox is distinct from the Production environment, you will want to create a new account at\nhttps://requestersandbox.mturk.com with the same email address as your Production account and also link it to the same\nAWS account.\n\nTo create a single MTurk task in the Sandbox, you can use the following command. The `--sandbox` flag lets Task\nAssembly know to create the task in the appropriate environment and the `--assignments 1` flag indicates that we\nonly need to ask one Worker to provide an answer to the task (just you). Finally, `number=4` provides the input\ndata to our task interface which refers to `{{ number }}` from the html template.\n\n```shell\ntask-assembly create_task --sandbox --assignments 1 number=4\n```\n\nTake note of the task ID that is returned. You\'ll need that to retrieve the results.\n\nThis will create a new task for you in the Worker Sandbox, which you can view and work on by visiting\nhttps://workersandbox.mturk.com. You\'ll likely need to create an account there, feel free to use your personal \nAmazon account or any other account you wish. Once you\'re logged in, you can search for your username or\nthe title of the task. Then you can accept and complete the task.\n\nAfter you\'ve completed your task, give MTurk and Task Assembly a few moments to process the result. Then run \nthe following to get the output.\n\n```shell\ntask-assembly get_task <task_id>\n```\n\n## Create a *real* task\nNow we can repeat the same process to create a task in the Production environment by simply removing the\n`--sandbox` and `--assignments` flags.\n\n```shell\ntask-assembly create_task number=4\n```\n\nThe same `get_task` command can be used to retrieve results of the created task.\n\n## Result Processing\nYou may have noticed that the task result above includes `Result` and `Responses` values that looks a bit like this:\n\n```json\n{\n  "Result": {\n    "default": [\n      {\n        "WorkerId": "A12NBGVI9QN3DQ",\n        "Result": {\n          "default": {\n            "numberAsText": "Four"\n          }\n        },\n        "AssignmentId": "3ERMJ6L4DYRPC5L5VMTQ0TYQKQG7M2"\n      },\n      {\n        "WorkerId": "ACKG8OU1KHKO2",\n        "Result": {\n          "default": {\n            "numberAsText": "four"\n          }\n        },\n        "AssignmentId": "3FTYUGLFSUK7L719U0FQJJX0K6T5D3"\n      }\n    ]\n  },\n  "Responses": [\n    {\n      "WorkerId": "A12NBGVI9QN3DQ",\n      "Result": {\n        "default": {\n          "numberAsText": "Four"\n        }\n      },\n      "AssignmentId": "3ERMJ6L4DYRPC5L5VMTQ0TYQKQG7M2"\n    },\n    {\n      "WorkerId": "ACKG8OU1KHKO2",\n      "Result": {\n        "default": {\n          "numberAsText": "four"\n        }\n      },\n      "AssignmentId": "3FTYUGLFSUK7L719U0FQJJX0K6T5D3"\n    }\n  ]\n}\n```\n\nThis output is the `default` structure that Task Assembly uses and ensures that all the relevant data is included.\nOf course, simplifying this output is preferred and will greatly simplify downstream data processing. To do this, we\'ll\nuse `handlers` that will perform data processing on each response and completed task. In most cases, AWS Lambda \nfunctions are the best option for these steps and Task Assembly helps this process by generating the Lambdas on your \nbehalf if you wish. This tutorial will use Task Assembly to manage the Lambdas, but you\'re welcome to create and deploy\nthem normally using your preferred approach (Console, Cloud Formation, CDK, etc).\n\nIf building the example application you\'ll find the handler code for this tutorial in the `handlers.py` file. The first \nfunction `process_response` simply retrieves the `numberAsText` form field and returns it. We can pull it into our \nproject adding the following to our `definition.json`:\n\n```yaml\nHandlerFile: handlers.py\nSubmissionHandlers:\n- Name:  value\n  FunctionName:  process_response\n```\n\nThis tells Task Assembly to run the `process_response` on the form outputs and return it in an attribute name `value`.\nTo test how this impacts task results, run the following to update the task definition and then *redrive* the task\nresults to reprocess the results:\n\n```shell\ntask-assembly update_task_definition\ntask-assembly redrive_task <task-id>\n```\n\nNow the `Responses` list should be the following, simpler, representation of the Worker responses.\n\n```json\n[\n  {\n    "WorkerId": "A12NBGVI9QN3DQ",\n    "Result": {\n      "value": "Four"\n    },\n    "AssignmentId": "3ERMJ6L4DYRPC5L5VMTQ0TYQKQG7M2"\n  },\n  {\n    "WorkerId": "ACKG8OU1KHKO2",\n    "Result": {\n      "value": "four"\n    },\n    "AssignmentId": "3FTYUGLFSUK7L719U0FQJJX0K6T5D3"\n  }\n]\n```\n\nNext, we want to consolidate these two responses into a single value as a result for this task. To do that, we\'ll \nadd a *consolidation* handler. The `consolidate_result` function in the example handles this step by simply looking \nfor agreement between Workers. If two Workers agree, then we\'ll use that result. If not, we\'ll return `{\'extend\': True}` \nwhich will prompt Task Assembly to ask an additional Worker to provide a response, up to the `MaxAssignments`. To add\nthis function to our task definition, we simply include the following in our `definition.json`.\n\n```yaml\nHandlerFile: handlers.py\nSubmissionHandlers:\n- Name:  value\n  FunctionName:  process_response\nConsolidationHandlers: \n- Name: value\n  FunctionName: consolidate_result\n```\n\nRunning `update_task_definition` and `redrive_task` will invoke this new consolidation logic, and we will now have a\nsingle value result for our task.\n\n```json\n{\n  "value": "four"\n}\n```\n\n## Create a batch\nNow that our task is set up, we can run it on a larger set of data. If using the example, the `batch.csv` file contains \na small batch of data. It is simply a single column CSV with column header `number` that corresponds to our task input, \nbut multiple values can be used for more complex projects. To submit the batch, the following command can be used. The \nfirst parameter is the name we want to give the batch `numbers1`, followed by the source data file for the batch, \n`batch.csv`. Finally, we provide an S3 location we will use to store the input and output files created by Task Assembly.\n\n```shell\ntask-assembly submit_batch numbers1 batch.csv s3://taskassembly-test/batches\n```\n\nThe result is a batch_id we can use to monitor progress and retrieve the results. To get batch status, the following\ncan be used:\n\n```shell\ntask-assembly get_batch_status <batch_id>\n```\n\nThis command can be used to monitor your batch progress. When the batch is complete, running the following\nto retrieve the results to a CSV file.\n\n```shell\ntask-assembly get_batch_results <batch_id> output.csv\n```\n\n## Improving Quality\nThe last step is to establish some test tasks that we can use to ensure we\'re getting Workers who will \ndo a good job on our task. For the example we\'ve created a data file of known or *gold* answers in `gold.json`.\nWhen Workers first start working on this task they will be prompted to answer at least two of these before\nthey can work on our *real* data.\n\nTo evaluate accuracy, we\'ll add a new handler `score_response` that scores each Worker\'s response. For\nthis we\'ll use a very simple comparison and give them a score of 100 if they provide an exact match,\n80 if they don\'t match on the inclusion of "and" or "-" in their answer (i.e. "twenty-four" or \n"one hundred and six"), and 0 if they still don\'t match.\n\nThe following statements can be added to our `definition.json` file to enable scoring of new workers.\nAs you can see, Workers will need to complete at least two tests with an average score of 80 before \nthey can begin working on the *real* tasks. Note that because it generally takes at least 10-15 seconds \nbefore scoring is complete so Workers will often be asked to do a third test if they quickly accept another\ntask after completing their second test.\n\n```yaml\nScoringHandler: \n  FunctionName: score_response\nGoldAnswersFile: gold.json\nTestPolicy:\n  MinTests: 2\n  MinScore: 80\n```\n\nWe can now create a new batch to begin using the new scoring that we\'ve created. However, because scoring\ndoesn\'t work well with small batches, expand the size of your `batch.csv` file to at least 20 numbers\nbefore submitting a new batch to Workers. This limitation is a result of how MTurk handles HITs with\nfewer than 10 Assignments and will be addressed in a future release of Task Assembly.\nEven with a larger set items in this test batch, you may not get responses for all of your items, however\nsubsequent batches should have higher yield.\n\nOnce you\'ve submitted a new batch, you\'ll note that `get_batch_results` will now show test responses in\nthe results. Because this is the first time we\'ve run a batch with scoring enabled, all Workers will need\nto complete 2-3 test tasks. This means that we\'ll see a larger number of test responses and fewer *Task*\nresponses in our counts. In future batches the portion of responses associated with tests will steadily\ndecline.\n\n```\n - Response Counts:\n     Task: 36\n     Test: 109\n     Total: 145\n```\n\nTo see how Workers did on your tests and their relative contribution to the final output, run\n`task_assembly list_workers workers.csv` to generate a report on how many responses\neach Worker has provided for this task definition and their scores on the test tasks.\n',
    'author': 'Dave Schultz',
    'author_email': 'dave@daveschultzconsulting.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
