"""
Display source code from files/context managers.
"""
import ast, re, os
import sys, linecache
import textwrap
import inspect
import pygments

from contextlib import contextmanager, suppress
from IPython.display import display

from .formatters import highlight, _HTML
    

# Do not use this in main work, just inside a function
class _Source(_HTML):
    "Returns the source code of the object as HTML."
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._raw = ''
    @property
    def raw(self):
        "Return raw source code."
        return self._raw
    
    @raw.setter
    def raw(self, value):
        "Set raw source code."
        self._raw = value
    
    def display(self,collapsed = False):
        "Display source object in IPython notebook."
        if collapsed:
            return _HTML(f"""<details style='max-height:100%;overflow:auto;'>
                <summary>Show Code</summary>
                {self.value}
                </details>""").display()
        else:
            return display(self) # Without collapsed

    def show_lines(self, lines):
        "Return source object with selected lines from list/tuple/range of lines."
        if not isinstance(lines,(list,tuple,range)):
            raise TypeError(f'lines must be list, tuple or range, not {type(lines)}')
        
        start, *middle = self.value.split('<code>')
        middle[-1], end = middle[-1].split('</code>')
        middle[-1] += '</code>'
        _max_index = len(middle) - 1
        
        new_lines = [start]
        picks = [-1,*sorted(lines)]
        for a, b in zip(picks[:-1],picks[1:]):
            if b - a > 1: # Not consecutive lines
                new_lines.append(f'<code class="code-no-focus"> + {b - a - 1} more lines ... </code>')
            new_lines.append('<code>' + middle[b])
        
        if lines and lines[-1] < _max_index:
            new_lines.append(f'<code class="code-no-focus"> + {_max_index - lines[-1]} more lines ... </code>')
        
        return self.__class__(''.join([*new_lines, end]))     
    
    def focus_lines(self, lines):
        "Return source object with focus on given list/tuple/range of lines."
        if not isinstance(lines,(list,tuple,range)):
            raise TypeError(f'lines must be list, tuple or range, not {type(lines)}')
        
        _lines = []
        for i, line in enumerate(self.value.split('<code>'), start = -1):
            if i == -1:
                _lines.append(line) # start things
            elif i not in lines:
                _lines.append('<code class="code-no-focus">' + line)
            else:
                _lines.append('<code class="code-focus">' + line)
        
        return self.__class__(''.join(_lines))

def _file2code(filename,language='python',name=None,**kwargs):
    "Only reads plain text or StringIO, return source object with `show_lines` and `focus_lines` methods."
    try:
        text = filename.read() # if stringIO
    except:
        with open(filename,'r') as f:
            text = f.read()
    
    return _str2code(text,language=language,name=name,**kwargs)


def _str2code(text,language='python',name=None,**kwargs):
    "Only reads plain text source code, return source object with `show_lines` and `focus_lines` methods."
    out = _Source(highlight(text,language = language, name = name, **kwargs).value)
    out.raw = text
    return out

class Source:
    current = None
    def __init__(self):
        raise Exception("""This class is not meant to be instantiated.
        Use Source.context() to get a context manager for source.
        Use Source.current to get the current source object.
        Use Source.from_file(filename) to get a source object from a file.
        Use Source.from_string(string) to get a source object from a string.
        Use Source.from_callable(callable) to get a source object from a callable.
        """)
    @classmethod
    def from_string(cls,text,language='python',name=None,**kwargs):
        "Creates source object from string. `name` is alternate used name for language. `kwargs` are passed to `ipyslides.formatter.highlight`."
        cls.current = _str2code(text,language=language,name=name,**kwargs)
        return cls.current
    
    @classmethod
    def from_file(cls, filename,language = None,name = None,**kwargs):
        """Returns source object with `show_lines` and `focus_lines` methods. `name` is alternate used name for language.  
        `kwargs` are passed to `ipyslides.formatter.highlight`.     
        
        It tries to auto detect lanaguage from filename extension, if `language` is not given.
        """
        _title = name or filename
        _lang = language or os.path.splitext(filename)[-1].replace('.','')
        
        if language is None:
            lexer = None
            with suppress(BaseException):
                lexer = pygments.lexers.get_lexer_by_name(_lang)
                
            if lexer is None:
                raise Exception(f'Failed to detect language from file {filename!r}. Use language argument!')
            
        cls.current = _file2code(filename,language = _lang,name = _title,**kwargs)
        return cls.current
    
    @classmethod       
    def from_callable(cls, callable,**kwargs):
        "Returns source object from a given callable [class,function,module,method etc.] with `show_lines` and `focus_lines` methods. `kwargs` are passed to `ipyslides.formatter.highlight`"
        for _type in ['class','function','module','method','builtin','generator']:
            if getattr(inspect,f'is{_type}')(callable):
                source = inspect.getsource(callable)
                cls.current = _str2code(source,language='python',name=None)
                return cls.current
    
    @classmethod
    @contextmanager 
    def context(cls, auto_display = True, **kwargs): 
        """Execute and displays source code in the context manager. `kwargs` are passed to `ipyslides.formatter.highlight` function.
        Useful when source is written inside context manager itself.
        If `auto_display` is True (by default), then source is displayed before the output of code. Otherwise you can assign the source to a variable and display it later anywhere.
        
        **Usage**:
        ```python
        with source.context(auto_display = False) as s: #if not used as `s`, still it is stored `source.current` attribute.`
            do_something()
            write(s) # or s.display(), write(s)
            
        #s.raw, s.value are accesible attributes.
        #s.focus_lines, s.show_lines are methods that are used to show selective lines.
        ```
        """ 
        frame = sys._getframe() 
        depth = 2 # default depth is 2 to catch under itself, others would be given from differnt context managers to get their source.
        if 'depth' in kwargs:
            depth = kwargs.pop('depth')
        
        for _ in range(depth):
            frame = frame.f_back # keep going back until required depth is reached.
              
        lines, n1 = linecache.getlines(frame.f_code.co_filename), frame.f_lineno
        offset = 0 # going back to zero indent level
        while re.match('^\t?^\s+', lines[n1 - offset]): 
             offset = offset + 1
             
        _source = ''.join(lines[n1 - offset:])
        tree = ast.parse(_source)
        with_node = tree.body[0] # Could be itself at top level
        
        for node in ast.walk(tree):
            if isinstance(node, ast.With) and node.lineno == offset: # that much gone up, so back same
                with_node = node
                break

        n2 = with_node.body[-1].end_lineno #can include multiline expressions in python 3.8+
        source = textwrap.dedent(''.join(lines[n1:][:n2 - offset]))
        source_html = _Source(highlight(source,language = 'python', **kwargs).value)
        source_html.raw = source # raw source code
        cls.current = source_html
        
        if auto_display:
            source_html.display()
            
        yield source_html
        # No need to try as it is not possible to get here if not in context manager