import re
from textwrap import dedent as _dedent

from ..formatters import _HTML

_icons = {
    'chevron': '''
        <svg width="{size}" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" fill="none" stroke="{color}" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" transform="rotate({rotation})">
            <path d="M9 4L17 12L9 20" />
        </svg>''',
    'pencil': '''
        <svg xmlns="http://www.w3.org/2000/svg" height="{size}" viewBox="0 0 25 25" fill="{color}" stroke-width="2" transform="rotate({rotation})">
            <rect x="9" y="0" width="8" height="6"/>
            <rect x="9" y="7" width="1" height="10"/>
            <rect x="12" y="7" width="2" height="10"/>
            <rect x="16" y="7" width="1" height="10"/>
            <polygon points="9 18,17 18,13 25,9 18"></polygon>
        </svg>''',
    'bars': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke-width="3" stroke="{color}" fill="{color}" transform="rotate({rotation})">
            <circle stroke="none" cx="3" cy="5" r="3"/>
            <circle stroke="none" cx="3" cy="13" r="3"/>
            <circle stroke="none" cx="3" cy="21" r="3"/>
            <line x1="8" y1="5" x2="24" y2="5"/>
            <line x1="8" y1="13" x2="24" y2="13" />
            <line x1="8" y1="21" x2="24" y2="21"/>
        </svg>''',
    'arrow': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke="{color}" stroke-width="3" stroke-linecap="round" stroke-linejoin="round" transform="rotate({rotation})">
            <line x1="4" y1="13" x2="21" y2="13" />
            <path fill="none" d="M13 4L21 13L13 21" />
        </svg>''',
    'close': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke="{color}" stroke-width="3" stroke-linecap="round" stroke-linejoin="round" transform="rotate({rotation})">
            <line x1="4" y1="4" x2="21" y2="21" />
            <line x1="4" y1="21" x2="21" y2="4" />
        </svg>''',
    'dots': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" fill="{color}" transform="rotate({rotation})">
          <circle cx="13" cy="5" r="3"/>
          <circle cx="13" cy="13" r="3"/>
          <circle cx="13" cy="21" r="3"/>
        </svg>''',
    'expand': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke="{color}" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round" transform="rotate({rotation})">
          <line x1="4" y1="21" x2="11" y2="14"/>
          <line x1="14" y1="11" x2="21" y2="4"/>
          <path fill="none" d="M13 4L21 4L21 13" />
          <path fill="none" d="M4 13L4 21L13 21" />
        </svg>''',
    'compress': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke="{color}" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round" transform="rotate({rotation})">
          <line x1="4" y1="21" x2="11" y2="14"/>
          <line x1="14" y1="11" x2="21" y2="4"/>
          <path fill="none" d="M14 4L14 11L21 11" />
          <path fill="none" d="M4 14L11 14L11 21" />
        </svg>''',
    'camera': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke="{color}" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <polygon points="4 8, 7 8,10 6, 16 6, 19 8, 22 8, 22 22, 4 22" fill="{color}"></polygon>
            <circle cx="13" cy="15" r="5" fill="black" stroke="white"/>
        </svg>''',
    'play': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" transform="rotate({rotation})">
            <polygon points="4 4,21 13,4 21" fill="{color}"></polygon>
        </svg>''',
    'pause': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" fill="{color}" transform="rotate({rotation})">
            <rect x="4" y="4" width="7" height="17"/>
            <rect x="14" y="4" width="7" height="17"/>
        </svg>''',
    'loading': '''
        <svg xmlns="http://www.w3.org/2000/svg" height="{size}" viewBox="0 0 50 50" transform="rotate({rotation})">
            <path fill="{color}" d="M25,5A20.14,20.14,0,0,1,45,22.88a2.51,2.51,0,0,0,2.49,2.26h0A2.52,2.52,0,0,0,50,22.33a25.14,25.14,0,0,0-50,0,2.52,2.52,0,0,0,2.5,2.81h0A2.51,2.51,0,0,0,5,22.88,20.14,20.14,0,0,1,25,5Z">
                <animateTransform attributeName="transform" type="rotate" from="0 25 25" to="360 25 25" dur="0.2s" repeatCount="indefinite"/>
            </path>
        </svg>''',
    'circle': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg">
            <circle cx="13" cy="13" r="11" fill="none" stroke="{color}" stroke-width="2"/>
        </svg>''',
    'refresh': '''
        <svg height="{size}" viewBox="0 0 40 40" xmlns="http://www.w3.org/2000/svg">
            <path d=" M 37 18.8 A 17 17 4.2 1 1 32.2 8.2" stroke="{color}" stroke-width="3.5" fill="none"/>
            <path d="M36 11L30 0L24 10" fill="{color}" stroke="none"/>
        </svg>''',
    'laser': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg">
            <circle cx="13" cy="13" r="10" fill="none" stroke="{color}" stroke-width="3" filter="drop-shadow(0 0 1px {color})" opacity="0.2"/>
            <circle cx="13" cy="13" r="7" fill="none" stroke="{color}" stroke-width="2" filter="drop-shadow(0 0 4px {color})" opacity="0.7"/>
            <circle cx="13" cy="13" r="4" fill="{color}" stroke="white" stroke-width="2" filter="drop-shadow(0 0 5px {color})"/>
        </svg>''',
    'zoom-in': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke-width="2" stroke-linecap="round" transform="rotate({rotation})">
            <circle cx="9" cy="9" r="8" fill="none" stroke="{color}"/>
            <line x1="17" y1="17" x2="21" y2="21" stroke="{color}" stroke-width="4"/>
            <line x1="6" y1="9" x2="12" y2="9" stroke="{color}"/>
            <line x1="9" y1="6" x2="9" y2="12" stroke="{color}"/>
        </svg>''',
    'zoom-out': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke-width="2" stroke-linecap="round" transform="rotate({rotation})">
            <circle cx="9" cy="9" r="8" fill="none" stroke="{color}"/>
            <line x1="17" y1="17" x2="21" y2="21" stroke="{color}" stroke-width="4"/>
            <line x1="6" y1="9" x2="12" y2="9" stroke="{color}"/>
        </svg>''',
    'win-maximize': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <polygon points="2 4,23 4,23 21,2 21,2 4" fill="none" stroke="{color}"></polygon>
            <polygon points="2 4,23 4,23 8,2 8,2 4" fill="{color}" stroke="{color}"></polygon>
        </svg>''',
    'win-restore': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <polygon points="2 6,19 6,19 21,2 21,2 6" fill="none" stroke="{color}"></polygon>
            <polygon points="2 8,19 8,19 10,2 10,2 8" fill="{color}" stroke="{color}"></polygon>
            <path d="M4 4L4 2L23 2L23 17L21 17" fill="none" stroke="{color}"/>
        </svg>''',
    'rows': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" transform="rotate({rotation})">
            <path d="M2 2L23 2L23 23L2 23L2 2" fill="none" stroke="{color}"/>
            <line x1="6" y1="7" x2="19" y2="7" stroke="{color}" stroke-width="3" stroke-linecap="butt"/>
            <line x1="6" y1="12.5" x2="19" y2="12.5" stroke="{color}" stroke-width="3" stroke-linecap="butt"/>
            <line x1="6" y1="18" x2="19" y2="18" stroke="{color}" stroke-width="3" stroke-linecap="butt"/>
        </svg>''',
    'columns': '''
        <svg height="{size}" viewBox="0 0 25 25" xmlns="http://www.w3.org/2000/svg" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" transform="rotate({rotation})">
            <polygon points="2 2,23 2,23 23,2 23,2 2" fill="none" stroke="{color}"></polygon>
            <polygon points="2 2,23 2,23 8,2 8,2 2" fill="{color}" stroke="{color}"></polygon>
            <line x1="12.5" y1="2" x2="12.5" y2="23" stroke="{color}"/>
        </svg>''',
}

loading_svg = _icons['loading'].format(size='4em',color='var(--accent-color, gray)',rotation=0) # Required outside

class Icon(_HTML):
    available = tuple(_icons.keys())
    def __init__(self, name: str, color:str = 'currentColor', size:str = '1em',rotation:int = 0) :
        "Get an icon from the available icon set with a given color and size. Not every icon supports rotation."
        if name not in _icons:
            raise KeyError(f'Icon {name} not found. Available icons: {", ".join(_icons.keys())}')
        
        _value = _icons[name].format(color=color, size=size, rotation = rotation).replace('#', '%23')  # replace # with %23 for svg
        super().__init__(_dedent(_value).strip()) # remove leading and trailing whitespace/newlines
        
    def __repr__(self):
        return f'Icon(css = {self.css}, svg = {self.value!r})'
    
    def __format__(self, spec):
        return f'{self._svg_inline:{spec}}' # important for f-strings be iinline to add in tables etc.
    
    @property
    def svg(self):
        "Get the SVG code of the icon."
        return self.value
    
    @property
    def _svg_inline(self):
        return re.sub(r' +', ' ', self.value).replace('\n', '') # remove newlines and extra spaces, keep 1
    
    @property
    def css(self):
        "Get the CSS code of the icon as dictionary of {'content': url(svg)}."
        return {'content': f"url('data:image/svg+xml;utf8,{self._svg_inline}')"}   