# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['bbrest']
install_requires = \
['aiohttp', 'maya', 'requests', 'requests-html>=0.10.0,<0.11.0', 'selenium']

setup_kwargs = {
    'name': 'bbrest',
    'version': '0.4.1',
    'description': 'Blackboard REST API for Humans',
    'long_description': "# BbRest\nBlackboard REST APIs... for humans? (TM)\n\n## Purpose\nThis Python library was created to make the Blackboard REST API more accessible.\nSpecifically, it helps solve a few problems:\n\n### Token Management\nBlackboard tokens expire in one hour, and cannot be expired manually.  \nIt's possible that two uses of one app would be called within an hour of\nstarting the other - in this case, the session expires in less than an hour.\nA call with an expired token returns a 401 error.\n\nThis wrapper has a self healing session that renews the token when needed.\n\n### API availability by version\nAll APIs documented on developer.blackboard.com are not available for all versions of self and managed hosted clients. The terminology is even a bit hard to decipher, as it doesn't match the QX 20XX format most admins are used to.  \n\nThis wrapper will create functions that only allows calling available functions, depending on the version of Blackboard it connects to. This has recently been improved by the distinct versions for .json.\n\n### Finding APIs\nIt can be difficult to find the correct API on developers.blackboard.com, and this wrapper allows you to tab complete generated APIs and to get hints about expected inputs.  \n\n### Convenience APIs\nThis wrapper also has some convenience APIs that make getting information easier.\nThis is also a work in progress, and probably will be built out separately.\n\n## Installation\nInstall poetry and bbrest\n```bash\n$ poetry init\n$ poetry add bbrest\n\nThis will create a virtual environment with python 3 and the dependencies of this project\n```\n#Install and run JupyterLab\n```bash\n$ poetry add --dev jupyterlab\n$ poetry run jupyter lab\n```\n\n## Usage\n\n### Setup\nThe key and secret are from your registration on developer.blackboard.com.\nThe url is the base url for your campus ie: https://blackboard.school.edu\n```python\nfrom bbrest import BbRest\nbb = BbRest(key, secret, url)\n```\n### Session Management\n```python\nbb.expiration()\n```\n'6 minutes ago'\n```python\nr = bb.GetUser(userId='test_user')\nr.status_code\n```\n200\n\nThe call method checks if the session is expired, and renews the token if so.  Other methods around token management:\n\n```python\nr = bb.is_expired() #returns boolean\nr = bb.refresh_token() #manually refreshes the token\n```\n\nNote, refresh_token will receive the same token from Blackboard if the token is not yet expired, even if there's only seconds left.\n\n### REST call discovery / usage with Tab completion\nFind all endpoints available in the current version that have 'GetUs' in the name.\n```python\nr = bb.GetUs<Tab>\n```\n  * bb.GetUser\n  * bb.GetUserGrades\n  * bb.GetUserMemberships\n  * bb.GetUsers\n\nFind the parameters of 'GetColumnGrade'\n```python\nr = bb.GetColumnGrade(<Shift-Tab>)\n```\n  * columnId=\n  * courseId=\n  * userId=\n\nAll available endpoints can be accessed this way.\nParams should be entered as `params={'key':'value'}`\nPayload should be entered as `payload={'key1':'value1','key2':'value2', etc}`\n\n### Calling API endpoints\n```python\n#Some convenience tricks for common calls\nr = bb.GetCourse(courseId='2832102')\n\n#same as above\nr = bb.call('GetCourse',courseId='courseId:2832102')\nr.json()\n\nr = bb.UpdateCourseMembership(courseId='2832102',\n                              userId='test_user',\n                              payload={'availability':{'available':'No'}})\n\n#same as above\nr = bb.call('UpdateCourseMembership',courseId='courseId:2832102',userId='userName:test_user',payload={'availability':{'available':'No'}})\nr.json()\n```\n### Asynchronous calls!\nOne of the big advantages of javascript over python was the idea of promises and asynchronous information gathering. Python now has await and async capabilities, but using it can be tricky. \n\nI've tried to make using it with BbRest to be as easy as possible.\n\n```python\nuser_info = await bb.GetUser('test_user', sync=False)\n```\n\nNOTE: Based on how this is setup - you get back a dict object on success or failure.  The failure has a status, but the success only has the info.  This is slightly different than how it works synchronously, and has the potential to cause logic errors in the code.\n\nHere's an example of multiple calls:\n```python\n#Assume users is a list of userNames\ntasks = []\nfor user in users:\n  tasks.append(bb.GetUser(user), sync=False)\nresps = await asynchio.gather(*tasks)\n```\n\nSince these calls are asynchronous, it's MUCH faster than synchronously going through all users. \n\n### Working on\n[x] Matching up the responses of Async and Sync functions (This is in place as of 3.7)\n[ ] Better exception handling, and a way to view failure history.\n[x] Calls will always be authenticated, if the session is expired, it will renew the session.\n",
    'author': 'Matt Deakyne',
    'author_email': 'm.d@ku.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'py_modules': modules,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
