import argparse
import sys
import pkg_resources
from os.path import expanduser, isfile
from sshush.sshush import read_file, Parser

GLOBAL_DEFAULT_CONFIG_PATH = '/etc/default/sshush'


def main():
    default_path = '{home}/.ssh/config'.format(home=expanduser('~'))
    default_yaml_path = '{home}/.ssh/config.yml'.format(home=expanduser('~'))

    arg_parser = argparse.ArgumentParser()

    arg_parser.add_argument(
        '--version', '-v',
        action='version',
        version=pkg_resources.require('sshush')[0].version
    )

    arg_parser.add_argument(
        '--source', '-s',
        help='Path to source YAML file if it differs from {}'.format(default_yaml_path),
        default=[default_yaml_path],
        nargs='+'
    )

    arg_parser.add_argument(
        '--path', '-p', '-o', '--output',
        help='Path to SSH config file if it differs from {}'.format(default_path),
        default=default_path
    )

    args = arg_parser.parse_args()

    if isfile(GLOBAL_DEFAULT_CONFIG_PATH):
        args.source.append(GLOBAL_DEFAULT_CONFIG_PATH)

    print('sshush running with path "{path}" and source YAML "{yaml}"'.format(
        path=args.path,
        yaml=args.source
    ))

    try:
        with open(args.path, 'w') as fh:
            fh.write('# Generated by sshush v{} ({})\n'.format(pkg_resources.require('sshush')[0].version, sys.argv[0]))
            fh.write('# From {}\n\n'.format(args.source))

            parser = Parser()

            for file in args.source:
                yaml_obj = read_file(file)
                config_file_contents = parser.process_yaml(yaml_obj)

                fh.write(config_file_contents)
                fh.write("\n")

            fh.write(parser.output_global_config())
            fh.write("\n")

            print('{} written successfully'.format(args.path))

    except IOError as exc:
        print('Error:', exc.strerror)
        exit(1)


if __name__ == '__main__':
    main()
