"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tasks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const fs = require("fs-extra");
const component_1 = require("./component");
const json_1 = require("./json");
const task_1 = require("./task");
const task_runtime_1 = require("./task-runtime");
/**
 * Defines project tasks.
 *
 * Tasks extend the projen CLI by adding subcommands to it. Task definitions are
 * synthesized into `.projen/tasks.json`.
 */
class Tasks extends component_1.Component {
    constructor(project) {
        super(project);
        this._tasks = {};
        this._env = {};
        new json_1.JsonFile(project, task_runtime_1.TaskRuntime.MANIFEST_FILE, {
            omitEmpty: true,
            obj: {
                tasks: (() => this.renderTasks()),
                env: (() => this._env),
            },
        });
    }
    /**
     * All tasks.
     */
    get all() {
        return Object.values(this._tasks);
    }
    /**
     * Adds a task to a project.
     * @param name The name of the task
     * @param options Task options.
     */
    addTask(name, options = {}) {
        const task = new task_1.Task(name, options);
        if (this._tasks[name]) {
            throw new Error(`A task with the name ${name} already exists. To override it, call removeTask first and then create the new task.`);
        }
        this._tasks[name] = task;
        return task;
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        const dependentTasks = this.all.filter((task) => task.steps.find((step) => step.spawn == name));
        if (dependentTasks.length > 0) {
            const errList = dependentTasks.map((depTask) => depTask.name).join(", ");
            throw new Error(`Unable to remove task "${name}" because the following tasks depend on it: ${errList}`);
        }
        const task = this._tasks[name];
        if (task) {
            delete this._tasks[name];
            return task;
        }
        else {
            return undefined;
        }
    }
    /**
     * Adds global environment.
     * @param name Environment variable name
     * @param value Value
     */
    addEnvironment(name, value) {
        this._env[name] = value;
    }
    /**
     * Returns a copy of the currently global environment for this project.
     */
    get env() {
        return {
            ...this._env,
        };
    }
    /**
     * Finds a task by name. Returns `undefined` if the task cannot be found.
     * @param name The name of the task
     */
    tryFind(name) {
        return this._tasks[name];
    }
    synthesize() {
        if (this.project.ejected) {
            // Insert a task-runner script so that tasks can be run after ejecting
            fs.mkdirpSync(path.join(this.project.outdir, "scripts"));
            fs.copyFileSync(path.join(__dirname, "..", "lib", "run-task.js"), path.join(this.project.outdir, "scripts", "run-task"));
            fs.chmodSync(path.join(this.project.outdir, "scripts", "run-task"), "755");
        }
    }
    renderTasks() {
        const tasks = {};
        for (const task of Object.values(this._tasks).sort((x, y) => x.name.localeCompare(y.name))) {
            tasks[task.name] = task._renderSpec();
        }
        return tasks;
    }
}
exports.Tasks = Tasks;
_a = JSII_RTTI_SYMBOL_1;
Tasks[_a] = { fqn: "projen.Tasks", version: "0.67.2" };
//# sourceMappingURL=data:application/json;base64,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