"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const publisher_1 = require("./publisher");
const release_trigger_1 = require("./release-trigger");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const version_1 = require("../version");
const BUILD_JOBID = "release";
const GIT_REMOTE_STEPID = "git_remote";
const LATEST_COMMIT_OUTPUT = "latest_commit";
/**
 * Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 */
class Release extends component_1.Component {
    constructor(project, options) {
        super(project);
        this._branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.github = project.github;
        this.buildTask = options.task;
        this.preBuildSteps = options.releaseWorkflowSetupSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.artifactsDirectory = options.artifactsDirectory ?? "dist";
        this.versionFile = options.versionFile;
        this.releaseTrigger = options.releaseTrigger ?? release_trigger_1.ReleaseTrigger.continuous();
        this.containerImage = options.workflowContainerImage;
        this.workflowRunsOn = options.workflowRunsOn;
        this._branchHooks = [];
        /**
         * Use manual releases with no changelog if releaseEveryCommit is explicitly
         * disabled and no other trigger is set.
         *
         * TODO: Remove this when releaseEveryCommit and releaseSchedule are removed
         */
        if (!((options.releaseEveryCommit ?? true) ||
            options.releaseSchedule ||
            options.releaseTrigger)) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.manual({ changelog: false });
        }
        if (options.releaseSchedule) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.scheduled({
                schedule: options.releaseSchedule,
            });
        }
        this.version = new version_1.Version(project, {
            versionInputFile: this.versionFile,
            artifactsDirectory: this.artifactsDirectory,
            versionrcOptions: options.versionrcOptions,
            tagPrefix: options.releaseTagPrefix,
        });
        this.publisher = new publisher_1.Publisher(project, {
            artifactName: this.artifactsDirectory,
            condition: `needs.${BUILD_JOBID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha`,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
            failureIssue: options.releaseFailureIssue,
            failureIssueLabel: options.releaseFailureIssueLabel,
            workflowRunsOn: options.workflowRunsOn,
            publishTasks: options.publishTasks,
            dryRun: options.publishDryRun,
            workflowNodeVersion: options.workflowNodeVersion,
        });
        const githubRelease = options.githubRelease ?? true;
        if (githubRelease) {
            this.publisher.publishToGitHubReleases({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
            });
        }
        // add the default branch (we need the internal method which does not require majorVersion)
        this.defaultBranch = this._addBranch(options.branch, {
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            minMajorVersion: options.minMajorVersion,
            workflowName: options.releaseWorkflowName ?? "release",
            tagPrefix: options.releaseTagPrefix,
            npmDistTag: options.npmDistTag,
        });
        for (const [name, opts] of Object.entries(options.releaseBranches ?? {})) {
            this.addBranch(name, opts);
        }
    }
    /**
     * Returns the `Release` component of a project or `undefined` if the project
     * does not have a Release component.
     */
    static of(project) {
        const isRelease = (c) => c instanceof Release;
        return project.components.find(isRelease);
    }
    /**
     * Add a hook that should be run for every branch (including those that will
     * be added by future `addBranch` calls).
     * @internal
     */
    _forEachBranch(hook) {
        for (const branch of this._branches) {
            hook(branch.name);
        }
        this._branchHooks.push(hook);
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    addBranch(branch, options) {
        this._addBranch(branch, options);
        // run all branch hooks
        for (const hook of this._branchHooks) {
            hook(branch);
        }
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    _addBranch(branch, options) {
        if (this._branches.find((b) => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch &&
            options.majorVersion &&
            this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        const releaseBranch = {
            name: branch,
            ...options,
            workflow: this.createWorkflow(branch, options),
        };
        this._branches.push(releaseBranch);
        return releaseBranch;
    }
    preSynthesize() {
        for (const branch of this._branches) {
            if (!branch.workflow) {
                continue;
            }
            branch.workflow.addJobs(this.publisher._renderJobsForBranch(branch.name, branch));
            branch.workflow.addJobs(this.jobs);
        }
    }
    /**
     * Adds jobs to all release workflows.
     * @param jobs The jobs to add (name => job)
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    /**
     * Retrieve all release branch names
     */
    get branches() {
        return this._branches.map((b) => b.name);
    }
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    createWorkflow(branchName, branch) {
        const workflowName = branch.workflowName ?? `release-${branchName}`;
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const noNewCommits = `\${{ steps.${GIT_REMOTE_STEPID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha }}`;
        // The arrays are being cloned to avoid accumulating values from previous branches
        const preBuildSteps = [...this.preBuildSteps];
        const env = {
            RELEASE: "true",
        };
        if (branch.majorVersion !== undefined) {
            env.MAJOR = branch.majorVersion.toString();
        }
        if (branch.minMajorVersion !== undefined) {
            if (branch.majorVersion !== undefined) {
                throw new Error(`minMajorVersion and majorVersion cannot be used together.`);
            }
            env.MIN_MAJOR = branch.minMajorVersion.toString();
        }
        if (branch.prerelease) {
            env.PRERELEASE = branch.prerelease;
        }
        if (branch.tagPrefix) {
            env.RELEASE_TAG_PREFIX = branch.tagPrefix;
        }
        // the "release" task prepares a release but does not publish anything. the
        // output of the release task is: `dist`, `.version.txt`, and
        // `.changelog.md`. this is what publish tasks expect.
        // if this is the release for "main" or "master", just call it "release".
        // otherwise, "release:BRANCH"
        const releaseTaskName = branchName === "main" || branchName === "master"
            ? "release"
            : `release:${branchName}`;
        const releaseTask = this.project.addTask(releaseTaskName, {
            description: `Prepare a release from "${branchName}" branch`,
            env,
        });
        releaseTask.exec(`rm -fr ${this.artifactsDirectory}`);
        releaseTask.spawn(this.version.bumpTask);
        releaseTask.spawn(this.buildTask);
        releaseTask.spawn(this.version.unbumpTask);
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        releaseTask.exec(Release.ANTI_TAMPER_CMD);
        if (this.releaseTrigger.isManual) {
            const publishTask = this.publisher.publishToGit({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
                projectChangelogFile: this.releaseTrigger.changelogPath,
                gitBranch: branchName,
                gitPushCommand: this.releaseTrigger.gitPushCommand,
            });
            releaseTask.spawn(publishTask);
        }
        const postBuildSteps = [...this.postBuildSteps];
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        postBuildSteps.push({
            name: "Check for new commits",
            id: GIT_REMOTE_STEPID,
            run: `echo "${LATEST_COMMIT_OUTPUT}=$(git ls-remote origin -h \${{ github.ref }} | cut -f1)" >> $GITHUB_OUTPUT`,
        });
        postBuildSteps.push({
            name: "Backup artifact permissions",
            if: noNewCommits,
            continueOnError: true,
            run: `cd ${this.artifactsDirectory} && getfacl -R . > ${constants_1.PERMISSION_BACKUP_FILE}`,
        }, {
            name: "Upload artifact",
            if: noNewCommits,
            uses: "actions/upload-artifact@v3",
            with: {
                name: constants_1.BUILD_ARTIFACT_NAME,
                path: this.artifactsDirectory,
            },
        });
        if (this.github && !this.releaseTrigger.isManual) {
            return new github_1.TaskWorkflow(this.github, {
                name: workflowName,
                jobId: BUILD_JOBID,
                outputs: {
                    latest_commit: {
                        stepId: GIT_REMOTE_STEPID,
                        outputName: LATEST_COMMIT_OUTPUT,
                    },
                },
                triggers: {
                    schedule: this.releaseTrigger.schedule
                        ? [{ cron: this.releaseTrigger.schedule }]
                        : undefined,
                    push: this.releaseTrigger.isContinuous
                        ? { branches: [branchName] }
                        : undefined,
                },
                container: this.containerImage
                    ? { image: this.containerImage }
                    : undefined,
                env: {
                    CI: "true",
                },
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                checkoutWith: {
                    // we must use 'fetch-depth=0' in order to fetch all tags
                    // otherwise tags are not checked out
                    "fetch-depth": 0,
                },
                preBuildSteps,
                task: releaseTask,
                postBuildSteps,
                runsOn: this.workflowRunsOn,
            });
        }
        else {
            return undefined;
        }
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.67.2" };
Release.ANTI_TAMPER_CMD = "git diff --ignore-space-at-eol --exit-code";
//# sourceMappingURL=data:application/json;base64,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