"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolvePythonImportName = exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const fs_extra_1 = require("fs-extra");
const common_1 = require("../common");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
/**
 * Allows writing projenrc files in python.
 *
 * This will install `projen` as a Python dependency and will add a
 * `synth` task which will run `.projenrc.py`.
 */
class Projenrc extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        const projenVersion = options.projenVersion ?? common_1.PROJEN_VERSION;
        this.rcfile = options.filename ?? ".projenrc.py";
        project.deps.addDependency(`projen@${projenVersion}`, dependencies_1.DependencyType.DEVENV);
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask?.exec("python .projenrc.py");
        // if this is a new project, generate a skeleton for projenrc.py
        this.generateProjenrc();
    }
    generateProjenrc() {
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const optionsTypeFqn = jsiiType.initializer?.parameters?.[0].type?.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const pythonFile = path_1.join(this.project.outdir, this.rcfile);
        // skip if file exists
        if (fs_extra_1.existsSync(pythonFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        // const openBlock = (line: string = '') => { emit(line + ' {'); indent++; };
        // const closeBlock = () => { indent--; emit('}'); };
        const optionFqns = {};
        for (const option of bootstrap.type.options) {
            if (option.fqn) {
                optionFqns[option.name] = option.fqn;
            }
        }
        const { renderedOptions, imports } = renderPythonOptions(indent, optionFqns, bootstrap.args);
        const importName = resolvePythonImportName(jsiiFqn, jsiiManifest);
        emit(toPythonImport(importName));
        for (const fqn of imports) {
            emit(toPythonImport(fqn));
        }
        emit();
        emit(`project = ${jsiiType.name}(${renderedOptions})`);
        emit();
        emit("project.synth()");
        fs_extra_1.mkdirpSync(path_1.dirname(pythonFile));
        fs_extra_1.outputFileSync(pythonFile, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${pythonFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.python.Projenrc", version: "0.67.2" };
function resolvePythonImportName(jsiiFqn, jsiiManifest) {
    const moduleName = jsiiManifest?.targets?.python?.module;
    // Module name prefix should take precedence in the event moduleName !== fqn prefix
    return moduleName
        ? [moduleName, ...jsiiFqn.split(".").slice(1)].join(".")
        : jsiiFqn;
}
exports.resolvePythonImportName = resolvePythonImportName;
function renderPythonOptions(indent, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        // no options
        return { renderedOptions: "", imports };
    }
    const lines = [""];
    for (const [name, value] of Object.entries(initOptions)) {
        const { pythonValue, importName } = toPythonValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`${toPythonProperty(name)}=${pythonValue},`);
    }
    const renderedOptions = lines
        .join(`\n${" ".repeat((indent + 1) * 4)}`)
        .concat("\n");
    return { renderedOptions, imports };
}
function toPythonProperty(prop) {
    return case_1.snake(prop);
}
function toPythonValue(value, name, optionFqns) {
    if (typeof value === "boolean") {
        return { pythonValue: value ? "True" : "False" };
    }
    else if (typeof value === "number") {
        return { pythonValue: JSON.stringify(value) };
    }
    else if (typeof value === "string") {
        if (optionFqns[name] !== undefined) {
            const parts = optionFqns[name].split(".");
            const base = parts[parts.length - 1];
            const choice = String(value).toUpperCase().replace(/-/g, "_");
            return { pythonValue: `${base}.${choice}`, importName: optionFqns[name] };
        }
        else {
            return { pythonValue: JSON.stringify(value) };
        }
    }
    else if (value === undefined || value === null) {
        return { pythonValue: "None" };
    }
    else {
        return { pythonValue: JSON.stringify(value) };
    }
}
function toPythonImport(fqn) {
    const parts = fqn.split(".");
    if (parts.length === 1) {
        return `import ${parts[0]}`;
    }
    else {
        return `from ${parts.slice(0, -1).join(".")} import ${parts[parts.length - 1]}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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