"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Eslint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const prettier_1 = require("./prettier");
const common_1 = require("../common");
const component_1 = require("../component");
const json_1 = require("../json");
const yaml_1 = require("../yaml");
/**
 * Represents eslint configuration.
 */
class Eslint extends component_1.Component {
    constructor(project, options) {
        super(project);
        this._plugins = new Array();
        this._extends = new Array();
        this.nodeProject = project;
        project.addDevDeps("eslint@^8", "@typescript-eslint/eslint-plugin@^5", "@typescript-eslint/parser@^5", "eslint-import-resolver-node", "eslint-import-resolver-typescript", "eslint-plugin-import", "json-schema");
        if (options.aliasMap) {
            project.addDevDeps("eslint-import-resolver-alias");
        }
        const devdirs = options.devdirs ?? [];
        const dirs = [...options.dirs, ...devdirs];
        const fileExtensions = options.fileExtensions ?? [".ts"];
        this._allowDevDeps = new Set((devdirs ?? []).map((dir) => `**/${dir}/**`));
        const lintProjenRc = options.lintProjenRc ?? true;
        const lintProjenRcFile = options.lintProjenRcFile ?? common_1.PROJEN_RC;
        this.eslintTask = project.addTask("eslint", {
            description: "Runs eslint against the codebase",
            exec: [
                "eslint",
                `--ext ${fileExtensions.join(",")}`,
                "--fix",
                "--no-error-on-unmatched-pattern",
                ...dirs,
                ...(lintProjenRc && lintProjenRcFile ? [lintProjenRcFile] : []),
            ].join(" "),
        });
        project.testTask.spawn(this.eslintTask);
        // exclude some files
        project.npmignore?.exclude("/.eslintrc.json");
        this._formattingRules = {
            // see https://github.com/typescript-eslint/typescript-eslint/blob/master/packages/eslint-plugin/docs/rules/indent.md
            indent: ["off"],
            "@typescript-eslint/indent": ["error", 2],
            // Style
            quotes: ["error", "single", { avoidEscape: true }],
            "comma-dangle": ["error", "always-multiline"],
            "comma-spacing": ["error", { before: false, after: true }],
            "no-multi-spaces": ["error", { ignoreEOLComments: false }],
            "array-bracket-spacing": ["error", "never"],
            "array-bracket-newline": ["error", "consistent"],
            "object-curly-spacing": ["error", "always"],
            "object-curly-newline": ["error", { multiline: true, consistent: true }],
            "object-property-newline": [
                "error",
                { allowAllPropertiesOnSameLine: true },
            ],
            "keyword-spacing": ["error"],
            "brace-style": ["error", "1tbs", { allowSingleLine: true }],
            "space-before-blocks": ["error"],
            curly: ["error", "multi-line", "consistent"],
            "@typescript-eslint/member-delimiter-style": ["error"],
            // Require semicolons
            semi: ["error", "always"],
            // Max line lengths
            "max-len": [
                "error",
                {
                    code: 150,
                    ignoreUrls: true,
                    ignoreStrings: true,
                    ignoreTemplateLiterals: true,
                    ignoreComments: true,
                    ignoreRegExpLiterals: true,
                },
            ],
            // Don't unnecessarily quote properties
            "quote-props": ["error", "consistent-as-needed"],
        };
        this.rules = {
            // Require use of the `import { foo } from 'bar';` form instead of `import foo = require('bar');`
            "@typescript-eslint/no-require-imports": ["error"],
            // Require all imported dependencies are actually declared in package.json
            "import/no-extraneous-dependencies": [
                "error",
                {
                    // Only allow importing devDependencies from "devdirs".
                    devDependencies: () => this.renderDevDepsAllowList(),
                    optionalDependencies: false,
                    peerDependencies: true,
                },
            ],
            // Require all imported libraries actually resolve (!!required for import/no-extraneous-dependencies to work!!)
            "import/no-unresolved": ["error"],
            // Require an ordering on all imports
            "import/order": [
                "warn",
                {
                    groups: ["builtin", "external"],
                    alphabetize: { order: "asc", caseInsensitive: true },
                },
            ],
            // Cannot import from the same module twice
            "no-duplicate-imports": ["error"],
            // Cannot shadow names
            "no-shadow": ["off"],
            "@typescript-eslint/no-shadow": ["error"],
            // Required spacing in property declarations (copied from TSLint, defaults are good)
            "key-spacing": ["error"],
            // No multiple empty lines
            "no-multiple-empty-lines": ["error"],
            // One of the easiest mistakes to make
            "@typescript-eslint/no-floating-promises": ["error"],
            // Make sure that inside try/catch blocks, promises are 'return await'ed
            // (must disable the base rule as it can report incorrect errors)
            "no-return-await": ["off"],
            "@typescript-eslint/return-await": ["error"],
            // Useless diff results
            "no-trailing-spaces": ["error"],
            // Must use foo.bar instead of foo['bar'] if possible
            "dot-notation": ["error"],
            // Are you sure | is not a typo for || ?
            "no-bitwise": ["error"],
            // Member ordering
            "@typescript-eslint/member-ordering": [
                "error",
                {
                    default: [
                        "public-static-field",
                        "public-static-method",
                        "protected-static-field",
                        "protected-static-method",
                        "private-static-field",
                        "private-static-method",
                        "field",
                        // Constructors
                        "constructor",
                        // Methods
                        "method",
                    ],
                },
            ],
        };
        // Overrides for .projenrc.js
        this.overrides = [
            {
                files: [lintProjenRcFile || common_1.PROJEN_RC],
                rules: {
                    "@typescript-eslint/no-require-imports": "off",
                    "import/no-extraneous-dependencies": "off",
                },
            },
        ];
        this.ignorePatterns = options.ignorePatterns ?? [
            "*.js",
            `!${lintProjenRcFile || common_1.PROJEN_RC}`,
            "*.d.ts",
            "node_modules/",
            "*.generated.ts",
            "coverage",
        ];
        const tsconfig = options.tsconfigPath ?? "./tsconfig.json";
        this.addPlugins("@typescript-eslint");
        this.addPlugins("import");
        this.addExtends("plugin:import/typescript");
        this.config = {
            env: {
                jest: true,
                node: true,
            },
            root: true,
            plugins: () => this._plugins,
            parser: "@typescript-eslint/parser",
            parserOptions: {
                ecmaVersion: 2018,
                sourceType: "module",
                project: tsconfig,
            },
            extends: () => this._extends,
            settings: {
                "import/parsers": {
                    "@typescript-eslint/parser": [".ts", ".tsx"],
                },
                "import/resolver": {
                    ...(options.aliasMap && {
                        alias: {
                            map: Object.entries(options.aliasMap).map(([k, v]) => [k, v]),
                            extensions: options.aliasExtensions,
                        },
                    }),
                    node: {},
                    typescript: {
                        project: tsconfig,
                        ...(options.tsAlwaysTryTypes !== false && { alwaysTryTypes: true }),
                    },
                },
            },
            ignorePatterns: this.ignorePatterns,
            rules: () => ({ ...this._formattingRules, ...this.rules }),
            overrides: this.overrides,
        };
        if (options.yaml) {
            new yaml_1.YamlFile(project, ".eslintrc.yml", {
                obj: this.config,
                marker: true,
            });
        }
        else {
            new json_1.JsonFile(project, ".eslintrc.json", {
                obj: this.config,
                // https://eslint.org/docs/latest/user-guide/configuring/configuration-files#comments-in-configuration-files
                marker: true,
                allowComments: true,
            });
        }
        // if the user enabled prettier explicitly _or_ if the project has a
        // `Prettier` component, we shall tweak our configuration accordingly.
        if (options.prettier || prettier_1.Prettier.of(project)) {
            this.enablePrettier();
        }
    }
    /**
     * Returns the singletone Eslint component of a project or undefined if there is none.
     */
    static of(project) {
        const isEslint = (c) => c instanceof Eslint;
        return project.components.find(isEslint);
    }
    /**
     * Add an eslint rule.
     */
    addRules(rules) {
        for (const [k, v] of Object.entries(rules)) {
            this.rules[k] = v;
        }
    }
    /**
     * Adds an eslint plugin
     * @param plugins The names of plugins to add
     */
    addPlugins(...plugins) {
        this._plugins.push(...plugins);
    }
    /**
     * Add an eslint override.
     */
    addOverride(override) {
        this.overrides.push(override);
    }
    /**
     * Do not lint these files.
     */
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    /**
     * Adds an `extends` item to the eslint configuration.
     * @param extendList The list of "extends" to add.
     */
    addExtends(...extendList) {
        this._extends.push(...extendList);
    }
    /**
     * Add a glob file pattern which allows importing dev dependencies.
     * @param pattern glob pattern.
     */
    allowDevDeps(pattern) {
        this._allowDevDeps.add(pattern);
    }
    /**
     * Enables prettier for code formatting.
     */
    enablePrettier() {
        this.nodeProject.addDevDeps("prettier", "eslint-plugin-prettier", "eslint-config-prettier");
        this.addPlugins("prettier");
        this._formattingRules = {
            "prettier/prettier": ["error"],
        };
        this.addExtends("prettier", "plugin:prettier/recommended");
    }
    renderDevDepsAllowList() {
        return Array.from(this._allowDevDeps);
    }
}
exports.Eslint = Eslint;
_a = JSII_RTTI_SYMBOL_1;
Eslint[_a] = { fqn: "projen.javascript.Eslint", version: "0.67.2" };
//# sourceMappingURL=data:application/json;base64,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