"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.getJavaImport = exports.generateJavaOptionNames = exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const common_1 = require("../common");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
/**
 * Allows writing projenrc files in java.
 *
 * This will install `org.projen/projen` as a Maven dependency and will add a
 * `synth` task which will compile & execute `main()` from
 * `src/main/java/projenrc.java`.
 */
class Projenrc extends component_1.Component {
    constructor(project, pom, options = {}) {
        super(project);
        const projenVersion = options.projenVersion ?? common_1.PROJEN_VERSION;
        this.className = options.className ?? "projenrc";
        this.testScope = options.testScope ?? true;
        const depType = this.testScope
            ? dependencies_1.DependencyType.TEST
            : dependencies_1.DependencyType.RUNTIME;
        const execOpts = this.testScope ? ' -Dexec.classpathScope="test"' : "";
        const compileGoal = this.testScope
            ? "compiler:testCompile"
            : "compiler:compile";
        project.deps.addDependency(`io.github.cdklabs/projen@${projenVersion}`, depType);
        pom.addPlugin("org.codehaus.mojo/exec-maven-plugin@3.0.0");
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask?.exec(`mvn ${compileGoal} --quiet`);
        project.defaultTask?.exec(`mvn exec:java --quiet -Dexec.mainClass=${this.className}${execOpts}`);
        // if this is a new project, generate a skeleton for projenrc.java
        this.generateProjenrc();
    }
    generateProjenrc() {
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const javaTarget = jsiiManifest.targets.java;
        const optionsTypeFqn = jsiiType.initializer?.parameters?.[0].type?.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const jsiiOptionsType = jsiiManifest.types[optionsTypeFqn];
        if (!jsiiOptionsType) {
            this.project.logger.warn(`cannot find jsii type for project options: ${optionsTypeFqn}`);
            return;
        }
        const dir = this.testScope ? "src/test/java" : "src/main/java";
        const split = this.className.split(".");
        let javaClass, javaPackage;
        if (split.length === 1) {
            javaClass = split[0];
            javaPackage = [];
        }
        else {
            javaPackage = split.slice(0, split.length - 2);
            javaClass = split[split.length - 1];
        }
        const javaFile = path_1.join(this.project.outdir, dir, ...javaPackage, javaClass + ".java");
        // skip if file exists
        if (fs_extra_1.existsSync(javaFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        const openBlock = (line = "") => {
            emit(line + " {");
            indent++;
        };
        const closeBlock = () => {
            indent--;
            emit("}");
        };
        const optionFqns = generateJavaOptionNames(bootstrap.type.options, jsiiManifest);
        if (javaPackage.length > 0) {
            emit(`package ${javaPackage.join(".")};`);
            emit();
        }
        const { renderedOptions, imports } = renderJavaOptions(2, jsiiOptionsType.name, optionFqns, bootstrap.args);
        emit(`import ${getJavaImport(jsiiType, jsiiManifest)};`);
        emit(`import ${getJavaImport(jsiiOptionsType, jsiiManifest)};`);
        for (const optionTypeName of imports) {
            emit(`import ${javaTarget.package}.${optionTypeName};`);
        }
        emit();
        openBlock(`public class ${javaClass}`);
        openBlock("public static void main(String[] args)");
        emit(`${jsiiType.name} project = new ${jsiiType.name}(${renderedOptions});`);
        emit("project.synth();");
        closeBlock();
        closeBlock();
        fs_extra_1.mkdirpSync(path_1.dirname(javaFile));
        fs_extra_1.outputFileSync(javaFile, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${javaFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.java.Projenrc", version: "0.67.2" };
function generateJavaOptionNames(options, jsiiManifest) {
    const optionFqns = {};
    for (const option of options) {
        if (option.fqn && jsiiManifest.types[option.fqn]) {
            optionFqns[option.name] = toJavaFullTypeName(jsiiManifest.types[option.fqn]);
        }
    }
    return optionFqns;
}
exports.generateJavaOptionNames = generateJavaOptionNames;
function renderJavaOptions(indent, optionsTypeName, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        return { renderedOptions: "", imports }; // no options
    }
    const lines = [`${optionsTypeName}.builder()`];
    for (const [name, value] of Object.entries(initOptions)) {
        const { javaValue, importName } = toJavaValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`.${toJavaProperty(name)}(${javaValue})`);
    }
    lines.push(".build()");
    const renderedOptions = lines.join(`\n${" ".repeat((indent + 1) * 4)}`);
    return { renderedOptions, imports };
}
function toJavaProperty(prop) {
    return prop;
}
function toJavaValue(value, name, optionFqns) {
    if (typeof value === "string" && optionFqns[name] !== undefined) {
        const parts = optionFqns[name].split(".");
        const base = parts[parts.length - 1];
        const choice = String(value).toUpperCase().replace(/-/g, "_");
        return { javaValue: `${base}.${choice}`, importName: optionFqns[name] };
    }
    else {
        return { javaValue: JSON.stringify(value) };
    }
}
function toJavaFullTypeName(jsiiType) {
    return [jsiiType.namespace, jsiiType.name].filter((x) => x).join(".");
}
function getJavaImport(jsiiType, jsiiManifest) {
    const packageName = jsiiManifest?.submodules?.[`${jsiiType.assembly}.${jsiiType?.namespace}`]
        ?.targets?.java?.package ||
        [jsiiManifest.targets.java.package, jsiiType.namespace]
            .filter((x) => x)
            .join(".");
    return `${packageName}.${jsiiType.name}`;
}
exports.getJavaImport = getJavaImport;
//# sourceMappingURL=data:application/json;base64,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