"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutoMerge = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
/**
 * Sets up mergify to merging approved pull requests.
 *
 * If `buildJob` is specified, the specified GitHub workflow job ID is required
 * to succeed in order for the PR to be merged.
 *
 * `approvedReviews` specified the number of code review approvals required for
 * the PR to be merged.
 */
class AutoMerge extends component_1.Component {
    constructor(github, options = {}) {
        super(github.project);
        this.lazyConditions = new Array();
        const mergify = github.mergify;
        if (!mergify) {
            throw new Error("auto merging requires mergify to be enabled");
        }
        const blockingLabels = options.blockingLabels ?? ["do-not-merge"];
        const blockingCondition = blockingLabels?.length
            ? [`-label~=(${blockingLabels.join("|")})`]
            : [];
        const mergeAction = {
            delete_head_branch: {},
            queue: {
                // squash all commits into a single commit when merging
                // method: "squash",
                method: "squash",
                name: "default",
                // use PR title+body as the commit message
                commit_message_template: [
                    "{{ title }} (#{{ number }})",
                    "",
                    "{{ body }}",
                ].join("\n"),
            },
        };
        const approvedReviews = options.approvedReviews ?? 1;
        // add initial conditions (additional conditions can be added later)
        this.addConditions(`#approved-reviews-by>=${approvedReviews}`);
        this.addConditions(...blockingCondition);
        mergify.addRule({
            name: "Automatic merge on approval and successful build",
            actions: mergeAction,
            conditions: (() => this.renderConditions()),
        });
        mergify.addQueue({
            name: "default",
            conditions: (() => this.renderConditions()),
        });
        this.project.addPackageIgnore("/.mergify.yml");
    }
    /**
     * Adds conditions to the auto merge rule.
     * @param conditions The conditions to add (mergify syntax)
     */
    addConditions(...conditions) {
        this.addConditionsLater({ render: () => conditions });
    }
    /**
     * Adds conditions that will be rendered only during synthesis.
     * @param later The later
     */
    addConditionsLater(later) {
        this.lazyConditions.push(later);
    }
    renderConditions() {
        const output = new Array();
        for (const later of this.lazyConditions) {
            output.push(...later.render());
        }
        return output;
    }
}
exports.AutoMerge = AutoMerge;
_a = JSII_RTTI_SYMBOL_1;
AutoMerge[_a] = { fqn: "projen.github.AutoMerge", version: "0.67.2" };
//# sourceMappingURL=data:application/json;base64,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