"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const fs_extra_1 = require("fs-extra");
const _resolve_1 = require("./_resolve");
const common_1 = require("./common");
const component_1 = require("./component");
const util_1 = require("./util");
class FileBase extends component_1.Component {
    constructor(project, filePath, options = {}) {
        super(project);
        this.readonly = !project.ejected && (options.readonly ?? true);
        this.executable = options.executable ?? false;
        this.path = filePath;
        // `marker` is empty if project is being ejected or if explicitly disabled
        this.marker =
            project.ejected || options.marker === false
                ? undefined
                : `${common_1.PROJEN_MARKER}. To modify, edit ${common_1.PROJEN_RC} and run "npx projen".`;
        const globPattern = `/${this.path}`;
        const committed = options.committed ?? project.commitGenerated ?? true;
        if (committed && filePath !== ".gitattributes") {
            project.root.annotateGenerated(`/${filePath}`);
        }
        this.absolutePath = path.resolve(project.outdir, filePath);
        // verify file path is unique within project tree
        const existing = project.root.tryFindFile(this.absolutePath);
        if (existing && existing !== this) {
            throw new Error(`there is already a file under ${path.relative(project.root.outdir, this.absolutePath)}`);
        }
        const editGitignore = options.editGitignore ?? true;
        if (editGitignore) {
            this.project.addGitIgnore(`${committed ? "!" : ""}${globPattern}`);
        }
        else {
            if (options.committed != null) {
                throw new Error('"gitignore" is disabled, so it does not make sense to specify "committed"');
            }
        }
    }
    /**
     * Writes the file to the project's output directory
     */
    synthesize() {
        const outdir = this.project.outdir;
        const filePath = path.join(outdir, this.path);
        const resolver = {
            resolve: (obj, options) => _resolve_1.resolve(obj, options),
        };
        const content = this.synthesizeContent(resolver);
        if (content === undefined) {
            // remove file (if exists) and skip rest of synthesis
            fs_extra_1.removeSync(filePath);
            return;
        }
        // check if the file was changed.
        const prev = util_1.tryReadFileSync(filePath);
        const prevReadonly = !util_1.isWritable(filePath);
        const prevExecutable = util_1.isExecutable(filePath);
        if (prev !== undefined &&
            content === prev &&
            prevReadonly === this.readonly &&
            prevExecutable === this.executable) {
            this.project.logger.debug(`no change in ${filePath}`);
            this._changed = false;
            return;
        }
        util_1.writeFile(filePath, content, {
            readonly: this.readonly,
            executable: this.executable,
        });
        this.checkForProjenMarker();
        this._changed = true;
    }
    /**
     * For debugging, check whether a file was incorrectly generated with
     * or without the projen marker. The projen marker does not *need* to be
     * included on projen-generated files, but it's recommended since it signals
     * that it probably should not be edited directly.
     */
    checkForProjenMarker() {
        const filePath = path.join(this.project.outdir, this.path);
        const contents = util_1.tryReadFileSync(filePath);
        const containsMarker = contents?.includes(common_1.PROJEN_MARKER);
        if (this.marker && !containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to contain marker but found none.`);
        }
        else if (!this.marker && containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to not contain marker but found one anyway.`);
        }
    }
    /**
     * Indicates if the file has been changed during synthesis. This property is
     * only available in `postSynthesize()` hooks. If this is `undefined`, the
     * file has not been synthesized yet.
     */
    get changed() {
        return this._changed;
    }
}
exports.FileBase = FileBase;
_a = JSII_RTTI_SYMBOL_1;
FileBase[_a] = { fqn: "projen.FileBase", version: "0.67.2" };
//# sourceMappingURL=data:application/json;base64,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