"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.discoverTaskCommands = void 0;
const chalk = require("chalk");
const helpers_1 = require("yargs/helpers");
const logging = require("../logging");
/**
 * Reads .projen/tasks.json and adds CLI commands for all tasks.
 * @param ya yargs
 */
function discoverTaskCommands(runtime, ya) {
    const tasks = runtime.manifest.tasks ?? {};
    for (const task of Object.values(tasks)) {
        ya.command(task.name, task.description ?? "", taskCommandHandler(task));
    }
    function taskCommandHandler(task) {
        return (args) => {
            const taskReceivesArgs = task.steps?.reduce((receiveArgs, step) => receiveArgs || (step.receiveArgs ?? false), false);
            let taskArgs = [];
            if (taskReceivesArgs) {
                args.strict(false);
                args.strictCommands(false);
                taskArgs = helpers_1.hideBin(process.argv).slice(1);
            }
            args.option("inspect", {
                alias: "i",
                desc: "show all steps in this task",
            });
            const argv = args.argv;
            if (argv.inspect) {
                return inspectTask(task.name);
            }
            else {
                try {
                    runtime.runTask(task.name, [], taskArgs);
                }
                catch (e) {
                    logging.error(e.message);
                    process.exit(1);
                }
            }
        };
    }
    function inspectTask(name, indent = 0) {
        const writeln = (s) => console.log(" ".repeat(indent) + s);
        const task = runtime.tryFindTask(name);
        if (!task) {
            throw new Error(`${name}: unable to resolve subtask with name "${name}"`);
        }
        if (task.description) {
            writeln(`${chalk.underline("description")}: ${task.description}`);
        }
        for (const [k, v] of Object.entries(task.env ?? {})) {
            writeln(`${chalk.underline("env")}: ${k}=${v}`);
        }
        for (const step of task.steps ?? []) {
            if (step.spawn) {
                writeln(`- ${chalk.bold(step.spawn)}`);
                inspectTask(step.spawn, indent + 2);
            }
            else if (step.exec) {
                writeln(`- exec: ${step.exec}`);
            }
            else if (step.builtin) {
                writeln(`- builtin: ${step.builtin}`);
            }
        }
    }
}
exports.discoverTaskCommands = discoverTaskCommands;
//# sourceMappingURL=data:application/json;base64,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