"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const yargs = require("yargs");
const synth_1 = require("./synth");
const tasks_1 = require("./tasks");
const common_1 = require("../common");
const logging = require("../logging");
const task_runtime_1 = require("../task-runtime");
const util_1 = require("../util");
const DEFAULT_RC = path_1.resolve(common_1.PROJEN_RC);
async function main() {
    const ya = yargs;
    ya.commandDir("cmds");
    const runtime = new task_runtime_1.TaskRuntime(".");
    tasks_1.discoverTaskCommands(runtime, ya);
    ya.recommendCommands();
    ya.strictCommands();
    ya.showHelpOnFail(false);
    ya.wrap(yargs.terminalWidth());
    ya.option("post", {
        type: "boolean",
        default: true,
        desc: "Run post-synthesis steps such as installing dependencies. Use --no-post to skip",
    });
    ya.option("watch", {
        type: "boolean",
        default: false,
        desc: "Keep running and resynthesize when projenrc changes",
        alias: "w",
    });
    ya.options("debug", { type: "boolean", default: false, desc: "Debug logs" });
    ya.options("rc", {
        desc: "path to .projenrc.js file",
        default: DEFAULT_RC,
        type: "string",
    });
    ya.completion();
    ya.help();
    // do not use the default yargs '--version' implementation since it is
    // global by default (it appears on all subcommands)
    ya.version(false);
    ya.option("version", {
        type: "boolean",
        description: "Show version number",
        global: false,
    });
    const args = ya.argv;
    if (args.debug) {
        process.env.DEBUG = "true";
    }
    const nodeVersion = util_1.getNodeMajorVersion();
    if (nodeVersion && nodeVersion < 14) {
        logging.warn(`WARNING: You are using Node v${nodeVersion}, which reaches end of life on April 30, 2022. Support for EOL Node releases may be dropped by projen in the future. Please consider upgrading to Node >= 14 as soon as possible.`);
    }
    // no command means synthesize
    if (args._.length === 0) {
        if (args.version) {
            console.log(common_1.PROJEN_VERSION);
            process.exit(0);
        }
        await synth_1.synth(runtime, {
            post: args.post,
            watch: args.watch,
            rcfile: args.rc,
        });
    }
}
main().catch((e) => {
    console.error(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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