"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaExtension = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const internal_1 = require("./internal");
const lambda_function_1 = require("./lambda-function");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
/**
 * Create a Lambda Extension
 */
class LambdaExtension extends component_1.Component {
    constructor(project, options) {
        super(project);
        const basePath = path_1.join(path_1.dirname(options.entrypoint), path_1.basename(options.entrypoint, internal_1.TYPESCRIPT_LAMBDA_EXTENSION_EXT));
        const name = options.name ?? path_1.basename(basePath);
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const compatibleRuntimes = options.compatibleRuntimes ?? [
            lambda_function_1.LambdaRuntime.NODEJS_18_X,
            lambda_function_1.LambdaRuntime.NODEJS_16_X,
            lambda_function_1.LambdaRuntime.NODEJS_14_X,
            lambda_function_1.LambdaRuntime.NODEJS_12_X,
        ];
        if (compatibleRuntimes.length === 0) {
            throw new Error("Compatible runtimes must include at least one runtime");
        }
        // Use the lowest runtime version to bundle
        const [bundlerRuntime] = compatibleRuntimes.sort((a, b) => a.functionRuntime.localeCompare(b.functionRuntime));
        // Allow extension code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(options.entrypoint);
        const bundle = bundler.addBundle(options.entrypoint, {
            platform: bundlerRuntime.esbuildPlatform,
            target: bundlerRuntime.esbuildTarget,
            externals: bundlerRuntime.defaultExternals,
            outfile: `extensions/${name}`,
            // Make the output executable because Lambda expects to run
            // extensions as stand-alone programs alongside the main lambda
            // process.
            executable: true,
            ...options.bundlingOptions,
        });
        const constructFile = options.constructFile ?? `${basePath}-layer-version.ts`;
        new LambdaLayerConstruct(project, {
            constructFile: constructFile,
            constructName: options.constructName,
            assetDir: bundle.outdir,
            compatibleRuntimes: compatibleRuntimes,
            description: `Provides a Lambda Extension \`${name}\` from ${internal_1.convertToPosixPath(options.entrypoint)}`,
            cdkDeps: options.cdkDeps,
        });
    }
}
exports.LambdaExtension = LambdaExtension;
_a = JSII_RTTI_SYMBOL_1;
LambdaExtension[_a] = { fqn: "projen.awscdk.LambdaExtension", version: "0.67.2" };
class LambdaLayerConstruct extends source_code_1.SourceCode {
    constructor(project, options) {
        super(project, options.constructFile);
        const src = this;
        const cdkDeps = options.cdkDeps;
        const constructName = options.constructName ?? case_1.pascal(path_1.basename(options.constructFile, ".ts"));
        const propsType = `${constructName}Props`;
        const assetDir = path_1.relative(path_1.dirname(options.constructFile), options.assetDir);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        src.open(`export interface ${propsType} extends lambda.LayerVersionOptions {`);
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * ${options.description}`);
        src.line(" */");
        src.open(`export class ${constructName} extends lambda.LayerVersion {`);
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: ${encodeCodeString(options.description)},`);
        src.line("...props,");
        src.open("compatibleRuntimes: [");
        for (const runtime of options.compatibleRuntimes) {
            src.line(`new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        }
        src.close("],");
        src.open(`code: lambda.Code.fromAsset(path.join(__dirname,`);
        src.line(`${encodeCodeString(internal_1.convertToPosixPath(assetDir))})),`);
        src.close();
        src.close("});");
        src.close("}");
        src.close("}");
    }
}
/**
 * Encodes a string for embedding in source code.
 */
function encodeCodeString(value) {
    const json = JSON.stringify(value);
    const escapedString = json.substring(1, json.length - 1).replace(/'/g, "\\'");
    return `'${escapedString}'`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGFtYmRhLWV4dGVuc2lvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9hd3NjZGsvbGFtYmRhLWV4dGVuc2lvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLCtCQUF5RDtBQUN6RCwrQkFBOEI7QUFFOUIseUNBR29CO0FBQ3BCLHVEQUFrRDtBQUNsRCw0Q0FBeUM7QUFDekMsOENBQWlFO0FBRWpFLGdEQUE0QztBQWtFNUM7O0dBRUc7QUFDSCxNQUFhLGVBQWdCLFNBQVEscUJBQVM7SUFDNUMsWUFBWSxPQUFnQixFQUFFLE9BQStCO1FBQzNELEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVmLE1BQU0sUUFBUSxHQUFHLFdBQUksQ0FDbkIsY0FBTyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsRUFDM0IsZUFBUSxDQUFDLE9BQU8sQ0FBQyxVQUFVLEVBQUUsMENBQStCLENBQUMsQ0FDOUQsQ0FBQztRQUVGLE1BQU0sSUFBSSxHQUFHLE9BQU8sQ0FBQyxJQUFJLElBQUksZUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRWhELE1BQU0sT0FBTyxHQUFHLG9CQUFPLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDWixNQUFNLElBQUksS0FBSyxDQUNiLG1FQUFtRSxDQUNwRSxDQUFDO1NBQ0g7UUFFRCxNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsSUFBSTtZQUN2RCwrQkFBYSxDQUFDLFdBQVc7WUFDekIsK0JBQWEsQ0FBQyxXQUFXO1lBQ3pCLCtCQUFhLENBQUMsV0FBVztZQUN6QiwrQkFBYSxDQUFDLFdBQVc7U0FDMUIsQ0FBQztRQUVGLElBQUksa0JBQWtCLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLHVEQUF1RCxDQUFDLENBQUM7U0FDMUU7UUFFRCwyQ0FBMkM7UUFDM0MsTUFBTSxDQUFDLGNBQWMsQ0FBQyxHQUFHLGtCQUFrQixDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUN4RCxDQUFDLENBQUMsZUFBZSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDLENBQ25ELENBQUM7UUFFRixxRUFBcUU7UUFDckUsa0JBQWtCO1FBQ2xCLE1BQU0sTUFBTSxHQUFHLG1CQUFNLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ2xDLE1BQU0sRUFBRSxZQUFZLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRXpDLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLFVBQVUsRUFBRTtZQUNuRCxRQUFRLEVBQUUsY0FBYyxDQUFDLGVBQWU7WUFDeEMsTUFBTSxFQUFFLGNBQWMsQ0FBQyxhQUFhO1lBQ3BDLFNBQVMsRUFBRSxjQUFjLENBQUMsZ0JBQWdCO1lBQzFDLE9BQU8sRUFBRSxjQUFjLElBQUksRUFBRTtZQUM3QiwyREFBMkQ7WUFDM0QsK0RBQStEO1lBQy9ELFdBQVc7WUFDWCxVQUFVLEVBQUUsSUFBSTtZQUNoQixHQUFHLE9BQU8sQ0FBQyxlQUFlO1NBQzNCLENBQUMsQ0FBQztRQUVILE1BQU0sYUFBYSxHQUNqQixPQUFPLENBQUMsYUFBYSxJQUFJLEdBQUcsUUFBUSxtQkFBbUIsQ0FBQztRQUUxRCxJQUFJLG9CQUFvQixDQUFDLE9BQU8sRUFBRTtZQUNoQyxhQUFhLEVBQUUsYUFBYTtZQUM1QixhQUFhLEVBQUUsT0FBTyxDQUFDLGFBQWE7WUFDcEMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxNQUFNO1lBQ3ZCLGtCQUFrQixFQUFFLGtCQUFrQjtZQUN0QyxXQUFXLEVBQUUsaUNBQWlDLElBQUksV0FBVyw2QkFBa0IsQ0FDN0UsT0FBTyxDQUFDLFVBQVUsQ0FDbkIsRUFBRTtZQUNILE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTztTQUN6QixDQUFDLENBQUM7SUFDTCxDQUFDOztBQWhFSCwwQ0FpRUM7OztBQVdELE1BQU0sb0JBQXFCLFNBQVEsd0JBQVU7SUFDM0MsWUFBWSxPQUFnQixFQUFFLE9BQW9DO1FBQ2hFLEtBQUssQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRXRDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQztRQUNqQixNQUFNLE9BQU8sR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDO1FBRWhDLE1BQU0sYUFBYSxHQUNqQixPQUFPLENBQUMsYUFBYSxJQUFJLGFBQU0sQ0FBQyxlQUFRLENBQUMsT0FBTyxDQUFDLGFBQWEsRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQzFFLE1BQU0sU0FBUyxHQUFHLEdBQUcsYUFBYSxPQUFPLENBQUM7UUFFMUMsTUFBTSxRQUFRLEdBQUcsZUFBUSxDQUFDLGNBQU8sQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLEVBQUUsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRTVFLElBQUksR0FBRyxDQUFDLE1BQU0sRUFBRTtZQUNkLEdBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSxHQUFHLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztTQUM5QjtRQUNELEdBQUcsQ0FBQyxJQUFJLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUUxQyxJQUFJLE9BQU8sQ0FBQyxlQUFlLEtBQUssQ0FBQyxFQUFFO1lBQ2pDLEdBQUcsQ0FBQyxJQUFJLENBQUMsZ0RBQWdELENBQUMsQ0FBQztZQUMzRCxHQUFHLENBQUMsSUFBSSxDQUFDLDRDQUE0QyxDQUFDLENBQUM7WUFDdkQsT0FBTyxDQUFDLGlCQUFpQixDQUFDLHFCQUFxQixDQUFDLENBQUM7WUFDakQsT0FBTyxDQUFDLGlCQUFpQixDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQzVDO2FBQU07WUFDTCxHQUFHLENBQUMsSUFBSSxDQUFDLG1EQUFtRCxDQUFDLENBQUM7WUFDOUQsR0FBRyxDQUFDLElBQUksQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsR0FBRyxDQUFDLElBQUksRUFBRSxDQUFDO1FBRVgsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNoQixHQUFHLENBQUMsSUFBSSxDQUFDLGdCQUFnQixhQUFhLEVBQUUsQ0FBQyxDQUFDO1FBQzFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDaEIsR0FBRyxDQUFDLElBQUksQ0FDTixvQkFBb0IsU0FBUyx1Q0FBdUMsQ0FDckUsQ0FBQztRQUNGLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDZixHQUFHLENBQUMsSUFBSSxFQUFFLENBQUM7UUFFWCxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2hCLEdBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSxPQUFPLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQztRQUN0QyxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2hCLEdBQUcsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLGFBQWEsZ0NBQWdDLENBQUMsQ0FBQztRQUN4RSxHQUFHLENBQUMsSUFBSSxDQUNOLHFEQUFxRCxTQUFTLEtBQUssQ0FDcEUsQ0FBQztRQUVGLEdBQUcsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUMvQixHQUFHLENBQUMsSUFBSSxDQUFDLGdCQUFnQixnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ25FLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFFdEIsR0FBRyxDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2xDLEtBQUssTUFBTSxPQUFPLElBQUksT0FBTyxDQUFDLGtCQUFrQixFQUFFO1lBQ2hELEdBQUcsQ0FBQyxJQUFJLENBQ04sdUJBQXVCLE9BQU8sQ0FBQyxlQUFlLGtDQUFrQyxDQUNqRixDQUFDO1NBQ0g7UUFDRCxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRWhCLEdBQUcsQ0FBQyxJQUFJLENBQUMsa0RBQWtELENBQUMsQ0FBQztRQUM3RCxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsZ0JBQWdCLENBQUMsNkJBQWtCLENBQUMsUUFBUSxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDakUsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQ1osR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUVqQixHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2YsR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNqQixDQUFDO0NBQ0Y7QUFFRDs7R0FFRztBQUNILFNBQVMsZ0JBQWdCLENBQUMsS0FBYTtJQUNyQyxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ25DLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM5RSxPQUFPLElBQUksYUFBYSxHQUFHLENBQUM7QUFDOUIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGJhc2VuYW1lLCBkaXJuYW1lLCBqb2luLCByZWxhdGl2ZSB9IGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBwYXNjYWwgfSBmcm9tIFwiY2FzZVwiO1xuaW1wb3J0IHsgQXdzQ2RrRGVwcyB9IGZyb20gXCIuL2F3c2Nkay1kZXBzXCI7XG5pbXBvcnQge1xuICBjb252ZXJ0VG9Qb3NpeFBhdGgsXG4gIFRZUEVTQ1JJUFRfTEFNQkRBX0VYVEVOU0lPTl9FWFQsXG59IGZyb20gXCIuL2ludGVybmFsXCI7XG5pbXBvcnQgeyBMYW1iZGFSdW50aW1lIH0gZnJvbSBcIi4vbGFtYmRhLWZ1bmN0aW9uXCI7XG5pbXBvcnQgeyBDb21wb25lbnQgfSBmcm9tIFwiLi4vY29tcG9uZW50XCI7XG5pbXBvcnQgeyBCdW5kbGVyLCBCdW5kbGluZ09wdGlvbnMsIEVzbGludCB9IGZyb20gXCIuLi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQgeyBQcm9qZWN0IH0gZnJvbSBcIi4uL3Byb2plY3RcIjtcbmltcG9ydCB7IFNvdXJjZUNvZGUgfSBmcm9tIFwiLi4vc291cmNlLWNvZGVcIjtcblxuLyoqXG4gKiBDb21tb24gb3B0aW9ucyBmb3IgY3JlYXRpbmcgbGFtYmRhIGV4dGVuc2lvbnMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGFtYmRhRXh0ZW5zaW9uQ29tbW9uT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgZXh0ZW5zaW9uJ3MgY29tcGF0aWJsZSBydW50aW1lcy5cbiAgICovXG4gIHJlYWRvbmx5IGNvbXBhdGlibGVSdW50aW1lcz86IExhbWJkYVJ1bnRpbWVbXTtcblxuICAvKipcbiAgICogQnVuZGxpbmcgb3B0aW9ucyBmb3IgdGhpcyBBV1MgTGFtYmRhIGV4dGVuc2lvbi5cbiAgICpcbiAgICogSWYgbm90IHNwZWNpZmllZCB0aGUgZGVmYXVsdCBidW5kbGluZyBvcHRpb25zIHNwZWNpZmllZCBmb3IgdGhlIHByb2plY3RcbiAgICogYEJ1bmRsZXJgIGluc3RhbmNlIHdpbGwgYmUgdXNlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0c1xuICAgKi9cbiAgcmVhZG9ubHkgYnVuZGxpbmdPcHRpb25zPzogQnVuZGxpbmdPcHRpb25zO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGNyZWF0aW5nIGxhbWJkYSBleHRlbnNpb25zLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExhbWJkYUV4dGVuc2lvbk9wdGlvbnMgZXh0ZW5kcyBMYW1iZGFFeHRlbnNpb25Db21tb25PcHRpb25zIHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGV4dGVuc2lvblxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlcml2ZWQgZnJvbSB0aGUgZW50cnlwb2ludCBmaWxlbmFtZS5cbiAgICovXG4gIHJlYWRvbmx5IG5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEEgcGF0aCBmcm9tIHRoZSBwcm9qZWN0IHJvb3QgZGlyZWN0b3J5IHRvIGEgVHlwZVNjcmlwdCBmaWxlIHdoaWNoIGNvbnRhaW5zXG4gICAqIHRoZSBBV1MgTGFtYmRhIGV4dGVuc2lvbiBlbnRyeXBvaW50IChzdGFuZC1hbG9uZSBzY3JpcHQpLlxuICAgKlxuICAgKiBUaGlzIGlzIHJlbGF0aXZlIHRvIHRoZSByb290IGRpcmVjdG9yeSBvZiB0aGUgcHJvamVjdC5cbiAgICpcbiAgICogQGV4YW1wbGUgXCJzcmMvc3ViZGlyL2Zvby5sYW1iZGEtZXh0ZW5zaW9uLnRzXCJcbiAgICovXG4gIHJlYWRvbmx5IGVudHJ5cG9pbnQ6IHN0cmluZztcblxuICAvKipcbiAgICogQVdTIENESyBkZXBlbmRlbmN5IG1hbmFnZXIuXG4gICAqL1xuICByZWFkb25seSBjZGtEZXBzOiBBd3NDZGtEZXBzO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgZ2VuZXJhdGVkIFR5cGVTY3JpcHQgc291cmNlIGZpbGUuIFRoaXMgZmlsZSBzaG91bGQgYWxzbyBiZVxuICAgKiB1bmRlciB0aGUgc291cmNlIHRyZWUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVGhlIG5hbWUgb2YgdGhlIGVudHJ5cG9pbnQgZmlsZSwgd2l0aCB0aGUgYC1sYXllci12ZXJzaW9uLnRzYFxuICAgKiBzdWZmaXggaW5zdGVhZCBvZiBgLmxhbWJkYS1leHRlbnNpb24udHNgLlxuICAgKi9cbiAgcmVhZG9ubHkgY29uc3RydWN0RmlsZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGdlbmVyYXRlZCBgbGFtYmRhLkxheWVyVmVyc2lvbmAgc3ViY2xhc3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQSBwYXNjYWwgY2FzZWQgdmVyc2lvbiBvZiB0aGUgbmFtZSBvZiB0aGUgZW50cnlwb2ludCBmaWxlLCB3aXRoXG4gICAqIHRoZSBleHRlbnNpb24gYExheWVyVmVyc2lvbmAgKGUuZy4gYEFwcENvbmZpZ0xheWVyVmVyc2lvbmApLlxuICAgKi9cbiAgcmVhZG9ubHkgY29uc3RydWN0TmFtZT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBDcmVhdGUgYSBMYW1iZGEgRXh0ZW5zaW9uXG4gKi9cbmV4cG9ydCBjbGFzcyBMYW1iZGFFeHRlbnNpb24gZXh0ZW5kcyBDb21wb25lbnQge1xuICBjb25zdHJ1Y3Rvcihwcm9qZWN0OiBQcm9qZWN0LCBvcHRpb25zOiBMYW1iZGFFeHRlbnNpb25PcHRpb25zKSB7XG4gICAgc3VwZXIocHJvamVjdCk7XG5cbiAgICBjb25zdCBiYXNlUGF0aCA9IGpvaW4oXG4gICAgICBkaXJuYW1lKG9wdGlvbnMuZW50cnlwb2ludCksXG4gICAgICBiYXNlbmFtZShvcHRpb25zLmVudHJ5cG9pbnQsIFRZUEVTQ1JJUFRfTEFNQkRBX0VYVEVOU0lPTl9FWFQpXG4gICAgKTtcblxuICAgIGNvbnN0IG5hbWUgPSBvcHRpb25zLm5hbWUgPz8gYmFzZW5hbWUoYmFzZVBhdGgpO1xuXG4gICAgY29uc3QgYnVuZGxlciA9IEJ1bmRsZXIub2YocHJvamVjdCk7XG4gICAgaWYgKCFidW5kbGVyKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiTm8gYnVuZGxlciBmb3VuZC4gUGxlYXNlIGFkZCBhIEJ1bmRsZXIgY29tcG9uZW50IHRvIHlvdXIgcHJvamVjdC5cIlxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCBjb21wYXRpYmxlUnVudGltZXMgPSBvcHRpb25zLmNvbXBhdGlibGVSdW50aW1lcyA/PyBbXG4gICAgICBMYW1iZGFSdW50aW1lLk5PREVKU18xOF9YLFxuICAgICAgTGFtYmRhUnVudGltZS5OT0RFSlNfMTZfWCxcbiAgICAgIExhbWJkYVJ1bnRpbWUuTk9ERUpTXzE0X1gsXG4gICAgICBMYW1iZGFSdW50aW1lLk5PREVKU18xMl9YLFxuICAgIF07XG5cbiAgICBpZiAoY29tcGF0aWJsZVJ1bnRpbWVzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiQ29tcGF0aWJsZSBydW50aW1lcyBtdXN0IGluY2x1ZGUgYXQgbGVhc3Qgb25lIHJ1bnRpbWVcIik7XG4gICAgfVxuXG4gICAgLy8gVXNlIHRoZSBsb3dlc3QgcnVudGltZSB2ZXJzaW9uIHRvIGJ1bmRsZVxuICAgIGNvbnN0IFtidW5kbGVyUnVudGltZV0gPSBjb21wYXRpYmxlUnVudGltZXMuc29ydCgoYSwgYikgPT5cbiAgICAgIGEuZnVuY3Rpb25SdW50aW1lLmxvY2FsZUNvbXBhcmUoYi5mdW5jdGlvblJ1bnRpbWUpXG4gICAgKTtcblxuICAgIC8vIEFsbG93IGV4dGVuc2lvbiBjb2RlIHRvIGltcG9ydCBkZXYtZGVwcyBzaW5jZSB0aGV5IGFyZSBvbmx5IG5lZWRlZFxuICAgIC8vIGR1cmluZyBidW5kbGluZ1xuICAgIGNvbnN0IGVzbGludCA9IEVzbGludC5vZihwcm9qZWN0KTtcbiAgICBlc2xpbnQ/LmFsbG93RGV2RGVwcyhvcHRpb25zLmVudHJ5cG9pbnQpO1xuXG4gICAgY29uc3QgYnVuZGxlID0gYnVuZGxlci5hZGRCdW5kbGUob3B0aW9ucy5lbnRyeXBvaW50LCB7XG4gICAgICBwbGF0Zm9ybTogYnVuZGxlclJ1bnRpbWUuZXNidWlsZFBsYXRmb3JtLFxuICAgICAgdGFyZ2V0OiBidW5kbGVyUnVudGltZS5lc2J1aWxkVGFyZ2V0LFxuICAgICAgZXh0ZXJuYWxzOiBidW5kbGVyUnVudGltZS5kZWZhdWx0RXh0ZXJuYWxzLFxuICAgICAgb3V0ZmlsZTogYGV4dGVuc2lvbnMvJHtuYW1lfWAsXG4gICAgICAvLyBNYWtlIHRoZSBvdXRwdXQgZXhlY3V0YWJsZSBiZWNhdXNlIExhbWJkYSBleHBlY3RzIHRvIHJ1blxuICAgICAgLy8gZXh0ZW5zaW9ucyBhcyBzdGFuZC1hbG9uZSBwcm9ncmFtcyBhbG9uZ3NpZGUgdGhlIG1haW4gbGFtYmRhXG4gICAgICAvLyBwcm9jZXNzLlxuICAgICAgZXhlY3V0YWJsZTogdHJ1ZSxcbiAgICAgIC4uLm9wdGlvbnMuYnVuZGxpbmdPcHRpb25zLFxuICAgIH0pO1xuXG4gICAgY29uc3QgY29uc3RydWN0RmlsZSA9XG4gICAgICBvcHRpb25zLmNvbnN0cnVjdEZpbGUgPz8gYCR7YmFzZVBhdGh9LWxheWVyLXZlcnNpb24udHNgO1xuXG4gICAgbmV3IExhbWJkYUxheWVyQ29uc3RydWN0KHByb2plY3QsIHtcbiAgICAgIGNvbnN0cnVjdEZpbGU6IGNvbnN0cnVjdEZpbGUsXG4gICAgICBjb25zdHJ1Y3ROYW1lOiBvcHRpb25zLmNvbnN0cnVjdE5hbWUsXG4gICAgICBhc3NldERpcjogYnVuZGxlLm91dGRpcixcbiAgICAgIGNvbXBhdGlibGVSdW50aW1lczogY29tcGF0aWJsZVJ1bnRpbWVzLFxuICAgICAgZGVzY3JpcHRpb246IGBQcm92aWRlcyBhIExhbWJkYSBFeHRlbnNpb24gXFxgJHtuYW1lfVxcYCBmcm9tICR7Y29udmVydFRvUG9zaXhQYXRoKFxuICAgICAgICBvcHRpb25zLmVudHJ5cG9pbnRcbiAgICAgICl9YCxcbiAgICAgIGNka0RlcHM6IG9wdGlvbnMuY2RrRGVwcyxcbiAgICB9KTtcbiAgfVxufVxuXG5pbnRlcmZhY2UgTGFtYmRhTGF5ZXJDb25zdHJ1Y3RPcHRpb25zIHtcbiAgcmVhZG9ubHkgYXNzZXREaXI6IHN0cmluZztcbiAgcmVhZG9ubHkgY29tcGF0aWJsZVJ1bnRpbWVzOiBMYW1iZGFSdW50aW1lW107XG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGNvbnN0cnVjdEZpbGU6IHN0cmluZztcbiAgcmVhZG9ubHkgY29uc3RydWN0TmFtZT86IHN0cmluZztcbiAgcmVhZG9ubHkgY2RrRGVwczogQXdzQ2RrRGVwcztcbn1cblxuY2xhc3MgTGFtYmRhTGF5ZXJDb25zdHJ1Y3QgZXh0ZW5kcyBTb3VyY2VDb2RlIHtcbiAgY29uc3RydWN0b3IocHJvamVjdDogUHJvamVjdCwgb3B0aW9uczogTGFtYmRhTGF5ZXJDb25zdHJ1Y3RPcHRpb25zKSB7XG4gICAgc3VwZXIocHJvamVjdCwgb3B0aW9ucy5jb25zdHJ1Y3RGaWxlKTtcblxuICAgIGNvbnN0IHNyYyA9IHRoaXM7XG4gICAgY29uc3QgY2RrRGVwcyA9IG9wdGlvbnMuY2RrRGVwcztcblxuICAgIGNvbnN0IGNvbnN0cnVjdE5hbWUgPVxuICAgICAgb3B0aW9ucy5jb25zdHJ1Y3ROYW1lID8/IHBhc2NhbChiYXNlbmFtZShvcHRpb25zLmNvbnN0cnVjdEZpbGUsIFwiLnRzXCIpKTtcbiAgICBjb25zdCBwcm9wc1R5cGUgPSBgJHtjb25zdHJ1Y3ROYW1lfVByb3BzYDtcblxuICAgIGNvbnN0IGFzc2V0RGlyID0gcmVsYXRpdmUoZGlybmFtZShvcHRpb25zLmNvbnN0cnVjdEZpbGUpLCBvcHRpb25zLmFzc2V0RGlyKTtcblxuICAgIGlmIChzcmMubWFya2VyKSB7XG4gICAgICBzcmMubGluZShgLy8gJHtzcmMubWFya2VyfWApO1xuICAgIH1cbiAgICBzcmMubGluZShcImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XCIpO1xuXG4gICAgaWYgKGNka0RlcHMuY2RrTWFqb3JWZXJzaW9uID09PSAxKSB7XG4gICAgICBzcmMubGluZShcImltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcIik7XG4gICAgICBzcmMubGluZShcImltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1wiKTtcbiAgICAgIGNka0RlcHMuYWRkVjFEZXBlbmRlbmNpZXMoXCJAYXdzLWNkay9hd3MtbGFtYmRhXCIpO1xuICAgICAgY2RrRGVwcy5hZGRWMURlcGVuZGVuY2llcyhcIkBhd3MtY2RrL2NvcmVcIik7XG4gICAgfSBlbHNlIHtcbiAgICAgIHNyYy5saW5lKFwiaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYW1iZGEnO1wiKTtcbiAgICAgIHNyYy5saW5lKFwiaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XCIpO1xuICAgIH1cblxuICAgIHNyYy5saW5lKCk7XG5cbiAgICBzcmMubGluZShcIi8qKlwiKTtcbiAgICBzcmMubGluZShgICogUHJvcHMgZm9yICR7Y29uc3RydWN0TmFtZX1gKTtcbiAgICBzcmMubGluZShcIiAqL1wiKTtcbiAgICBzcmMub3BlbihcbiAgICAgIGBleHBvcnQgaW50ZXJmYWNlICR7cHJvcHNUeXBlfSBleHRlbmRzIGxhbWJkYS5MYXllclZlcnNpb25PcHRpb25zIHtgXG4gICAgKTtcbiAgICBzcmMuY2xvc2UoXCJ9XCIpO1xuICAgIHNyYy5saW5lKCk7XG5cbiAgICBzcmMubGluZShcIi8qKlwiKTtcbiAgICBzcmMubGluZShgICogJHtvcHRpb25zLmRlc2NyaXB0aW9ufWApO1xuICAgIHNyYy5saW5lKFwiICovXCIpO1xuICAgIHNyYy5vcGVuKGBleHBvcnQgY2xhc3MgJHtjb25zdHJ1Y3ROYW1lfSBleHRlbmRzIGxhbWJkYS5MYXllclZlcnNpb24ge2ApO1xuICAgIHNyYy5vcGVuKFxuICAgICAgYGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzPzogJHtwcm9wc1R5cGV9KSB7YFxuICAgICk7XG5cbiAgICBzcmMub3BlbihcInN1cGVyKHNjb3BlLCBpZCwge1wiKTtcbiAgICBzcmMubGluZShgZGVzY3JpcHRpb246ICR7ZW5jb2RlQ29kZVN0cmluZyhvcHRpb25zLmRlc2NyaXB0aW9uKX0sYCk7XG4gICAgc3JjLmxpbmUoXCIuLi5wcm9wcyxcIik7XG5cbiAgICBzcmMub3BlbihcImNvbXBhdGlibGVSdW50aW1lczogW1wiKTtcbiAgICBmb3IgKGNvbnN0IHJ1bnRpbWUgb2Ygb3B0aW9ucy5jb21wYXRpYmxlUnVudGltZXMpIHtcbiAgICAgIHNyYy5saW5lKFxuICAgICAgICBgbmV3IGxhbWJkYS5SdW50aW1lKCcke3J1bnRpbWUuZnVuY3Rpb25SdW50aW1lfScsIGxhbWJkYS5SdW50aW1lRmFtaWx5Lk5PREVKUyksYFxuICAgICAgKTtcbiAgICB9XG4gICAgc3JjLmNsb3NlKFwiXSxcIik7XG5cbiAgICBzcmMub3BlbihgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsYCk7XG4gICAgc3JjLmxpbmUoYCR7ZW5jb2RlQ29kZVN0cmluZyhjb252ZXJ0VG9Qb3NpeFBhdGgoYXNzZXREaXIpKX0pKSxgKTtcbiAgICBzcmMuY2xvc2UoKTtcbiAgICBzcmMuY2xvc2UoXCJ9KTtcIik7XG5cbiAgICBzcmMuY2xvc2UoXCJ9XCIpO1xuICAgIHNyYy5jbG9zZShcIn1cIik7XG4gIH1cbn1cblxuLyoqXG4gKiBFbmNvZGVzIGEgc3RyaW5nIGZvciBlbWJlZGRpbmcgaW4gc291cmNlIGNvZGUuXG4gKi9cbmZ1bmN0aW9uIGVuY29kZUNvZGVTdHJpbmcodmFsdWU6IHN0cmluZykge1xuICBjb25zdCBqc29uID0gSlNPTi5zdHJpbmdpZnkodmFsdWUpO1xuICBjb25zdCBlc2NhcGVkU3RyaW5nID0ganNvbi5zdWJzdHJpbmcoMSwganNvbi5sZW5ndGggLSAxKS5yZXBsYWNlKC8nL2csIFwiXFxcXCdcIik7XG4gIHJldHVybiBgJyR7ZXNjYXBlZFN0cmluZ30nYDtcbn1cbiJdfQ==