"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCdkDeps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const semver = require("semver");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
/**
 * Manages dependencies on the AWS CDK.
 */
class AwsCdkDeps extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.cdkDependenciesAsDeps = options.cdkDependenciesAsDeps ?? true;
        this.dependencyType = options.dependencyType;
        this._packageNames = this.packageNames();
        const framework = determineFrameworkVersion(options);
        this.cdkVersion = framework.range;
        this.cdkMajorVersion = framework.major;
        this.cdkMinimumVersion = framework.minimum;
        this.addFrameworkDependency(options);
        // assert/assertions library
        this.addV1AssertionLibraryDependency(options);
        // constructs library
        this.addConstructsDependency(options.constructsVersion);
        // user-defined v1 dependencies (will only fail in CDK v2 if these have values)
        this.addV1Dependencies(...(options.cdkDependencies ?? []));
        this.addV1DevDependencies(...(options.cdkTestDependencies ?? []));
    }
    preSynthesize() {
        // Log a warning if any AWS CDK v1-only deps are found in the dependencies.
        const depNames = Array.from(new Set(this.project.deps.all.map((dep) => dep.name)));
        const v1Deps = depNames
            .filter((dep) => [PACKAGE_AWS_CDK_VERSION.V1].includes(cdkVersionOfPackage(dep)))
            .sort();
        if (this.cdkMajorVersion === 2 && v1Deps.length > 0) {
            this.project.logger.warn(`WARNING: Found CDK v1 deps in your project, even though your "cdkVersion" is 2.x: [${v1Deps.join(", ")}]. Check out https://docs.aws.amazon.com/cdk/v2/guide/migrating-v2.html for more information about using CDK v2 dependencies.`);
        }
    }
    /**
     * Adds dependencies to AWS CDK modules.
     *
     * The type of dependency is determined by the `dependencyType` option.
     *
     * This method is not supported in CDK v2. Use `project.addPeerDeps()` or
     * `project.addDeps()` as appropriate.
     *
     * @param deps names of cdk modules (e.g. `@aws-cdk/aws-lambda`).
     */
    addV1Dependencies(...deps) {
        if (deps.length > 0 && this.cdkMajorVersion !== 1) {
            throw new Error("addV1Dependencies() is not supported for CDK 2.x and above, use addDeps() or addPeerDeps() instead");
        }
        // this will add dependencies based on the type requested by the user
        // for libraries, this will be "peer" and for apps it will be "runtime"
        this.addV1DependenciesByType(this.dependencyType, ...deps);
        // add deps as runtime deps if `cdkDepsAsDeps` is true
        if (this.cdkDependenciesAsDeps) {
            this.addV1DependenciesByType(dependencies_1.DependencyType.RUNTIME, ...deps);
        }
    }
    /**
     * Adds AWS CDK modules as dev dependencies.
     *
     * This method is not supported in CDK v2. Use `project.addPeerDeps()` or
     * `project.addDeps()` as appropriate.
     *
     * @param deps fully qualified names of cdk modules (e.g. `@aws-cdk/aws-lambda`).
     */
    addV1DevDependencies(...deps) {
        if (deps.length > 0 && this.cdkMajorVersion !== 1) {
            throw new Error("addV1DevDependencies() is not supported for CDK 2.x and above, use addDevDeps()/addTestDeps() instead");
        }
        this.addV1DependenciesByType(dependencies_1.DependencyType.BUILD, ...deps);
    }
    addConstructsDependency(requestedVersion) {
        if (requestedVersion && !semver.parse(requestedVersion)) {
            throw new Error(`"constructsVersion" cannot be parsed as a semver version: ${requestedVersion}`);
        }
        const defaultVersion = this.cdkMajorVersion === 1 ? "3.2.27" : "10.0.5";
        const versionRequirement = `^${requestedVersion ?? defaultVersion}`;
        const constructsMajorVersion = semver.minVersion(versionRequirement)?.major;
        if (!constructsMajorVersion) {
            throw new Error(`Cannot determine major version of constructs version '${versionRequirement}'`);
        }
        switch (this.cdkMajorVersion) {
            case 1:
                if (constructsMajorVersion !== 3) {
                    throw new Error("AWS CDK 1.x requires constructs 3.x");
                }
                break;
            case 2:
                if (constructsMajorVersion !== 10) {
                    throw new Error("AWS CDK 2.x requires constructs 10.x");
                }
                break;
        }
        this.project.deps.addDependency(`${this._packageNames.constructs}@${versionRequirement}`, this.dependencyType);
        return versionRequirement;
    }
    /**
     * Adds a dependency on the AWS CDK framework (e.g. `@aws-cdk/core` for V1 or `aws-cdk-lib` for V1).
     */
    addFrameworkDependency(options) {
        switch (this.cdkMajorVersion) {
            case 1:
                this.addV1Dependencies(this._packageNames.coreV1);
                break;
            case 2:
                if (options.cdkDependencies !== undefined) {
                    throw new Error('cdkDependencies is not used for CDK 2.x. Use "peerDeps" or "deps" instead');
                }
                if (options.cdkDependenciesAsDeps !== undefined) {
                    throw new Error("cdkDependenciesAsDeps is not used for CDK 2.x");
                }
                if (options.cdkTestDependencies !== undefined) {
                    throw new Error('cdkTestDependencies is not used for CDK 2.x. Use "devDeps" or "testDeps" instead');
                }
                this.project.deps.addDependency(`${this._packageNames.coreV2}@${this.cdkVersion}`, this.dependencyType);
                break;
            default:
                throw new Error(`Unsupported AWS CDK major version ${this.cdkMajorVersion}.x`);
        }
    }
    addV1AssertionLibraryDependency(options) {
        if (this.cdkMajorVersion !== 1) {
            if (options.cdkAssert !== undefined) {
                throw new Error("cdkAssert is not used for CDK 2.x. Use the assertions library that is provided in aws-cdk-lib");
            }
            if (options.cdkAssertions !== undefined) {
                throw new Error("cdkAssertion is not used for CDK 2.x. Use the assertions library that is provided in aws-cdk-lib");
            }
            return;
        }
        const testDeps = new Array();
        if ((options.cdkAssert ?? true) && this._packageNames.assert) {
            testDeps.push(this._packageNames.assert);
        }
        // @aws-cdk/assertions is only available starting v1.111.0
        if (semver.gte(this.cdkMinimumVersion, "1.111.0") &&
            (options.cdkAssertions ?? true)) {
            testDeps.push(this._packageNames.assertions);
        }
        this.addV1DependenciesByType(dependencies_1.DependencyType.TEST, ...testDeps);
    }
    /**
     * Adds a set of dependencies with the user-specified dependency type.
     * @param deps The set of dependency specifications
     */
    addV1DependenciesByType(type, ...modules) {
        for (const module of modules) {
            this.project.deps.addDependency(`${module}@${this.cdkVersion}`, type);
        }
    }
}
exports.AwsCdkDeps = AwsCdkDeps;
_a = JSII_RTTI_SYMBOL_1;
AwsCdkDeps[_a] = { fqn: "projen.awscdk.AwsCdkDeps", version: "0.67.2" };
/**
 * Which AWS CDK version a construct library package belongs to.
 */
var PACKAGE_AWS_CDK_VERSION;
(function (PACKAGE_AWS_CDK_VERSION) {
    PACKAGE_AWS_CDK_VERSION["V1"] = "v1";
    PACKAGE_AWS_CDK_VERSION["V2"] = "v2";
    PACKAGE_AWS_CDK_VERSION["EITHER"] = "either";
    PACKAGE_AWS_CDK_VERSION["UNKNOWN"] = "unknown";
})(PACKAGE_AWS_CDK_VERSION || (PACKAGE_AWS_CDK_VERSION = {}));
function cdkVersionOfPackage(packageName) {
    if (packageName === "aws-cdk-lib") {
        return PACKAGE_AWS_CDK_VERSION.V2;
    }
    else if (packageName.startsWith("@aws-cdk/")) {
        if (packageName.endsWith("-alpha")) {
            return PACKAGE_AWS_CDK_VERSION.V2;
        }
        else if (AWS_CDK_V1_V2_SCOPED_PACKAGES.includes(packageName)) {
            return PACKAGE_AWS_CDK_VERSION.EITHER;
        }
        else {
            return PACKAGE_AWS_CDK_VERSION.V1;
        }
    }
    else {
        return PACKAGE_AWS_CDK_VERSION.UNKNOWN;
    }
}
/**
 * A list of all known packages in the "@aws-cdk/" scope that are published
 * both for v1 and v2.
 */
const AWS_CDK_V1_V2_SCOPED_PACKAGES = [
    "@aws-cdk/cfnspec",
    "@aws-cdk/cx-api",
    "@aws-cdk/region-info",
    "@aws-cdk/cloud-assembly-schema",
    "@aws-cdk/assert",
    "@aws-cdk/cloudformation-diff",
    "@aws-cdk/integ-runner",
];
function determineFrameworkVersion(options) {
    const ver = semver.parse(options.cdkVersion);
    if (!ver) {
        throw new Error(`"cdkVersion" cannot be parsed as a semver version: ${options.cdkVersion}`);
    }
    return {
        minimum: ver.format(),
        range: options.cdkVersionPinning
            ? options.cdkVersion
            : `^${options.cdkVersion}`,
        major: ver.major,
    };
}
//# sourceMappingURL=data:application/json;base64,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