from pathlib import Path, PosixPath
from typing import Union
from .base import InputFile


class InputFileGenerated(InputFile):
    """A file generated by the application.

    Args:
        original_path (`:py:class:pathlib.PosixPath` | ``str``):
            Local path to a file from which the file is generated; may be empty if there is no such file.

        conversion (``str``):
            String specifying the conversion applied to the original file; must be persistent across application restarts. Conversions beginning with '#' are reserved for internal TDLib usage.

        expected_size (``int``):
            Expected size of the generated file, in bytes; 0 if unknown.
    """

    def __init__(
        self,
        original_path: Union[str, PosixPath],
        conversion: str,
        expected_size: int = 0,
    ) -> None:

        self.original_path = original_path
        if isinstance(self.original_path, PosixPath):
            self.original_path = str(self.original_path.resolve())
        else:
            self.original_path = str(Path(self.original_path).resolve())
        self.conversion = conversion
        self.expected_size = expected_size
        self.data = {
            "@type": "inputFileGenerated",
            "original_path": self.original_path,
            "conversion": self.conversion,
            "expected_size": self.expected_size,
        }
