r"""

:mod:`empymod.filters` -- Digital Linear Filters
================================================

Filters for the *Digital Linear Filter* (DLF) method for the Hankel
[Ghos70]_) and the Fourier ([Ande75]_) transforms.

To calculate the `dlf.factor` I used

.. code-block:: python

    np.around(np.average(dlf.base[1:]/dlf.base[:-1]), 15)


The filters `kong_61_2007` and `kong_241_2007` from [Kong07]_, and
`key_101_2009`, `key_201_2009`, `key_401_2009`, `key_81_CosSin_2009`,
`key_241_CosSin_2009`, and `key_601_CosSin_2009` from [Key09]_ are taken from
*DIPOLE1D*, [Key09]_, which can be downloaded at
https://marineemlab.ucsd.edu/Projects/Occam/1DCSEM (1DCSEM_). *DIPOLE1D* is
distributed under the license GNU GPL version 3 or later. Kerry Key gave his
written permission to re-distribute the filters under the Apache License,
Version 2.0 (email from Kerry Key to Dieter Werthmüller, 21 November 2016).

The filters `anderson_801_1982` from [Ande82]_ and `key_51_2012`,
`key_101_2012`, `key_201_2012`, `key_101_CosSin_2012`, and
`key_201_CosSin_2012`, all from [Key12]_, are taken from the software
distributed with [Key12]_ and available at https://software.seg.org/2012/0003
(SEG-2012-003_). These filters are distributed under the SEG license.

The filter `wer_201_2018` was designed with the add-on `fdesign`, see
https://github.com/emsig/article-fdesign.

.. _1DCSEM: https://marineemlab.ucsd.edu/Projects/Occam/1DCSEM
.. _SEG-2012-003: https://software.seg.org/2012/0003

"""
# Copyright 2016-2021 The empymod Developers.
#
# This file is part of empymod.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may not
# use this file except in compliance with the License.  You may obtain a copy
# of the License at
#
#     https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
# License for the specific language governing permissions and limitations under
# the License.


import os
import numpy as np

__all__ = ['DigitalFilter', 'kong_61_2007', 'kong_241_2007', 'key_101_2009',
           'key_201_2009', 'key_401_2009', 'anderson_801_1982', 'key_51_2012',
           'key_101_2012', 'key_201_2012', 'wer_201_2018',
           'key_81_CosSin_2009', 'key_241_CosSin_2009', 'key_601_CosSin_2009',
           'key_101_CosSin_2012', 'key_201_CosSin_2012']


# 0. Filter Class and saving/loading routines

class DigitalFilter:
    r"""Simple Class for Digital Linear Filters.


    Parameters
    ----------
    name : str
        Name of the DFL.

    savename = str
        Name with which the filter is saved. If None (default) it is set to the
        same value as `name`.

    filter_coeff = list of str
        By default, the following filter coefficients are checked:

            ``filter_coeff = ['j0', 'j1', 'sin', 'cos']``

        This accounts for the standard Hankel and Fourier DLF in CSEM
        modelling. However, additional coefficient names can be provided via
        this parameter (in list format).

    """

    def __init__(self, name, savename=None, filter_coeff=None):
        r"""Add filter name."""
        self.name = name
        if savename is None:
            self.savename = name
        else:
            self.savename = savename

        # Define coefficient names.
        self.filter_coeff = ['j0', 'j1', 'sin', 'cos']
        if filter_coeff is not None:  # Additional, user provided.
            self.filter_coeff.extend(filter_coeff)

    def tofile(self, path='filters'):
        r"""Save filter values to ASCII-files.

        Store the filter base and the filter coefficients in separate files
        in the directory `path`; `path` can be a relative or absolute path.

        Examples
        --------
        >>> import empymod
        >>> # Load a filter
        >>> filt = empymod.filters.wer_201_2018()
        >>> # Save it to pure ASCII-files
        >>> filt.tofile()
        >>> # This will save the following three files:
        >>> #    ./filters/wer_201_2018_base.txt
        >>> #    ./filters/wer_201_2018_j0.txt
        >>> #    ./filters/wer_201_2018_j1.txt

        """

        # Get name of filter
        name = self.savename

        # Get absolute path, create if it doesn't exist
        path = os.path.abspath(path)
        os.makedirs(path, exist_ok=True)

        # Save filter base
        basefile = os.path.join(path, name + '_base.txt')
        with open(basefile, 'w') as f:
            self.base.tofile(f, sep="\n")

        # Save filter coefficients
        for val in self.filter_coeff:
            if hasattr(self, val):
                attrfile = os.path.join(path, name + '_' + val + '.txt')
                with open(attrfile, 'w') as f:
                    getattr(self, val).tofile(f, sep="\n")

    def fromfile(self, path='filters'):
        r"""Load filter values from ASCII-files.

        Load filter base and filter coefficients from ASCII files in the
        directory `path`; `path` can be a relative or absolute path.

        Examples
        --------
        >>> import empymod
        >>> # Create an empty filter;
        >>> # Name has to be the base of the text files
        >>> filt = empymod.filters.DigitalFilter('my-filter')
        >>> # Load the ASCII-files
        >>> filt.fromfile()
        >>> # This will load the following three files:
        >>> #    ./filters/my-filter_base.txt
        >>> #    ./filters/my-filter_j0.txt
        >>> #    ./filters/my-filter_j1.txt
        >>> # and store them in filt.base, filt.j0, and filt.j1.

        """

        # Get name of filter
        name = self.savename

        # Get absolute path
        path = os.path.abspath(path)

        # Get filter base
        basefile = os.path.join(path, name + '_base.txt')
        with open(basefile, 'r') as f:
            self.base = np.fromfile(f, sep="\n")

        # Get filter coefficients
        for val in self.filter_coeff:
            attrfile = os.path.join(path, name + '_' + val + '.txt')
            if os.path.isfile(attrfile):
                with open(attrfile, 'r') as f:
                    setattr(self, val, np.fromfile(f, sep="\n"))

        # Add factor
        self.factor = np.around(np.average(self.base[1:]/self.base[:-1]), 15)


# 1. Hankel DLF

def kong_61_2007():
    r"""Kong 61 pt Hankel filter, as published in [Kong07]_.

    Taken from file `FilterModules.f90` provided with 1DCSEM_.

    License: `Apache License, Version 2.0,
    <https://www.apache.org/licenses/LICENSE-2.0>`_.

    """

    dlf = DigitalFilter('Kong 61', 'kong_61_2007')

    dlf.base = np.array([
            2.3517745856009100e-02, 2.6649097336355482e-02,
            3.0197383422318501e-02, 3.4218118311666032e-02,
            3.8774207831722009e-02, 4.3936933623407420e-02,
            4.9787068367863938e-02, 5.6416139503777350e-02,
            6.3927861206707570e-02, 7.2439757034251456e-02,
            8.2084998623898800e-02, 9.3014489210663506e-02,
            1.0539922456186430e-01, 1.1943296826671961e-01,
            1.3533528323661270e-01, 1.5335496684492850e-01,
            1.7377394345044520e-01, 1.9691167520419400e-01,
            2.2313016014842979e-01, 2.5283959580474641e-01,
            2.8650479686019009e-01, 3.2465246735834979e-01,
            3.6787944117144239e-01, 4.1686201967850839e-01,
            4.7236655274101469e-01, 5.3526142851899028e-01,
            6.0653065971263342e-01, 6.8728927879097224e-01,
            7.7880078307140488e-01, 8.8249690258459546e-01,
            1.0000000000000000e+00, 1.1331484530668261e+00,
            1.2840254166877421e+00, 1.4549914146182010e+00,
            1.6487212707001280e+00, 1.8682459574322221e+00,
            2.1170000166126748e+00, 2.3988752939670981e+00,
            2.7182818284590451e+00, 3.0802168489180310e+00,
            3.4903429574618419e+00, 3.9550767229205772e+00,
            4.4816890703380636e+00, 5.0784190371800806e+00,
            5.7546026760057307e+00, 6.5208191203301116e+00,
            7.3890560989306504e+00, 8.3728974881272649e+00,
            9.4877358363585262e+00, 1.0751013186076360e+01,
            1.2182493960703470e+01, 1.3804574186067100e+01,
            1.5642631884188170e+01, 1.7725424121461639e+01,
            2.0085536923187671e+01, 2.2759895093526730e+01,
            2.5790339917193059e+01, 2.9224283781234941e+01,
            3.3115451958692312e+01, 3.7524723159601002e+01,
            4.2521082000062783e+01])

    dlf.factor = np.array([1.1331484530668261])

    dlf.j0 = np.array([
            1.4463210615326699e+02, -1.1066222143752420e+03,
            3.7030010025325978e+03, -6.8968188464424520e+03,
            7.1663544112656937e+03, -2.4507884783377681e+03,
            -4.0166567754046082e+03, 6.8623845298546094e+03,
            -5.0013321011775661e+03, 2.1291291365196648e+03,
            -1.3845222435542289e+03, 2.1661554291595580e+03,
            -2.2260393789657141e+03, 8.0317156013986391e+02,
            1.0142221718890841e+03, -1.9350455051432630e+03,
            1.6601169447226580e+03, -7.5159684285420133e+02,
            -9.0315984178183285e+01, 5.0705574889546148e+02,
            -5.1207646422722519e+02, 2.9722959494490038e+02,
            -5.0248319908072993e+01, -1.2290725861955920e+02,
            1.9695244755899429e+02, -1.9175679966946601e+02,
            1.4211755630338590e+02, -7.7463216543224149e+01,
            1.7638009334931201e+01, 2.8855056499202671e+01,
            -5.9225643887809561e+01, 7.5987941373668960e+01,
            -8.1687962781233580e+01, 8.0599209238447102e+01,
            -7.4895905328771619e+01, 6.7516291538794434e+01,
            -5.9325033647358048e+01, 5.1617042242841528e+01,
            -4.4664967446820263e+01, 3.8366152052928278e+01,
            -3.3308787868993100e+01, 2.8278671651033459e+01,
            -2.4505863388620480e+01, 2.0469632532079750e+01,
            -1.7074034940700429e+01, 1.4206119215530070e+01,
            -1.0904435643084650e+01, 8.7518389425802283e+00,
            -6.7721665239085622e+00, 4.5096884588095891e+00,
            -3.2704247166629590e+00, 2.6827195063720430e+00,
            -1.8406031821386459e+00, 9.1586697140412443e-01,
            -3.2436011485890798e-01, 8.0675176189581893e-02,
            -1.2881307195759690e-02, 7.0489137468452920e-04,
            2.3846917590855061e-04, -6.9102205995825531e-05,
            6.7792635718095777e-06])

    dlf.j1 = np.array([
            4.6440396425864918e+01, -4.5034239857914162e+02,
            1.7723440076223640e+03, -3.7559735516994660e+03,
            4.4736494009764137e+03, -2.2476603569606068e+03,
            -1.5219842155931799e+03, 3.4904608559273802e+03,
            -2.4814243247472318e+03, 5.7328164634108396e+02,
            5.3132044837659631e-01, 6.8895205008006235e+02,
            -1.2012013872160269e+03, 7.9679138423597340e+02,
            4.9874460187939818e+01, -5.6367338332457007e+02,
            4.7971936503711203e+02, -5.8979702298044558e+01,
            -3.1935800954986922e+02, 4.5762551999442371e+02,
            -3.7239927283248380e+02, 1.8255852885279569e+02,
            -2.3504740340815669e-01, -1.1588151583545380e+02,
            1.5740956677133170e+02, -1.4334746114883359e+02,
            9.9857411013284818e+01, -4.8246322019171487e+01,
            2.0371404343057380e+00, 3.3003938094974323e+01,
            -5.5476151884197712e+01, 6.7354852323852583e+01,
            -7.0735403363284121e+01, 6.8872932663164747e+01,
            -6.3272750944993042e+01, 5.6501568721817442e+01,
            -4.8706577819918110e+01, 4.1737211284663481e+01,
            -3.4776621242200903e+01, 2.9161717578906430e+01,
            -2.3886749056000909e+01, 1.9554007583544220e+01,
            -1.5966397353366460e+01, 1.2429310210239199e+01,
            -1.0139180791868180e+01, 7.4716493393871861e+00,
            -5.5509479014742613e+00, 4.3380799768234208e+00,
            -2.5911516181746550e+00, 1.6300524630626780e+00,
            -1.4041567266387460e+00, 7.5225141726873213e-01,
            4.6808777208492733e-02, -3.6630197849601159e-01,
            2.8948389902792782e-01, -1.3705521898064801e-01,
            4.6292091649913013e-02, -1.1721281347435180e-02,
            2.2002397354029149e-03, -2.8146036357227600e-04,
            1.8788896009128770e-05])

    return dlf


def kong_241_2007():
    r"""Kong 241 pt Hankel filter, as published in [Kong07]_.

    Taken from file `FilterModules.f90` provided with 1DCSEM_.

    License: `Apache License, Version 2.0,
    <https://www.apache.org/licenses/LICENSE-2.0>`_.

    """

    dlf = DigitalFilter('Kong 241', 'kong_241_2007')

    dlf.base = np.array([
            4.0973497897978643e-04, 4.3725238042414673e-04,
            4.6661782370309847e-04, 4.9795542150327349e-04,
            5.3139762179825294e-04, 5.6708576763830356e-04,
            6.0517069453505309e-04, 6.4581336796593124e-04,
            6.8918556369279311e-04, 7.3547059377007120e-04,
            7.8486408131093165e-04, 8.3757478728596971e-04,
            8.9382549284889267e-04, 9.5385394091834482e-04,
            1.0179138409954376e-03, 1.0862759414638579e-03,
            1.1592291739045914e-03, 1.2370818742617167e-03,
            1.3201630860205026e-03, 1.4088239509056637e-03,
            1.5034391929775724e-03, 1.6044087023989032e-03,
            1.7121592255655228e-03, 1.8271461687448972e-03,
            1.9498555228451206e-03, 2.0808059174495293e-03,
            2.2205508127982939e-03, 2.3696808389813539e-03,
            2.5288262922292556e-03, 2.6986597988524685e-03,
            2.8798991580882404e-03, 3.0733103758703055e-03,
            3.2797109023435700e-03, 3.4999730868071656e-03,
            3.7350278646880674e-03, 3.9858686921282905e-03,
            4.2535557448151254e-03, 4.5392203988006852e-03,
            4.8440700122489664e-03, 5.1693930283203291e-03,
            5.5165644207607716e-03, 5.8870515052116155e-03,
            6.2824201408011177e-03, 6.7043413482289257e-03,
            7.1545983723145792e-03, 7.6350942188599617e-03,
            8.1478596976799818e-03, 8.6950620057954561e-03,
            9.2790138870647437e-03, 9.9021834069673818e-03,
            1.0567204383852655e-02, 1.1276887520740558e-02,
            1.2034232284723775e-02, 1.2842439584178571e-02,
            1.3704925297364945e-02, 1.4625334709594208e-02,
            1.5607557919982831e-02, 1.6655746282908664e-02,
            1.7774329953659442e-02, 1.8968036612429941e-02,
            2.0241911445804381e-02, 2.1601338470175833e-02,
            2.3052063287225571e-02, 2.4600217367638302e-02,
            2.6252343965687961e-02, 2.8015425774221808e-02,
            2.9896914436926308e-02, 3.1904762042607962e-02,
            3.4047454734599344e-02, 3.6334048577339996e-02,
            3.8774207831722009e-02, 4.1378245800970381e-02,
            4.4157168419692860e-02, 4.7122720770327912e-02,
            5.0287436723591865e-02, 5.3664691912730114e-02,
            5.7268760265467331e-02, 6.1114874332588359e-02,
            6.5219289668127525e-02, 6.9599353533269015e-02,
            7.4273578214333877e-02, 7.9261719264731550e-02,
            8.4584859001564691e-02, 9.0265495609784266e-02,
            9.6327638230493035e-02, 1.0279690843528640e-01,
            1.0970064851551141e-01, 1.1706803704412637e-01,
            1.2493021219858241e-01, 1.3332040336594936e-01,
            1.4227407158651356e-01, 1.5182905942943059e-01,
            1.6202575093388075e-01, 1.7290724229171636e-01,
            1.8451952399298926e-01, 1.9691167520419406e-01,
            2.1013607120076472e-01, 2.2424860473053532e-01,
            2.3930892224375450e-01, 2.5538067598807768e-01,
            2.7253179303401259e-01, 2.9083476236785155e-01,
            3.1036694126548503e-01, 3.3121088224198103e-01,
            3.5345468195878016e-01, 3.7719235356315689e-01,
            4.0252422403363597e-01, 4.2955735821073915e-01,
            4.5840601130522352e-01, 4.8919211179633149e-01,
            5.2204577676101604e-01, 5.5710586181217392e-01,
            5.9452054797019427e-01, 6.3444796794822822e-01,
            6.7705687449816465e-01, 7.2252735364207221e-01,
            7.7105158580356625e-01, 8.2283465805601841e-01,
            8.7809543092056130e-01, 9.3706746337740343e-01,
            1.0000000000000000e+00, 1.0671590243841926e+00,
            1.1388283833246218e+00, 1.2153109864897309e+00,
            1.2969300866657718e+00, 1.3840306459807514e+00,
            1.4769807938826425e+00, 1.5761733830339912e+00,
            1.6820276496988864e+00, 1.7949909856399000e+00,
            1.9155408290138962e+00, 2.0441866822585570e+00,
            2.1814722654982011e+00, 2.3279778145702346e+00,
            2.4843225333848169e+00, 2.6511672109826070e+00,
            2.8292170143515598e+00, 3.0192244688065686e+00,
            3.2219926385284996e+00, 3.4383785207051250e+00,
            3.6692966676192444e+00, 3.9157230519927220e+00,
            4.1786991919232470e+00, 4.4593365528478257e+00,
            4.7588212451378542e+00, 5.0784190371800815e+00,
            5.4194807051312059e+00, 5.7834477419567749e+00,
            6.1718584498835538e+00, 6.5863544420150681e+00,
            7.0286875805892945e+00, 7.5007273812029629e+00,
            8.0044689142963534e+00, 8.5420412372940930e+00,
            9.1157163930403051e+00, 9.7279190125598838e+00,
            1.0381236562731843e+01, 1.1078430282186428e+01,
            1.1822446851646363e+01, 1.2616430848036902e+01,
            1.3463738035001692e+01, 1.4367949545996751e+01,
            1.5332887019907195e+01, 1.6362628753157214e+01,
            1.7461526936579997e+01, 1.8634226049899006e+01,
            1.9885682491564729e+01, 2.1221185526912038e+01,
            2.2646379643175401e+01, 2.4167288405845099e+01,
            2.5790339917193062e+01, 2.7522393984568446e+01,
            2.9370771113289432e+01, 3.1343283446669389e+01,
            3.3448267783944921e+01, 3.5694620815655881e+01,
            3.8091836725399020e+01, 4.0650047316878776e+01,
            4.3380064835851620e+01, 4.6293427667950432e+01,
            4.9402449105530195e+01, 5.2720269389647328e+01,
            5.6260911247127851e+01, 6.0039339157450577e+01,
            6.4071522599936642e+01, 6.8374503548558152e+01,
            7.2966468499632811e+01, 7.7866825336828100e+01,
            8.3096285358343764e+01, 8.8676950812960627e+01,
            9.4632408314924064e+01, 1.0098782853248096e+02,
            1.0777007257140046e+02, 1.1500780550310940e+02,
            1.2273161751726525e+02, 1.3097415321081860e+02,
            1.3977024956000301e+02, 1.4915708315838788e+02,
            1.5917432734329714e+02, 1.6986431987468302e+02,
            1.8127224187515122e+02, 1.9344630878742183e+02,
            2.0643797415630826e+02, 2.2030214709649516e+02,
            2.3509742436523857e+02, 2.5088633802084459e+02,
            2.6773561971364717e+02, 2.8571648272651305e+02,
            3.0490492295690876e+02, 3.2538204011263201e+02,
            3.4723438047873475e+02, 3.7055430270433595e+02,
            3.9544036815532411e+02, 4.2199775748276141e+02,
            4.5033871516762099e+02, 4.8058302392070897e+02,
            5.1285851094282907e+02, 5.4730158818487951e+02,
            5.8405782889129489e+02, 6.2328258286358414e+02,
            6.6514163304436181e+02, 7.0981189619693009e+02,
            7.5748217064180938e+02, 8.0835393421053413e+02,
            8.6264219578923701e+02, 9.2057640405108020e+02,
            9.8240141721825944e+02, 1.0483785379522853e+03,
            1.1187866177464875e+03, 1.1939232354884318e+03,
            1.2741059551734540e+03, 1.3596736680849924e+03,
            1.4509880251144575e+03, 1.5484349652742915e+03,
            1.6524263468644833e+03, 1.7634016881866382e+03,
            1.8818300251626902e+03, 2.0082118937094979e+03,
            2.1430814452477584e+03, 2.2870087042864643e+03,
            2.4406019776245007e+03])

    dlf.factor = np.array([1.0671590243841931])

    dlf.j0 = np.array([
            2.0521734894828349e+01, -1.8902686822627982e+02,
            7.3615711333621391e+02, -1.5743636401162100e+03,
            1.9044172285355407e+03, -9.3177949828687565e+02,
            -8.4756674787751410e+02, 1.8896580243120454e+03,
            -1.4364973619770130e+03, 3.9999475225222255e+02,
            -1.3542454844021287e+02, 9.2532241451272546e+02,
            -1.9710056537324476e+03, 2.4446279171484089e+03,
            -2.1906123203102729e+03, 1.5850478978775775e+03,
            -1.0505328230896389e+03, 7.6984613632493506e+02,
            -7.0040976495766233e+02, 7.1606276337453505e+02,
            -7.1676220487815579e+02, 6.6234563164374151e+02,
            -5.5792724680287802e+02, 4.2758743543890813e+02,
            -2.9556591300691770e+02, 1.7825665706981184e+02,
            -8.3414675513698882e+01, 1.2417017098470755e+01,
            3.7033552727335611e+01, -6.8808606022416711e+01,
            8.7059849275066583e+01, -9.5546538383371242e+01,
            9.7381161313521631e+01, -9.4989208296887298e+01,
            9.0188673544147917e+01, -8.4292718636942382e+01,
            7.8226832173348868e+01, -7.2614408878060885e+01,
            6.7859682477117360e+01, -6.4196278592205076e+01,
            6.1741547025682522e+01, -6.0519141387284385e+01,
            6.0491769334480402e+01, -6.1560425911383518e+01,
            6.3581206383691445e+01, -6.6354594873531482e+01,
            6.9646829420226268e+01, -7.3188103112902326e+01,
            7.6708283429767803e+01, -7.9940731778724427e+01,
            8.2660409124105158e+01, -8.4678217919647295e+01,
            8.5870333056460041e+01, -8.6157409171557106e+01,
            8.5526114160418373e+01, -8.3997300873449348e+01,
            8.1646015415233720e+01, -7.8563495286564020e+01,
            7.4879819626555161e+01, -7.0721125046688840e+01,
            6.6236029868394425e+01, -6.1547931592552032e+01,
            5.6787391190038285e+01, -5.2042250889790630e+01,
            4.7401124144337359e+01, -4.2904131591225969e+01,
            3.8598898252452933e+01, -3.4488213955995406e+01,
            3.0594494176351635e+01, -2.6898802810010366e+01,
            2.3412439481509015e+01, -2.0106495314306827e+01,
            1.6993408313817032e+01, -1.4047029053019282e+01,
            1.1294767677459204e+01, -8.7239710357940154e+00,
            6.3824509509500222e+00, -4.2667780201601095e+00,
            2.4325051404370148e+00, -8.6492951505166771e-01,
            -3.9467042614078562e-01, 1.3928518582065126e+00,
            -2.1151897072040806e+00, 2.6442660480353983e+00,
            -2.9851029846728716e+00, 3.2414793692894390e+00,
            -3.4159478759422446e+00, 3.6114813897567299e+00,
            -3.8065481471148583e+00, 4.0881401855578137e+00,
            -4.3995650940379534e+00, 4.8090112941999275e+00,
            -5.2256078193681690e+00, 5.7072204037420695e+00,
            -6.1390449324831593e+00, 6.5839615678399346e+00,
            -6.9164266653291033e+00, 7.2188877597238257e+00,
            -7.3636288680068356e+00, 7.4607479507817942e+00,
            -7.3815887701068963e+00, 7.2656732754121194e+00,
            -6.9791586417386533e+00, 6.6886849969172548e+00,
            -6.2474308336775373e+00, 5.8445619106015023e+00,
            -5.3113072582819267e+00, 4.8557012512460096e+00,
            -4.2807487698944051e+00, 3.8139742889863348e+00,
            -3.2285888930306363e+00, 2.7762320886023009e+00,
            -2.2014921362632469e+00, 1.7852627321359549e+00,
            -1.2441166658179819e+00, 8.9066883548526044e-01,
            -4.1204139236605647e-01, 1.5134230126345752e-01,
            2.3623794557833316e-01, -3.8120367816755335e-01,
            6.6106381680388782e-01, -6.8548466976773659e-01,
            8.6073950596627435e-01, -7.8244282877711779e-01,
            8.7517842089752496e-01, -7.3021880955390983e-01,
            7.7262232832472633e-01, -6.0480748494904057e-01,
            6.2658579242671697e-01, -4.7604045939428874e-01,
            4.9376650403683020e-01, -3.8947933764667242e-01,
            4.0212565692397595e-01, -3.6042458247357878e-01,
            3.5204231781815948e-01, -3.7930843668720610e-01,
            3.2733689195982146e-01, -4.2315508593901219e-01,
            3.0989441332277490e-01, -4.6647831068082313e-01,
            2.9127018318261688e-01, -4.8705387071535239e-01,
            2.7698618486052751e-01, -4.6732646495222491e-01,
            2.8247063840665948e-01, -3.9734195755492102e-01,
            3.2057830222839734e-01, -2.8520009926766782e-01,
            3.8137726386396886e-01, -1.7293887127458049e-01,
            4.1451464616516831e-01, -1.3721835010891198e-01,
            3.4710931937928907e-01, -2.3427502149669574e-01,
            1.7531537381430198e-01, -3.8202053727848895e-01,
            6.9544164681199228e-02, -3.4148035441124835e-01,
            2.2691549707658903e-01, -7.7194405672412439e-02,
            3.9072641204282460e-01, -8.3147250278941501e-02,
            9.2847198484461116e-02, -3.6217207710792204e-01,
            1.0822725873639430e-02, 5.1804674714713058e-03,
            3.1786543576503390e-01, -7.5374320952788773e-02,
            -1.9068492965240180e-01, -8.3243368421653877e-02,
            2.8203125407893237e-01, -1.6118506235275606e-02,
            -2.6963245268577724e-01, 1.9777266987246947e-01,
            8.3472930934463230e-02, -2.7664291725690993e-01,
            2.8131050087465398e-01, -1.7190500226396740e-01,
            4.7929281626787730e-02, 4.0609103740114338e-02,
            -8.8052363812875192e-02, 1.0655635530077956e-01,
            -1.0911039889769197e-01, 1.0429389492393636e-01,
            -9.6633502650203901e-02, 8.8193660973234708e-02,
            -7.9818272272792340e-02, 7.1826372817129266e-02,
            -6.4338746209327960e-02, 5.7410843811670373e-02,
            -5.1077264811879450e-02, 4.5360450730759908e-02,
            -4.0267658416644959e-02, 3.5786523769444523e-02,
            -3.1883184387833892e-02, 2.8503987928848736e-02,
            -2.5580391582215669e-02, 2.3035937972928848e-02,
            -2.0793869607217731e-02, 1.8783956187539304e-02,
            -1.6947463041594817e-02, 1.5239810059902827e-02,
            -1.3631136080120022e-02, 1.2105408954537602e-02,
            -1.0658744979296539e-02, 9.2973012802827770e-03,
            -8.0347421903655229e-03, 6.8891191289946248e-03,
            -5.8791440085348362e-03, 5.0201744968206546e-03,
            -4.3205477515280673e-03, 3.7790104974130003e-03,
            -3.3838316346466528e-03, 3.1138040173687830e-03,
            -2.9408759849158255e-03, 2.8337622799129800e-03,
            -2.7617063348841968e-03, 2.6976528553792770e-03,
            -2.6203758209448719e-03, 2.5154519816519300e-03,
            -2.3752432671978732e-03, 2.1981931889228849e-03,
            -1.9877598903184075e-03, 1.7512378319139862e-03,
            -1.4986007967920582e-03, 1.2413783399553321e-03,
            -9.9151066433110242e-04, 7.6014790547529541e-04,
            -5.5645455379022023e-04, 3.8658922048399793e-04,
            -2.5308541294203246e-04, 1.5481736123335846e-04,
            -8.7598356988947244e-05, 4.5276882194925013e-05,
            -2.1044884609057091e-05, 8.6194148443616213e-06,
            -3.0265483432492266e-06, 8.7586378588036060e-07,
            -1.9631579670478718e-07, 3.0405955678339503e-08,
            -2.4535953018971818e-09])

    dlf.j1 = np.array([
            -6.8036776043707992e+00, 1.2311367914708828e+02,
            -6.0393880274694629e+02, 1.4366739793969700e+03,
            -1.8485584122626103e+03, 1.0200389392810666e+03,
            5.5616172944428206e+02, -1.3603216652028611e+03,
            7.0167261135195611e+02, 5.3402931472384910e+02,
            -1.1429297666385887e+03, 9.0089651035093164e+02,
            -4.1234322841018411e+02, 2.0384051761852257e+02,
            -2.8956552394430940e+02, 4.0377046372586426e+02,
            -3.6323786432883537e+02, 1.8298774315085831e+02,
            1.9289907486739775e+01, -1.4580899521803201e+02,
            1.6824962609150825e+02, -1.1236235241859872e+02,
            2.2347944322515357e+01, 6.4941329832499832e+01,
            -1.2923900641568204e+02, 1.6503157425386775e+02,
            -1.7577683834104877e+02, 1.6854431620724952e+02,
            -1.5058653072004714e+02, 1.2774003008692418e+02,
            -1.0402698302316379e+02, 8.1858707646356891e+01,
            -6.2444337865410809e+01, 4.6197726342228336e+01,
            -3.3061362052896769e+01, 2.2733224772330665e+01,
            -1.4810952273048434e+01, 8.8755599393231179e+00,
            -4.5352205696513836e+00, 1.4449219838043461e+00,
            6.8743711356557613e-01, -2.1019399621163908e+00,
            2.9910240715100782e+00, -3.5052807631070255e+00,
            3.7597471619400826e+00, -3.8399078959004260e+00,
            3.8072695188293357e+00, -3.7042465619688505e+00,
            3.5586759844836791e+00, -3.3877703451391699e+00,
            3.2017608368233237e+00, -3.0067309393330288e+00,
            2.8068645418014970e+00, -2.6056123347708602e+00,
            2.4063835683466084e+00, -2.2124155853618990e+00,
            2.0267333591783983e+00, -1.8516694378892813e+00,
            1.6889204130272601e+00, -1.5392291077389748e+00,
            1.4027911155907105e+00, -1.2791049665324250e+00,
            1.1676349053464947e+00, -1.0675953584302726e+00,
            9.7863461730665857e-01, -9.0027354444851670e-01,
            8.3249709182475107e-01, -7.7482174407455295e-01,
            7.2701622202684046e-01, -6.8800213341352723e-01,
            6.5698937864345264e-01, -6.3225165202126643e-01,
            6.1273246350052379e-01, -5.9649977526660647e-01,
            5.8279442137494575e-01, -5.6994253522747218e-01,
            5.5790401373575793e-01, -5.4545052833277607e-01,
            5.3336442347695479e-01, -5.2069797861987954e-01,
            5.0887851227988856e-01, -4.9685712521558462e-01,
            4.8646743314063406e-01, -4.7619096686921952e-01,
            4.6816529269778301e-01, -4.6015156373071614e-01,
            4.5466141540832178e-01, -4.4857363825706975e-01,
            4.4499201335812283e-01, -4.3980181069680696e-01,
            4.3704907382079500e-01, -4.3150722634129624e-01,
            4.2859194645669874e-01, -4.2173773973311074e-01,
            4.1817784119266610e-01, -4.0960162465920091e-01,
            4.0553804034464430e-01, -3.9532815510851277e-01,
            3.9136208758411489e-01, -3.7983009113722693e-01,
            3.7671762005172676e-01, -3.6406330003120191e-01,
            3.6244935256730459e-01, -3.4854450330689657e-01,
            3.4888232036420985e-01, -3.3325449310423266e-01,
            3.3591600342224537e-01, -3.1786716813473143e-01,
            3.2333450491673743e-01, -3.0206709086775535e-01,
            3.1108664637173294e-01, -2.8574426655786522e-01,
            2.9938562076378405e-01, -2.6899179718067823e-01,
            2.8862849386853856e-01, -2.5197407940080729e-01,
            2.7924443578937680e-01, -2.3479292380166417e-01,
            2.7158086404171844e-01, -2.1743177724854762e-01,
            2.6585734716833231e-01, -1.9977886402875333e-01,
            2.6215186968647208e-01, -1.8169913895160320e-01,
            2.6037686710257568e-01, -1.6314391449016330e-01,
            2.6022362236599678e-01, -1.4430969552882361e-01,
            2.6106112640293244e-01, -1.2585051878275111e-01,
            2.6176557155826263e-01, -1.0912157374630198e-01,
            2.6047541087804182e-01, -9.6410349758274366e-02,
            2.5434822704405619e-01, -9.1058907636613021e-02,
            2.3952590152331843e-01, -9.7233907394681532e-02,
            2.1169127037476246e-01, -1.1886530901829440e-01,
            1.6780342130647322e-01, -1.5708700322500274e-01,
            1.0962587924131692e-01, -2.0578176345887339e-01,
            4.8885661268589009e-02, -2.4642892106977957e-01,
            1.1194065592765716e-02, -2.4747581882772479e-01,
            3.0611266302904498e-02, -1.7906330306689583e-01,
            1.2284053143305627e-01, -5.2283741701101033e-02,
            2.3805370442772930e-01, 3.8699307341758679e-02,
            2.4690725918834208e-01, -3.8424035688006668e-02,
            7.1201119703791560e-02, -2.3889263981505893e-01,
            -7.7114690286621504e-02, -2.1836754494419724e-01,
            1.2315414896678636e-01, 7.5338270471735747e-02,
            2.7696240054976468e-01, -6.8197423407317256e-02,
            -9.2154357023181005e-02, -2.7364306064385413e-01,
            1.4776915895856893e-01, 1.5391397605873250e-01,
            1.1202666775507018e-01, -3.3501468763636766e-01,
            5.3448876160720617e-02, 1.8824613952112992e-01,
            -4.6814043305072224e-03, -3.0449707752313021e-01,
            3.7651614016005935e-01, -2.0743158205566595e-01,
            -1.1420479303107758e-02, 1.4728687125987736e-01,
            -1.8431611574819040e-01, 1.6270659034524407e-01,
            -1.2222143749935742e-01, 8.4181157909789264e-02,
            -5.5281360387327068e-02, 3.5292561498256686e-02,
            -2.1995586173467183e-02, 1.3230592337602392e-02,
            -7.4157047320195287e-03, 3.5088522519723777e-03,
            -8.5018538826118518e-04, -9.7550322159493529e-04,
            2.2326305963338822e-03, -3.0933635050624424e-03,
            3.6729552029512069e-03, -4.0509173928643906e-03,
            4.2837535729585508e-03, -4.4126872566431202e-03,
            4.4684173716055568e-03, -4.4741053242745125e-03,
            4.4473069057176040e-03, -4.4012645981419867e-03,
            4.3457940693551285e-03, -4.2878952176510369e-03,
            4.2321726027587606e-03, -4.1811395672186466e-03,
            4.1354762911556932e-03, -4.0942937154137878e-03,
            4.0554223224550864e-03, -4.0157136359124266e-03,
            3.9713297394453244e-03, -3.9180044788809047e-03,
            3.8512768724064841e-03, -3.7667091650295829e-03,
            3.6601079883520934e-03, -3.5277760443730628e-03,
            3.3668346001750932e-03, -3.1756548702066185e-03,
            2.9543911277564146e-03, -2.7055140265840505e-03,
            2.4341413354852411e-03, -2.1479350621587340e-03,
            1.8564376110443456e-03, -1.5699285271508928e-03,
            1.2980859369581127e-03, -1.0488156933979571e-03,
            8.2752974192331252e-04, -6.3697315746016306e-04,
            4.7751718968428873e-04, -3.4772734444372060e-04,
            2.4499928240303252e-04, -1.6610663484386119e-04,
            1.0758513633705277e-04, -6.5954676827986505e-05,
            3.7834534141493396e-05, -2.0025414475625305e-05,
            9.6121169858568779e-06, -4.0939692455506115e-06,
            1.5034842918522733e-06, -4.5718370169166360e-07,
            1.0805490301555136e-07, -1.7684735514194769e-08,
            1.5081394131412287e-09])

    return dlf


def key_101_2009():
    r"""Key 101 pt Hankel filter, as published in [Key09]_.

    Taken from file `FilterModules.f90` provided with 1DCSEM_.

    License: `Apache License, Version 2.0,
    <https://www.apache.org/licenses/LICENSE-2.0>`_.

    """

    dlf = DigitalFilter('Key 101 (2009)', 'key_101_2009')

    dlf.base = np.array([
        3.182780796509667e-03, 3.570677233218250e-03, 4.005847942090417e-03,
        4.494054401183452e-03, 5.041760259690979e-03, 5.656216913953108e-03,
        6.345559512909110e-03, 7.118914664064660e-03, 7.986521265955502e-03,
        8.959866066878890e-03, 1.005183574463358e-02, 1.127688752074056e-02,
        1.265124056800531e-02, 1.419309074557768e-02, 1.592285150451168e-02,
        1.786342415331403e-02, 2.004050106168401e-02, 2.248290581673552e-02,
        2.522297483522721e-02, 2.829698548418685e-02, 3.174563637806794e-02,
        3.561458621137186e-02, 3.995505826065390e-02, 4.482451855926687e-02,
        5.028743672359186e-02, 5.641613950377735e-02, 6.329176835964070e-02,
        7.100535373963698e-02, 7.965902028589801e-02, 8.936733892175319e-02,
        1.002588437228037e-01, 1.124777336542896e-01, 1.261857817050387e-01,
        1.415644766941340e-01, 1.588174261069207e-01, 1.781730517728984e-01,
        1.998876140751445e-01, 2.242486047305353e-01, 2.515785530597565e-01,
        2.822392961405233e-01, 3.166367693790532e-01, 3.552263809249515e-01,
        3.985190410845142e-01, 4.470879265593564e-01, 5.015760690660555e-01,
        5.627048688069557e-01, 6.312836455069260e-01, 7.082203534678000e-01,
        7.945336025033340e-01, 8.913661439068313e-01, 1.000000000000000e+00,
        1.121873437571938e+00, 1.258600009929478e+00, 1.411989919667659e+00,
        1.584073984994482e+00, 1.777130526914038e+00, 1.993715533243082e+00,
        2.236696498819987e+00, 2.509290389936297e+00, 2.815106235624064e+00,
        3.158192909689768e+00, 3.543092736108982e+00, 3.974901627494749e+00,
        4.459336552847826e+00, 5.002811227833588e+00, 5.612521029693157e+00,
        6.296538261026657e+00, 7.063919023701211e+00, 7.924823117849490e+00,
        8.890648553371371e+00, 9.974182454814724e+00, 1.118977035755271e+01,
        1.255350613666823e+01, 1.408344508312441e+01, 1.579984294826040e+01,
        1.772542412146164e+01, 1.988568249156473e+01, 2.230921897527583e+01,
        2.502812018133782e+01, 2.807838322380105e+01, 3.150039230874794e+01,
        3.533945340427969e+01, 3.964639407257260e+01, 4.447823640552875e+01,
        4.989895197340787e+01, 5.598030878164415e+01, 6.280282144920171e+01,
        7.045681718843254e+01, 7.904363169956446e+01, 8.867695081296063e+01,
        9.948431564193386e+01, 1.116088111737080e+02, 1.252109606547652e+02,
        1.404708508514462e+02, 1.575905163233670e+02, 1.767966142764323e+02,
        1.983434254093812e+02, 2.225162204838159e+02, 2.496350371896939e+02,
        2.800589173104007e+02, 3.141906602856942e+02])

    dlf.factor = np.array([1.1218734375719379])

    dlf.j0 = np.array([
        1.761499629714979e+00, -1.261104080802587e+01, 4.589440735380221e+01,
        -1.137201615271146e+02, 2.171373845748458e+02, -3.427418757004401e+02,
        4.682970092296048e+02, -5.721810218704758e+02, 6.404705872914975e+02,
        -6.689903305418517e+02, 6.614840943418809e+02, -6.261458042620394e+02,
        5.724742839763362e+02, -5.091188391804840e+02, 4.429121852279642e+02,
        -3.786064826795332e+02, 3.191588258655895e+02, -2.660734942256455e+02,
        2.198909349916423e+02, -1.804655347160418e+02, 1.473251045730838e+02,
        -1.197617622162473e+02, 9.706708172663220e+01, -7.848237715099181e+01,
        6.337970122621439e+01, -5.112377096515730e+01, 4.125319165745542e+01,
        -3.328022871692455e+01, 2.690546850964645e+01, -2.176077775919447e+01,
        1.767900438275361e+01, -1.437298500741315e+01, 1.177821110514306e+01,
        -9.653214292533669e+00, 8.014188621856462e+00, -6.636716121370467e+00,
        5.605112419659892e+00, -4.690136885102554e+00, 4.040369173672801e+00,
        -3.404103887583860e+00, 2.995747642598741e+00, -2.524176278751386e+00,
        2.274303867444810e+00, -1.898578152750098e+00, 1.759903177266298e+00,
        -1.438255316175359e+00, 1.383718859899911e+00, -1.090411693791455e+00,
        1.103187499775871e+00, -8.232645366851894e-01, 8.897253309567960e-01,
        -6.181710614766990e-01, 7.214321035775552e-01, -4.660523646863802e-01,
        5.784691305884657e-01, -3.658095952962029e-01, 4.407748104824833e-01,
        -3.217298446491200e-01, 2.909042759264825e-01, -3.338253349563184e-01,
        1.293953529828388e-01, -3.718639185374572e-01, 8.698186744359777e-03,
        -3.389828291018994e-01, 4.963953772877858e-02, -1.147290346722791e-01,
        2.818501136331604e-01, 1.360129862128020e-01, 2.683528605579231e-01,
        -1.505769588191184e-01, -2.031064772010313e-01, -2.437687529694006e-01,
        2.876770799753536e-01, 2.011329588874316e-01, -2.275463527816749e-01,
        -1.749917271598988e-01, 4.306671898041470e-01, -3.992878212851492e-01,
        2.550550974806911e-01, -1.336748872779371e-01, 6.423008702535922e-02,
        -3.108808914792973e-02, 1.639821648563433e-02, -9.842274279418138e-03,
        6.685502920904526e-03, -4.955006305606947e-03, 3.851240446211632e-03,
        -3.047184325211406e-03, 2.405935290382968e-03, -1.869259018281964e-03,
        1.412821020240033e-03, -1.027340635931714e-03, 7.099124273512882e-04,
        -4.593396038963651e-04, 2.731571829548821e-04, -1.457155899557221e-04,
        6.748422483369748e-05, -2.590402870915282e-05, 7.676168023731497e-06,
        -1.549433571510781e-06, 1.584912057504873e-07])

    dlf.j1 = np.array([
        3.382421751388876e-04, -2.080089427136974e-03, 6.616516890510206e-03,
        -1.444442106541980e-02, 2.445603365138576e-02, -3.439020151920317e-02,
        4.205277853503258e-02, -4.616883971957859e-02, 4.665044563723574e-02,
        -4.417942425707860e-02, 3.983124989286448e-02, -3.456241855974430e-02,
        2.919015247012813e-02, -2.413480064473489e-02, 1.973163931127073e-02,
        -1.595784764119277e-02, 1.292753867241687e-02, -1.040069021237603e-02,
        8.496859564365926e-03, -6.857608112687733e-03, 5.734515396281575e-03,
        -4.635710275375116e-03, 4.034369992101164e-03, -3.217728935403965e-03,
        2.990802607299357e-03, -2.269398404454681e-03, 2.363714769044985e-03,
        -1.583785021876379e-03, 2.025371816097564e-03, -1.027952174606392e-03,
        1.922153595502174e-03, -5.043945522132147e-04, 2.054567905217619e-03,
        7.633835394740574e-05, 2.471902278652444e-03, 8.190345324336234e-04,
        3.277371965548541e-03, 1.870779100759767e-03, 4.642867014031185e-03,
        3.454727180711420e-03, 6.839165720545689e-03, 5.911273173907802e-03,
        1.028566504388743e-02, 9.752018620581887e-03, 1.561636209584471e-02,
        1.572789508971819e-02, 2.375132082749520e-02, 2.489168113861482e-02,
        3.593610268802203e-02, 3.858602166700044e-02, 5.363612015669736e-02,
        5.817318184012853e-02, 7.800216974165830e-02, 8.408281670359681e-02,
        1.083042686943279e-01, 1.133712057428050e-01, 1.383564865699443e-01,
        1.348125772346644e-01, 1.504073961323417e-01, 1.224262032649067e-01,
        1.106446266767018e-01, 3.772825296930285e-02, -1.353204196275998e-02,
        -1.266046406313744e-01, -1.730074119319237e-01, -2.327661941998823e-01,
        -1.312677137162940e-01, -1.104509755532312e-02, 2.294237929047981e-01,
        2.327988918717147e-01, 9.149367519642598e-02, -3.004964869169849e-01,
        -1.797040516746428e-01, 1.817308988818578e-01, 2.935346562382805e-01,
        -4.502465801331593e-01, 2.151483212376926e-01, 4.924585434461976e-02,
        -1.679842097056792e-01, 1.754997616383300e-01, -1.428351980172456e-01,
        1.076274808049898e-01, -8.030885482867917e-02, 6.079630210587774e-02,
        -4.687267027003955e-02, 3.664115351116358e-02, -2.884253032421485e-02,
        2.269535305564210e-02, -1.772060472957579e-02, 1.362455706705945e-02,
        -1.022825775681532e-02, 7.424720522409015e-03, -5.150709666353805e-03,
        3.365543012321504e-03, -2.033399376196629e-03, 1.108895838202336e-03,
        -5.283266194475448e-04, 2.099779511098949e-04, -6.484566005000160e-05,
        1.372984806560325e-05, -1.482645851470641e-06])

    return dlf


def key_201_2009():
    r"""Key 201 pt Hankel filter, as published in [Key09]_.

    Taken from file `FilterModules.f90` provided with 1DCSEM_.

    License: `Apache License, Version 2.0,
    <https://www.apache.org/licenses/LICENSE-2.0>`_.

    """

    dlf = DigitalFilter('Key 201 (2009)', 'key_201_2009')

    dlf.base = np.array([
        6.112527611295728e-04, 6.582011330626792e-04, 7.087554594672159e-04,
        7.631927021868981e-04, 8.218110956199024e-04, 8.849317805958155e-04,
        9.529005637454616e-04, 1.026089812002297e-03, 1.104900492614410e-03,
        1.189764369843323e-03, 1.281146370384213e-03, 1.379547130466501e-03,
        1.485505738589109e-03, 1.599602688916441e-03, 1.722463061515276e-03,
        1.854759946855503e-03, 1.997218133335814e-03, 2.150618078036461e-03,
        2.315800182452862e-03, 2.493669396634629e-03, 2.685200176953821e-03,
        2.891441824663527e-03, 3.113524234494096e-03, 3.352664084780651e-03,
        3.610171503034560e-03, 3.887457243476130e-03, 4.186040415850672e-03,
        4.507556807870229e-03, 4.853767846875483e-03, 5.226570249814254e-03,
        5.628006414404065e-03, 6.060275608406676e-03, 6.525746018315052e-03,
        7.026967723461506e-03, 7.566686666625634e-03, 8.147859697679989e-03,
        8.773670772690183e-03, 9.447548397216066e-03, 1.017318440937716e-02,
        1.095455420558538e-02, 1.179593851975157e-02, 1.270194687528350e-02,
        1.367754283835671e-02, 1.472807121080859e-02, 1.585928731163164e-02,
        1.707738850748481e-02, 1.838904816496257e-02, 1.980145221062947e-02,
        2.132233849911398e-02, 2.296003920493998e-02, 2.472352647033940e-02,
        2.662246155912740e-02, 2.866724778592989e-02, 3.086908751073575e-02,
        3.324004351101861e-02, 3.579310506765532e-02, 3.854225912669247e-02,
        4.150256692682124e-02, 4.469024651236346e-02, 4.812276158381668e-02,
        5.181891717272583e-02, 5.579896266503613e-02, 6.008470273734039e-02,
        6.469961681378547e-02, 6.966898769808187e-02, 7.502004008532698e-02,
        8.078208971247929e-02, 8.698670396460390e-02, 9.366787481677047e-02,
        1.008622050590664e-01, 1.086091088249580e-01, 1.169510275215943e-01,
        1.259336623450285e-01, 1.356062246541897e-01, 1.460217055752807e-01,
        1.572371663136276e-01, 1.693140507634552e-01, 1.823185221282210e-01,
        1.963218253956815e-01, 2.114006776535105e-01, 2.276376883838127e-01,
        2.451218120391174e-01, 2.639488353792869e-01, 2.842219022392174e-01,
        3.060520786022707e-01, 3.295589610751891e-01, 3.548713320980274e-01,
        3.821278654786651e-01, 4.114778861171706e-01, 4.430821880821678e-01,
        4.771139155210344e-01, 5.137595112299984e-01, 5.532197380808739e-01,
        5.957107789003212e-01, 6.414654208273198e-01, 6.907343306373547e-01,
        7.437874280201796e-01, 8.009153643346592e-01, 8.624311149420454e-01,
        9.286716938412872e-01, 1.000000000000000e+00, 1.076806805496220e+00,
        1.159512896362974e+00, 1.248571377864284e+00, 1.344470156832053e+00,
        1.447734614663324e+00, 1.558930485621915e+00, 1.678666956213205e+00,
        1.807600002612004e+00, 1.946435984427591e+00, 2.095935514494364e+00,
        2.256917625888753e+00, 2.430264259001381e+00, 2.616925093246914e+00,
        2.817922749882108e+00, 3.034358394435675e+00, 3.267417769442918e+00,
        3.518377690535413e+00, 3.788613041474606e+00, 4.079604306451588e+00,
        4.392945680918757e+00, 4.730353805388542e+00, 5.093677170047323e+00,
        5.484906241707685e+00, 5.906184368579528e+00, 6.359819522601831e+00,
        6.848296943665372e+00, 7.374292754997836e+00, 7.940688624303139e+00,
        8.550587550976035e+00, 9.207330865882248e+00, 9.914516536837411e+00,
        1.067601888007134e+01, 1.149600978566695e+01, 1.237898157325731e+01,
        1.332977160319577e+01, 1.435358877803146e+01, 1.545604207947845e+01,
        1.664317129721834e+01, 1.792148011788406e+01, 1.929797175550276e+01,
        2.078018731859920e+01, 2.237624712415386e+01, 2.409489518475410e+01,
        2.594554711266131e+01, 2.793834170323650e+01, 3.008419648032392e+01,
        3.239486750789821e+01, 3.488301379565316e+01, 3.756226665137786e+01,
        4.044730436006738e+01, 4.355393259889750e+01, 4.689917102861648e+01,
        5.050134653574537e+01, 5.438019363641357e+01, 5.855696259189233e+01,
        6.305453582813728e+01, 6.789755329714343e+01, 7.311254746690634e+01,
        7.872808867953015e+01, 8.477494167382795e+01, 9.128623412992303e+01,
        9.829763815922249e+01, 1.058475657340557e+02, 1.139773791276396e+02,
        1.227316175172651e+02, 1.321582409921502e+02, 1.423088933027569e+02,
        1.532391847910440e+02, 1.650089970516890e+02, 1.776828109933644e+02,
        1.913300600973533e+02, 2.060255108088307e+02, 2.218496721447841e+02,
        2.388892367626087e+02, 2.572375559057747e+02, 2.769951508285525e+02,
        2.982702635016372e+02, 3.211794496157137e+02, 3.458482171317310e+02,
        3.724117138761822e+02, 4.010154679483840e+02, 4.318161849960710e+02,
        4.649826067271839e+02, 5.006964353612040e+02, 5.391533290846429e+02,
        5.805639739642864e+02, 6.251552381906738e+02, 6.731714149753277e+02,
        7.248755609109528e+02, 7.805509371268035e+02, 8.405025611345947e+02,
        9.050588778667341e+02, 9.745735590616712e+02, 1.049427440854279e+03,
        1.130030610186370e+03, 1.216824651467729e+03, 1.310285065796017e+03,
        1.410923875989213e+03, 1.519292431702289e+03, 1.635984429995926e+03])

    dlf.factor = np.array([1.0768068054962201])

    dlf.j0 = np.array([
        1.104702818216325e-01, -3.002860174042879e-01, 0.000000000000000e+00,
        9.304611998317738e-01, -1.237989456789895e+00, 0.000000000000000e+00,
        1.522782496923861e+00, -1.481262207187227e+00, 0.000000000000000e+00,
        1.200938682543965e+00, -1.042356281617017e+00, 0.000000000000000e+00,
        8.186482154233182e-01, -7.837979439107681e-01, 0.000000000000000e+00,
        1.072087935017888e+00, -2.017973482744253e+00, 2.638551914228436e+00,
        -2.916973810417140e+00, 2.932547717791041e+00, -2.782030176901836e+00,
        2.548634287940637e+00, -2.286132540834447e+00, 2.028668029433413e+00,
        -1.790466549754032e+00, 1.579778206432123e+00, -1.395100794535786e+00,
        1.236899777150533e+00, -1.099633354001948e+00, 9.829180606149369e-01,
        -8.808127495830721e-01, 7.939417322911940e-01, -7.166399975760631e-01,
        6.508052453577512e-01, -5.909087763650980e-01, 5.400346833052113e-01,
        -4.925062292020488e-01, 4.525012829058494e-01, -4.139282151279450e-01,
        3.820303825432799e-01, -3.500694920581675e-01, 3.243907465091445e-01,
        -2.973991010648227e-01, 2.766317890637754e-01, -2.534315625567141e-01,
        2.366617712595763e-01, -2.163875214294563e-01, 2.029716420595088e-01,
        -1.849663778885546e-01, 1.744470283910267e-01, -1.581901786171230e-01,
        1.502386381629129e-01, -1.352956168139171e-01, 1.296732438173242e-01,
        -1.156619860385481e-01, 1.121981827706036e-01, -9.877037098650596e-02,
        9.735307747326948e-02, -8.418324552747586e-02, 8.475347956447533e-02,
        -7.152829800825916e-02, 7.407480907190452e-02, -6.048336872780390e-02,
        6.504162604072715e-02, -5.077074526100943e-02, 5.742706238192050e-02,
        -4.215804851336973e-02, 5.105276831205472e-02, -3.445402816016690e-02,
        4.578139059809960e-02, -2.749842403920171e-02, 4.150734318146184e-02,
        -2.115449634951816e-02, 3.815224456264026e-02, -1.530500719575409e-02,
        3.566006962242232e-02, -9.844572094042197e-03, 3.398735588004622e-02,
        -4.669634342048067e-03, 3.309505748308261e-02, 3.248775057265341e-04,
        3.294860285661111e-02, 5.240451270870069e-03, 3.352191931272241e-02,
        1.016839266998432e-02, 3.479748010810809e-02, 1.518640694860538e-02,
        3.675989077470884e-02, 2.035623099573585e-02, 3.938537294158406e-02,
        2.571720552065411e-02, 4.262879025301092e-02, 3.127265047233792e-02,
        4.640586621922704e-02, 3.696789312241264e-02, 5.056664152720505e-02,
        4.265736230160079e-02, 5.485681544583107e-02, 4.805697373905592e-02,
        5.886326726197704e-02, 5.267934237797620e-02, 6.194081754102766e-02,
        5.575263672089661e-02, 6.312348308951768e-02, 5.613171205140025e-02,
        6.103877063939782e-02, 5.223073389072554e-02, 5.387340754058757e-02,
        4.205003959253396e-02, 3.949537291826706e-02, 2.344464896766128e-02,
        1.592975900866907e-02, -5.112454956546657e-03, -1.750424757311479e-02,
        -4.293028583599693e-02, -5.806536835195215e-02, -8.443837687273714e-02,
        -9.651641994648259e-02, -1.160353562902818e-01, -1.145716151033802e-01,
        -1.152746912280352e-01, -8.753115163764941e-02, -5.917367601312370e-02,
        -6.660378399098311e-04, 4.966494483361900e-02, 1.157206357429209e-01,
        1.448966611107768e-01, 1.588998330585965e-01, 1.031590212340854e-01,
        2.389465737393602e-02, -1.007600802615937e-01, -1.701081679147513e-01,
        -1.806596339730873e-01, -5.044220811489127e-02, 1.067558829070489e-01,
        2.243843205181916e-01, 1.124861192136151e-01, -1.060750025290042e-01,
        -2.494689576049384e-01, -2.532563907981785e-02, 2.313093162590108e-01,
        1.265041517428334e-01, -2.716895564501260e-01, -5.741673301698751e-02,
        2.828355798553848e-01, -1.151529184187685e-01, -1.895238214387882e-01,
        3.498981818372285e-01, -3.283594487966644e-01, 2.228689935519562e-01,
        -1.191882703384384e-01, 4.914136861200422e-02, -1.085046330669215e-02,
        -6.989673166059397e-03, 1.396225098602933e-02, -1.583247654053119e-02,
        1.553798887482501e-02, -1.443534141636538e-02, 1.310934487425667e-02,
        -1.179562561923210e-02, 1.057909167982074e-02, -9.482129155265809e-03,
        8.502636346417821e-03, -7.630076138978968e-03, 6.852043148244981e-03,
        -6.156761038065365e-03, 5.533849408416496e-03, -4.974383010461074e-03,
        4.470699747473056e-03, -4.016161870832697e-03, 3.604955260613827e-03,
        -3.231951863754572e-03, 2.892630190613161e-03, -2.583034645430386e-03,
        2.299750969077188e-03, -2.039879838855092e-03, 1.801000017171617e-03,
        -1.581121784044337e-03, 1.378637896797070e-03, -1.192281463551114e-03,
        1.021096380613436e-03, -8.644169540225837e-04, 7.218438967094129e-04,
        -5.932006323351060e-04, 4.784596555937276e-04, -3.776406369869407e-04,
        2.906937486164167e-04, -2.173880970439557e-04, 1.572247021446439e-04,
        -1.093880480994978e-04, 7.274278828090175e-05, -4.587508579334320e-05,
        2.717284254234203e-05, -1.493569664311702e-05, 7.502072444202401e-06,
        -3.374980009202976e-06, 1.323028566570774e-06, -4.342813347631881e-07,
        1.120529389225740e-07, -2.023679217396607e-08, 1.923133952677995e-09])

    dlf.j1 = np.array([
        1.289633927144360e-05, -4.692852957012775e-05, 5.712407500240781e-05,
        0.000000000000000e+00, -5.401898356504400e-05, 0.000000000000000e+00,
        1.163813605855986e-04, -1.341585158986474e-04, 0.000000000000000e+00,
        1.563529882751311e-04, -1.701932285008713e-04, 0.000000000000000e+00,
        2.685212722282496e-04, -5.148623386148907e-04, 6.653519984942213e-04,
        -7.072232255490351e-04, 6.684047762347991e-04, -5.847964888682289e-04,
        4.877023048974497e-04, -3.939313465485314e-04, 3.136345180982827e-04,
        -2.470064062519923e-04, 1.953997902093073e-04, -1.539602296158165e-04,
        1.234529832565310e-04, -9.817008106125012e-05, 8.058804362211881e-05,
        -6.451459975546781e-05, 5.456430430893415e-05, -4.344419043404446e-05,
        3.815523049837875e-05, -2.949393655677912e-05, 2.732187176767718e-05,
        -1.959842972127203e-05, 1.982911726363689e-05, -1.200904791302960e-05,
        1.443251606646564e-05, -5.683761857979987e-06, 1.043594555177340e-05,
        4.513240229146117e-08, 7.457609318727016e-06, 5.653216751529018e-06,
        5.308688404222969e-06, 1.152796570665543e-05, 3.932961600888582e-06,
        1.803352844385906e-05, 3.380683574927984e-06, 2.555716029426899e-05,
        3.803729181402945e-06, 3.454895491783953e-05, 5.466139820523614e-06,
        4.556234770039426e-05, 8.768101084525483e-06, 5.930097379335330e-05,
        1.428416302046287e-05, 7.667672615961499e-05, 2.281866040881824e-05,
        9.888436530447273e-05, 3.548288867698162e-05, 1.274995405373852e-04,
        5.380041779170104e-05, 1.646090070216299e-04, 7.984956415148433e-05,
        2.129842798716441e-04, 1.164554629053559e-04, 2.763133928114338e-04,
        1.674485387417859e-04, 3.595100112861982e-04, 2.380118276339850e-04,
        4.691253417077570e-04, 3.351463657538389e-04, 6.138970952511212e-04,
        4.682919161509869e-04, 8.054805703670692e-04, 6.501519986304682e-04,
        1.059417913885959e-03, 8.977881651498095e-04, 1.396414494631849e-03,
        1.234064772364374e-03, 1.844009097024346e-03, 1.689538327387063e-03,
        2.438744274863229e-03, 2.304895516259244e-03, 3.228952488337877e-03,
        3.134049181497694e-03, 4.278260457055239e-03, 4.247981356211152e-03,
        5.669864127830826e-03, 5.739341687299852e-03, 7.511500876566557e-03,
        7.727614768189288e-03, 9.940768446104526e-03, 1.036426807004136e-02,
        1.312987922331795e-02, 1.383652144093020e-02, 1.728788292640136e-02,
        1.836695924406547e-02, 2.265651442167874e-02, 2.420372071766719e-02,
        2.949262366900291e-02, 3.159191631258840e-02, 3.802499854417571e-02,
        4.071065391868849e-02, 4.836588871048378e-02, 5.155150426795294e-02,
        6.034835565661112e-02, 6.370530818484274e-02, 7.325374787602650e-02,
        7.602246037354336e-02, 8.540182884134039e-02, 8.613756797984258e-02,
        9.363045212001810e-02, 8.994780552031496e-02, 9.285438346694046e-02,
        8.138538169597652e-02, 7.625701418518774e-02, 5.331840281941071e-02,
        3.728410637826844e-02, 1.088918303862871e-03, -2.482185562667677e-02,
        -6.966985038573692e-02, -9.508370436510773e-02, -1.320457796877950e-01,
        -1.333188913518803e-01, -1.345679831656020e-01, -8.403605511993056e-02,
        -3.270153309655380e-02, 6.374700977990792e-02, 1.243720031600276e-01,
        1.818967705583269e-01, 1.375740652175917e-01, 5.811873304105276e-02,
        -1.043087758627017e-01, -1.815595823147628e-01, -1.819917129979687e-01,
        1.631402693428312e-02, 1.752375632474410e-01, 2.152308945759099e-01,
        -6.135834728025826e-02, -2.260012732339317e-01, -1.058818826360049e-01,
        2.751720328057493e-01, 7.282078436121274e-02, -2.453904197631155e-01,
        -2.981552944151773e-02, 3.295321158355015e-01, -3.395688755289581e-01,
        1.437347757121746e-01, 5.569137331864311e-02, -1.618975349034848e-01,
        1.840881618330268e-01, -1.638475650685530e-01, 1.323098979187242e-01,
        -1.035844311836129e-01, 8.132823321716304e-02, -6.502108525281801e-02,
        5.316632459343718e-02, -4.441483689628730e-02, 3.779017256454915e-02,
        -3.263631595413427e-02, 2.852104339616387e-02, -2.515785440611284e-02,
        2.235339482483744e-02, -1.997416046894760e-02, 1.792574037444691e-02,
        -1.613980618851269e-02, 1.456582802493779e-02, -1.316569266298338e-02,
        1.191013283575226e-02, -1.077631048875747e-02, 9.746151537603939e-03,
        -8.805179856684300e-03, 7.941688885035929e-03, -7.146145811536250e-03,
        6.410759357461669e-03, -5.729165010828318e-03, 5.096195694120956e-03,
        -4.507714364593528e-03, 3.960490050623528e-03, -3.452101711144719e-03,
        2.980856037463172e-03, -2.545706551739226e-03, 2.146162475389078e-03,
        -1.782177147988815e-03, 1.454007747414243e-03, -1.162041366288416e-03,
        9.065878213110874e-04, -6.876473801626548e-04, 5.046718335603190e-04,
        -3.563489117729183e-04, 2.404501739881472e-04, -1.537865721775565e-04,
        9.230834105005988e-05, -5.136252914208434e-05, 2.608318212185880e-05,
        -1.184499677743133e-05, 4.678587718668116e-06, -1.544064223784288e-06,
        3.995921803021977e-07, -7.219416019819105e-08, 6.844864105885603e-09])

    return dlf


def key_401_2009():
    r"""Key 401 pt Hankel filter, as published in [Key09]_.

    Taken from file `FilterModules.f90` provided with 1DCSEM_.

    License: `Apache License, Version 2.0,
    <https://www.apache.org/licenses/LICENSE-2.0>`_.

    """

    dlf = DigitalFilter('Key 401 (2009)', 'key_401_2009')

    dlf.base = np.array([
        6.825603376334870e-08, 7.375625734276621e-08, 7.969970121723440e-08,
        8.612208106759941e-08, 9.306199062400369e-08, 1.005611335855232e-07,
        1.086645742284082e-07, 1.174210082088922e-07, 1.268830551878997e-07,
        1.371075750361032e-07, 1.481560095194920e-07, 1.600947515187244e-07,
        1.729955440010014e-07, 1.869359111419852e-07, 2.019996241884814e-07,
        2.182772048613791e-07, 2.358664693239236e-07, 2.548731159841640e-07,
        2.754113606638370e-07, 2.976046229505788e-07, 3.215862678579245e-07,
        3.475004072499321e-07, 3.755027658463905e-07, 4.057616170126568e-07,
        4.384587939575313e-07, 4.737907824157171e-07, 5.119699013810637e-07,
        5.532255789859379e-07, 5.978057311938045e-07, 6.459782515899233e-07,
        6.980326212227157e-07, 7.542816481697429e-07, 8.150633472817832e-07,
        8.807429714008947e-07, 9.517152062585649e-07, 1.028406542243639e-06,
        1.111277837292621e-06, 1.200827086303379e-06, 1.297592413714413e-06,
        1.402155307232813e-06, 1.515144112143249e-06, 1.637237807196195e-06,
        1.769170084765623e-06, 1.911733759794935e-06, 2.065785534025608e-06,
        2.232251144137991e-06, 2.412130924740802e-06, 2.606505819638747e-06,
        2.816543877501467e-06, 3.043507270968020e-06, 3.288759881366484e-06,
        3.553775494627146e-06, 3.840146657640719e-06, 4.149594248281688e-06,
        4.483977816605428e-06, 4.845306759362147e-06, 5.235752394978102e-06,
        5.657661011565692e-06, 6.113567966371402e-06, 6.606212921388751e-06,
        7.138556306690845e-06, 7.713797110415130e-06, 8.335392102304759e-06,
        9.007076606325931e-06, 9.732886947188948e-06, 1.051718470566020e-05,
        1.136468292842132e-05, 1.228047444997721e-05, 1.327006249680666e-05,
        1.433939375766396e-05, 1.549489411875887e-05, 1.674350727855744e-05,
        1.809273647424976e-05, 1.955068957062923e-05, 2.112612778233479e-05,
        2.282851832224015e-05, 2.466809129236741e-05, 2.665590115919790e-05,
        2.880389318280080e-05, 3.112497519896879e-05, 3.363309518571897e-05,
        3.634332508027547e-05, 3.927195135021125e-05, 4.243657286301518e-05,
        4.585620664220731e-05, 4.955140214551744e-05, 5.354436475185514e-05,
        5.785908919913498e-05, 6.252150377482026e-05, 6.755962612566159e-05,
        7.300373162293296e-05, 7.888653529491450e-05, 8.524338841989974e-05,
        9.211249096110730e-05, 9.953512112007229e-05, 1.075558833879609e-04,
        1.162229765854152e-04, 1.255884835016470e-04, 1.357086838732945e-04,
        1.466443925838176e-04, 1.584613251157513e-04, 1.712304924519203e-04,
        1.850286277986747e-04, 1.999386476954356e-04, 2.160501502814794e-04,
        2.334599537141681e-04, 2.522726779741279e-04, 2.726013735535827e-04,
        2.945682008058004e-04, 3.183051640380327e-04, 3.439549047593057e-04,
        3.716715588498817e-04, 4.016216828033581e-04, 4.339852546074165e-04,
        4.689567552777785e-04, 5.067463375445843e-04, 5.475810887141261e-04,
        5.917063952947988e-04, 6.393874175876612e-04, 6.909106831027882e-04,
        7.465858083766792e-04, 8.067473595375511e-04, 8.717568627991351e-04,
        9.420049769645596e-04, 1.017913840995439e-03, 1.099939610753318e-03,
        1.188575200157420e-03, 1.284353243230985e-03, 1.387849294835929e-03,
        1.499685289329845e-03, 1.620533277929307e-03, 1.751119467238237e-03,
        1.892228583209940e-03, 2.044708586766894e-03, 2.209475769415756e-03,
        2.387520259478373e-03, 2.579911972027180e-03, 2.787807038279695e-03,
        3.012454753087960e-03, 3.255205082272187e-03, 3.517516774912128e-03,
        3.800966129344984e-03, 4.107256465546973e-03, 4.438228360820604e-03,
        4.795870710296421e-03, 5.182332678714725e-03, 5.599936615308509e-03,
        6.051192009396498e-03, 6.538810570549064e-03, 7.065722523947535e-03,
        7.635094218859962e-03, 8.250347156047146e-03, 8.915178548439553e-03,
        9.633583538639471e-03, 1.040987920675908e-02, 1.124873051286366e-02,
        1.215517832991493e-02, 1.313466973567140e-02, 1.419309074557768e-02,
        1.533680168334325e-02, 1.657267540176125e-02, 1.790813858344623e-02,
        1.935121636967760e-02, 2.091058058553484e-02, 2.259560185112186e-02,
        2.441640589203004e-02, 2.638393438742415e-02, 2.851001072140295e-02,
        3.080741103275108e-02, 3.328994099003863e-02, 3.597251875342965e-02,
        3.887126462173843e-02, 4.200359790344555e-02, 4.538834159379689e-02,
        4.904583548701673e-02, 5.299805840335580e-02, 5.726876026546736e-02,
        6.188360481779279e-02, 6.687032384659380e-02, 7.225888382737922e-02,
        7.808166600115317e-02, 8.437366096160970e-02, 9.117267892259785e-02,
        9.851957692940828e-02, 1.064585043792528e-01, 1.150371683263324e-01,
        1.243071201657794e-01, 1.343240654192323e-01, 1.451481984836237e-01,
        1.568445643547723e-01, 1.694834494994701e-01, 1.831408042249150e-01,
        1.978986990836147e-01, 2.138458180564171e-01, 2.310779914773302e-01,
        2.496987719026136e-01, 2.698200563846869e-01, 2.915627588902595e-01,
        3.150575369034133e-01, 3.404455765799855e-01, 3.678794411714423e-01,
        3.975239878166446e-01, 4.295573582107391e-01, 4.641720491043618e-01,
        5.015760690660556e-01, 5.419941884591870e-01, 5.856692901447937e-01,
        6.328638290270813e-01, 6.838614092123558e-01, 7.389684882589442e-01,
        7.985162187593771e-01, 8.628624383213754e-01, 9.323938199059482e-01,
        1.007528195444534e+00, 1.088717066698399e+00, 1.176448318448691e+00,
        1.271249150321405e+00, 1.373689244865351e+00, 1.484384190920914e+00,
        1.603999182851514e+00, 1.733253017867395e+00, 1.872922415462686e+00,
        2.023846684922348e+00, 2.186932768947246e+00, 2.363160693705795e+00,
        2.553589458062927e+00, 2.759363397376282e+00, 2.981719060101298e+00,
        3.221992638528500e+00, 3.481627998306172e+00, 3.762185354999911e+00,
        4.065350649828702e+00, 4.392945680918757e+00, 4.746939050956379e+00,
        5.129457997027160e+00, 5.542801173730021e+00, 5.989452466383113e+00,
        6.472095917328692e+00, 6.993631855032968e+00, 7.557194322904825e+00,
        8.166169912567650e+00, 8.824218114758272e+00, 9.535293310146759e+00,
        1.030366853222556e+01, 1.113396114506531e+01, 1.203116059024153e+01,
        1.300065836967064e+01, 1.404828044452976e+01, 1.518032224495389e+01,
        1.640358650089262e+01, 1.772542412146164e+01, 1.915377836844383e+01,
        2.069723258938951e+01, 2.236506179715657e+01, 2.416728840584508e+01,
        2.611474245805797e+01, 2.821912670540861e+01, 3.049308693336139e+01,
        3.295028795300461e+01, 3.560549571641010e+01, 3.847466604903214e+01,
        4.157504053236070e+01, 4.492525011301342e+01, 4.854542706087923e+01,
        5.245732594909905e+01, 5.668445438288377e+01, 6.125221426275198e+01,
        6.618805443107456e+01, 7.152163561921920e+01, 7.728500868650357e+01,
        8.351280722204120e+01, 9.024245566687466e+01, 9.751439420705401e+01,
        1.053723217891024e+02, 1.138634587182101e+02, 1.230388304171765e+02,
        1.329535740512827e+02, 1.436672698616796e+02, 1.552442991983601e+02,
        1.677542314042285e+02, 1.812722418751512e+02, 1.958795638082189e+02,
        2.116639763528940e+02, 2.287203320984661e+02, 2.471511270676237e+02,
        2.670671166413821e+02, 2.885879811166153e+02, 3.118430448957007e+02,
        3.369720536300714e+02, 3.641260139877280e+02, 3.934681010910957e+02,
        4.251746390782468e+02, 4.594361606799342e+02, 4.964585521797127e+02,
        5.364642906374987e+02, 5.796937808113659e+02, 6.264067998114887e+02,
        6.768840581675209e+02, 7.314288866902692e+02, 7.903690592644506e+02,
        8.540587625261516e+02, 9.228807242613036e+02, 9.972485133152526e+02,
        1.077609024834175e+03, 1.164445165772805e+03, 1.258278756806354e+03,
        1.359673668084992e+03, 1.469239207674401e+03, 1.587633783044448e+03,
        1.715568857608799e+03, 1.853813226091298e+03, 2.003197634410938e+03,
        2.164619771847478e+03, 2.339049665486888e+03, 2.527535509363266e+03,
        2.731209963326043e+03, 2.951296959483918e+03, 3.189119057127294e+03,
        3.446105390326745e+03, 3.723800255966753e+03, 4.023872393822313e+03,
        4.348125014444879e+03, 4.698506635117757e+03, 5.077122788996905e+03,
        5.486248677800499e+03, 5.928342844080489e+03, 6.406061945236226e+03,
        6.922276718051178e+03, 7.480089229687659e+03, 8.082851518805124e+03,
        8.734185738821492e+03, 9.438005924363450e+03, 1.019854151170577e+04,
        1.102036275454031e+04, 1.190840818780434e+04, 1.286801430460541e+04,
        1.390494762457918e+04, 1.502543934638713e+04, 1.623622279258970e+04,
        1.754457387191111e+04, 1.895835480204384e+04, 2.048606135573400e+04,
        2.213687391406209e+04, 2.392071263371084e+04, 2.584829705973495e+04,
        2.793121054206043e+04, 3.018196984280969e+04, 3.261410035274024e+04,
        3.524221736879158e+04, 3.808211392115859e+04, 4.115085567766677e+04,
        4.446688349575319e+04, 4.805012423831561e+04, 5.192211051935027e+04,
        5.610611009895962e+04, 6.062726570529947e+04, 6.551274612369082e+04,
        7.079190946082873e+04, 7.649647956518646e+04, 8.266073666376984e+04,
        8.932172336080558e+04, 9.651946723626507e+04, 1.042972213818742e+05,
        1.127017243200503e+05, 1.217834808676890e+05, 1.315970656325818e+05,
        1.422014509662509e+05, 1.536603613439579e+05, 1.660426563014429e+05,
        1.794227442295626e+05, 1.938810295134221e+05, 2.095043957029773e+05,
        2.263867276186048e+05, 2.446294755291000e+05, 2.643422647924012e+05,
        2.856435546225249e+05, 3.086613499414067e+05, 3.335339705933576e+05,
        3.604108825445382e+05, 3.894535960623372e+05, 4.208366362720539e+05,
        4.547485919232073e+05, 4.913932486677662e+05, 5.309908136604783e+05,
        5.737792388402272e+05, 6.200156508443463e+05, 6.699778961486300e+05,
        7.239662107181761e+05, 7.823050242024113e+05, 8.453449095161883e+05,
        9.134646895224825e+05, 9.870737134762700e+05, 1.066614316909347e+06,
        1.152564479738165e+06, 1.245440698567906e+06, 1.345801090453257e+06,
        1.454248746767143e+06, 1.571435357331702e+06, 1.698065126589807e+06,
        1.834899005350438e+06, 1.982759263537569e+06])

    dlf.factor = np.array([1.0805822324585601])

    dlf.j0 = np.array([
        0.000000000000000e+00, 0.000000000000000e+00, 9.824913062932289e-08,
        0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
        0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
        1.587400592121677e-08, 0.000000000000000e+00, 0.000000000000000e+00,
        0.000000000000000e+00, 8.566005894708157e-08, 0.000000000000000e+00,
        0.000000000000000e+00, 0.000000000000000e+00, 4.185852284487868e-08,
        0.000000000000000e+00, 6.042919228413999e-08, 0.000000000000000e+00,
        0.000000000000000e+00, 9.223778633040080e-08, 0.000000000000000e+00,
        0.000000000000000e+00, 9.605835817155985e-08, 0.000000000000000e+00,
        7.119233736108770e-08, 0.000000000000000e+00, 1.084534194934211e-07,
        2.978122322309773e-08, 0.000000000000000e+00, 2.229467165084615e-07,
        -1.762389145190478e-07, 3.656694527170595e-07, -2.190824497653861e-07,
        3.591123314594687e-07, -1.330459122900614e-07, 2.693078682355359e-07,
        0.000000000000000e+00, 1.681753537860265e-07, 1.292951481169703e-07,
        8.732830659371336e-08, 2.394635732137388e-07, 3.263161048145093e-08,
        3.320472817306875e-07, 0.000000000000000e+00, 4.140474794253468e-07,
        -1.691089584764466e-08, 4.929867154878337e-07, -2.351486069903310e-08,
        5.748749293789135e-07, -2.296118919509815e-08, 6.634488608548862e-07,
        -1.578522272463715e-08, 7.602932506949016e-07, -3.950291386691670e-10,
        8.657561597587320e-07, 2.577126803166016e-08, 9.803339051961460e-07,
        6.493462588145047e-08, 1.105842139938396e-06, 1.182703157662404e-07,
        1.245710190589679e-06, 1.861786291659876e-07, 1.404457760420360e-06,
        2.690715579072592e-07, 1.587041404534934e-06, 3.680016417418645e-07,
        1.798553595216537e-06, 4.849140624691158e-07, 2.044272068810943e-06,
        6.226640605691353e-07, 2.329872825482119e-06, 7.849774436589454e-07,
        2.661676063084173e-06, 9.764401550293115e-07, 3.046886487275798e-06,
        1.202531755803571e-06, 3.493832828910101e-06, 1.469704682136747e-06,
        4.012202093180585e-06, 1.785529348521925e-06, 4.613260893326240e-06,
        2.158900926480479e-06, 5.310102517657744e-06, 2.600268757751806e-06,
        6.117954165362566e-06, 3.121898205359148e-06, 7.054521021164678e-06,
        3.738203205552756e-06, 8.140366436119160e-06, 4.466138097026229e-06,
        9.399370737137856e-06, 5.325632087195220e-06, 1.085928513468306e-05,
        6.340092174159829e-06, 1.255236713273140e-05, 7.537013084429620e-06,
        1.451609696381449e-05, 8.948708367122905e-06, 1.679400873721405e-05,
        1.061315399682074e-05, 1.943668323451985e-05, 1.257494571773497e-05,
        2.250293226239968e-05, 1.488640124393654e-05, 2.606118359025671e-05,
        1.760885939507792e-05, 3.019107498684298e-05, 2.081422548042667e-05,
        3.498528720118145e-05, 2.458680024846277e-05, 4.055166655188445e-05,
        2.902542214383600e-05, 4.701571256263205e-05, 3.424594329627977e-05,
        5.452351542272840e-05, 4.038409635240394e-05, 6.324519057873240e-05,
        4.759885125947362e-05, 7.337888648387832e-05, 5.607631517546404e-05,
        8.515549372910889e-05, 6.603428945751648e-05, 9.884406328190645e-05,
        7.772775341712903e-05, 1.147579427511257e-04, 9.145538883362232e-05,
        1.332619721204058e-04, 1.075670047677990e-04, 1.547811559467743e-04,
        1.264719996713977e-04, 1.798108274980929e-04, 1.486493783884393e-04,
        2.089282897741272e-04, 1.746596606473501e-04, 2.428062685411137e-04,
        2.051588438598003e-04, 2.822286254590445e-04, 2.409146476076457e-04,
        3.281087927133202e-04, 2.828254175118406e-04, 3.815112277320206e-04,
        3.319424735329088e-04, 4.436760084416005e-04, 3.894966102002753e-04,
        5.160474740163103e-04, 4.569285549031484e-04, 6.003086167616667e-04,
        5.359233992792353e-04, 6.984220304753900e-04, 6.284506339729007e-04,
        8.126770850250102e-04, 7.368119832657103e-04, 9.457437019690870e-04,
        8.636982308314989e-04, 1.100734372093935e-03, 1.012255913625649e-03,
        1.281276054621223e-03, 1.186165424832822e-03, 1.491593633007820e-03,
        1.389731874050673e-03, 1.736607751791357e-03, 1.627989098002781e-03,
        2.022051249754969e-03, 1.906816944033515e-03, 2.354607536102306e-03,
        2.233075556223528e-03, 2.742069502777252e-03, 2.614764820469374e-03,
        3.193516622456554e-03, 3.061214433236700e-03, 3.719512717457636e-03,
        3.583305236990679e-03, 4.332330454742194e-03, 4.193717895916075e-03,
        5.046212230367296e-03, 4.907200820233952e-03, 5.877674969863473e-03,
        5.740851469703553e-03, 6.845854941792223e-03, 6.714414341214463e-03,
        7.972865827753512e-03, 7.850607650897248e-03, 9.284131207419897e-03,
        9.175463215382743e-03, 1.080866736231174e-02, 1.071861578305127e-02,
        1.257928364139867e-02, 1.251344881977102e-02, 1.463261489338917e-02,
        1.459697067058206e-02, 1.700882559413141e-02, 1.700922269488248e-02,
        1.975072507757058e-02, 1.979189868224237e-02, 2.290187743495209e-02,
        2.298566772167304e-02, 2.650305108874451e-02, 2.662540459598260e-02,
        3.058599251785477e-02, 3.073210898400032e-02, 3.516298150334437e-02,
        3.529966137617352e-02, 4.020993030999442e-02, 4.027366645135865e-02,
        4.563986763942291e-02, 4.551858910431959e-02, 5.126253569381094e-02,
        5.076838045690198e-02, 5.672499952578218e-02, 5.555543753580153e-02,
        6.142864225226326e-02, 5.911480957167643e-02, 6.442218515556448e-02,
        6.026861996000471e-02, 6.428390136013452e-02, 5.731690882023245e-02,
        5.903878467303038e-02, 4.800846542566535e-02, 4.622345209535847e-02,
        2.975631767989811e-02, 2.332973372901844e-02, 4.075631045367046e-04,
        -1.097998927420109e-02, -3.996770310336364e-02, -5.439690105073641e-02,
        -8.586172077870871e-02, -9.710630904000793e-02, -1.221886100159675e-01,
        -1.181757180654116e-01, -1.226608581391981e-01, -8.819223320765970e-02,
        -5.976246996708556e-02, 1.003867346106077e-02, 6.143837055743897e-02,
        1.358126160861001e-01, 1.546019688831242e-01, 1.597454471983037e-01,
        7.378168765574926e-02, -2.155543820940989e-02, -1.576840287359735e-01,
        -1.867618104598368e-01, -1.378646829885588e-01, 6.349123295842654e-02,
        1.950187860384398e-01, 1.936424769502024e-01, -7.533434525407418e-02,
        -2.318813953601980e-01, -1.072893080046963e-01, 2.602429230035626e-01,
        1.145512544892857e-01, -2.400299276793784e-01, -8.781414190173367e-02,
        3.482215332524013e-01, -2.648837943057846e-01, 1.177899549516786e-02,
        1.809751035116563e-01, -2.458202901044509e-01, 2.258610969476037e-01,
        -1.773155121294782e-01, 1.311747401438518e-01, -9.665134793648206e-02,
        7.308596000248034e-02, -5.738192263561636e-02, 4.678168146095997e-02,
        -3.939779158727641e-02, 3.405264070131274e-02, -3.003070542957877e-02,
        2.689468371509007e-02, -2.437148059023670e-02, 2.228571230862690e-02,
        -2.052141594646854e-02, 1.899976167605663e-02, -1.766579040557837e-02,
        1.648030895118207e-02, -1.541479785692305e-02, 1.444812372540324e-02,
        -1.356436078983561e-02, 1.275131275416127e-02, -1.199948916264833e-02,
        1.130138518277636e-02, -1.065096922990164e-02, 1.004331590462835e-02,
        -9.474341780324707e-03, 8.940614242462378e-03, -8.439212039530420e-03,
        7.967622109909870e-03, -7.523661440170574e-03, 7.105415845769664e-03,
        -6.711190184736050e-03, 6.339466716099865e-03, -5.988869875713372e-03,
        5.658136700741345e-03, -5.346092874868108e-03, 5.051634994011519e-03,
        -4.773719694137324e-03, 4.511359360882715e-03, -4.263622706076311e-03,
        4.029637517014100e-03, -3.808592909460376e-03, 3.599739234584605e-03,
        -3.402384890213290e-03, 3.215890296536024e-03, -3.039660023908146e-03,
        2.873134442084575e-03, -2.715782351627641e-03, 2.567095892455205e-03,
        -2.426588488831973e-03, 2.293795626711503e-03, -2.168277194783892e-03,
        2.049619594802782e-03, -1.937436245806754e-03, 1.831366196380340e-03,
        -1.731071555824182e-03, 1.636234772852003e-03, -1.546556467549061e-03,
        1.461754019926524e-03, -1.381560792781408e-03, 1.305725738551140e-03,
        -1.234013089394381e-03, 1.166201823788569e-03, -1.102084687325930e-03,
        1.041466730622445e-03, -9.841635538957642e-04, 9.299996161562829e-04,
        -8.788069763395656e-04, 8.304246517808708e-04, -7.846985037183290e-04,
        7.414813684955915e-04, -7.006331617591514e-04, 6.620208464608574e-04,
        -6.255183200732592e-04, 5.910063209799918e-04, -5.583723812133077e-04,
        5.275107580030028e-04, -4.983222495996592e-04, 4.707138540343669e-04,
        -4.445983047168279e-04, 4.198935530394733e-04, -3.965222590204658e-04,
        3.744113329795447e-04, -3.534915670228068e-04, 3.336973877578390e-04,
        -3.149667279395557e-04, 2.972409648555553e-04, -2.804648423393001e-04,
        2.645863035783382e-04, -2.495562056240159e-04, 2.353279391742063e-04,
        -2.218570165506096e-04, 2.091007054227963e-04, -1.970177752442844e-04,
        1.855683950070826e-04, -1.747141837505830e-04, 1.644183743725741e-04,
        -1.546460137084759e-04, 1.453641062486393e-04, -1.365416355278548e-04,
        1.281494594613588e-04, -1.201601331637577e-04, 1.125477274134751e-04,
        -1.052876878881750e-04, 9.835675748306749e-05, -9.173298412621861e-05,
        8.539584183369774e-05, -7.932647157887555e-05, 7.350799507002284e-05,
        -6.792579804522193e-05, 6.256766460994959e-05, -5.742369885284320e-05,
        5.248607676312167e-05, -4.774876610278635e-05, 4.320737034558452e-05,
        -3.885918466176108e-05, 3.470344555454732e-05, -3.074167696093530e-05,
        2.697802016509551e-05, -2.341946443717172e-05, 2.007591620962982e-05,
        -1.696002345446848e-05, 1.408663272782194e-05, -1.147175200077509e-05,
        9.130950853154427e-06, -7.077245367140833e-06, 5.318671603956403e-06,
        -3.855913768030704e-06, 2.680452779662483e-06, -1.773679649172923e-06,
        1.107284379526082e-06, -6.450375038180866e-07, 3.458519700889004e-07,
        -1.677370250259620e-07, 7.195813987460663e-08, -2.650936383269536e-08,
        8.053026529021296e-09, -1.901548822685746e-09, 3.176144531356312e-10,
        -3.147399480514734e-11, 1.181734681811153e-12])

    dlf.j1 = np.array([
        0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
        -1.761153377678605e-10, 0.000000000000000e+00, 0.000000000000000e+00,
        0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
        0.000000000000000e+00, 3.440216255617151e-10, 0.000000000000000e+00,
        0.000000000000000e+00, 0.000000000000000e+00, -5.012130889133619e-10,
        0.000000000000000e+00, 0.000000000000000e+00, 5.096249946359556e-10,
        0.000000000000000e+00, 0.000000000000000e+00, -5.137927435894773e-10,
        0.000000000000000e+00, 4.901252024472790e-10, 0.000000000000000e+00,
        -3.412499448385044e-10, 0.000000000000000e+00, 2.331695173457025e-10,
        0.000000000000000e+00, -2.490134046395816e-10, 1.954047749927753e-10,
        0.000000000000000e+00, -8.103005437723991e-11, 0.000000000000000e+00,
        1.279088845538237e-10, -1.851480627564174e-10, 1.332990147779435e-10,
        0.000000000000000e+00, -1.624963592448070e-10, 3.098241711700410e-10,
        -4.149316491270542e-10, 4.685626300171698e-10, -4.712327624043219e-10,
        4.305062986549322e-10, -3.542442130313303e-10, 2.522911525991727e-10,
        -1.313966839434664e-10, 0.000000000000000e+00, 1.374783013515889e-10,
        -2.736921677833286e-10, 4.061582817302266e-10, -5.279722941941932e-10,
        6.386355968571155e-10, -7.315427954629247e-10, 8.087932062353638e-10,
        -8.640922282673907e-10, 9.028429079409167e-10, -9.186974584267662e-10,
        9.208119531836431e-10, -9.017596435374981e-10, 8.746151932449153e-10,
        -8.291880695503223e-10, 7.826366656707373e-10, -7.199240415978938e-10,
        6.630127031117425e-10, -5.898849811863955e-10, 5.290634200040055e-10,
        -4.493240826470479e-10, 3.886261259968761e-10, -3.037437366057724e-10,
        2.460094930171363e-10, -1.562353890563393e-10, 1.042595244297130e-10,
        -9.277742086714826e-12, -3.359801793948191e-11, 1.346759112142224e-10,
        -1.644581113778544e-10, 2.735615970185790e-10, -2.856750577185324e-10,
        4.064174138388151e-10, -3.956626260990240e-10, 5.339967379460912e-10,
        -4.942289770979441e-10, 6.590721699414525e-10, -5.825238873978475e-10,
        7.864186241426929e-10, -6.626363292105279e-10, 9.225771435176782e-10,
        -7.369977848939469e-10, 1.075704953982971e-09, -8.079491312165849e-10,
        1.255853770105183e-09, -8.775982413533787e-10, 1.475677833588153e-09,
        -9.478165158795805e-10, 1.751454564451092e-09, -1.020150878388706e-09,
        2.104270247990516e-09, -1.095430340362651e-09, 2.561356147221457e-09,
        -1.172968828002544e-09, 3.157686036019299e-09, -1.249340900707534e-09,
        3.938151654389299e-09, -1.316804538118081e-09, 4.960584143866221e-09,
        -1.361050365193285e-09, 6.299594474808514e-09, -1.357732636276717e-09,
        8.051372567815041e-09, -1.267357938643338e-09, 1.034000380570114e-08,
        -1.028333203364199e-09, 1.332639788605693e-08, -5.481128505786461e-10,
        1.722132657425943e-08, 3.079572462647345e-10, 2.230389235352557e-08,
        1.731612035551629e-09, 2.894640331467214e-08, 3.992876253106220e-09,
        3.764713519770196e-08, 7.470342606008511e-09, 4.907355253852564e-08,
        1.269390990642245e-08, 6.411984689966026e-08, 2.040424455749986e-08,
        8.398456134765634e-08, 3.163423793932673e-08, 1.102763405195791e-07,
        4.782016944498328e-08, 1.451576997806275e-07, 7.095396726449770e-08,
        1.915405053682995e-07, 1.037907010433357e-07, 2.533530424909100e-07,
        1.501316966404849e-07, 3.359015461416407e-07, 2.152163676698347e-07,
        4.463548364795692e-07, 3.062649070955455e-07, 5.943990986479385e-07,
        4.332201278554725e-07, 7.931323845043337e-07, 6.097551748617194e-07,
        1.060288415924516e-06, 8.546425300312143e-07, 1.419911983644722e-06,
        1.193611725125921e-06, 1.904650353313021e-06, 1.661883236417129e-06,
        2.558861699766762e-06, 2.307654632812588e-06, 3.442811157821135e-06,
        3.196895656510743e-06, 4.638358992076794e-06, 4.419902312928682e-06,
        6.256715142637839e-06, 6.100218940817017e-06, 8.449032402277294e-06,
        8.406776377975507e-06, 1.142087099410692e-05, 1.157042201253195e-05,
        1.545191683298927e-05, 1.590647868284089e-05, 2.092279678518197e-05,
        2.184558773386808e-05, 2.835151124366015e-05, 2.997585693259872e-05,
        3.844299184771377e-05, 4.110034139799833e-05, 5.215663114685968e-05,
        5.631528820209043e-05, 7.079837414439344e-05, 7.711659421807641e-05,
        9.614618225427727e-05, 1.055447661767286e-04, 1.306206751127902e-04,
        1.443823581310115e-04, 1.775170452103382e-04, 1.974225796898412e-04,
        2.413201291349723e-04, 2.698342899158978e-04, 3.281318482161766e-04,
        3.686575352518240e-04, 4.462498902719190e-04, 5.034750551777089e-04,
        6.069499176331384e-04, 6.873187128242988e-04, 8.255410973454176e-04,
        9.378875323431232e-04, 1.122784104067260e-03, 1.279178262961256e-03,
        1.526781204453462e-03, 1.743655859738250e-03, 2.075473273444583e-03,
        2.375110960283700e-03, 2.819903510936917e-03, 3.232361399449634e-03,
        3.828410021593598e-03, 4.393943043291972e-03, 5.191863182780206e-03,
        5.963857250198409e-03, 7.029925656135096e-03, 8.078221794986575e-03,
        9.497983356558860e-03, 1.091218414985025e-02, 1.279369144280816e-02,
        1.468546702846075e-02, 1.716070017133079e-02, 1.966301954392782e-02,
        2.288457156694697e-02, 2.614381914454830e-02, 3.027139461924632e-02,
        3.442503605732608e-02, 3.959217858802792e-02, 4.471956265829849e-02,
        5.096501947760914e-02, 5.699213341078931e-02, 6.413328819589428e-02,
        7.066611680136001e-02, 7.808858155732867e-02, 8.415253392588408e-02,
        9.050430770970036e-02, 9.420119432022898e-02, 9.704525747507635e-02,
        9.525207334318243e-02, 9.091107970583592e-02, 7.941214687854993e-02,
        6.360896948482617e-02, 3.851999787085855e-02, 8.930119389229475e-03,
        -2.931232114631221e-02, -6.745472857764939e-02, -1.071387129122266e-01,
        -1.337787672525563e-01, -1.467487065654186e-01, -1.280178925639679e-01,
        -8.266114319901335e-02, -1.829018947010989e-03, 8.538653035086596e-02,
        1.634948424549636e-01, 1.768283682900049e-01, 1.165478617534897e-01,
        -2.909864536607605e-02, -1.650763395456089e-01, -2.108189429067250e-01,
        -6.548403763878953e-02, 1.474231756411919e-01, 2.387075880611132e-01,
        8.364392500807047e-03, -2.375301184090862e-01, -1.324045834042435e-01,
        2.623684401711879e-01, 9.558257428232823e-02, -2.795911533969242e-01,
        4.155824322494574e-02, 2.715606356255319e-01, -3.742520323983440e-01,
        2.853293391721109e-01, -1.417457299548064e-01, 3.117732200130105e-02,
        2.889458199693983e-02, -5.278084724642508e-02, 5.771601780075224e-02,
        -5.477531833561626e-02, 4.939546060797920e-02, -4.378947822887341e-02,
        3.869238569366965e-02, -3.425716943762586e-02, 3.044093551672354e-02,
        -2.715226194815648e-02, 2.430080341386173e-02, -2.181052920166688e-02,
        1.962073026812987e-02, -1.768361271316385e-02, 1.596144663779621e-02,
        -1.442412760579961e-02, 1.304729403368628e-02, -1.181093160074335e-02,
        1.069835428430446e-02, -9.695462302234538e-03, 8.790199228345667e-03,
        -7.972150985921739e-03, 7.232245468946166e-03, -6.562523603657329e-03,
        5.955961653845067e-03, -5.406331000039078e-03, 4.908085823543291e-03,
        -4.456271717308975e-03, 4.046450048794231e-03, -3.674634444757500e-03,
        3.337237040467819e-03, -3.031022902436713e-03, 2.753071298464685e-03,
        -2.500742536477930e-03, 2.271649199484839e-03, -2.063630842184971e-03,
        1.874731507609227e-03, -1.703179664969160e-03, 1.547370305159833e-03,
        -1.405848965175527e-03, 1.277297436350464e-03, -1.160520897100170e-03,
        1.054436232968270e-03, -9.580613717996333e-04, 8.705055444165467e-04,
        -7.909604401097324e-04, 7.186922398324291e-04, -6.530344915884751e-04,
        5.933817682686722e-04, -5.391840290971849e-04, 4.899415886335732e-04,
        -4.452005847911536e-04, 4.045488440443307e-04, -3.676120749678280e-04,
        3.340503647933679e-04, -3.035549812866991e-04, 2.758454813737592e-04,
        -2.506671088369273e-04, 2.277884460323346e-04, -2.069992810457240e-04,
        1.881086606885530e-04, -1.709431124465889e-04, 1.553450267830473e-04,
        -1.411711924592810e-04, 1.282914747613330e-04, -1.165876238497402e-04,
        1.059521991589964e-04, -9.628759515338238e-05, 8.750515384295626e-05,
        -7.952435126636077e-05, 7.227204913633232e-05, -6.568180805443277e-05,
        5.969326263717584e-05, -5.425155895475095e-05, 4.930685077009869e-05,
        -4.481384639404849e-05, 4.073139627769274e-05, -3.702211350786090e-05,
        3.365202295484529e-05, -3.059023779502484e-05, 2.780866410092890e-05,
        -2.528173545505838e-05, 2.298617982129317e-05, -2.090081950209839e-05,
        1.900640153317426e-05, -1.728545097076178e-05, 1.572213560023288e-05,
        -1.430213090870262e-05, 1.301247986370528e-05, -1.184145017024185e-05,
        1.077839734116676e-05, -9.813642618247219e-06, 8.938371729153144e-06,
        -8.144555804633981e-06, 7.424890780058305e-06, -6.772748023670692e-06,
        6.182129388552279e-06, -5.647624656180153e-06, 5.164374046764935e-06,
        -4.728037195336446e-06, 4.334761689817321e-06, -3.981136442356169e-06,
        3.664118637205433e-06, -3.380939384592361e-06, 3.129009529723152e-06,
        -2.905847492954934e-06, 2.709034327844308e-06, -2.536183084429876e-06,
        2.384905895111026e-06, -2.252772087223033e-06, 2.137259365710216e-06,
        -2.035698487395405e-06, 1.945208220753021e-06, -1.862627051737732e-06,
        1.784472814922380e-06, -1.706983978589542e-06, 1.626297101583022e-06,
        -1.538789746921996e-06, 1.441575114294373e-06, -1.333077139855850e-06,
        1.213536734497111e-06, -1.085223736705093e-06, 9.521451230126823e-07,
        -8.192296419664399e-07, 6.912547372495382e-07, -5.719422069867315e-07,
        4.635461014353267e-07, -3.669835113430816e-07, 2.823229600705464e-07,
        -2.093541854410099e-07, 1.479865821668501e-07, -9.831343971854163e-08,
        6.033173947686275e-08, -3.349527235750482e-08, 1.640107598073682e-08,
        -6.856420858188184e-09, 2.340536901536035e-09, -6.097093750212437e-10,
        1.074334228952679e-10, -9.576603341064843e-12])
    return dlf


def anderson_801_1982():
    r"""Anderson 801 pt Hankel filter, as published in [Ande82]_.

    Taken from file `wa801Hankel.txt` provided with SEG-2012-003_.

    License: https://software.seg.org/disclaimer.txt.

    """

    dlf = DigitalFilter('Anderson 801', 'anderson_801_1982')

    dlf.base = np.array([
        8.9170998013276122e-14, 9.8549193740053558e-14, 1.0891370292131004e-13,
        1.2036825704856271e-13, 1.3302749714952577e-13, 1.4701812115404721e-13,
        1.6248015192957424e-13, 1.7956833867707853e-13, 1.9845370571306597e-13,
        2.1932526413842384e-13, 2.4239190352504611e-13, 2.6788448255287750e-13,
        2.9605813952118386e-13, 3.2719484585839544e-13, 3.6160622818694338e-13,
        3.9963668718723693e-13, 4.4166684447542651e-13, 4.8811735199248203e-13,
        5.3945310203018623e-13, 5.9618788002945774e-13, 6.5888950671773090e-13,
        7.2818552104964630e-13, 8.0476946082782683e-13, 8.8940780386233457e-13,
        9.8294763913818316e-13, 1.0863251447666380e-12, 1.2005749575704075e-12,
        1.3268405280767117e-12, 1.4663855645545188e-12, 1.6206066806315963e-12,
        1.7910473730731519e-12, 1.9794134696162351e-12, 2.1875902014670651e-12,
        2.4176610713286508e-12, 2.6719287057960420e-12, 2.9529379008172930e-12,
        3.2635010908666289e-12, 3.6067264967339290e-12, 3.9860492336432057e-12,
        4.4052656910401999e-12, 4.8685715281340575e-12, 5.3806036654648819e-12,
        5.9464866927629836e-12, 6.5718841575654989e-12, 7.2630552479034764e-12,
        8.0269174363596475e-12, 8.8711157124590269e-12, 9.8040990962935900e-12,
        1.0835205199155430e-11, 1.1974753677488654e-11, 1.3234149515479892e-11,
        1.4625997169973320e-11, 1.6164226720111148e-11, 1.7864233284248175e-11,
        1.9743031099470126e-11, 2.1819423805797495e-11, 2.4114192639334892e-11,
        2.6650304417866614e-11, 2.9453141400489177e-11, 3.2550755321790535e-11,
        3.5974148143039067e-11, 3.9757582330231898e-11, 4.3938923764370285e-11,
        4.8560020715925063e-11, 5.3667122676391448e-11, 5.9311343238746032e-11,
        6.5549171659464899e-11, 7.2443038221988632e-11, 8.0061939059984521e-11,
        8.8482126693839770e-11, 9.7787873191516816e-11, 1.0807231359173380e-10,
        1.1943837803073504e-10, 1.3199982190169390e-10, 1.4588236435691723e-10,
        1.6122494654738060e-10, 1.7818112219246607e-10, 1.9692059439719579e-10,
        2.1763091409795137e-10, 2.4051935713527569e-10, 2.6581499874015753e-10,
        2.9377100619593643e-10, 3.2466717262157026e-10, 3.5881271723520484e-10,
        3.9654938012404966e-10, 4.3825484249402556e-10, 4.8434650663021947e-10,
        5.3528567339925320e-10, 5.9158215910339261e-10, 6.5379939789347122e-10,
        7.2256008080723437e-10, 7.9855238787054355e-10, 8.8253687563439063e-10,
        9.7535408908047105e-10, 1.0779329740779026e-09, 1.1913001745856907e-09,
        1.3165903076505441e-09, 1.4550573190356532e-09, 1.6080870331313201e-09,
        1.7772110227512878e-09, 1.9641219376282043e-09, 2.1706904450210777e-09,
        2.3989839519819838e-09, 2.6512872966606692e-09, 2.9301256157327775e-09,
        3.2382896168163712e-09, 3.5788635088117781e-09, 3.9552558697009517e-09,
        4.3712337607414862e-09, 4.8309604284819404e-09, 5.3390369719325178e-09,
        5.9005483919104740e-09, 6.5211144834374943e-09, 7.2069460805370025e-09,
        7.9649072163487805e-09, 8.8025838206795459e-09, 9.7283596425382361e-09,
        1.0751500157514075e-08, 1.1882245299770274e-08, 1.3131911946747182e-08,
        1.4513007182275169e-08, 1.6039353471673485e-08, 1.7726227001629234e-08,
        1.9590510569407876e-08, 2.1650862551563211e-08, 2.3927903643240804e-08,
        2.6444423237026018e-08, 2.9225607506845074e-08, 3.2299291479658440e-08,
        3.5696237617767112e-08, 3.9450443699874199e-08, 4.3599483082281526e-08,
        4.8184880745668805e-08, 5.3252528891056283e-08, 5.8853146244378704e-08,
        6.5042785666540466e-08, 7.1883395149287942e-08, 7.9443437811533216e-08,
        8.7798577101257606e-08, 9.7032434060732531e-08, 1.0723742423401467e-07,
        1.1851568259277344e-07, 1.3098008573741766e-07, 1.4475538160404861e-07,
        1.5997943798373734e-07, 1.7680462234971339e-07, 1.9539932680225051e-07,
        2.1594965339340700e-07, 2.3866127669890898e-07, 2.6376150227843976e-07,
        2.9150154162607574e-07, 3.2215902637935602e-07, 3.5604078695003014e-07,
        3.9348592338594013e-07, 4.3486919919828399e-07, 4.8060479212078988e-07,
        5.3115043933968794e-07, 5.8701201868132739e-07, 6.4874861160748036e-07,
        7.1697809869054185e-07, 7.9238334356995964e-07, 8.7571902728106165e-07,
        9.6781920135652544e-07, 1.0696056352944290e-06, 1.1820970419372323e-06,
        1.3064192730922798e-06, 1.4438165874351121e-06, 1.5956641034685138e-06,
        1.7634815621706487e-06, 1.9489485370736159e-06, 2.1539212439998414e-06,
        2.3804511186939409e-06, 2.6308053482811889e-06, 2.9074895620382395e-06,
        3.2132729085731683e-06, 3.5512157703954198e-06, 3.9247003932526156e-06,
        4.3374647367828553e-06, 4.7936398852710445e-06, 5.2977913929290508e-06,
        5.8549649774966706e-06, 6.4707370194807457e-06, 7.1512703724456250e-06,
        7.9033760429228938e-06, 8.7345813572541863e-06, 9.6532052976030589e-06,
        1.0668441761124655e-05, 1.1790451575578730e-05, 1.3030464192308785e-05,
        1.4400890074365772e-05, 1.5915444904593322e-05, 1.7589286856791756e-05,
        1.9439168303816492e-05, 2.1483603480955855e-05, 2.3743053782621199e-05,
        2.6240132546858983e-05, 2.8999831377238759e-05, 3.2049770267221979e-05,
        3.5420474030339236e-05, 3.9145677802784702e-05, 4.3262664675997137e-05,
        4.7812638838370539e-05, 5.2841137960621413e-05, 5.8398488952101788e-05,
        6.4540311649424992e-05, 7.1328075478483536e-05, 7.8829714661124577e-05,
        8.7120308123676523e-05, 9.6282830912076676e-05, 1.0640898463402227e-04,
        1.1760011523948004e-04, 1.2996822732501783e-04, 1.4363710511345465e-04,
        1.5874355132796520e-04, 1.7543875635971559e-04, 1.9388981143211701e-04,
        2.1428138090594652e-04, 2.3681755046234279e-04, 2.6172386966089376e-04,
        2.8924960931544049e-04, 3.1967025628016824e-04, 3.5329027061463029e-04,
        3.9044613272236556e-04, 4.3150971095986352e-04, 4.7689198342006864e-04,
        5.2704715113927460e-04, 5.8247718389374547e-04, 6.4373684408196936e-04,
        7.1143923897319119e-04, 7.8626195689104005e-04, 8.6895384874522731e-04,
        9.6034252278312808e-04, 1.0613426275713149e-03, 1.1729650061058118e-03,
        1.2963268126685650e-03, 1.4326626936829981e-03, 1.5833371444703658e-03,
        1.7498581655775909e-03, 1.9338923553535578e-03, 2.1372815898255634e-03,
        2.3620614568137010e-03, 2.6104816287778987e-03, 2.8850283782960811e-03,
        3.1884494615157760e-03, 3.5237816186211930e-03, 3.8943809665496808e-03,
        4.3039565881380368e-03, 4.7566076538702448e-03, 5.2568644477534289e-03,
        5.8097337079228871e-03, 6.4207487357601832e-03, 7.0960247750331334e-03,
        7.8423202153109061e-03, 8.6671042321983614e-03, 9.5786315413559902e-03,
        1.0586025014468769e-02, 1.1699366984012220e-02, 1.2929800150624698e-02,
        1.4289639103000540e-02, 1.5792493566432773e-02, 1.7453404613518294e-02,
        1.9288995200267750e-02, 2.1317636534236663e-02, 2.3559631939745293e-02,
        2.6037420060372646e-02, 2.8775799432443357e-02, 3.1802176677114116e-02,
        3.5146840795050142e-02, 3.8843266308924179e-02, 4.2928448287690595e-02,
        4.7443272605670044e-02, 5.2432925142121563e-02, 5.7947344016710180e-02,
        6.4041719386992949e-02, 7.0777045810065983e-02, 7.8220732696592896e-02,
        8.6447278966843372e-02, 9.5539018660927857e-02, 1.0558694496554405e-01,
        1.1669162090437316e-01, 1.2896418580662170e-01, 1.4252746762678245e-01,
        1.5751721224808826e-01, 1.7408344207293630e-01, 1.9239195749751575e-01,
        2.1262599629790074e-01, 2.3498806753530016e-01, 2.5970197833480985e-01,
        2.8701507382234365e-01, 3.1720071263778921e-01, 3.5056100280015567e-01,
        3.8742982530616760e-01, 4.2817617572350486e-01, 4.7320785722246550e-01,
        5.2297556200716200e-01, 5.7797738199458382e-01, 6.3876379388591331e-01,
        7.0594316852237826e-01, 7.8018785966510817e-01, 8.6224093313756178e-01,
        9.5292360367804352e-01, 1.0531434539328177e+00, 1.1639035178482904e+00,
        1.2863123193718702e+00, 1.4215949669322252e+00, 1.5711054147362096e+00,
        1.7363400135976370e+00, 1.9189524869191823e+00, 2.1207704817120194e+00,
        2.3438138603014091e+00, 2.5903149157877352e+00, 2.8627407135861742e+00,
        3.1638177826465657e+00, 3.4965594034715632e+00, 3.8642957660407120e+00,
        4.2707072994710495e+00, 4.7198615069887886e+00, 5.2162536748687076e+00,
        5.7648518627701177e+00, 6.3711466257477678e+00, 7.0412059655722228e+00,
        7.7817360613311779e+00, 8.6001483871237490e+00, 9.5046338885843582e+00,
        1.0504244960619683e+01, 1.1608986046819558e+01, 1.2829913767290948e+01,
        1.4179247577028322e+01, 1.5670492062326302e+01, 1.7318572099218304e+01,
        1.9139982226652400e+01, 2.1152951729381002e+01, 2.3377627082769852e+01,
        2.5836273585494894e+01, 2.8553498198134982e+01, 3.1556495817904207e+01,
        3.4875321454323519e+01, 3.8543191029858036e+01, 4.2596813816033297e+01,
        4.7076759832162928e+01, 5.2027865883738308e+01, 5.7499684304247715e+01,
        6.3546978891585326e+01, 7.0230273002547193e+01, 7.7616455290928428e+01,
        8.5779449151652855e+01, 9.4800952570955488e+01, 1.0477125578728878e+02,
        1.1579014494637651e+02, 1.2796790079449919e+02, 1.4142640240527015e+02,
        1.5630034698636834e+02, 1.7273859797446690e+02, 1.9090567491054190e+02,
        2.1098340000673463e+02, 2.3317271788416465e+02, 2.5769570669423615e+02,
        2.8479780075142190e+02, 3.1475024692237423e+02, 3.4785281935573693e+02,
        3.8443681972258241e+02, 4.2486839299488850e+02, 4.6955219194748605e+02,
        5.1893542705903599e+02, 5.7351234234481308e+02, 6.3382916191693221e+02,
        7.0048955677885397e+02, 7.7416068656768959e+02, 8.5557987671208730e+02,
        9.4556199783294710e+02, 1.0450076212424817e+03, 1.1549120321646019e+03,
        1.2763751908839649e+03, 1.4106127415182118e+03, 1.5589681785928883e+03,
        1.7229262931862229e+03, 1.9041280332172894e+03, 2.1043869266043293e+03,
        2.3257072316616973e+03, 2.5703039963907313e+03, 2.8406252274246508e+03,
        3.1393763905017463e+03, 3.4695474876758281e+03, 3.8344429822617512e+03,
        4.2377148710149440e+03, 4.6833992345424103e+03, 5.1759566317540202e+03,
        5.7203167426353284e+03, 6.3219277061417824e+03, 6.9868106470045868e+03,
        7.7216199371707708e+03, 8.5337097949942454e+03, 9.4312078887249354e+03,
        1.0423096680944427e+04, 1.1519303328070589e+04, 1.2730799034675636e+04,
        1.4069708856989042e+04, 1.5549433054535648e+04, 1.7184781204436986e+04,
        1.8992120420636755e+04, 2.0989539161478380e+04, 2.3197028265075824e+04,
        2.5636681024340593e+04, 2.8332914304083028e+04, 3.1312712913202089e+04,
        3.4605899677722737e+04, 3.8245433917662602e+04, 4.2267741314984683e+04,
        4.6713078474065609e+04, 5.1625935823322863e+04, 5.7055482890376195e+04,
        6.3056060407206460e+04, 6.9687724170465866e+04, 7.7016846100076247e+04,
        8.5116778511712138e+04, 9.4068588251430439e+04, 1.0396186803991348e+05,
        1.1489563314653051e+05, 1.2697931236743393e+05, 1.4033384322573140e+05,
        1.5509288235486555e+05, 1.7140414317912518e+05, 1.8943087427924355e+05,
        2.0935349323906421e+05, 2.3137139232536047e+05, 2.5570493407265941e+05,
        2.8259765674555401e+05, 3.1231871175151068e+05, 3.4516555739862064e+05,
        3.8146693595832621e+05, 4.2158616382856754e+05, 4.6592476772640843e+05,
        5.1492650330237771e+05, 5.6908179639617179e+05, 6.2893265138329600e+05,
        6.9507807573702815e+05, 7.6818007509654609e+05, 8.4897027884186991e+05,
        9.3825726248660800e+05, 1.0369346401734686e+06, 1.1459900082649530e+06,
        1.2665148295396979e+06, 1.3997153569188099e+06, 1.5469247060505433e+06,
        1.7096161975797811e+06, 1.8894181026362439e+06, 2.0881299391192470e+06,
        2.3077404818776632e+06, 2.5504476670370763e+06, 2.8186805896832580e+06,
        3.1151238150622533e+06, 3.4427442466116664e+06, 3.8048208197274678e+06,
        4.2049773184515461e+06, 4.6472186435203683e+06, 5.1359708947576685e+06,
        5.6761256689691413e+06, 6.2730890166873736e+06, 6.9328355477426499e+06,
        7.6619682271662261e+06, 8.4677844598837383e+06, 9.3583491255964171e+06,
        1.0342575294807823e+07, 1.1430313433829281e+07, 1.2632449991557509e+07,
        1.3961016354714315e+07, 1.5429309262008758e+07, 1.7052023882367309e+07,
        1.8845400889123756e+07, 2.0827389002136521e+07, 2.3017824624609854e+07,
        2.5438630372484341e+07, 2.8114034483345557e+07, 3.1070813300769445e+07,
        3.4338559260968395e+07, 3.7949977063839465e+07, 4.1941210992593251e+07,
        4.6352206657888710e+07, 5.1227110786931038e+07, 5.6614713058755472e+07,
        6.2568934407734118e+07, 6.9149366682410479e+07, 7.6421869060749277e+07,
        8.4459227190925196e+07, 9.3341881654554456e+07, 1.0315873304307236e+08,
        1.1400803170473291e+08, 1.2599836106711736e+08, 1.3924972437657300e+08,
        1.5389474573103800e+08, 1.7007999742658958e+08, 1.8796746690224770e+08,
        2.0773617796471131e+08, 2.2958398251878601e+08, 2.5372954073575008e+08,
        2.8041450947784126e+08, 3.0990596088135868e+08, 3.4249905530436677e+08,
        3.7851999539076889e+08, 4.1832929081600583e+08, 4.6232536638905960e+08,
        5.1094854962185794e+08, 5.6468547767500913e+08, 6.2407396778607523e+08,
        6.8970839992524862e+08, 7.6224566554987264e+08, 8.4241174199493706e+08,
        9.3100895829824984e+08, 1.0289240251791286e+09, 1.1371369095373080e+09,
        1.2567306422910509e+09, 1.3889021577146010e+09, 1.5349742727586930e+09,
        1.6964089262472236e+09, 1.8748218104523671e+09, 2.0719985414858897e+09,
        2.2899125303453670e+09, 2.5307447334747176e+09, 2.7969054805093622e+09,
        3.0910585976653414e+09, 3.4161480682074280e+09, 3.7754274968231587e+09,
        4.1724926727920933e+09, 4.6113175578535490e+09, 5.0962940589513578e+09,
        5.6322759839147396e+09, 6.2246276199984837e+09, 6.8792774214727545e+09,
        7.6027773435861149e+09, 8.4023684167357969e+09, 9.2860532171337223e+09,
        1.0262675959279013e+10, 1.1342011011829260e+10, 1.2534860722767443e+10,
        1.3853163532931267e+10, 1.5310113459941727e+10, 1.6920292148366150e+10,
        1.8699814807717983e+10, 2.0666491498890266e+10, 2.2840005383231117e+10,
        2.5242109718238258e+10, 2.7896845571471642e+10, 3.0830782431637867e+10,
        3.4073284124963757e+10, 3.7656802698238571e+10, 4.1617203209805832e+10,
        4.5994122679121964e+10, 5.0831366787369171e+10, 5.6177348299436752e+10,
        6.2085571595143913e+10, 6.8615168159056526e+10, 7.5831488388259552e+10,
        8.3806755641095596e+10, 9.2620789072811050e+10, 1.0236180249248946e+11,
        1.1312728723650264e+11, 1.2502498789457329e+11, 1.3817398065384225e+11,
        1.5270586505337357e+11, 1.6876608107657275e+11, 1.8651536476342502e+11,
        2.0613135691080905e+11, 2.2781038096129733e+11, 2.5176940787416034e+11,
        2.7824822764364838e+11, 3.0751184919785205e+11, 3.3985315269713074e+11,
        3.7559582077719043e+11, 4.1509757807370447e+11, 4.5875377145069446e+11,
        5.0700132676482880e+11, 5.6032312176625806e+11, 6.1925281890142700e+11,
        6.8438020638622375e+11, 7.5635710100466516e+11, 8.3590387171035938e+11,
        9.2381664932112744e+11, 1.0209752944637970e+12, 1.1283522035151108e+12,
        1.2470220406714768e+12, 1.3781724935494604e+12, 1.5231161599626641e+12,
        1.6833036848417888e+12, 1.8603382787767229e+12, 2.0559917634869438e+12,
        2.2722223048088311e+12, 2.5111940106775435e+12, 2.7752985902465620e+12,
        3.0671792909168477e+12, 3.3897573528451914e+12, 3.7462612456975898e+12,
        4.1402589802588306e+12, 4.5756938182836025e+12, 5.0569237379855439e+12,
        5.5887650501480273e+12, 6.1765406013811748e+12, 6.8261330469599551e+12,
        7.5440437264152627e+12, 8.3374577311251680e+12, 9.2143158151245215e+12,
        1.0183393868840104e+13, 1.1254390751151955e+13, 1.2438025358832701e+13,
        1.3746143904869295e+13, 1.5191838479344387e+13, 1.6789578079473949e+13,
        1.8555353420195023e+13, 2.0506836974615066e+13, 2.2663559846062926e+13,
        2.5047107241935781e+13, 2.7681334505709309e+13, 3.0592605869233902e+13,
        3.3810058314827727e+13, 3.7365893187989258e+13, 4.1295698479286742e+13,
        4.5638805000928352e+13, 5.0438680022751523e+13, 5.5743362307268203e+13,
        6.1605942897746914e+13, 6.8085096471218984e+13, 7.5245668574365969e+13,
        8.3159324619548062e+13, 9.1905267136336859e+13, 1.0157102845705283e+14,
        1.1225334676976897e+14, 1.2405913430660934e+14, 1.3710654735730572e+14,
        1.5152616881705603e+14, 1.6746231510403103e+14, 1.8507448052659562e+14,
        2.0453893355595084e+14, 2.2605048098024438e+14, 2.4982441759637878e+14,
        2.7609868095270331e+14, 3.0513623270797488e+14, 3.3722769044001631e+14,
        3.7269423624412369e+14, 4.1189083123142100e+14, 4.5520976809897019e+14,
        5.0308459732694231e+14, 5.5599446629753312e+14, 6.1446891476302538e+14,
        6.7909317465760050e+14, 7.5051402729524488e+14, 8.2944627657453862e+14,
        9.1667990297630850e+14, 1.0130879699538239e+15, 1.1196353618452631e+15,
        1.2373884407604868e+15, 1.3675257190914632e+15, 1.5113496544603695e+15,
        1.6702996851532818e+15, 1.8459666365023202e+15, 2.0401086424002802e+15,
        2.2546687412955835e+15, 2.4917943227741000e+15, 2.7538586193559425e+15,
        3.0434844586041465e+15, 3.3635705132645030e+15, 3.7173203121567135e+15,
        4.1082743021674790e+15, 4.5403452822330300e+15, 5.0178575639459200e+15,
        5.5455902507189340e+15, 6.1288250686584140e+15, 6.7733992278542490e+15,
        7.4857638431405730e+15, 8.2730484990211680e+15, 9.1431326049475640e+15,
        1.0104724255097300e+16, 1.1167447381907124e+16, 1.2341938075623802e+16,
        1.3639951033869966e+16, 1.5074477206608918e+16, 1.6659873813938426e+16,
        1.8412008037974732e+16, 2.0348415826944768e+16, 2.2488477400849620e+16,
        2.4853611215220372e+16, 2.7467488324220356e+16, 3.0356269288510672e+16,
        3.3548865998934976e+16, 3.7077231036439896e+16, 4.0976677464245088e+16,
        4.5286232252849512e+16, 5.0049026875068592e+16, 5.5312728980312400e+16,
        6.1130019468441424e+16, 6.7559119737919472e+16, 7.4664374385139184e+16,
        8.2516895186767952e+16, 9.1195272810312464e+16, 1.0078636337593230e+17,
        1.1138615774168470e+17, 1.2310074221229675e+17, 1.3604736028655734e+17,
        1.5035558606966320e+17, 1.6616862109441197e+17, 1.8364472753027533e+17,
        2.0295881212438685e+17, 2.2430417672705101e+17, 2.4789445292163760e+17,
        2.7396574012126704e+17, 3.0277896853109434e+17, 3.3462251062550758e+17,
        3.6981506727676966e+17, 4.0870885742047546e+17, 4.5169314318103648e+17,
        4.9919812573785997e+17, 5.5169925092335405e+17, 6.0972196764460902e+17,
        6.7384698675268378e+17, 7.4471609299197325e+17, 8.2303856819764672e+17,
        9.0959829002666112e+17, 1.0052615772688020e+18, 1.1109858602563373e+18,
        1.2278292631485612e+18, 1.3569611939940383e+18, 1.4996740485594204e+18,
        1.6573961450606344e+18, 1.8317060192517033e+18, 2.0243482229410977e+18,
        2.2372507840526221e+18, 2.4725445029768842e+18, 2.7325842783378632e+18,
        3.0199726756097413e+18, 3.3375859744669926e+18, 3.6886029555580964e+18,
        4.0765367148106660e+18, 4.5052698236763945e+18, 4.9790931872109588e+18,
        5.5027489888941455e+18, 6.0814781519959931e+18, 6.7210727924983665e+18,
        7.4279341885386875e+18, 8.2091368465528033e+18, 9.0724993053134008e+18,
        1.0026662386493903e+19, 1.1081175674915967e+19, 1.2246593094004429e+19,
        1.3534578532999780e+19, 1.4958022583082340e+19, 1.6531171550741404e+19,
        1.8269770039598803e+19, 2.0191218527694397e+19, 2.2314747517318074e+19,
        2.4661610000340734e+19, 2.7255294165300175e+19, 3.0121758475086463e+19,
        3.3289691467964277e+19, 3.6790798882105184e+19, 4.0660120977272758e+19,
        4.4936383229519495e+19, 4.9662383908766908e+19, 5.4885422418277278e+19,
        6.0657772682977313e+19, 6.7037206324470079e+19, 7.4087570858841309e+19,
        8.1879428704059785e+19, 9.0490763392375407e+19, 1.0000776005571789e+20,
        1.1052566799546697e+20, 1.2214975396947463e+20, 1.3499635573715799e+20,
        1.4919404640690183e+20, 1.6488492123893672e+20, 1.8222601978246680e+20,
        2.0139089758026023e+20, 2.2257136317085365e+20, 2.4597939777288110e+20,
        2.7184927686435029e+20, 3.0043991489037533e+20, 3.3203745656596601e+20,
        3.6695814070850958e+20, 4.0555146526215674e+20, 4.4820368519070253e+20,
        4.9534167824709799e+20, 5.4743721730947816e+20, 6.0501169204269036e+20,
        6.6864132714132197e+20, 7.3896294938009534e+20, 8.1668036119028944e+20,
        9.0257138455102489e+20, 9.9749564569306608e+20, 1.1024031785270605e+21,
        1.2183439329022653e+21, 1.3464782828574936e+21, 1.4880886400345398e+21,
        1.6445922884849167e+21, 1.8175555693249954e+21, 2.0087095572044141e+21,
        2.2199673854829336e+21, 2.4534433935121722e+21, 2.7114742876544838e+21,
        2.9966425278256010e+21, 3.3118021736215536e+21, 3.6601074487062634e+21,
        4.0450443093422228e+21, 4.4704653330124248e+21,
        4.9406282763106685e+21])

    dlf.factor = np.array([1.1051709180756479])

    dlf.j0 = np.array([
        0.21035620538389819885E-28, -0.12644693616088940552E-13,
        0.46157312567885668321E-13, -0.27987033742576678494E-13,
        0.54657649654108409156E-13, -0.26529331099287291499E-13,
        0.56749134340673213135E-13, -0.21572768289772080733E-13,
        0.58318460867739760925E-13, -0.15465892848687829700E-13,
        0.60573024556529743179E-13, -0.85025312590830646706E-14,
        0.63880180611476449908E-13, -0.56596576350102877128E-15,
        0.68485006047914070374E-13, 0.85728977321682762439E-14,
        0.74650681546818133979E-13, 0.19208372932613381433E-13,
        0.82693454289757706437E-13, 0.31701165629228998860E-13,
        0.93000040396952081623E-13, 0.46490696394179916916E-13,
        0.10604419444905640479E-12, 0.64112165895974571186E-13,
        0.12240608340017008854E-12, 0.85217767515070225126E-13,
        0.14279579404871719178E-12, 0.11060266069684630524E-12,
        0.16808202030984049793E-12, 0.14123670281595459178E-12,
        0.19932710117763080694E-12, 0.17830320429641190908E-12,
        0.23782981967994220694E-12, 0.22324626027650970672E-12,
        0.28517768171070337866E-12, 0.27782855757859949941E-12,
        0.34331077352335570356E-12, 0.34420197641399489491E-12,
        0.41459976123278392377E-12, 0.42499381982249688902E-12,
        0.50194116319499513568E-12, 0.52341213107734220974E-12,
        0.60887371932475596308E-12, 0.64337432538102752823E-12,
        0.73972052807969328662E-12, 0.78966429788659195358E-12,
        0.89976265596232089469E-12, 0.96812431295714248033E-12,
        0.10954511874715580138E-11, 0.11858893754914550946E-11,
        0.13346662261643230004E-11, 0.14516734901185850330E-11,
        0.16270332417806469835E-11, 0.17761192965262209981E-11,
        0.19843094598657728049E-11, 0.21722251127126358340E-11,
        0.24208558013562638359E-11, 0.26558665246211218806E-11,
        0.29542133130007581834E-11, 0.32464334551104291636E-11,
        0.36058072230543221354E-11, 0.39676082798212998573E-11,
        0.44018068787207206107E-11, 0.48483162182209038194E-11,
        0.53741760778847699533E-11, 0.59238861421284167256E-11,
        0.65619559488550109069E-11, 0.72374683888302430790E-11,
        0.80128318647926483616E-11, 0.88417664804021047188E-11,
        0.97850472788000974558E-11, 0.10801152249024990298E-10,
        0.11949741288775620211E-10, 0.13194249255521529763E-10,
        0.14593803746893311587E-10, 0.16117088182600741411E-10,
        0.17823362499440759998E-10, 0.19686960839662181492E-10,
        0.21768042712348459703E-10, 0.24047127453754399616E-10,
        0.26586169224245549843E-10, 0.29372566166660639848E-10,
        0.32471120715874496926E-10, 0.35876995485484108464E-10,
        0.39659090711123993131E-10, 0.43821451522206353096E-10,
        0.48438566886024388833E-10, 0.53524764256840081139E-10,
        0.59161909123775483693E-10, 0.65376353273289546079E-10,
        0.72259490983916676274E-10, 0.79851856505621554776E-10,
        0.88256972132554113981E-10, 0.97532219231111761839E-10,
        0.10779639493701439540E-09, 0.11912700941828918848E-09,
        0.13166195190543400137E-09, 0.14550289515667290553E-09,
        0.16081145810919829686E-09, 0.17771842706736448843E-09,
        0.19641479168712890847E-09, 0.21706652163468219836E-09,
        0.23990084518390418007E-09, 0.26512635046403231588E-09,
        0.29301487204485379059E-09, 0.32382671796405940321E-09,
        0.35788852978338560143E-09, 0.39552347102192978760E-09,
        0.43712543089936042400E-09, 0.48309404739375797825E-09,
        0.53390563500721587301E-09, 0.59005295736900699180E-09,
        0.65211327580989620296E-09, 0.72069283339348376395E-09,
        0.79649244503723412979E-09, 0.88025670846750671949E-09,
        0.97283758951862838429E-09, 0.10751484374562240226E-08,
        0.11882260626931209546E-08, 0.13131897062580609792E-08,
        0.14513021636655630472E-08, 0.16039339435115999074E-08,
        0.17726240632935599375E-08, 0.19590497332198700268E-08,
        0.21650875406672578739E-08, 0.23927891159868709863E-08,
        0.26444435360147638239E-08, 0.29225595734392348364E-08,
        0.32299302912485969586E-08, 0.35696226515762041770E-08,
        0.39450454481729238111E-08, 0.43599472612559631445E-08,
        0.48184890913636877388E-08, 0.53252519017629532640E-08,
        0.58853155833436851597E-08, 0.65042776355473761329E-08,
        0.71883404192425699700E-08, 0.79443429030829021263E-08,
        0.87798585629591097164E-08, 0.97032425780220928695E-08,
        0.10723743227689399320E-07, 0.11851567478400949647E-07,
        0.13098009332253010086E-07, 0.14475537424021579590E-07,
        0.15997944513720238914E-07, 0.17680461540553181382E-07,
        0.19539933354871179973E-07, 0.21594964684505000780E-07,
        0.23866128306162058448E-07, 0.26376149610345229076E-07,
        0.29150154762698309478E-07, 0.32215902055657983021E-07,
        0.35604079260985092872E-07, 0.39348591789544360374E-07,
        0.43486920453657868292E-07, 0.48060478694380069026E-07,
        0.53115044437493142888E-07, 0.58701201380017479006E-07,
        0.64874861635712681668E-07, 0.71697809408859342572E-07,
        0.79238334805050121668E-07, 0.87571902294266750180E-07,
        0.96781920558355529704E-07, 0.10696056312048640378E-06,
        0.11820970459254829996E-06, 0.13064192692376749733E-06,
        0.14438165911984879199E-06, 0.15956640998357920017E-06,
        0.17634815657222329376E-06, 0.19489485336503760544E-06,
        0.21539212473518470874E-06, 0.23804511154682969141E-06,
        0.26308053514448537707E-06, 0.29074895589985508713E-06,
        0.32132729115584617374E-06, 0.35512157675298091583E-06,
        0.39247003960676882480E-06, 0.43374647340783931675E-06,
        0.47936398879211592319E-06, 0.52977913903701991577E-06,
        0.58549649799821895163E-06, 0.64707370170465826468E-06,
        0.71512703747583724585E-06, 0.79033760405820345763E-06,
        0.87345813593705873095E-06, 0.96532052953043588492E-06,
        0.10668441762993300587E-05, 0.11790451573234470162E-05,
        0.13030464193827640161E-05, 0.14400890071820729325E-05,
        0.15915444905568059534E-05, 0.17589286853766969218E-05,
        0.19439168303884759854E-05, 0.21483603476946079820E-05,
        0.23743053781113989137E-05, 0.26240132540945578888E-05,
        0.28999831372928098690E-05, 0.32049770257732388059E-05,
        0.35420474020978068237E-05, 0.39145677786671023699E-05,
        0.43262664657480666650E-05, 0.47812638810077579977E-05,
        0.52841137925459449193E-05, 0.58398488901504216978E-05,
        0.64540311583956847791E-05, 0.71328075387126483640E-05,
        0.78829714540444842174E-05, 0.87120307957926337232E-05,
        0.96282830690788715692E-05, 0.10640898433258409191E-04,
        0.11760011483484689510E-04, 0.12996822677619209683E-04,
        0.14363710437469910021E-04, 0.15874355032820411062E-04,
        0.17543875501207770228E-04, 0.19388980961051210830E-04,
        0.21428137844875080257E-04, 0.23681754714302638776E-04,
        0.26172386518181301401E-04, 0.28924960326686671307E-04,
        0.31967024811679323171E-04, 0.35329025959273332330E-04,
        0.39044611784553647914E-04, 0.43150969087563300697E-04,
        0.47689195631023510063E-04, 0.52704711454198507913E-04,
        0.58247713449355698436E-04, 0.64373677739542765810E-04,
        0.71143914895687183756E-04, 0.78626183537770216584E-04,
        0.86895368472094227166E-04, 0.96034230136811580777E-04,
        0.10613423286950590218E-03, 0.11729646026571620215E-03,
        0.12963262680752149015E-03, 0.14326619585462989793E-03,
        0.15833361521503999387E-03, 0.17498568260644121238E-03,
        0.19338905472215898854E-03, 0.21372791490647819607E-03,
        0.23620581621746139576E-03, 0.26104771814107751186E-03,
        0.28850223750643329973E-03, 0.31884413578705622606E-03,
        0.35237706800135917308E-03, 0.38943662007281590462E-03,
        0.43039366566958368833E-03, 0.47565807487700027594E-03,
        0.52568281303167982161E-03, 0.58096846836193664822E-03,
        0.64206825611437718954E-03, 0.70959354469423893946E-03,
        0.78421996374912559657E-03, 0.86669414659246905939E-03,
        0.95784118347590099622E-03, 0.10585728435434579864E-02,
        0.11698966653840010659E-02, 0.12929259749833240008E-02,
        0.14288909657393959393E-02, 0.15791508896276298946E-02,
        0.17452075486231769744E-02, 0.19287201026563420645E-02,
        0.21315214730933669356E-02, 0.23556362776736309780E-02,
        0.26033007312297349808E-02, 0.28769842725422829777E-02,
        0.31794136300501061286E-02, 0.35135987233342810820E-02,
        0.38828616256736492134E-02, 0.42908672540316900729E-02,
        0.47416579734868603835E-02, 0.52396893384671194144E-02,
        0.57898709851889693795E-02, 0.63976070720003401851E-02,
        0.70688437831027441105E-02, 0.78101127966257229834E-02,
        0.86285849757206656285E-02, 0.95321125348557418644E-02,
        0.10529286971052299882E-01, 0.11629470426845669312E-01,
        0.12842853010275019979E-01, 0.14180454004084899408E-01,
        0.15654168426849128515E-01, 0.17276700258609969246E-01,
        0.19061578784670331344E-01, 0.21022951738450229575E-01,
        0.23175536220878610594E-01, 0.25534136813168170632E-01,
        0.28113470567056909888E-01, 0.30927161344728969217E-01,
        0.33987341082207328524E-01, 0.37302669043183162012E-01,
        0.40877565846017260842E-01, 0.44708454582277110112E-01,
        0.48782456555642082774E-01, 0.53070463888373879680E-01,
        0.57525214796560822372E-01, 0.62068889249144526543E-01,
        0.66590986905823254527E-01, 0.70926870977039285782E-01,
        0.74857621553299974471E-01, 0.78074664211727651253E-01,
        0.80188872113380424422E-01, 0.80676406709186576638E-01,
        0.78917673064227769619E-01, 0.74124063016304961304E-01,
        0.65458647531413310938E-01, 0.51957717257333460581E-01,
        0.32847972741848592559E-01, 0.74970763258312700383E-02,
        -0.23866128698945488634E-01, -0.60174943784761181220E-01,
        -0.98178997988506697125E-01, -0.13281477972726110637E+00,
        -0.15546285697725620301E+00, -0.15639821579874499391E+00,
        -0.12430498665290500016E+00, -0.54868159863436967438E-01,
        0.46862558991703072431E-01, 0.15112182958062059246E+00,
        0.21193155344105990556E+00, 0.16951341358877961008E+00,
        0.13861974203314799889E-01, -0.18693504518381348634E+00,
        -0.24558896069253360883E+00, -0.53092694018998139172E-01,
        0.25199984157985949595E+00, 0.19682248760574280744E+00,
        -0.20143336189691599114E+00, -0.24584508272586030886E+00,
        0.34335593140766357267E+00, -0.47700665106262918336E-01,
        -0.20966284735074519618E+00, 0.25090851237942479734E+00,
        -0.16764126613351920669E+00, 0.74951997868967626393E-01,
        -0.16951351027511458308E-01, -0.88285242013749955919E-02,
        0.16167873561835501006E-01, -0.15564117559925290737E-01,
        0.12513111252163700016E-01, -0.92993152469780984010E-02,
        0.66505259967794931597E-02, -0.46677074086718837662E-02,
        0.32488855455568548675E-02, -0.22555291085446750252E-02,
        0.15668752111009150857E-02, -0.10911324863680910667E-02,
        0.76253830490409491537E-03, -0.53525405778204085232E-03,
        0.37771045672146697511E-03, -0.26825282115352968703E-03,
        0.19202652749967551236E-03, -0.13882137232089440237E-03,
        0.10159989178100049974E-03, -0.75497250626312589804E-04,
        0.57141729200529359656E-04, -0.44191609676806338589E-04,
        0.35017761995785832974E-04, -0.28484969541399899885E-04,
        0.23801048173241150913E-04, -0.20412749240321660898E-04,
        0.17933634897918600856E-04, -0.16093677807779768610E-04,
        0.14703946920177649808E-04, -0.13632085400878389976E-04,
        0.12785408514801159958E-04, -0.12099103232765920005E-04,
        0.11527807463720849231E-04, -0.11039642499677090445E-04,
        0.10612147244787119201E-04, -0.10229546722558040478E-04,
        0.98808019172690629615E-05, -0.95581362683731644660E-05,
        0.92559894628748428139E-05, -0.89703680398349605195E-05,
        0.86984392262176360162E-05, -0.84381998993781744882E-05,
        0.81881850397080960968E-05, -0.79472759767526677708E-05,
        0.77146204892770847038E-05, -0.74895903346701209313E-05,
        0.72717128455631313132E-05, -0.70605952027200397989E-05,
        0.68558909586342899260E-05, -0.66573078263653524112E-05,
        0.64646106359474119065E-05, -0.62775967052488337714E-05,
        0.60960690045641779074E-05, -0.59198350149128057049E-05,
        0.57487213891732378171E-05, -0.55825764283683409239E-05,
        0.54212557290731716876E-05, -0.52646114832336199488E-05,
        0.51124978277574427215E-05, -0.49647806501954759280E-05,
        0.48213365261592663842E-05, -0.46820435304229967753E-05,
        0.45467775878200409083E-05, -0.44154180000760036035E-05,
        0.42878525031129105819E-05, -0.41639746681098469138E-05,
        0.40436784078298272207E-05, -0.39268575265170754164E-05,
        0.38134098513603608910E-05, -0.37032392398348510301E-05,
        0.35962529271829669603E-05, -0.34923586009061261710E-05,
        0.33914651799141980567E-05, -0.32934854312419398030E-05,
        0.31983363564463991755E-05, -0.31059371103484061739E-05,
        0.30162076790696608685E-05, -0.29290699115622908088E-05,
        0.28444489540948809787E-05, -0.27622729358204111026E-05,
        0.26824715684158770123E-05, -0.26049757091405561401E-05,
        0.25297182336640620443E-05, -0.24566346909850541750E-05,
        0.23856627938558707926E-05, -0.23167415846242317899E-05,
        0.22498114158806991699E-05, -0.21848145526023768863E-05,
        0.21216953789870668849E-05, -0.20603999269499001719E-05,
        0.20008754358939798722E-05, -0.19430704833474811314E-05,
        0.18869353395796189952E-05, -0.18324219538318030139E-05,
        0.17794835990689859392E-05, -0.17280746718615620423E-05,
        0.16781508408187099176E-05, -0.16296692208678659180E-05,
        0.15825882749397249515E-05, -0.15368675777174579453E-05,
        0.14924677441823879966E-05, -0.14493505440316709914E-05,
        0.14074789632159439540E-05, -0.13668170916979060623E-05,
        0.13273299804037580486E-05, -0.12889836283545319635E-05,
        0.12517450527435769451E-05, -0.12155822896230419416E-05,
        0.11804642977313919266E-05, -0.11463608782321180614E-05,
        0.11132426817551759487E-05, -0.10810812413606430626E-05,
        0.10498489463079790445E-05, -0.10195189699197359345E-05,
        0.99006522696480728454E-06, -0.96146238238294861780E-06,
        0.93368585903962674737E-06, -0.90671180391877650539E-06,
        0.88051703794158972103E-06, -0.85507903355065637431E-06,
        0.83037591856657379114E-06, -0.80638646950736030707E-06,
        0.78309007959684342440E-06, -0.76046672516600126341E-06,
        0.73849695280747625374E-06, -0.71716187755802708277E-06,
        0.69644316956135183135E-06, -0.67632302736954520768E-06,
        0.65678415552512063648E-06, -0.63780975556830963862E-06,
        0.61938351997573715339E-06, -0.60148961697207375561E-06,
        0.58411266955661727328E-06, -0.56723774005096348097E-06,
        0.55085032232163946943E-06, -0.53493633333997110944E-06,
        0.51948209868244907623E-06, -0.50447433644131324069E-06,
        0.48990014588904722006E-06, -0.47574699997612418437E-06,
        0.46200273605516768571E-06, -0.44865554313913170035E-06,
        0.43569394953682061395E-06, -0.42310681390672771474E-06,
        0.41088331789010578938E-06, -0.39901295702377770089E-06,
        0.38748552999527151342E-06, -0.37629112895780040410E-06,
        0.36542013202237861199E-06, -0.35486319617658597678E-06,
        0.34461124894551899091E-06, -0.33465547948734137742E-06,
        0.32498733079349469753E-06, -0.31559849314212400870E-06,
        0.30648089749286697872E-06, -0.29762670812339488137E-06,
        0.28902831526969758359E-06, -0.28067832866533619338E-06,
        0.27256957173628210936E-06, -0.26469507560502209121E-06,
        0.25704807272840818962E-06, -0.24962199077868072385E-06,
        0.24241044716871418470E-06, -0.23540724388966429376E-06,
        0.22860636218196940360E-06, -0.22200195709699679458E-06,
        0.21558835236008589946E-06, -0.20936003566018609291E-06,
        0.20331165407816460064E-06, -0.19743800942055008854E-06,
        0.19173405358761880565E-06, -0.18619488421725469657E-06,
        0.18081574059851621267E-06, -0.17559199964945570756E-06,
        0.17051917187001359107E-06, -0.16559289739464161294E-06,
        0.16080894226704971095E-06, -0.15616319488355259655E-06,
        0.15165166247713490835E-06, -0.14727046762624959541E-06,
        0.14301584488191908927E-06, -0.13888413756276189610E-06,
        0.13487179465891340034E-06, -0.13097536777476139733E-06,
        0.12719150812468619494E-06, -0.12351696364172060964E-06,
        0.11994857620999400758E-06, -0.11648327897320799497E-06,
        0.11311809368621049480E-06, -0.10985012813125349456E-06,
        0.10667657363213940060E-06, -0.10359470265995370341E-06,
        0.10060186649751670646E-06, -0.97695492950586510549E-07,
        0.94873084124749734212E-07, -0.92132214283314405060E-07,
        0.89470527774724840543E-07, -0.86885737009431156146E-07,
        0.84375620484431720137E-07, -0.81938020868743667602E-07,
        0.79570843154679285033E-07, -0.77272052863928245642E-07,
        0.75039674297478696595E-07, -0.72871788831488877790E-07,
        0.70766533266929795021E-07, -0.68722098232649864085E-07,
        0.66736726633350816695E-07, -0.64808712137167796518E-07,
        0.62936397705674278282E-07, -0.61118174170089166613E-07,
        0.59352478851276259248E-07, -0.57637794217708362227E-07,
        0.55972646579199561646E-07, -0.54355604818630417116E-07,
        0.52785279162883091201E-07, -0.51260319990189909104E-07,
        0.49779416670227040800E-07, -0.48341296436233499673E-07,
        0.46944723290471018939E-07, -0.45588496942848318593E-07,
        0.44271451780258727462E-07, -0.42992455864451837250E-07,
        0.41750409958276353087E-07, -0.40544246580819631899E-07,
        0.39372929090650920317E-07, -0.38235450795273419860E-07,
        0.37130834085523768675E-07, -0.36058129594866006882E-07,
        0.35016415383563508384E-07, -0.34004796146768310693E-07,
        0.33022402445156628786E-07, -0.32068389957362020693E-07,
        0.31141938754091167181E-07, -0.30242252593599009938E-07,
        0.29368558237615870861E-07, -0.28520104786762129308E-07,
        0.27696163034962460481E-07, -0.26896024842649531045E-07,
        0.26119002528299780677E-07, -0.25364428277531399452E-07,
        0.24631653569082881086E-07, -0.23920048617305701087E-07,
        0.23229001830889051246E-07, -0.22557919287333441164E-07,
        0.21906224222551861542E-07, -0.21273356535101449785E-07,
        0.20658772304732728872E-07, -0.20061943324939738367E-07,
        0.19482356649058811450E-07, -0.18919514149424371449E-07,
        0.18372932089200169455E-07, -0.17842140706600119178E-07,
        0.17326683811178871537E-07, -0.16826118391794460252E-07,
        0.16340014235853439520E-07, -0.15867953559529469194E-07,
        0.15409530648689289756E-07, -0.14964351510224189348E-07,
        0.14532033533448680711E-07, -0.14112205161252269515E-07,
        0.13704505570744190592E-07, -0.13308584363144380637E-07,
        0.12924101262648650824E-07, -0.12550725823983400048E-07,
        0.12188137148392460070E-07, -0.11836023607829919776E-07,
        0.11494082577135149689E-07, -0.11162020173950579471E-07,
        0.10839551006143939185E-07, -0.10526397926519000691E-07,
        0.10222291794616780561E-07, -0.99269712454067339727E-08,
        0.96401824646609211080E-08, -0.93616789708108062008E-08,
        0.90912214031028235422E-08, -0.88285773158782117222E-08,
        0.85735209788006156203E-08, -0.83258331828536520172E-08,
        0.80853010519388850005E-08, -0.78517178599159764754E-08,
        0.76248828529317467947E-08, -0.74046010768841071059E-08,
        0.71906832098687622728E-08, -0.69829453994641022983E-08,
        0.67812091047173317459E-08, -0.65853009426977313290E-08,
        0.63950525394835678792E-08, -0.62103003854524726628E-08,
        0.60308856947513333758E-08, -0.58566542688268000505E-08,
        0.56874563638996400563E-08, -0.55231465622676621705E-08,
        0.53635836473256503502E-08, -0.52086304821955339059E-08,
        0.50581538918636123925E-08, -0.49120245487234261852E-08,
        0.47701168614250102390E-08, -0.46323088669346301918E-08,
        0.44984821257128528939E-08, -0.43685216199214189850E-08,
        0.42423156545711511770E-08, -0.41197557615253826557E-08,
        0.40007366062763786183E-08, -0.38851558974150727748E-08,
        0.37729142987165276171E-08, -0.36639153437652961865E-08,
        0.35580653530469988486E-08, -0.34552733534349709264E-08,
        0.33554510000030559535E-08, -0.32585125000973940249E-08,
        0.31643745396017359674E-08, -0.30729562113327868796E-08,
        0.29841789455041530200E-08, -0.28979664421992758989E-08,
        0.28142446057953132032E-08, -0.27329414812814390995E-08,
        0.26539871924168161407E-08, -0.25773138816751668664E-08,
        0.25028556519244291169E-08, -0.24305485097912959499E-08,
        0.23603303106619080563E-08, -0.22921407052714058628E-08,
        0.22259210878365399183E-08, -0.21616145456867578372E-08,
        0.20991658103504500059E-08, -0.20385212100542671023E-08,
        0.19796286235947108123E-08, -0.19224374355423869606E-08,
        0.18668984927404269626E-08, -0.18129640620596530537E-08,
        0.17605877893741759270E-08, -0.17097246597221680675E-08,
        0.16603309586176070024E-08, -0.16123642344797300183E-08,
        0.15657832621478869464E-08, -0.15205480074504270167E-08,
        0.14766195927971890460E-08, -0.14339602637660350412E-08,
        0.13925333566546899287E-08, -0.13523032669700050132E-08,
        0.13132354188275479691E-08, -0.12752962352352419048E-08,
        0.12384531092355140920E-08, -0.12026743758811379070E-08,
        0.11679292850206920657E-08, -0.11341879748702300079E-08,
        0.11014214463484779337E-08, -0.10696015381534739059E-08,
        0.10387009025592480307E-08, -0.10086929819117270958E-08,
        0.97955198580366660312E-09, -0.95125286890900208679E-09,
        0.92377130945756002788E-09, -0.89708368833163302057E-09,
        0.87116706876644941479E-09, -0.84599917663709281674E-09,
        0.82155838131493233922E-09, -0.79782367707710818756E-09,
        0.77477466505309300292E-09, -0.75239153569281363935E-09,
        0.73065505174126690031E-09, -0.70954653170499806391E-09,
        0.68904783379622916597E-09, -0.66914134034083904798E-09,
        0.64980994263679380996E-09, -0.63103702625001653087E-09,
        0.61280645673505631634E-09, -0.59510256576828648337E-09,
        0.57791013768171107199E-09, -0.56121439638580773724E-09,
        0.54500099267016873900E-09, -0.52925599187102496775E-09,
        0.51396586189505248004E-09, -0.49911746158916965897E-09,
        0.48469802944632955612E-09, -0.47069517263760014005E-09,
        0.45709685636110621144E-09, -0.44389139349867930081E-09,
        0.43106743457132480471E-09, -0.41861395798487421376E-09,
        0.40652026055743952578E-09, -0.39477594832052842209E-09,
        0.38337092758591272230E-09, -0.37229539627057491552E-09,
        0.36153983547227521766E-09, -0.35109500128849910154E-09,
        0.34095191687175519511E-09, -0.33110186471439289704E-09,
        0.32153637915631239045E-09, -0.31224723910912398970E-09,
        0.30322646099050710905E-09, -0.29446629186269331473E-09,
        0.28595920276917881651E-09, -0.27769788226393698380E-09,
        0.26967523012757238777E-09, -0.26188435126501301881E-09,
        0.25431854977949852574E-09, -0.24697132321776888896E-09,
        0.23983635698150977934E-09, -0.23290751890024982077E-09,
        0.22617885396104611209E-09, -0.21964457919042880688E-09,
        0.21329907868420528750E-09, -0.20713689878085250208E-09,
        0.20115274337434959303E-09, -0.19534146936242159446E-09,
        0.18969808222628370222E-09, -0.18421773173808480806E-09,
        0.17889570779236339873E-09, -0.17372743635793169822E-09,
        0.16870847554670939670E-09, -0.16383451179612808855E-09,
        0.15910135616182540748E-09, -0.15450494071744330869E-09,
        0.15004131505843449035E-09, -0.14570664290687508993E-09,
        0.14149719881436189859E-09, -0.13740936496016368823E-09,
        0.13343962804187350973E-09, -0.12958457625588848896E-09,
        0.12584089636512460714E-09, -0.12220537085144369393E-09,
        0.11867487515034671199E-09, -0.11524637496555659739E-09,
        0.11191692366118209830E-09, -0.10868365972922199935E-09,
        0.10554380433023220449E-09, -0.10249465890504320142E-09,
        0.99533602855475026004E-10, -0.96658091292055187607E-10,
        0.93865652846806834872E-10, -0.91153887549225778384E-10,
        0.88520464763624569567E-10, -0.85963121186075071965E-10,
        0.83479658899239781350E-10, -0.81067943483446593655E-10,
        0.78725902182445575131E-10, -0.76451522122414838220E-10,
        0.74242848583018786633E-10, -0.72097983319807716229E-10,
        0.70015082938317186447E-10, -0.67992357322650702613E-10,
        0.66028068126905105916E-10, -0.64120527350693063962E-10,
        0.62268096049913068230E-10, -0.60469183303296924744E-10,
        0.58722245716347976830E-10, -0.57025788118346391824E-10,
        0.55378366976800887211E-10, -0.53778600071222512844E-10,
        0.52225190653632309517E-10, -0.50716985205207501023E-10,
        0.49253109171572777739E-10, -0.47833283755322262690E-10,
        0.46458563164427748566E-10, -0.45133048314954522821E-10,
        0.43867868281155817508E-10, -0.42690428488605550129E-10,
        0.41665890740660430845E-10, -0.40947061319725302450E-10,
        0.40890256062850869212E-10, -0.42324395200868817252E-10,
        0.47175970286618397865E-10, -0.59920514722972034865E-10,
        0.90953607290146280299E-10])

    dlf.j1 = np.array([
        -0.23779001100582381051E-28, 0.79721192934236443185E-27,
        0.38325661863384939290E-27, 0.84934033213920264104E-27,
        0.82321643716951481883E-27, 0.11003687170198490196E-26,
        0.13254287328216880532E-26, 0.15922464286287840307E-26,
        0.19986019650982509534E-26, 0.23685465814544270716E-26,
        0.29798793392175001283E-26, 0.35431019643707156492E-26,
        0.44257411307894436460E-26, 0.53180794470588352748E-26,
        0.65556190066651513897E-26, 0.79947762199059232857E-26,
        0.97071583223006191650E-26, 0.12004979935771289997E-25,
        0.14406653509631588960E-25, 0.17970467112699249374E-25,
        0.21453903977135419404E-25, 0.26816147371136119450E-25,
        0.32037836290813990881E-25, 0.39922197526685471353E-25,
        0.47937803770632117330E-25, 0.59351247450921383090E-25,
        0.71773343544187673606E-25, 0.88252267695870270015E-25,
        0.10736460080008319237E-24, 0.13139234241813519897E-24,
        0.16038389953077929496E-24, 0.19586330559120450202E-24,
        0.23934895950562681781E-24, 0.29216907136422889152E-24,
        0.35704555610974980825E-24, 0.43591578004654766279E-24,
        0.53258407091337795497E-24, 0.65037749121846816863E-24,
        0.79446117848346841533E-24, 0.97029807615169898533E-24,
        0.11851621862641970763E-23, 0.14475349036735769831E-23,
        0.17680470446219830604E-23, 0.21594646724325061636E-23,
        0.26376279226241200337E-23, 0.32215255903424431280E-23,
        0.39348989123644471831E-23, 0.48059305631057603546E-23,
        0.58701957447819091530E-23, 0.71696014006070338140E-23,
        0.87572874338578454769E-23, 0.10695830329268070127E-22,
        0.13064267975684400067E-22, 0.15956396898683340405E-22,
        0.19489490174852410262E-22, 0.23804258576458750974E-22,
        0.29074795736995329016E-22, 0.35511877512642321550E-22,
        0.43374414815306912102E-22, 0.52977566601218580972E-22,
        0.64706963771419739714E-22, 0.79033290084770208947E-22,
        0.96531416601155305002E-22, 0.11790381653809730581E-21,
        0.14400798270141680420E-21, 0.17589176532470089827E-21,
        0.21483473765595870697E-21, 0.26239961572838840870E-21,
        0.32049580752851479664E-21, 0.39145421602105668850E-21,
        0.47812354861088417904E-21, 0.58398109523939098113E-21,
        0.71327648305877594511E-21, 0.87119745891754148321E-21,
        0.10640834348623049967E-20, 0.12996739243852600375E-20,
        0.15874259111262610786E-20, 0.19388856933392509112E-20,
        0.23681611511416871136E-20, 0.28924775863608318981E-20,
        0.35328812735771197862E-20, 0.43150695199493149714E-20,
        0.52704395186827729765E-20, 0.64373272953957132288E-20,
        0.78625718444181387840E-20, 0.96033638142877597977E-20,
        0.11729578932087649353E-19, 0.14326535217816349936E-19,
        0.17498475666460750925E-19, 0.21372678938714691346E-19,
        0.26104658300704859397E-19, 0.31884290164425472060E-19,
        0.38943574121348422642E-19, 0.47565771349593827251E-19,
        0.58096985951299433442E-19, 0.70959792070530976791E-19,
        0.86670519080596190619E-19, 0.10585956951243269563E-18,
        0.12929722209660508847E-18, 0.15792391867645680267E-18,
        0.19288879140366910337E-18, 0.23559479941190520470E-18,
        0.28775626657086499509E-18, 0.35146613570216158086E-18,
        0.42928191135495900673E-18, 0.52432585404586203783E-18,
        0.64041336717328802188E-18, 0.78220224663792857958E-18,
        0.95538449298914256672E-18, 0.11669086110519650175E-17,
        0.14252662063096260770E-17, 0.17408230557831051757E-17,
        0.21262473650918901610E-17, 0.25970027813123101601E-17,
        0.31719883922980728330E-17, 0.38742728138556379626E-17,
        0.47320507187765319669E-17, 0.57797357502250896216E-17,
        0.70593902823169582028E-17, 0.86223523498694457235E-17,
        0.10531373010190409793E-16, 0.12863037889413019372E-16,
        0.15710962731510451527E-16, 0.19189397138461720101E-16,
        0.23438002820075089885E-16, 0.28627215837238828004E-16,
        0.34965392409988710645E-16, 0.42706786428448497367E-16,
        0.52162237447471309093E-16, 0.63711036879213345092E-16,
        0.77816916463307065235E-16, 0.95045695365657204522E-16,
        0.11608920160259351019E-15, 0.14179151105966549619E-15,
        0.17318474398326101217E-15, 0.21152807068685210363E-15,
        0.25836128767539467929E-15, 0.31556278833430849848E-15,
        0.38542976466718232176E-15, 0.47076434267244616602E-15,
        0.57499366556778888035E-15, 0.70229784368391582701E-15,
        0.85778978838088164829E-15, 0.10477052216252109330E-14,
        0.12796720504321799823E-14, 0.15629924515539289125E-14,
        0.19090464627227088193E-14, 0.23317106244241540845E-14,
        0.28479628092949809297E-14, 0.34785033119698000940E-14,
        0.42486614900323137393E-14, 0.51893168582342936762E-14,
        0.63382585119205047931E-14, 0.77415505884680514824E-14,
        0.94555711726406619067E-14, 0.11549035630408280097E-13,
        0.14106055531142479946E-13, 0.17229135422830471373E-13,
        0.21043763493017538471E-13, 0.25702847898914450407E-13,
        0.31393608430354023054E-13, 0.38344140375861572417E-13,
        0.46833764078042948155E-13, 0.57202731000069601939E-13,
        0.69867771751590042453E-13, 0.85336439559542886244E-13,
        0.10423047667756279584E-12, 0.12730699655547051243E-12,
        0.15549361393599358810E-12, 0.18991970329598081032E-12,
        0.23196923668250597755E-12, 0.28332687489467591536E-12,
        0.34605747293074408206E-12, 0.42267398347606569328E-12,
        0.51625714279965715259E-12, 0.63055541477094275274E-12,
        0.77016524760249830236E-12, 0.94067802567402927053E-12,
        0.11489516827349049439E-11, 0.14033265286503660695E-11,
        0.17140347264354340417E-11, 0.20935168852125301327E-11,
        0.25570397012560628610E-11, 0.31231597365723921107E-11,
        0.38146555551079558782E-11, 0.46592061049755500161E-11,
        0.56907982819762827155E-11, 0.69507175913176593709E-11,
        0.84896748701239594140E-11, 0.10369250352182410092E-10,
        0.12665108932133729727E-10, 0.15469100894640800720E-10,
        0.18894125922585180356E-10, 0.23077182209924898792E-10,
        0.28186729422033271362E-10, 0.34427103160373659936E-10,
        0.42049668170180476841E-10, 0.51359191342712271588E-10,
        0.62730747867977636297E-10, 0.76618892012631225296E-10,
        0.93583301711963271940E-10, 0.11430192677538819367E-09,
        0.13960991678261530046E-09, 0.17051839200598010096E-09,
        0.20827357888452780935E-09, 0.25438347678424908063E-09,
        0.31070775906120548243E-09, 0.37949543960166480335E-09,
        0.46352165150554629058E-09, 0.56614048928061041213E-09,
        0.69149327357632214648E-09, 0.84458207882202713872E-09,
        0.10315871013042910049E-08, 0.12599679520777700732E-08,
        0.15389476810193000824E-08, 0.18796505921747109347E-08,
        0.22958410530823679168E-08, 0.28041080392694730677E-08,
        0.34249937978079458891E-08, 0.41832358233125644067E-08,
        0.51094925678131643906E-08, 0.62406516508546298375E-08,
        0.76224707236306606749E-08, 0.93099537030146089046E-08,
        0.11371395622837669325E-07, 0.13888811603603460790E-07,
        0.16964137540446880094E-07, 0.20719660567011048447E-07,
        0.25307532985450009557E-07, 0.30910082760196839954E-07,
        0.37754423983369756739E-07, 0.46112394911463966981E-07,
        0.56323015017988116740E-07, 0.68791561040010018689E-07,
        0.84024115221375004322E-07, 0.10262487036321569423E-06,
        0.12534932630323100331E-06, 0.15309818493173440690E-06,
        0.18699933285775400780E-06, 0.22839543319719978897E-06,
        0.27897038209650021238E-06, 0.34072557173474452451E-06,
        0.41617511631609012044E-06, 0.50830216571405257032E-06,
        0.62086055851273829343E-06, 0.75829654716288451693E-06,
        0.92621529324795927674E-06, 0.11312433484295780322E-05,
        0.13817506779893170808E-05, 0.16876126549713120575E-05,
        0.20613285048301949073E-05, 0.25176142217059249495E-05,
        0.30751364937151691114E-05, 0.37558230263550338492E-05,
        0.45875527884742903516E-05, 0.56029966733669012944E-05,
        0.68437947583602416914E-05, 0.83586201447068474255E-05,
        0.10209670082926559548E-04, 0.12469450254224749995E-04,
        0.15230868871022190138E-04, 0.18601920832727638478E-04,
        0.22721395800711319485E-04, 0.27750126132155010419E-04,
        0.33895447798826631189E-04, 0.41396841136470921454E-04,
        0.50564046316342118052E-04, 0.61753575721436104002E-04,
        0.75428141719150856360E-04, 0.92118338007583430976E-04,
        0.11251534681498950364E-03, 0.13740856459760000547E-03,
        0.16783029354650548710E-03, 0.20495440270807768974E-03,
        0.25032223970649432505E-03, 0.30567824291028637771E-03,
        0.37332275750272562662E-03, 0.45584569317397240131E-03,
        0.55667758728412204596E-03, 0.67965842762967730497E-03,
        0.82989825334180589590E-03, 0.10130793883812590737E-02,
        0.12367998790718090669E-02, 0.15094441288791659965E-02,
        0.18422782918639890679E-02, 0.22476208557991408411E-02,
        0.27421062075955880862E-02, 0.33437164053099180556E-02,
        0.40768527847886001578E-02, 0.49675073471938113734E-02,
        0.60511176276551595712E-02, 0.73646815739407444149E-02,
        0.89588637538056889453E-02, 0.10885052122857430054E-01,
        0.13213887638701850638E-01, 0.16013879136554731025E-01,
        0.19379495481742049084E-01, 0.23395566125555520426E-01,
        0.28179341735834879251E-01, 0.33820790606197338735E-01,
        0.40444641201110451134E-01, 0.48109746447546619663E-01,
        0.56899341828119280595E-01, 0.66751684235759015107E-01,
        0.77591144073115236868E-01, 0.89047612974465659152E-01,
        0.10065405918556659393E+00, 0.11140105420732690500E+00,
        0.12006932932879070541E+00, 0.12458597468885210247E+00,
        0.12268765549548429905E+00, 0.11115942050671910601E+00,
        0.87319954786645864742E-01, 0.48465000302246638575E-01,
        -0.49561566373757083917E-02, -0.69471081820978636268E-01,
        -0.13351269533458298699E+00, -0.17957507285653320772E+00,
        -0.18191106295048839470E+00, -0.12098476462017759836E+00,
        0.57914272269164483384E-02, 0.15348128813633118983E+00,
        0.23522097310698558692E+00, 0.15150400266403860061E+00,
        -0.82790857821451305232E-01, -0.27115431040159521103E+00,
        -0.13312798063762890100E+00, 0.22420582796659790925E+00,
        0.23042922487820899358E+00, -0.25728138031301517330E+00,
        -0.14755494381924760816E+00, 0.36119355267312441837E+00,
        -0.24353715522887228739E+00, 0.38114547256147109089E-01,
        0.82736503214593373756E-01, -0.10914677811197480428E+00,
        0.89643285228612118454E-01, -0.60977888159322063411E-01,
        0.37801231818370660542E-01, -0.22438621211053739329E-01,
        0.13156278189915569798E-01, -0.77896933116771488112E-02,
        0.47370324617059750630E-02, -0.29965293695965681403E-02,
        0.19878544860469768082E-02, -0.13869827135561199356E-02,
        0.10157400415733200925E-02, -0.77631830897451124893E-03,
        0.61460402632747919080E-03, -0.50020690522478944255E-03,
        0.41570616801530989705E-03, -0.35086447759715891749E-03,
        0.29949364033671969747E-03, -0.25773573414907620556E-03,
        0.22310488448276601017E-03, -0.19394286742651010405E-03,
        0.16910356690987170454E-03, -0.14776630535110279563E-03,
        0.12932306435161670878E-03, -0.11330886340153000630E-03,
        0.99357870873694533213E-04, -0.87175171695487733894E-04,
        0.76518203465147482511E-04, -0.67184226920608184730E-04,
        0.59001625485470532949E-04, -0.51823700437172301393E-04,
        0.45524131447897800190E-04, -0.39993558219308811687E-04,
        0.35136924058234570150E-04, -0.30871356264610351938E-04,
        0.27124444492712818783E-04, -0.23832821622649580026E-04,
        0.20940974022723349952E-04, -0.18400228343346008335E-04,
        0.16167881311147351016E-04, -0.14206450345841819494E-04,
        0.12483025923096489624E-04, -0.10968708114345040680E-04,
        0.96381137476371433619E-05, -0.84689451209231461783E-05,
        0.74416131426310713740E-05, -0.65389076895947963034E-05,
        0.57457083047103120349E-05, -0.50487298647161201493E-05,
        0.44362992864155191501E-05, -0.38981597741651851768E-05,
        0.34252990553006400639E-05, -0.30097983612462208859E-05,
        0.26446996057275441348E-05, -0.23238887351302340716E-05,
        0.20419933647589090787E-05, -0.17942928642745300170E-05,
        0.15766392837610099303E-05, -0.13853878318735780200E-05,
        0.12173358208997559817E-05, -0.10696690610117151049E-05,
        0.93991475334313001620E-06, -0.82590006822179900410E-06,
        0.72571573986650604055E-06, -0.63768409395308406728E-06,
        0.56033096589579911757E-06, -0.49236101932854606775E-06,
        0.43263604794155821972E-06, -0.38015590945392312373E-06,
        0.33404177953802387996E-06, -0.29352143989559692218E-06,
        0.25791634724263857612E-06, -0.22663026755387299218E-06,
        0.19913929019334100136E-06, -0.17498305605429440553E-06,
        0.15375705002049011184E-06, -0.13510582667935480436E-06,
        0.11871705693399870160E-06, -0.10431629758749350038E-06,
        0.91662396276475694184E-07, -0.80543453607028647290E-07,
        0.70773274428481856046E-07, -0.62188249446989189968E-07,
        0.54644615505335029221E-07, -0.48016048443646231508E-07,
        0.42191547784261369808E-07, -0.37073577780952398937E-07,
        0.32576433988135847120E-07, -0.28624808133909398953E-07,
        0.25152527127147581427E-07, -0.22101444936595110588E-07,
        0.19420468820503340806E-07, -0.17064703711658711163E-07,
        0.14994700450203629605E-07, -0.13175795208633650552E-07,
        0.11577529007302610104E-07, -0.10173137628831489796E-07,
        0.89391034277419880612E-08, -0.78547615254943383801E-08,
        0.69019537690386798258E-08, -0.60647246504485460851E-08,
        0.53290541108617648523E-08, -0.46826227654829832319E-08,
        0.41146056122897838021E-08, -0.36154907601472161940E-08,
        0.31769201408808051073E-08, -0.27915495445014601903E-08,
        0.24529256362912200658E-08, -0.21553778928767579448E-08,
        0.18939236446342600392E-08, -0.16641846353897590312E-08,
        0.14623137046133780063E-08, -0.12849303645740829450E-08,
        0.11290641920775340868E-08, -0.99210508606754888054E-09,
        0.87175955886335064848E-09, -0.76601232990455893213E-09,
        0.67309257867770258852E-09, -0.59144429108839117984E-09,
        0.51970020258742657674E-09, -0.45665890207353687588E-09,
        0.40126471338250552359E-09, -0.35259001730495680327E-09,
        0.30981971788986289322E-09, -0.27223759293244408980E-09,
        0.23921430022514762304E-09, -0.21019683877088560052E-09,
        0.18469928839465210504E-09, -0.16229467261876449390E-09,
        0.14260780857078909327E-09, -0.12530902424439949164E-09,
        0.11010863790384879428E-09, -0.96752107149079824750E-10,
        0.85015766394465004630E-10, -0.74703081400610626905E-10,
        0.65641358155343214620E-10, -0.57678850985617859679E-10,
        0.50682221459754282741E-10, -0.44534305525909120100E-10,
        0.39132151504364007283E-10, -0.34385296082594761887E-10,
        0.30214249439239317785E-10, -0.26549164124680559140E-10,
        0.23328665408564599349E-10, -0.20498823510896081544E-10,
        0.18012250506600650469E-10, -0.15827306778509629286E-10,
        0.13907403728951850551E-10, -0.12220391074163700618E-10,
        0.10738018462452050425E-10, -0.94354624003716234485E-11,
        0.82909105640953193728E-11, -0.72851965346947523766E-11,
        0.64014788411436639934E-11, -0.56249589367146987257E-11,
        0.49426333858928202302E-11, -0.43430761116809759680E-11,
        0.38162470568545827330E-11, -0.33533240552467311054E-11,
        0.29465550976975220819E-11, -0.25891285186428910646E-11,
        0.22750589294212621855E-11, -0.19990869881930792011E-11,
        0.17565913281172699705E-11, -0.15435111689614020217E-11,
        0.13562783161689700960E-11, -0.11917574086305238998E-11,
        0.10471934145827610682E-11, -0.92016549643814607864E-12,
        0.80854647197038528016E-12, -0.71046719298279401734E-12,
        0.62428524494839663246E-12, -0.54855744348363701401E-12,
        0.48201566709453555482E-12, -0.42354562149103289211E-12,
        0.37216817985480538490E-12, -0.32702298658923321260E-12,
        0.28735404998800310208E-12, -0.25249708256169828971E-12,
        0.22186837702370389530E-12, -0.19495503165314128815E-12,
        0.17130636135177430412E-12, -0.15052635056794931119E-12,
        0.13226702170589389284E-12, -0.11622260796808300942E-12,
        0.10212443304988860003E-12, -0.89736411943535084833E-13,
        0.78851097509314687508E-13, -0.69286206610625636538E-13,
        0.60881567639821682927E-13, -0.53496438318744010298E-13,
        0.47007148858659282416E-13, -0.41305031012602458038E-13,
        0.36294598340360191951E-13, -0.31891947213064369610E-13,
        0.28023351780988580919E-13, -0.24624029376283279094E-13,
        0.21637055676372050574E-13, -0.19012411461508211104E-13,
        0.16706145003659701281E-13, -0.14679636060291389006E-13,
        0.12898949148078830678E-13, -0.11334265266611379683E-13,
        0.99593825558268706535E-14, -0.87512775252863140427E-14,
        0.76897195077421143642E-14, -0.67569318807328292835E-14,
        0.59372943830891981248E-14, -0.52170815413984568005E-14,
        0.45842328261723863703E-14, -0.40281506888097158469E-14,
        0.35395230973258291850E-14, -0.31101675990681859105E-14,
        0.27328942991223000800E-14, -0.24013854598749791415E-14,
        0.21100897055373030150E-14, -0.18541290599993040151E-14,
        0.16292172612908320081E-14, -0.14315879847593459752E-14,
        0.12579317730060459754E-14, -0.11053406164233719234E-14,
        0.97125925629132513426E-15, -0.85344239496416429968E-15,
        0.74991709659831521783E-15, -0.65894974879250159170E-15,
        0.57901703188703568188E-15, -0.50878040978025863482E-15,
        0.44706371509062259811E-15, -0.39283345331036878858E-15,
        0.34518149612852799007E-15, -0.30330987410939582363E-15,
        0.26651741406787180955E-15, -0.23418799737396022297E-15,
        0.20578024256254929268E-15, -0.18081843947571900690E-15,
        0.15888458312267581200E-15, -0.13961137385800899984E-15,
        0.12267606666074510376E-15, -0.10779506651556279801E-15,
        0.94719179391603797610E-16, -0.83229439292786975771E-16,
        0.73133441500293154255E-16, -0.64262120604484965377E-16,
        0.56466919371882109638E-16, -0.49617301037651720597E-16,
        0.43598563364994760812E-16, -0.38309917866133909996E-16,
        0.33662802019951652661E-16, -0.29579396223041629810E-16,
        0.25991320639354369445E-16, -0.22838490126158601174E-16,
        0.20068108061152711033E-16, -0.17633782221571330287E-16,
        0.15494747909989438574E-16, -0.13615185317456429361E-16,
        0.11963619692670830272E-16, -0.10512394272558709913E-16,
        0.92372071480646483496E-17, -0.81167043096073984687E-17,
        0.71321220574247113037E-17, -0.62669727886873747397E-17,
        0.55067688996430172615E-17, -0.48387801793162163364E-17,
        0.42518206321314128635E-17, -0.37360611596341719162E-17,
        0.32828649645187700314E-17, -0.28846429206752079096E-17,
        0.25347264873153391346E-17, -0.22272560390247178611E-17,
        0.19570827417369101585E-17, -0.17196823314652230881E-17,
        0.15110793519499210062E-17, -0.13277805825221979748E-17,
        0.11667165413822330724E-17, -0.10251900847168659743E-17,
        0.90083124090830315999E-18, -0.79155752349720990603E-18,
        0.69553905830937584557E-18, -0.61116794077226843712E-18,
        0.53703131028706945667E-18, -0.47188769076765341320E-18,
        0.41464620115119749872E-18, -0.36434828774274431505E-18,
        0.32015167248500091253E-18, -0.28131624835279290921E-18,
        0.24719168567911820595E-18, -0.21720654187174320654E-18,
        0.19085869215550189721E-18, -0.16770692109721899111E-18,
        0.14736353410746910352E-18, -0.12948786519216900428E-18,
        0.11378057223455149305E-18, -0.99978624276428549682E-19,
        0.87850896856935804010E-19, -0.77194301650481244444E-19,
        0.67830385593241861930E-19, -0.59602342548611573044E-19,
        0.52372387468676202365E-19, -0.46019449077998791288E-19,
        0.40437142440414807125E-19, -0.35531987457892000724E-19,
        0.31221843469976618630E-19, -0.27434533741424270108E-19,
        0.24106636804928459342E-19, -0.21182424419102399996E-19,
        0.18612928356413968765E-19, -0.16355120393022199627E-19,
        0.14371191768318589670E-19, -0.12627920047644379558E-19,
        0.11096112785317059883E-19, -0.97501186719036165739E-20,
        0.85673979808058807363E-20, -0.75281451224231245595E-20,
        0.66149569855872109159E-20, -0.58125415113657893624E-20,
        0.51074616176521397340E-20, -0.44879101854824920831E-20,
        0.39435123390814399527E-20, -0.34651517093082292638E-20,
        0.30448177717368800233E-20, -0.26754717030230440586E-20,
        0.23509285093699011334E-20, -0.20657534540814548706E-20,
        0.18151710504142631532E-20, -0.15949850948474660527E-20,
        0.14015083996708770500E-20, -0.12315010469494319376E-20,
        0.10821161305372590522E-20, -0.95085207960662486644E-21,
        0.83551076750596628203E-21, -0.73416070535124345550E-21,
        0.64510470249568779156E-21, -0.56685144894552668532E-21,
        0.49809054123125183129E-21, -0.43767053435643358195E-21,
        0.38457965596782228367E-21, -0.33792886257541818238E-21,
        0.29693695347390241880E-21, -0.26091749061412948769E-21,
        0.22926730418560010001E-21, -0.20145639198759520765E-21,
        0.17701904387406560389E-21, -0.15554604225258168932E-21,
        0.13667780796729550641E-21, -0.12009837779858251158E-21,
        0.10553011379060780116E-21, -0.92729055585803990884E-22,
        0.81480836646395174933E-22, -0.71597094891875700200E-22,
        0.62912317547278588468E-22, -0.55281068025898321612E-22,
        0.48575549320856098216E-22, -0.42683463880645999618E-22,
        0.37506134255287311446E-22, -0.32956852260392350507E-22,
        0.28959427878761049247E-22, -0.25446913153691919200E-22,
        0.22360480394754259887E-22, -0.19648436888446069881E-22,
        0.17265359799157371192E-22, -0.15171336029025659557E-22,
        0.13331293566397819119E-22, -0.11714413365284629986E-22,
        0.10293612991331600087E-22, -0.90450942661818855010E-23,
        0.79479472072663889120E-23, -0.69838025973983943288E-23,
        0.61365260759698707238E-23, -0.53919477932897098293E-23,
        0.47376234574340661915E-23, -0.41626240529454467843E-23,
        0.36573513415638608941E-23, -0.32133750545472298724E-23,
        0.28232876298439711191E-23, -0.24805746766402519629E-23,
        0.21795021088492381699E-23, -0.19150207966391389939E-23,
        0.16826867531148651179E-23, -0.14785926095987099109E-23,
        0.12993065708576839773E-23, -0.11418169017687040810E-23,
        0.10034812450227850599E-23, -0.88198046609572075136E-24,
        0.77527684778131062791E-24, -0.68157678580025223715E-24,
        0.59929859006652480651E-24, -0.52704590637343920302E-24,
        0.46358624111797927966E-24, -0.40783289235105509668E-24,
        0.35882870194420578191E-24, -0.31573129057568951175E-24,
        0.27780031811537501369E-24, -0.24438721059930989498E-24,
        0.21492709123302868776E-24, -0.18893185489531158875E-24,
        0.16598283447655300523E-24, -0.14572197983278360944E-24,
        0.12784189795680629463E-24, -0.11207611655384429414E-24,
        0.98190635833986939235E-25, -0.85977069575473510383E-25,
        0.75247618025397713641E-25, -0.65832296928724367365E-25,
        0.57578153084183532562E-25, -0.50349088175171190647E-25,
        0.44024756065989681357E-25, -0.38498191971011007846E-25,
        0.33673291115583042791E-25, -0.29463411791224339510E-25,
        0.25791217847239958719E-25, -0.22588731181811621204E-25,
        0.19796670767579898895E-25, -0.17363295672276158774E-25,
        0.15243569288426309447E-25, -0.13398787953334260281E-25,
        0.11796045820480120133E-25, -0.10407147490908270710E-25,
        0.92071944889040744035E-26, -0.81731249295591442672E-26,
        0.72825159703487444195E-26, -0.65131006447129183541E-26,
        0.58430301262485386924E-26, -0.52513930081011199285E-26,
        0.47187919679302219783E-26, -0.42284069142292412393E-26,
        0.37676954747502728537E-26, -0.33299134001135826479E-26,
        0.29143004338630110718E-26, -0.25243668234065648260E-26,
        0.21646903157623978348E-26, -0.18376734028052161580E-26,
        0.15423017366879009263E-26, -0.12758944190588119779E-26,
        0.10371662830395429178E-26, -0.82748173500706471962E-27,
        0.64909427840638429032E-27, -0.50234666376562531025E-27,
        0.38452110973130778698E-27, -0.29091636205058451455E-27,
        0.21678009467166822096E-27, -0.15874578619111640200E-27,
        0.11537394235020309101E-27, -0.86900364189788941857E-28,
        0.74355723662636970071E-28, -0.78471428315427385396E-28,
        0.98949852128730596091E-28, -0.13427202805922251023E-27,
        0.18174242734021449278E-27, -0.23763619609508699858E-27,
        0.29778198107110528091E-27, -0.35868369680655068529E-27,
        0.41844766665021181973E-27, -0.47660851066880372752E-27,
        0.53287774452581460041E-27, -0.58588249858342115784E-27,
        0.63294098597067369917E-27, -0.67092689718604759703E-27,
        0.69754161721794173542E-27, -0.71229292612293631973E-27,
        0.71678213553979546309E-27, -0.71422093211775989035E-27,
        0.70847857438160299782E-27, -0.70324864718393631473E-27,
        0.70176504791510964044E-27, -0.70696084721646144402E-27,
        0.72149205056137611593E-27])

    return dlf


def key_51_2012():
    r"""Key 51 pt Hankel filter, as published in [Key12]_.

    Taken from file `kk51Hankel.txt` provided with SEG-2012-003_.

    License: https://software.seg.org/disclaimer.txt.

    """

    dlf = DigitalFilter('Key 51 (2012)', 'key_51_2012')

    dlf.base = np.array([
        4.9915939069102170e-03, 6.1703482511978082e-03, 7.6274629409160176e-03,
        9.4286721829279947e-03, 1.1655233178024955e-02, 1.4407591843112350e-02,
        1.7809914185937643e-02, 2.2015687754376449e-02, 2.7214645856121149e-02,
        3.3641326918204623e-02, 4.1585655121173182e-02, 5.1406019627642255e-02,
        6.3545442443014022e-02, 7.8551564282309541e-02, 9.7101350057185065e-02,
        1.2003162851145673e-01, 1.4837684372696586e-01, 1.8341572156771246e-01,
        2.2672895630607542e-01, 2.8027051982381157e-01, 3.4645581033005746e-01,
        4.2827061721265969e-01, 5.2940581770869455e-01, 6.5442388190885858e-01,
        8.0896469756649991e-01, 1.0000000000000000e+00, 1.2361478850785035e+00,
        1.5280615937840571e+00, 1.8889101074258496e+00, 2.3349722343978727e+00,
        2.8863709892679585e+00, 3.5679813939355345e+00, 4.4105526541128617e+00,
        5.4520953354089947e+00, 6.7395961181122024e+00, 8.3311374876876929e+00,
        1.0298517985703377e+01, 1.2730491227470161e+01, 1.5736769806847677e+01,
        1.9452974714702009e+01, 2.4046753552064487e+01, 2.9725343546388515e+01,
        3.6744920558100112e+01, 4.5422155835273074e+01, 5.6148501871478999e+01,
        6.9407851838755207e+01, 8.5798369258319340e+01, 1.0605947270185592e+02,
        1.3110519287294045e+02, 1.6206540689269471e+02,
        2.0033680997479166e+02])

    dlf.factor = np.array([1.2361478850785039])

    dlf.j0 = np.array([
        6.5314496156480717e-02, -2.2265895125268051e-01,
        4.2794928430042073e-01, -5.8890658905204007e-01,
        6.7897324897038758e-01, -6.8296549495756353e-01,
        6.4702210424566342e-01, -5.6843090153083187e-01,
        4.9960500362966542e-01, -4.0923517268678855e-01,
        3.5487831784012780e-01, -2.7408257835871558e-01,
        2.4668368971888163e-01, -1.7348161774657314e-01,
        1.7509173908347012e-01, -1.0007796678766585e-01,
        1.3297302593795621e-01, -4.3822390117083297e-02,
        1.1415804492335258e-01, 4.2584271963990363e-03,
        1.1496183925974515e-01, 5.1313336759457658e-02,
        1.3312389010285514e-01, 1.0111673764393908e-01,
        1.6331446488042209e-01, 1.4704406144951229e-01,
        1.8341391219815678e-01, 1.5073655145087875e-01,
        1.2356364992866918e-01, 1.1529465836107913e-02,
        -1.2852626053530605e-01, -2.9874216118407571e-01,
        -3.1330553036684633e-01, -3.2332453606385231e-02,
        4.2056051544727674e-01, 1.3883637369942961e-01,
        -5.0223684329542873e-01, 3.7996073748731690e-01,
        -1.7984404170109178e-01, 7.0557286965350657e-02,
        -2.7522277139686569e-02, 1.1909130661901874e-02,
        -5.8977428045126450e-03, 3.2648420034561555e-03,
        -1.9304039780209286e-03, 1.1594138315987755e-03,
        -6.7050941490060716e-04, 3.5089803807606258e-04,
        -1.5287051174029577e-04, 4.8238483411813232e-05,
        -8.0433917146487977e-06])

    dlf.j1 = np.array([
        3.8409924166118657e-05, -9.4382633059055269e-05,
        1.6377612356675232e-04, -1.7853487610334151e-04,
        1.9759569268356316e-04, -1.2796247718553291e-04,
        1.3163370416771999e-04, 1.7066927559149181e-05,
        4.1368007808198310e-05, 2.3582259002758631e-04,
        -1.9929745662753875e-05, 5.8196110748926486e-04,
        1.1164151202379049e-05, 1.2053942009238238e-03,
        2.8328713800714110e-04, 2.4375075610646724e-03,
        1.1709289200267103e-03, 5.0080095645252103e-03,
        3.5856849047131551e-03, 1.0536604059335859e-02,
        9.6674718436839564e-03, 2.2583365267100426e-02,
        2.4201911801947001e-02, 4.8602551164692191e-02,
        5.6856974938076565e-02, 1.0202078096072606e-01,
        1.2202639829343022e-01, 1.9561985197118459e-01,
        2.1570853920674185e-01, 2.8552388758852060e-01,
        2.0559964796530625e-01, 1.1666011451374350e-01,
        -2.3636714743601936e-01, -3.4036030266164258e-01,
        -1.9475364255836261e-01, 5.7608115965662166e-01,
        -2.1471464584834585e-01, -1.2611070492338947e-01,
        2.0019703515125831e-01, -1.6147664664730754e-01,
        1.1398023016900673e-01, -7.9924455203352035e-02,
        5.7024556382799692e-02, -4.1006636508593382e-02,
        2.9192063091319696e-02, -2.0089573135329029e-02,
        1.2965400526524978e-02, -7.5154910983038946e-03,
        3.6269079130258285e-03, -1.2668720233377250e-03,
        2.3403502580547994e-04])

    return dlf


def key_101_2012():
    r"""Key 101 pt Hankel filter, as published in [Key12]_.

    Taken from file `kk101Hankel.txt` provided with SEG-2012-003_.

    License: https://software.seg.org/disclaimer.txt.

    """

    dlf = DigitalFilter('Key 101 (2012)', 'key_101_2012')

    dlf.base = np.array([
        5.5308437014783363e-04, 6.4259236035555788e-04, 7.4658580837667996e-04,
        8.6740895730700251e-04, 1.0077854290485113e-03, 1.1708796207911744e-03,
        1.3603680375478939e-03, 1.5805221687362171e-03, 1.8363047770289071e-03,
        2.1334817700377098e-03, 2.4787521766663585e-03, 2.8798991580882430e-03,
        3.3459654574712720e-03, 3.8874572434761303e-03, 4.5165809426126703e-03,
        5.2475183991813838e-03, 6.0967465655156379e-03, 7.0834089290521185e-03,
        8.2297470490200302e-03, 9.5616019305435132e-03, 1.1108996538242306e-02,
        1.2906812580479873e-02, 1.4995576820477703e-02, 1.7422374639493515e-02,
        2.0241911445804391e-02, 2.3517745856009110e-02, 2.7323722447292573e-02,
        3.1745636378067953e-02, 3.6883167401240015e-02, 4.2852126867040180e-02,
        4.9787068367863944e-02, 5.7844320874838456e-02, 6.7205512739749784e-02,
        7.8081666001153169e-02, 9.0717953289412512e-02, 1.0539922456186433e-01,
        1.2245642825298191e-01, 1.4227407158651359e-01, 1.6529888822158656e-01,
        1.9204990862075413e-01, 2.2313016014842982e-01, 2.5924026064589156e-01,
        3.0119421191220208e-01, 3.4993774911115533e-01, 4.0656965974059917e-01,
        4.7236655274101469e-01, 5.4881163609402650e-01, 6.3762815162177333e-01,
        7.4081822068171788e-01, 8.6070797642505781e-01, 1.0000000000000000e+00,
        1.1618342427282831e+00, 1.3498588075760032e+00, 1.5683121854901687e+00,
        1.8221188003905089e+00, 2.1170000166126748e+00, 2.4596031111569494e+00,
        2.8576511180631639e+00, 3.3201169227365472e+00, 3.8574255306969740e+00,
        4.4816890703380645e+00, 5.2069798271798486e+00, 6.0496474644129448e+00,
        7.0286875805892928e+00, 8.1661699125676499e+00, 9.4877358363585262e+00,
        1.1023176380641601e+01, 1.2807103782663029e+01, 1.4879731724872830e+01,
        1.7287781840567639e+01, 2.0085536923187668e+01, 2.3336064580942711e+01,
        2.7112638920657883e+01, 3.1500392308747923e+01, 3.6598234443677974e+01,
        4.2521082000062783e+01, 4.9402449105530167e+01, 5.7397457045446188e+01,
        6.6686331040925154e+01, 7.7478462925260828e+01, 9.0017131300521811e+01,
        1.0458498557711414e+02, 1.2151041751873485e+02, 1.4117496392147686e+02,
        1.6402190729990167e+02, 1.9056626845862999e+02, 2.2140641620418697e+02,
        2.5723755590577474e+02, 2.9886740096706029e+02, 3.4723438047873447e+02,
        4.0342879349273511e+02, 4.6871738678241655e+02, 5.4457191012592898e+02,
        6.3270229281225352e+02, 7.3509518924197266e+02, 8.5405876252615155e+02,
        9.9227471560502534e+02, 1.1528587427833875e+03, 1.3394307643944169e+03,
        1.5561965278371533e+03, 1.8080424144560632e+03])

    dlf.factor = np.array([1.1618342427282831])

    dlf.j0 = np.array([
        5.1818808036862153e-02, -2.9258492652779172e-01,
        8.7289913080159265e-01, -1.8525183961240297e+00,
        3.2096055660873608e+00, -4.9248158048315069e+00,
        7.0219167445719828e+00, -9.4781099645811544e+00,
        1.2098564895534539e+01, -1.4476661837664896e+01,
        1.6146079632847474e+01, -1.6810370923047440e+01,
        1.6506202923977266e+01, -1.5534747212675301e+01,
        1.4267433361428399e+01, -1.2939053370802849e+01,
        1.1628520257389562e+01, -1.0325503688577493e+01,
        9.0574401975545840e+00, -7.8759096646654951e+00,
        6.8446506699312666e+00, -5.9542991234257840e+00,
        5.1830565431894602e+00, -4.4805481279340267e+00,
        3.8608298760715378e+00, -3.3125736597743227e+00,
        2.8648088091590589e+00, -2.4710577422247479e+00,
        2.1450315408533416e+00, -1.8334353019592327e+00,
        1.5850020809005190e+00, -1.3468969891769780e+00,
        1.1802299801807625e+00, -9.9988174688478793e-01,
        8.8503813256539254e-01, -7.3165077099639619e-01,
        6.5909386165012596e-01, -5.3038429622108840e-01,
        5.0119791866928076e-01, -3.8269840749371159e-01,
        3.8622329595139565e-01, -2.6413330193320073e-01,
        3.0353632730294311e-01, -1.7255655176111537e-01,
        2.5119774955173280e-01, -9.8536724013140092e-02,
        2.1755876398162183e-01, -3.4268716744688760e-02,
        1.9925218957836366e-01, 1.8319735495994052e-02,
        1.8963099219918259e-01, 5.7228135568732394e-02,
        1.7266316834158107e-01, 6.8141322181604499e-02,
        1.2468074033081850e-01, 2.1678852838804014e-02,
        1.1114005126278175e-02, -1.0424806190544497e-01,
        -1.6473681128667569e-01, -2.3603153458714107e-01,
        -2.2093965064386309e-01, -7.5633542915308175e-02,
        1.3358180223826097e-01, 3.3949946683489907e-01,
        1.2815187347111337e-01, -2.2988301163130145e-01,
        -3.3425750283830036e-01, 4.4350537603751145e-01,
        -8.8943820418669634e-02, -1.8646079800862164e-01,
        2.4494657419883456e-01, -2.0169045144304532e-01,
        1.4661206655803793e-01, -1.0564640256629308e-01,
        7.8487036158411266e-02, -6.0533748554794195e-02,
        4.8243702864613026e-02, -3.9347085461672239e-02,
        3.2489421068688185e-02, -2.6982535264609676e-02,
        2.2530196869950080e-02, -1.8947601435977172e-02,
        1.6017457455233256e-02, -1.3519422964231352e-02,
        1.1318328695851308e-02, -9.3839154851987506e-03,
        7.7256174599644543e-03, -6.3199190181430056e-03,
        5.0986473606509210e-03, -3.9929354358609528e-03,
        2.9783618569305080e-03, -2.0802747552777045e-03,
        1.3429398653973710e-03, -7.9345560431147465e-04,
        4.2530878973518046e-04, -2.0457624610548748e-04,
        8.6750842046621606e-05, -3.1406113278782205e-05,
        9.1385289919391839e-06, -1.8949818224609619e-06,
        2.0794387557779629e-07])

    dlf.j1 = np.array([
        4.1746363961646286e-06, -2.5993370843637242e-05,
        9.2506838353721647e-05, -2.4318740269846489e-04,
        5.2276050788732404e-04, -9.5803859755432658e-04,
        1.5308357107396591e-03, -2.1584605764249656e-03,
        2.7161344302423967e-03, -3.0845361397036195e-03,
        3.2133980102756368e-03, -3.1305779240645847e-03,
        2.9239120736637542e-03, -2.6680602560576316e-03,
        2.4131379039476304e-03, -2.1549566068119440e-03,
        1.9058200916563901e-03, -1.6533122444456158e-03,
        1.4391633836110848e-03, -1.2363666069459527e-03,
        1.0907227256323094e-03, -9.2712769282234358e-04,
        8.2888526218099687e-04, -6.7675613963809832e-04,
        6.3558641899842260e-04, -4.8591507683128695e-04,
        5.1892328126100203e-04, -3.3190335155472070e-04,
        4.6189809360776176e-04, -1.8326476919948130e-04,
        4.7669939315323097e-04, -1.7380345519175726e-05,
        5.9009214750893949e-04, 2.1645139320505843e-04,
        8.5203135609860704e-04, 5.9560990889710838e-04,
        1.3651107378683781e-03, 1.2606987195022564e-03,
        2.3131990435032663e-03, 2.4572002938857678e-03,
        4.0408430887840449e-03, 4.6278078809669602e-03,
        7.1437051276295131e-03, 8.5723466849983495e-03,
        1.2684763573137947e-02, 1.5643387012124067e-02,
        2.2480928663078600e-02, 2.8124131719595692e-02,
        3.9373999883133115e-02, 4.9445586323210833e-02,
        6.7326587032714463e-02, 8.3541701874077570e-02,
        1.0961776964853986e-01, 1.3118216481048459e-01,
        1.6144181878027153e-01, 1.7704476632134886e-01,
        1.9032654055463888e-01, 1.6333927603825021e-01,
        1.1034442879456068e-01, -1.1282682226217962e-02,
        -1.4579378661401318e-01, -2.7560051939484942e-01,
        -2.3333751023296440e-01, -5.9615109981928909e-03,
        3.2380356528743487e-01, 2.1970582679149839e-01,
        -2.8302631898505476e-01, -2.2706962776315812e-01,
        4.7351258711649669e-01, -3.5417866045339608e-01,
        1.6582378621661517e-01, -5.1221107687420810e-02,
        3.5484710794282063e-03, 1.1614894738175324e-02,
        -1.4776256489774963e-02, 1.4194293010344678e-02,
        -1.2609914713728791e-02, 1.0854082569077052e-02,
        -9.2082270347668624e-03, 7.7880648968821110e-03,
        -6.6089268276601255e-03, 5.6159700146012417e-03,
        -4.7432063636330263e-03, 3.9609893136500528e-03,
        -3.2753768787828126e-03, 2.6945716306237466e-03,
        -2.2042319512562145e-03, 1.7733135127602079e-03,
        -1.3773733375919272e-03, 1.0137885881408651e-03,
        -6.9751488719640639e-04, 4.4545125438205673e-04,
        -2.6373575275283686e-04, 1.4519595835116970e-04,
        -7.4640168948626184e-05, 3.5826939350044606e-05,
        -1.5854604622793124e-05, 6.2412989356183992e-06,
        -2.0349222128793154e-06, 4.8072849734177625e-07,
        -6.0462736574031818e-08])

    return dlf


def key_201_2012():
    r"""Key 201 pt Hankel filter, as published in [Key12]_.

    Taken from file `kk201Hankel.txt` provided with SEG-2012-003_.

    License: https://software.seg.org/disclaimer.txt.

    """

    dlf = DigitalFilter('Key 201 (2012)', 'key_201_2012')

    dlf.base = np.array([
        4.1185887075357082e-06, 4.6623077830484039e-06, 5.2778064058937756e-06,
        5.9745606155328603e-06, 6.7632974390298035e-06, 7.6561600412698171e-06,
        8.6668946776271221e-06, 9.8110623273521872e-06, 1.1106278265924788e-05,
        1.2572483264760042e-05, 1.4232250593579901e-05, 1.6111133551969887e-05,
        1.8238058880617237e-05, 2.0645772109076137e-05, 2.3371341696506312e-05,
        2.6456729726989950e-05, 2.9949437945688260e-05, 3.3903239082024059e-05,
        3.8379004719130766e-05, 4.3445642455208216e-05, 4.9181156785051293e-05,
        5.5673850034776868e-05, 6.3023681838995322e-05, 7.1343808090545215e-05,
        8.0762323056016586e-05, 9.1424231478173270e-05, 1.0349368102719458e-04,
        1.1715648947091054e-04, 1.3262300547161834e-04, 1.5013134705348249e-04,
        1.6995106759902750e-04, 1.9238730581535294e-04, 2.1778548356175115e-04,
        2.4653662386513240e-04, 2.7908337099788406e-04, 3.1592680530155527e-04,
        3.5763415767542709e-04, 4.0484754250000483e-04, 4.5829384344501882e-04,
        5.1879590436097246e-04, 5.8728519754599128e-04, 6.6481616442494776e-04,
        7.5258244942583781e-04, 8.5193527698548301e-04, 9.6440425461164468e-04,
        1.0917209222795108e-03, 1.2358454107222641e-03, 1.3989966190390965e-03,
        1.5836863762264436e-03, 1.7927581125735061e-03, 2.0294306362957340e-03,
        2.2973476893787268e-03, 2.6006340455800008e-03, 2.9439590142573526e-03,
        3.3326083277104117e-03, 3.7725655187922052e-03, 4.2706040416570154e-03,
        4.8343915539089625e-03, 5.4726079656493079e-03, 6.1950790728715292e-03,
        7.0129278325854246e-03, 7.9387456086585442e-03, 8.9867860248264830e-03,
        1.0173184409377162e-02, 1.1516206210016314e-02, 1.3036528203437736e-02,
        1.4757556829019875e-02, 1.6705788547622769e-02, 1.8911217773465227e-02,
        2.1407798659484324e-02, 2.4233967845691123e-02, 2.7433236218606032e-02,
        3.1054858792331431e-02, 3.5154593024557458e-02, 3.9795557242315927e-02,
        4.5049202393557801e-02, 5.0996412085361466e-02, 5.7728747844643331e-02,
        6.5349858773045680e-02, 7.3977077298642363e-02, 8.3743225592195963e-02,
        9.4798660459030126e-02, 1.0731358818908403e-01, 1.2148068500391276e-01,
        1.3751806344428075e-01, 1.5567263036799733e-01, 1.7622388825676111e-01,
        1.9948823835583873e-01, 2.2582385189647586e-01, 2.5563618439697788e-01,
        2.8938421793905067e-01, 3.2758752752368947e-01, 3.7083428029819565e-01,
        4.1979029080811425e-01, 4.7520927168614446e-01, 5.3794443759467447e-01,
        6.0896164107289685e-01, 6.8935424252422239e-01, 7.8035994327803426e-01,
        8.8337984088275090e-01, 1.0000000000000000e+00, 1.1320158709991752e+00,
        1.2814599321940212e+00, 1.4506329812931589e+00, 1.6421395578187052e+00,
        1.8589280418463421e+00, 2.1043360464154781e+00, 2.3821418024579781e+00,
        2.6966223273530128e+00, 3.0526192726543444e+00, 3.4556134647626755e+00,
        3.9118092861497971e+00, 4.4282301962435247e+00, 5.0128268625854631e+00,
        5.6745995670177445e+00, 6.4237367714291338e+00, 7.2717719763787807e+00,
        8.2317612875478190e+00, 9.3184844237807365e+00, 1.0548672261378394e+01,
        1.1941264417849103e+01, 1.3517700840802913e+01, 1.5302251891207787e+01,
        1.7322392002874359e+01, 1.9609222670922968e+01, 2.2197951281441636e+01,
        2.5128433154258410e+01, 2.8445785143962375e+01, 3.2201080245997971e+01,
        3.6452133901787732e+01, 4.1264394108610787e+01, 4.6711949038112273e+01,
        5.2878667676447755e+01, 5.9859491047029913e+01, 6.7761893895170928e+01,
        7.6707539338295589e+01, 8.6834151956244213e+01, 9.8297638159222487e+01,
        1.1127448647797397e+02, 1.2596448473034971e+02, 1.4259379589698909e+02,
        1.6141844006140866e+02, 1.8272823602144376e+02, 2.0685126325595743e+02,
        2.3415891294197226e+02, 2.6507160578622688e+02, 3.0006526470124555e+02,
        3.3967864197737873e+02, 3.8452161375783919e+02, 4.3528456951608860e+02,
        4.9274904109325632e+02, 5.5779973493719069e+02, 6.3143815278803334e+02,
        7.1479801051045558e+02, 8.0916269245647072e+02, 9.1598501008114988e+02,
        1.0369095690092008e+03, 1.1737980889093292e+03, 1.3287580659938624e+03,
        1.5041752194232211e+03, 1.7027502211507524e+03, 1.9275402746900081e+03,
        2.1820061829391980e+03, 2.4700656297055034e+03, 2.7961534952362003e+03,
        3.1652901343571971e+03, 3.5831586684094545e+03, 4.0561924809477755e+03,
        4.5916742642604040e+03, 5.1978481416012310e+03, 5.8840465913361650e+03,
        6.6608341270911405e+03, 7.5401699459601114e+03, 8.5355920488578286e+03,
        9.6624256676814348e+03, 1.0938019208165191e+04, 1.2382011340936813e+04,
        1.4016633352832258e+04, 1.5867051413382505e+04, 1.7961754025908867e+04,
        2.0332990628312182e+04, 2.3017268096126892e+04, 2.6055912791858576e+04,
        2.9495706813754354e+04, 3.3389608239508460e+04, 3.7797566453568354e+04,
        4.2787445110585410e+04, 4.8436066944688770e+04, 5.4830396510166145e+04,
        6.2068879062685897e+04, 7.0262956194088882e+04, 7.9538781555028458e+04,
        9.0039163080228551e+04, 1.0192576161830177e+05, 1.1538157981559623e+05,
        1.3061377957221285e+05, 1.4785687144693290e+05, 1.6737632511421290e+05,
        1.8947265645880660e+05, 2.1448605423174356e+05,
        2.4280161749832361e+05])

    dlf.factor = np.array([1.132015870999175])

    dlf.j0 = np.array([
        1.5020099209519960e-03, -1.0381698214761684e-02,
        3.6840860097595164e-02, -8.9903380392274704e-02,
        1.7082286536833860e-01, -2.7115749656836280e-01,
        3.7649328091859574e-01, -4.7220778569122657e-01,
        5.4778211089647089e-01, -5.9823516853035097e-01,
        6.2345791612185331e-01, -6.2650436648257724e-01,
        6.1197225351173345e-01, -5.8470173866140451e-01,
        5.4911055789616858e-01, -5.0878861684860843e-01,
        4.6652106345430200e-01, -4.2426478870289325e-01,
        3.8339538911933357e-01, -3.4472344791723936e-01,
        3.0876891466890510e-01, -2.7570439368364807e-01,
        2.4562331000974616e-01, -2.1839207265400126e-01,
        1.9393194321380827e-01, -1.7198162962644575e-01,
        1.5242270280410272e-01, -1.3494945901825492e-01,
        1.1946763189654602e-01, -1.0565870880444576e-01,
        9.3482355548033913e-02, -8.2612525279836244e-02,
        7.3077763526174969e-02, -6.4536481281701225e-02,
        5.7096310621587334e-02, -5.0384859605213800e-02,
        4.4599557396733047e-02, -3.9316995063444334e-02,
        3.4838544754296139e-02, -3.0664946477420088e-02,
        2.7221072240291456e-02, -2.3901586108434601e-02,
        2.1281566646364738e-02, -1.8612484595521447e-02,
        1.6655386303461400e-02, -1.4472420076504743e-02,
        1.3057747649606617e-02, -1.1226286001060300e-02,
        1.0266818557930830e-02, -8.6738022421653585e-03,
        8.1103368670848164e-03, -6.6573628142085938e-03,
        6.4551189810889585e-03, -5.0524015117839084e-03,
        5.1989014270506263e-03, -3.7597430078747883e-03,
        4.2640545165292636e-03, -2.6994965688974474e-03,
        3.5928000560587229e-03, -1.8061291289989019e-03,
        3.1436470898586703e-03, -1.0244200001620755e-03,
        2.8888297674091743e-03, -3.0605070741955765e-04,
        2.8125907445050334e-03, 3.9337862066047565e-04,
        2.9102041487017727e-03, 1.1170884769911178e-03,
        3.1876763415953362e-03, 1.9097806429850762e-03,
        3.6621020321465411e-03, 2.8203783587849949e-03,
        4.3626885599390430e-03, 3.9050024915686802e-03,
        5.3324923486866077e-03, 5.2303399394270488e-03,
        6.6309348458396984e-03, 6.8775540687037451e-03,
        8.3371689988226087e-03, 8.9468564745410102e-03,
        1.0554326601376745e-02, 1.1562765604509096e-02,
        1.3414536265660639e-02, 1.4879837399735491e-02,
        1.7084241336519569e-02, 1.9088092654441269e-02,
        2.1768513715084332e-02, 2.4416120601223806e-02,
        2.7711234465350204e-02, 3.1127164234404547e-02,
        3.5184140220277452e-02, 3.9497913113638941e-02,
        4.4449762157204989e-02, 4.9758433269983887e-02,
        5.5667457114992429e-02, 6.1949790434151823e-02,
        6.8682064905571841e-02, 7.5616743950118387e-02,
        8.2584983563572259e-02, 8.9193635262987042e-02,
        9.4874686942005279e-02, 9.8891681869094042e-02,
        1.0004294654495730e-01, 9.7016844329802857e-02,
        8.7789596149914384e-02, 7.0509767592855419e-02,
        4.2778853484849347e-02, 3.5584532926218175e-03,
        -4.7210453264879347e-02, -1.0489787743225988e-01,
        -1.6020950407348281e-01, -1.9459781573132096e-01,
        -1.8490774599542381e-01, -1.0754165020025190e-01,
        3.6037727487476613e-02, 1.9759013047489976e-01,
        2.6132313321851336e-01, 1.1713996822458939e-01,
        -1.8758779281301441e-01, -3.0238114997462151e-01,
        4.8163135684567732e-02, 3.6399529664885466e-01,
        -1.4910233461562913e-01, -2.6373490348543854e-01,
        4.0362661807718703e-01, -3.1409794650104578e-01,
        1.8179369405131079e-01, -9.0738718042631769e-02,
        4.2946487545160242e-02, -2.0586135807067835e-02,
        1.0392667161913182e-02, -5.6117848068723023e-03,
        3.2402025511569896e-03, -1.9858724388273777e-03,
        1.2807317326135252e-03, -8.6253791756068252e-04,
        6.0296590782143555e-04, -4.3548936996943465e-04,
        3.2375891570874245e-04, -2.4698212240059978e-04,
        1.9279062274925971e-04, -1.5357911509105972e-04,
        1.2453787849367440e-04, -1.0255126402954421e-04,
        8.5558482209476271e-05, -7.2170928476334345e-05,
        6.1436863283080017e-05, -5.2693349406473615e-05,
        4.5471255142623729e-05, -3.9433284158653591e-05,
        3.4332971164795619e-05, -2.9987212220165472e-05,
        2.6257657435614391e-05, -2.3037978256349448e-05,
        2.0245071331016652e-05, -1.7812925644382519e-05,
        1.5688305607849465e-05, -1.3827679492110470e-05,
        1.2195005442258958e-05, -1.0760110818279243e-05,
        9.4974857670959145e-06, -8.3853711343938338e-06,
        7.4050612540713192e-06, -6.5403682860228500e-06,
        5.7772102413267325e-06, -5.1032933218796731e-06,
        4.5078641320869195e-06, -3.9815111951817114e-06,
        3.5159993210292339e-06, -3.1041249680128287e-06,
        2.7395854235553632e-06, -2.4168587823583312e-06,
        2.1310948134441218e-06, -1.8780185742021288e-06,
        1.6538488252707735e-06, -1.4552318768651144e-06,
        1.2791890865306748e-06, -1.1230741959148155e-06,
        9.8453630538596172e-07, -8.6148574493190796e-07,
        7.5206242699701732e-07, -6.5460810334682633e-07,
        5.6764451596029765e-07, -4.8985884555184997e-07,
        4.2009674106879252e-07, -3.5736212641225517e-07,
        3.0082215969939671e-07, -2.4981510481639669e-07,
        2.0385823466866512e-07, -1.6265189071584773e-07,
        1.2607416700611311e-07, -9.4158417913450858e-08,
        6.7043911217063425e-08, -4.4891090827293947e-08,
        2.7761325666544702e-08, -1.5480404355710375e-08,
        7.5327300141098751e-09, -3.0524770418657847e-09,
        9.5877856096830783e-10, -2.0575286298055636e-10,
        2.2414416956474645e-11])

    dlf.j1 = np.array([
        4.7827871332506182e-10, -2.9784175503440788e-09,
        9.7723832770897220e-09, -2.2382340996085809e-08,
        4.0446774329470848e-08, -6.1734815854553919e-08,
        8.3293912185608189e-08, -1.0249453502284080e-07,
        1.1780779749909977e-07, -1.2870061460834850e-07,
        1.3559243438349901e-07, -1.3921010821521872e-07,
        1.4065745722769670e-07, -1.4074881908375281e-07,
        1.4051720878600928e-07, -1.4040746687777830e-07,
        1.4127886061686993e-07, -1.4315595655055356e-07,
        1.4689283208027915e-07, -1.5210916706348747e-07,
        1.5989801550138741e-07, -1.6940918407911260e-07,
        1.8227089415749844e-07, -1.9695856878603369e-07,
        2.1603952427106760e-07, -2.3691320619292838e-07,
        2.6369843208466607e-07, -2.9202021404039016e-07,
        3.2852445086324662e-07, -3.6589094553627693e-07,
        4.1487501036863030e-07, -4.6327136995173986e-07,
        5.2852697369750518e-07, -5.9034983719954077e-07,
        6.7710211611560898e-07, -7.5512942807901028e-07,
        8.7062473466409799e-07, -9.6788530918130449e-07,
        1.1222658353725904e-06, -1.2417228058919743e-06,
        1.4493467036899140e-06, -1.5932456559208080e-06,
        1.8747045814274419e-06, -2.0433320340041385e-06,
        2.4285695351967146e-06, -2.6179926456520528e-06,
        3.1511729661391781e-06, -3.3492412032881268e-06,
        4.0964186613549959e-06, -4.2758275751239154e-06,
        5.3371197552207799e-06, -5.4435423773626499e-06,
        6.9725766921672099e-06, -6.9045562968161745e-06,
        9.1397025436977438e-06, -8.7148373033635960e-06,
        1.2029590806160379e-05, -1.0927976968519436e-05,
        1.5912526719455299e-05, -1.3582559661331659e-05,
        2.1176226828087565e-05, -1.6678205993448338e-05,
        2.8384979250408712e-05, -2.0132088397797457e-05,
        3.8372045311188108e-05, -2.3702184335455945e-05,
        5.2385308850007940e-05, -2.6854373943701261e-05,
        7.2318581557902158e-05, -2.8535361884516687e-05,
        1.0108123118106823e-04, -2.6788477540644352e-05,
        1.4319185407094621e-04, -1.8108424211338017e-05,
        2.0573561552327273e-04, 3.6361648565843316e-06,
        2.9991264692859609e-04, 4.8993332079278846e-05,
        4.4354733854670903e-04, 1.3589101811995494e-04,
        6.6515823521273764e-04, 2.9451991608624389e-04,
        1.0105553806271136e-03, 5.7533964792254050e-04,
        1.5535077418254303e-03, 1.0621193133794828e-03,
        2.4128970258747457e-03, 1.8929698186109245e-03,
        3.7800177772191607e-03, 3.2937343278959356e-03,
        5.9612179800391544e-03, 5.6295935320552241e-03,
        9.4422526803168080e-03, 9.4810228247137925e-03,
        1.4979159139973408e-02, 1.5745093424331037e-02,
        2.3708966370000140e-02, 2.5740590762136872e-02,
        3.7232782843117498e-02, 4.1225008900614292e-02,
        5.7507103212277359e-02, 6.4044642846235691e-02,
        8.6091796551857253e-02, 9.4717139804457451e-02,
        1.2172497389177185e-01, 1.2853597000398900e-01,
        1.5450777327408322e-01, 1.4755964090969320e-01,
        1.5621399202016978e-01, 1.1147620703185755e-01,
        7.7489831356083380e-02, -2.7628266850147711e-02,
        -1.0198730178317840e-01, -2.2039889971111640e-01,
        -2.1185762869925318e-01, -1.6052415083152241e-01,
        9.1649025798681089e-02, 2.3792823877700942e-01,
        2.6075777853738125e-01, -1.5662188259001042e-01,
        -2.8932081756330175e-01, 1.3148519116247689e-02,
        4.2691302759079564e-01, -4.0005050006489040e-01,
        1.1513789407450359e-01, 9.3748244358717620e-02,
        -1.6037231301955096e-01, 1.5071857939129532e-01,
        -1.2120369075996129e-01, 9.4110656079982341e-02,
        -7.3742238434584328e-02, 5.9038567576124905e-02,
        -4.8288117528475852e-02, 4.0197054299576880e-02,
        -3.3919787720641081e-02, 2.8918247156763971e-02,
        -2.4845271759013743e-02, 2.1470449751150148e-02,
        -1.8635828020057092e-02, 1.6229579362363859e-02,
        -1.4170085406786529e-02, 1.2396084121011890e-02,
        -1.0860414401084047e-02, 9.5259444245356633e-03,
        -8.3628577447233381e-03, 7.3468029551253195e-03,
        -6.4576043210966359e-03, 5.6783439955994880e-03,
        -4.9946949167265437e-03, 4.3944258108608806e-03,
        -3.8670264019660858e-03, 3.4034180355556670e-03,
        -2.9957260668529964e-03, 2.6370977166248776e-03,
        -2.3215540117372982e-03, 2.0438677474690805e-03,
        -1.7994616759226389e-03, 1.5843226896713463e-03,
        -1.3949288614414647e-03, 1.2281869708886250e-03,
        -1.0813786997088304e-03, 9.5211407460757294e-04,
        -8.3829103020448140e-04, 7.3806018220987763e-04,
        -6.4979406671247244e-04, 5.7206022901230412e-04,
        -5.0359764543483573e-04, 4.4329604122300101e-04,
        -3.9017773206623073e-04, 3.4338166974098888e-04,
        -3.0214941633163506e-04, 2.6581280850704716e-04,
        -2.3378310479588391e-04, 2.0554143583887405e-04,
        -1.8063040107732216e-04, 1.5864667598176302e-04,
        -1.3923451235168759e-04, 1.2208003098625932e-04,
        -1.0690622166177854e-04, 9.3468580362568137e-05,
        -8.1551328581234316e-05, 7.0964174631531072e-05,
        -6.1539592468666665e-05, 5.3130609116145443e-05,
        -4.5609105983126460e-05, 3.8864648584181207e-05,
        -3.2803856352344075e-05, 2.7350296775876261e-05,
        -2.2444816150805301e-05, 1.8046076281584239e-05,
        -1.4130826937491561e-05, 1.0693106849359383e-05,
        -7.7412053314530284e-06, 5.2910576443698300e-06,
        -3.3552268362550323e-06, 1.9282956206367452e-06,
        -9.7253712572058755e-07, 4.1100807632959352e-07,
        -1.3553176263207053e-07, 3.0748587523233524e-08,
        -3.5668195345476294e-09])

    return dlf


def wer_201_2018():
    r"""Werthmüller 201 pt Hankel filter, 2018.

    Designed with the empymod add-on `fdesign`, see
    https://github.com/emsig/article-fdesign.

    License: `Apache License, Version 2.0,
    <https://www.apache.org/licenses/LICENSE-2.0>`_.

    """

    dlf = DigitalFilter('Werthmüller 201', 'wer_201_2018')

    dlf.base = np.array([
        8.653980893285999343e-04, 9.170399868578506730e-04,
        9.717635708540675208e-04, 1.029752738345341345e-03,
        1.091202360259058363e-03, 1.156318936280037154e-03,
        1.225321288786770891e-03, 1.298441298197734088e-03,
        1.375924682198844014e-03, 1.458031821470676028e-03,
        1.545038634690233410e-03, 1.637237505747725754e-03,
        1.734938266294200121e-03, 1.838469236921894991e-03,
        1.948178330476121357e-03, 2.064434221206373505e-03,
        2.187627583685520637e-03, 2.318172405660454734e-03,
        2.456507379246002792e-03, 2.603097375137131825e-03,
        2.758435004793547123e-03, 2.923042275846299189e-03,
        3.097472346289414230e-03, 3.282311383351382318e-03,
        3.478180533293271509e-03, 3.685738008752825322e-03,
        3.905681300649099623e-03, 4.138749522080598965e-03,
        4.385725892093550113e-03, 4.647440367666977670e-03,
        4.924772432759196537e-03, 5.218654053788364042e-03,
        5.530072811478751842e-03, 5.860075219597365645e-03,
        6.209770241733286907e-03, 6.580333017937914017e-03,
        6.973008813749223544e-03, 7.389117204870794542e-03,
        7.830056511567909730e-03, 8.297308497682591086e-03,
        8.792443350058264107e-03, 9.317124955107483966e-03,
        9.873116490254248145e-03, 1.046228634904101083e-02,
        1.108661441981135046e-02, 1.174819873906770597e-02,
        1.244926254186268406e-02, 1.319216173291635694e-02,
        1.397939280356631266e-02, 1.481360122115480543e-02,
        1.569759031904567614e-02, 1.663433071714528685e-02,
        1.762697030458526201e-02, 1.867884481811322647e-02,
        1.979348905174003331e-02, 2.097464873531334692e-02,
        2.222629312193481060e-02, 2.355262832652090660e-02,
        2.495811146033087569e-02, 2.644746560896089893e-02,
        2.802569570413705746e-02, 2.969810534264444649e-02,
        3.147031460891126092e-02, 3.334827896114080786e-02,
        3.533830924445728605e-02, 3.744709289831891358e-02,
        3.968171642946591998e-02, 4.204968922592208086e-02,
        4.455896879207719985e-02, 4.721798748965116282e-02,
        5.003568087440296575e-02, 5.302151772380819111e-02,
        5.618553185661335353e-02, 5.953835585119460899e-02,
        6.309125677603129312e-02, 6.685617405236506106e-02,
        7.084575957628110043e-02, 7.507342023504076645e-02,
        7.955336296053971967e-02, 8.430064247129397115e-02,
        8.933121186338736919e-02, 9.466197622039211612e-02,
        1.003108494224146802e-01, 1.062968143451746283e-01,
        1.126399866514113390e-01, 1.193616823889895873e-01,
        1.264844896228644322e-01, 1.340323443416230609e-01,
        1.420306108936851552e-01, 1.505061672234652703e-01,
        1.594874951939309615e-01, 1.690047762990831426e-01,
        1.790899930879975566e-01, 1.897770366412597776e-01,
        2.011018204609656135e-01, 2.131024011570106513e-01,
        2.258191063352318062e-01, 2.392946701171655144e-01,
        2.535743767468323084e-01, 2.687062127671349665e-01,
        2.847410282772538936e-01, 3.017327078129410922e-01,
        3.197383514239506286e-01, 3.388184665571116749e-01,
        3.590371713898611872e-01, 3.804624102975323052e-01,
        4.031661821784713884e-01, 4.272247824042622599e-01,
        4.527190592081252185e-01, 4.797346853730769523e-01,
        5.083624461328500876e-01, 5.386985442530570767e-01,
        5.708449233178135573e-01, 6.049096103082166609e-01,
        6.410070786239048246e-01, 6.792586327676195523e-01,
        7.197928159854947161e-01, 7.627458422329327359e-01,
        8.082620539176789132e-01, 8.564944069583262376e-01,
        9.076049847882753374e-01, 9.617655430324436594e-01,
        1.019158086687097953e+00, 1.079975481742401877e+00,
        1.144422103303022853e+00, 1.212714522384784388e+00,
        1.285082233695329812e+00, 1.361768426844476521e+00,
        1.443030803575896748e+00, 1.529142443766406734e+00,
        1.620392723103028176e+00, 1.717088285521651159e+00,
        1.819554073675149652e+00, 1.928134420893808709e+00,
        2.043194208307565596e+00, 2.165120091018537973e+00,
        2.294321797444361266e+00, 2.431233506198735572e+00,
        2.576315305136151590e+00, 2.730054737463883274e+00,
        2.892968440116886253e+00, 3.065603879901345419e+00,
        3.248541193241110570e+00, 3.442395135709440002e+00,
        3.647817147897402634e+00, 3.865497544561224075e+00,
        4.096167834405143537e+00, 4.340603178295351583e+00,
        4.599624994165753655e+00, 4.874103717369282940e+00,
        5.164961725750846000e+00, 5.473176439271492555e+00,
        5.799783604600056819e+00, 6.145880775710010013e+00,
        6.512631002177978523e+00, 6.901266737578352739e+00,
        7.313093981108034214e+00, 7.749496666359123154e+00,
        8.211941311987910552e+00, 8.701981949908562441e+00,
        9.221265347572652260e+00, 9.771536541883744320e+00,
        1.035464470334362019e+01, 1.097254935013656407e+01,
        1.162732693303372145e+01, 1.232117781324618733e+01,
        1.305643365667540934e+01, 1.383556526940939513e+01,
        1.466119090079532228e+01, 1.553608504199116247e+01,
        1.646318774956322173e+01, 1.744561452546165370e+01,
        1.848666678657500739e+01, 1.958984295904654616e+01,
        2.075885023463468926e+01, 2.199761702862400270e+01,
        2.331030618115176267e+01, 2.470132894631230158e+01,
        2.617535981604930129e+01, 2.773735222865149197e+01,
        2.939255521463919862e+01, 3.114653103598043415e+01,
        3.300517387791185087e+01, 3.497472965617872376e+01,
        3.706181700625468523e+01, 3.927344952507589682e+01,
        4.161705934003131091e+01, 4.410052208441296528e+01,
        4.673218336325475519e+01, 4.952088679849771324e+01,
        5.247600374772711262e+01, 5.560746479634944706e+01,
        5.892579312903905020e+01, 6.244213989259677078e+01,
        6.616832166905808776e+01, 7.011686018497628936e+01,
        7.430102439032442874e+01, 7.873487504841911289e+01,
        8.343331198671111792e+01, 8.841212416722630962e+01,
        9.368804274491759543e+01])

    dlf.factor = np.array([1.0596741524693181])

    dlf.j0 = np.array([
        2.940900904253498815e+00, -1.601154970027019786e+01,
        3.574488144287594338e+01, -3.775710631592443178e+01,
        9.347313619702582344e+00, 1.903333998229986435e+01,
        -1.266160545445113073e+01, -1.274822633210828471e+01,
        1.499040669570122830e+01, 1.128114232815630835e+01,
        -3.141814347069891511e+01, 2.185520874118305557e+01,
        4.204688908817206361e+00, -2.039396651211594147e+01,
        1.703214350560853418e+01, -3.858035665251962953e+00,
        -5.664929771474184861e+00, 6.236013601339658763e+00,
        -8.349084962847823643e-01, -5.076648434675173682e+00,
        8.069437229646323928e+00, -7.693517618528387558e+00,
        5.264540454351635645e+00, -2.378232295562936471e+00,
        9.488986688091295696e-02, 1.211665432819921451e+00,
        -1.641788465968915700e+00, 1.505783472315168181e+00,
        -1.120376089133631847e+00, 7.202627618735454318e-01,
        -4.293308092527763353e-01, 2.880519253847271255e-01,
        -2.765909110132210857e-01, 3.540771325316399709e-01,
        -4.702415607316920432e-01, 5.886079250132733032e-01,
        -6.804407177384068639e-01, 7.365579045711867501e-01,
        -7.516232294769148448e-01, 7.332518369461620278e-01,
        -6.840940481215002089e-01, 6.145375558739786248e-01,
        -5.263752170486778459e-01, 4.305607160730071659e-01,
        -3.307504282674695872e-01, 2.429511351824290011e-01,
        -1.749591466871897039e-01, 1.456366850874822316e-01,
        -1.595894233786172844e-01, 2.280395161025021433e-01,
        -3.413711397379035062e-01, 4.950015142033588056e-01,
        -6.617826176259098414e-01, 8.243458750063681340e-01,
        -9.468449720632085009e-01, 1.012979648696065826e+00,
        -9.946084380621768029e-01, 8.932243609342348512e-01,
        -7.019325270091921753e-01, 4.480078578272302381e-01,
        -1.456381084707468188e-01, -1.604285443826692914e-01,
        4.506359421881006022e-01, -6.820827325803067165e-01,
        8.491728004462286705e-01, -9.252486609867097700e-01,
        9.260533367474167443e-01, -8.402031897556576645e-01,
        6.982570225067336045e-01, -4.946654104012083719e-01,
        2.669962789856030194e-01, -1.042528939201309117e-02,
        -2.319933928270803414e-01, 4.654858099944349514e-01,
        -6.436180173401860882e-01, 7.780252822887177011e-01,
        -8.272840865922600484e-01, 8.197748606729908794e-01,
        -7.271116816505861502e-01, 5.992292042110317629e-01,
        -4.191980233493379782e-01, 2.522483458861179417e-01,
        -8.121597742330896597e-02, -2.652177515646544220e-02,
        1.028351781199190462e-01, -8.958878867947789315e-02,
        4.230973855067972356e-02, 8.503909842995018009e-02,
        -2.142619566112575480e-01, 3.865561094837716705e-01,
        -5.104727910435372662e-01, 6.353554205789004872e-01,
        -6.678424785380696616e-01, 6.769052092946485910e-01,
        -5.740789051340543514e-01, 4.533865101372042683e-01,
        -2.311110970314779189e-01, 2.495118155831516429e-02,
        2.508326503987150513e-01, -4.617568849058834579e-01,
        7.064398492531979157e-01, -8.419497320020807862e-01,
        9.894625411708843910e-01, -1.001995584015555441e+00,
        1.027606495406535592e+00, -9.137586391106879979e-01,
        8.347545540345707726e-01, -6.273557157264747497e-01,
        4.895538883108097039e-01, -2.415916082478662963e-01,
        1.027651673270286170e-01, 1.284590129111473078e-01,
        -2.133464520148750654e-01, 3.796869867196227544e-01,
        -3.706104772018420923e-01, 4.433984947033918766e-01,
        -3.235750823461782666e-01, 2.994380356487793549e-01,
        -7.888021235143724552e-02, -1.910472056257344828e-02,
        3.043509756878240990e-01, -4.298489130677841108e-01,
        7.233655581626995401e-01, -8.161936207950235556e-01,
        1.054311731055406431e+00, -1.057859070622381603e+00,
        1.189505276805579825e+00, -1.071819355511486993e+00,
        1.077289848919564141e+00, -8.458743245226547636e-01,
        7.443908003078970603e-01, -4.461507052923350813e-01,
        2.868208498373531756e-01, 8.293917061440093941e-03,
        -1.688393814622326516e-01, 3.930003549460732160e-01,
        -5.148213607425790039e-01, 6.232961968619017412e-01,
        -6.962255871378753014e-01, 6.749680427277989780e-01,
        -7.157027353444623818e-01, 5.769543174180707945e-01,
        -6.151602424632486299e-01, 3.892025319337197864e-01,
        -4.506933666706669506e-01, 1.804687704243498336e-01,
        -2.727244303574414830e-01, 1.174346932607069939e-02,
        -1.137579568068424057e-01, -7.811421895993872488e-02,
        1.779393134675293781e-02, -8.675295089134033022e-02,
        1.335093001151206882e-01, -5.514694464669991220e-02,
        2.441624548660136784e-01, -5.518933795483266236e-02,
        3.328770195966135881e-01, -1.532145946078544430e-01,
        3.579812932040986606e-01, -3.542418291888540516e-01,
        3.138078445399646865e-01, -5.530414472417091165e-01,
        2.996207005996078809e-01, -5.735406549188309944e-01,
        4.429014477665748628e-01, -3.892397979507834505e-01,
        6.319796721730875921e-01, -3.224362675105876264e-01,
        5.280557016061210307e-01, -5.925152169592526885e-01,
        3.097933933610434454e-01, -6.161398121500974989e-01,
        5.824585099080735739e-01, -2.863603061506506120e-01,
        5.579069742591284964e-01, -5.790177379949956737e-01,
        1.778203303660013113e-01, -2.248159329174458654e-01,
        4.570279405020428731e-01, -2.082694991220853942e-01,
        -1.639463813834285966e-01, 1.219471051626438846e-01,
        1.315907246465166380e-01, -1.561922951483753763e-01,
        -9.000173966500184253e-02, 3.555651953426591794e-01,
        -4.594671107387718334e-01, 4.051183609553802301e-01,
        -2.826594312508529105e-01, 1.664628231342878961e-01,
        -8.573386548433219179e-02, 3.940897144775467459e-02,
        -1.632396319789397934e-02, 6.095399479649578345e-03,
        -2.034197210251703809e-03, 5.959585007671103435e-04,
        -1.489396688112877424e-04, 3.040429693056684047e-05,
        -4.735729642331119566e-06, 4.982902654694162416e-07,
        -2.645962918790746080e-08])

    dlf.j1 = np.array([
        -2.594301879688918743e-03, 2.339490069567079153e-02,
        -9.478827695764932559e-02, 2.269100107268227362e-01,
        -3.508900104631907935e-01, 3.515648778060092017e-01,
        -1.994483426338835574e-01, 9.293484158449249674e-03,
        8.048790331434139966e-02, -5.691508909482047296e-02,
        2.075411236619714023e-02, -5.275228907131672418e-02,
        1.348060780914397960e-01, -1.939017682982676627e-01,
        1.854125006516780250e-01, -1.238162114378456441e-01,
        5.435433909585837137e-02, -1.270327896980103649e-02,
        7.560629247467166685e-03, -2.713648547163816441e-02,
        5.383176198347366936e-02, -7.463014212375274070e-02,
        8.413254388281748986e-02, -8.280914671407160754e-02,
        7.393816085156304507e-02, -6.119942568117115594e-02,
        4.746907894866293082e-02, -3.450474036399159977e-02,
        2.313104032185092293e-02, -1.354138672902785445e-02,
        5.604638952874172256e-03, 9.486899997092870969e-04,
        -6.382314415042419746e-03, 1.091601370650388016e-02,
        -1.467479730415811347e-02, 1.771829574440562591e-02,
        -2.001990396213982823e-02, 2.152720536808908763e-02,
        -2.214634467574278301e-02, 2.181839780171816040e-02,
        -2.049006667874660528e-02, 1.819226618052163791e-02,
        -1.498058384418661862e-02, 1.101097898088004151e-02,
        -6.444381466006598655e-03, 1.527769500667217105e-03,
        3.533045890696502513e-03, -8.470577184056853060e-03,
        1.310891443998619434e-02, -1.722186749311952272e-02,
        2.071070403809925978e-02, -2.340849145682623311e-02,
        2.528895672008608583e-02, -2.620974028011485712e-02,
        2.617463680378217042e-02, -2.501974545161460978e-02,
        2.276381800709064221e-02, -1.923696679546721411e-02,
        1.454384146700743972e-02, -8.605818651281786635e-03,
        1.738844503682727928e-03, 5.947517515245405624e-03,
        -1.385885800081448037e-02, 2.171703574368676753e-02,
        -2.872814698074024550e-02, 3.461838232135941440e-02,
        -3.858941536214401807e-02, 4.060565074712448042e-02,
        -4.004001014668581715e-02, 3.725276070158690944e-02,
        -3.184536604257427045e-02, 2.462644816656397312e-02,
        -1.539955242663786951e-02, 5.437594699869644464e-03,
        5.315602443775078317e-03, -1.514798704965643165e-02,
        2.414409295479183482e-02, -3.032245442310742278e-02,
        3.416403682006976389e-02, -3.375211066545635158e-02,
        3.045425146826126819e-02, -2.272683240323554107e-02,
        1.317397219169818418e-02, -6.119848802195663844e-04,
        -1.115685958739305421e-02, 2.344848055710842955e-02,
        -3.169397030448774938e-02, 3.817520279763731567e-02,
        -3.809651169502087376e-02, 3.552394037620545952e-02,
        -2.568176960339346032e-02, 1.487169880055576841e-02,
        1.997965754252607837e-03, -1.644546122863936588e-02,
        3.480415781283187349e-02, -4.677537576184215284e-02,
        6.111820051573001178e-02, -6.590953690231306228e-02,
        7.332112878498593667e-02, -6.941405787722150500e-02,
        7.037058951810715168e-02, -5.921683716330301134e-02,
        5.655416878814193554e-02, -4.097804417398832194e-02,
        3.810136479256000241e-02, -2.058552251762190213e-02,
        2.012009960638341116e-02, -1.904932224648239053e-03,
        5.324133880430357950e-03, 1.357085636140308721e-02,
        -5.592398994521355186e-03, 2.589415408390925710e-02,
        -1.292224858788840886e-02, 3.561835458678683924e-02,
        -1.700336052804993919e-02, 4.302272543873187499e-02,
        -1.764501119714675936e-02, 4.776824187802861110e-02,
        -1.404967795338326469e-02, 4.909105531804618811e-02,
        -5.268363386320836297e-03, 4.641444115511134810e-02,
        9.124170345995139680e-03, 3.991380510079723526e-02,
        2.859680534219202416e-02, 3.073032173944831302e-02,
        5.151734601415858261e-02, 2.097520864952298614e-02,
        7.511832720419686638e-02, 1.354148686126329000e-02,
        9.576070647813632319e-02, 1.140272504589405489e-02,
        1.097360846564945647e-01, 1.641156037670641471e-02,
        1.142607074214286172e-01, 2.822722328175052836e-02,
        1.079741026261866604e-01, 4.397838549427537935e-02,
        9.072532008032478668e-02, 5.865331390992817306e-02,
        6.310580311949447185e-02, 6.592890500839264367e-02,
        2.624768245556352575e-02, 5.954092122676180737e-02,
        -1.789442088462365327e-02, 3.555984804371448149e-02,
        -6.551973841104229146e-02, -4.489823587807304991e-03,
        -1.089659607197719371e-01, -5.052377390460756346e-02,
        -1.345089168583019634e-01, -8.286532978746213862e-02,
        -1.229744228392625760e-01, -7.728334735138943368e-02,
        -5.831402820224085293e-02, -2.088719549713474385e-02,
        5.205792146829352207e-02, 6.258826423151815643e-02,
        1.537593434905694390e-01, 9.979165672991972824e-02,
        1.545936043232943868e-01, 1.442374184540378551e-02,
        1.426608350698001064e-02, -1.462978689171700042e-01,
        -1.285617304470880462e-01, -1.519560295563492092e-01,
        -4.700754607069132507e-02, 1.106161196544740571e-01,
        1.168039951766200457e-01, 2.366820009670577984e-01,
        -1.283956636561735531e-01, 1.406024618127853579e-02,
        -4.011626826391421763e-01, 2.656020926376266300e-01,
        -1.291178852844470371e-01, 5.190017993719785450e-01,
        -5.307617026829943851e-01, 2.388177316428383157e-01,
        -4.213777919843004205e-01, 7.425023205388556757e-01,
        -6.039491377476802203e-01, 2.966551251977549986e-01,
        -3.157763414193298090e-01, 5.842678377857475347e-01,
        -7.375386163194860289e-01, 6.321787190293964853e-01,
        -3.871768116388242809e-01, 1.635153404860406612e-01,
        -3.134030444480916111e-02, -2.007964312111037639e-02,
        2.747060309825222202e-02, -1.969027573928272892e-02,
        1.080160200622757964e-02, -4.917741573140644099e-03,
        1.900872502605262596e-03, -6.228037791451519409e-04,
        1.698280169898911708e-04, -3.718088395106743311e-05,
        6.139140918052498045e-06, -6.796809441105533029e-07,
        3.780807126974975489e-08])

    return dlf


# 2. Fourier DLF (cosine/sine)


def key_81_CosSin_2009():
    r"""Key 81 pt CosSin filter, as published in [Key09]_.

    Taken from file `FilterModules.f90` provided with 1DCSEM_.

    License: `Apache License, Version 2.0,
    <https://www.apache.org/licenses/LICENSE-2.0>`_.

    """

    dlf = DigitalFilter('Key 81 CosSin (2009)', 'key_81_CosSin_2009')

    dlf.base = np.array([
        3.354626279025119e-04, 4.097349789797864e-04,  5.004514334406104e-04,
        6.112527611295723e-04, 7.465858083766792e-04,  9.118819655545162e-04,
        1.113775147844802e-03, 1.360368037547893e-03,  1.661557273173934e-03,
        2.029430636295734e-03, 2.478752176666358e-03,  3.027554745375813e-03,
        3.697863716482929e-03, 4.516580942612666e-03,  5.516564420760772e-03,
        6.737946999085467e-03, 8.229747049020023e-03,  1.005183574463358e-02,
        1.227733990306844e-02, 1.499557682047770e-02,  1.831563888873418e-02,
        2.237077185616559e-02, 2.732372244729256e-02,  3.337326996032607e-02,
        4.076220397836620e-02, 4.978706836786394e-02,  6.081006262521795e-02,
        7.427357821433388e-02, 9.071795328941247e-02,  1.108031583623339e-01,
        1.353352832366127e-01, 1.652988882215865e-01,  2.018965179946554e-01,
        2.465969639416064e-01, 3.011942119122020e-01,  3.678794411714423e-01,
        4.493289641172216e-01, 5.488116360940264e-01,  6.703200460356393e-01,
        8.187307530779818e-01, 1e0,  1.221402758160170e+00,
        1.491824697641270e+00, 1.822118800390509e+00,  2.225540928492468e+00,
        2.718281828459046e+00, 3.320116922736548e+00,  4.055199966844675e+00,
        4.953032424395115e+00, 6.049647464412947e+00,  7.389056098930650e+00,
        9.025013499434122e+00, 1.102317638064160e+01,  1.346373803500169e+01,
        1.644464677109706e+01, 2.008553692318767e+01,  2.453253019710935e+01,
        2.996410004739703e+01, 3.659823444367799e+01,  4.470118449330084e+01,
        5.459815003314424e+01, 6.668633104092515e+01,  8.145086866496814e+01,
        9.948431564193386e+01, 1.215104175187350e+02,  1.484131591025766e+02,
        1.812722418751512e+02, 2.214064162041872e+02,  2.704264074261528e+02,
        3.302995599096489e+02, 4.034287934927351e+02,  4.927490410932563e+02,
        6.018450378720822e+02, 7.350951892419732e+02,  8.978472916504184e+02,
        1.096633158428459e+03, 1.339430764394418e+03,  1.635984429995927e+03,
        1.998195895104119e+03, 2.440601977624501e+03,  2.980957987041728e+03])

    dlf.factor = np.array([1.2214027581601701])

    dlf.cos = np.array([
        1.746412733678043e-02, -7.658725022064888e-02, 1.761673907472465e-01,
        -2.840940679113589e-01,  3.680388960144733e-01, -4.115498161707958e-01,
        4.181209762362728e-01, -3.967204599348831e-01, 3.608829691008270e-01,
        -3.171870084102961e-01,  2.744932842186247e-01, -2.324673650676961e-01,
        1.971144816936984e-01, -1.634915360178986e-01, 1.381406405905393e-01,
        -1.125728533897677e-01,  9.619580319372194e-02, -7.640431432353632e-02,
        6.748891657821673e-02, -5.097864570224415e-02, 4.853609305288441e-02,
        -3.293272689265632e-02,  3.677175984620380e-02, -1.969323595300588e-02,
        3.053726798991684e-02, -9.301135480582538e-03, 2.895215492109734e-02,
        -1.875526095801418e-04,  3.181452657662026e-02, 9.025726238227111e-03,
        3.955376604096631e-02,  1.966766645672513e-02, 5.318782805621459e-02,
        3.300575875620110e-02,  7.409212944640006e-02, 4.972863917303501e-02,
        1.029344264288086e-01,  6.776855697600163e-02, 1.357865756912759e-01,
        7.511614666518443e-02,  1.522218287240260e-01, 3.034571997381229e-02,
        8.802563675323094e-02, -1.689255322598353e-01, -1.756581788680092e-01,
        -6.123863775740898e-01, -5.098359641153184e-01, -6.736869803920745e-01,
        4.599561125225532e-01,  8.907010262082216e-01, 1.039153770711999e+00,
        -2.178135931072732e+00,  8.040971159674268e-01, 5.659848584656202e-01,
        -9.349050336534268e-01,  8.006099486213468e-01, -5.944960111930493e-01,
        4.369614304892440e-01, -3.292566347310282e-01, 2.547426420681868e-01,
        -2.010899026277397e-01,  1.609467208423519e-01, -1.299975550484158e-01,
        1.056082501090365e-01, -8.608337452556068e-02, 7.027252107999236e-02,
        -5.735742622053085e-02,  4.673270108060494e-02, -3.793635725863799e-02,
        3.060786160620013e-02, -2.446220554726340e-02, 1.927399223200865e-02,
        -1.486843016804444e-02,  1.111747692371507e-02, -7.939442960305236e-03,
        5.298852472637883e-03, -3.200104589830043e-03, 1.665382777953919e-03,
        -6.913074254614758e-04,  1.999065225130592e-04,
        -2.955159288961187e-05])

    dlf.sin = np.array([
        7.478326513505658e-07, -2.572850425065560e-06, 5.225955618519281e-06,
        -7.352539610140040e-06,  8.768819961093828e-06, -8.560004370841340e-06,
        8.101932279460349e-06, -5.983552716117552e-06, 5.036792825138655e-06,
        -1.584355068233649e-06,  1.426050228179462e-06, 3.972863429067356e-06,
        -1.903788077376088e-06,  1.144652944379527e-05, -4.327773998196030e-06,
        2.297298998355334e-05, -4.391227697686659e-06, 4.291202395830839e-05,
        1.760279032167125e-06,  8.017887907026914e-05, 2.364651853689879e-05,
        1.535031685829202e-04,  8.375427119939347e-05, 3.030115685600468e-04,
        2.339455351760637e-04,  6.157392107422657e-04, 5.921808556382737e-04,
        1.281873037121434e-03,  1.424276189020714e-03, 2.718506171172064e-03,
        3.324504626808429e-03,  5.839859904586436e-03, 7.608663600764702e-03,
        1.263571470998938e-02,  1.714199295539484e-02, 2.735013970005427e-02,
        3.794840483226463e-02,  5.858519896601026e-02, 8.166914231915734e-02,
        1.215508018998907e-01,  1.658946642767184e-01, 2.324389477118542e-01,
        2.938956625118840e-01,  3.572525844816433e-01, 3.479235360502319e-01,
        2.294314115090992e-01, -1.250412450354792e-01, -6.340986743027450e-01,
        -9.703404081656508e-01, -2.734109755210948e-01, 1.321852608494946e+00,
        6.762199721133603e-01, -2.093257651144232e+00, 1.707842350925794e+00,
        -8.844618831465598e-01,  3.720792781726873e-01, -1.481509947473694e-01,
        6.124339615448667e-02, -2.726194382687923e-02, 1.307668436907975e-02,
        -6.682101544475918e-03,  3.599101395415812e-03, -2.030735143712865e-03,
        1.197624324158372e-03, -7.382202519234128e-04, 4.756906961407787e-04,
        -3.199977708080284e-04,  2.238628518300115e-04, -1.618377502708346e-04,
        1.199233854156409e-04, -9.025345928219504e-05, 6.830860296946832e-05,
        -5.143409372298764e-05,  3.804574823200909e-05, -2.720604959632104e-05,
        1.839913059679674e-05, -1.140157702141663e-05, 6.172802138985788e-06,
        -2.706562852604888e-06,  8.403636781016683e-07,
        -1.356300450956746e-07])

    return dlf


def key_241_CosSin_2009():
    r"""Key 241 pt CosSin filter, as published in [Key09]_.

    Taken from file `FilterModules.f90` provided with 1DCSEM_.

    License: `Apache License, Version 2.0,
    <https://www.apache.org/licenses/LICENSE-2.0>`_.

    """

    dlf = DigitalFilter('Key 241 CosSin (2009)', 'key_241_CosSin_2009')

    dlf.base = np.array([
        1.015631471002490e-06, 1.139409969679808e-06, 1.278273779488425e-06,
        1.434061399152754e-06,  1.608835391556724e-06, 1.804909691213134e-06,
        2.024880239788185e-06,  2.271659355282662e-06, 2.548514289903414e-06,
        2.859110487115151e-06,  3.207560110577854e-06, 3.598476487472605e-06,
        4.037035187022686e-06,  4.529042542864016e-06, 5.081012526472398e-06,
        5.700252989419669e-06,  6.394962416269964e-06, 7.174338469084136e-06,
        8.048699760616019e-06,  9.029622468426732e-06, 1.013009359863071e-05,
        1.136468292842132e-05,  1.274973590382314e-05, 1.430359004655643e-05,
        1.604681773515003e-05,  1.800249857462311e-05, 2.019652496079635e-05,
        2.265794488477607e-05,  2.541934651619925e-05, 2.851728965696073e-05,
        3.199278977768923e-05,  3.589186104541253e-05, 4.026612553187130e-05,
        4.517349666814366e-05,  5.067894599423484e-05, 5.685536335507487e-05,
        6.378452193155948e-05,  7.155816088324133e-05, 8.027919993640778e-05,
        9.006310199818260e-05,  1.010394018370932e-04, 1.133534210691922e-04,
        1.271681921554341e-04,  1.426666168832257e-04, 1.600538879095432e-04,
        1.795602054258330e-04,  2.014438249122027e-04, 2.259944763318926e-04,
        2.535372000347300e-04,  2.844366501553267e-04, 3.191019224812033e-04,
        3.579919707098018e-04,  4.016216828033581e-04, 4.505686978900301e-04,
        5.054810539641999e-04,  5.670857676383036e-04, 6.361984595385052e-04,
        7.137341527804347e-04,  8.007193874922814e-04, 8.983058117764621e-04,
        1.007785429048510e-03,  1.130607703621563e-03, 1.268398751007239e-03,
        1.422982867004443e-03,  1.596406680612247e-03, 1.790966250541270e-03,
        2.009237464070060e-03,  2.254110140714601e-03, 2.528826292229256e-03,
        2.837023045485535e-03,  3.182780796509667e-03, 3.570677233218250e-03,
        4.005847942090417e-03,  4.494054401183452e-03, 5.041760259690979e-03,
        5.656216913953108e-03,  6.345559512909110e-03, 7.118914664064660e-03,
        7.986521265955502e-03,  8.959866066878890e-03, 1.005183574463358e-02,
        1.127688752074056e-02,  1.265124056800531e-02, 1.419309074557768e-02,
        1.592285150451168e-02,  1.786342415331403e-02, 2.004050106168401e-02,
        2.248290581673552e-02,  2.522297483522721e-02, 2.829698548418685e-02,
        3.174563637806794e-02,  3.561458621137186e-02, 3.995505826065390e-02,
        4.482451855926687e-02,  5.028743672359186e-02, 5.641613950377735e-02,
        6.329176835964070e-02,  7.100535373963698e-02, 7.965902028589801e-02,
        8.936733892175319e-02,  1.002588437228037e-01, 1.124777336542896e-01,
        1.261857817050387e-01,  1.415644766941340e-01, 1.588174261069207e-01,
        1.781730517728984e-01,  1.998876140751445e-01, 2.242486047305353e-01,
        2.515785530597565e-01,  2.822392961405233e-01, 3.166367693790532e-01,
        3.552263809249515e-01,  3.985190410845142e-01, 4.470879265593564e-01,
        5.015760690660555e-01,  5.627048688069557e-01, 6.312836455069260e-01,
        7.082203534678000e-01,  7.945336025033340e-01, 8.913661439068313e-01,
        1e0,  1.121873437571938e+00, 1.258600009929478e+00,
        1.411989919667659e+00,  1.584073984994482e+00, 1.777130526914038e+00,
        1.993715533243082e+00,  2.236696498819987e+00, 2.509290389936297e+00,
        2.815106235624064e+00,  3.158192909689768e+00, 3.543092736108982e+00,
        3.974901627494749e+00,  4.459336552847826e+00, 5.002811227833588e+00,
        5.612521029693157e+00,  6.296538261026657e+00, 7.063919023701211e+00,
        7.924823117849490e+00,  8.890648553371371e+00, 9.974182454814724e+00,
        1.118977035755271e+01,  1.255350613666823e+01, 1.408344508312441e+01,
        1.579984294826040e+01,  1.772542412146164e+01, 1.988568249156473e+01,
        2.230921897527583e+01,  2.502812018133782e+01, 2.807838322380105e+01,
        3.150039230874794e+01,  3.533945340427969e+01, 3.964639407257260e+01,
        4.447823640552875e+01,  4.989895197340787e+01, 5.598030878164415e+01,
        6.280282144920171e+01,  7.045681718843254e+01, 7.904363169956446e+01,
        8.867695081296063e+01,  9.948431564193386e+01, 1.116088111737080e+02,
        1.252109606547652e+02,  1.404708508514462e+02, 1.575905163233670e+02,
        1.767966142764323e+02,  1.983434254093812e+02, 2.225162204838159e+02,
        2.496350371896939e+02,  2.800589173104007e+02, 3.141906602856942e+02,
        3.524821561077089e+02,  3.954403681553241e+02, 4.436340451771264e+02,
        4.977012512868071e+02,  5.583578136649854e+02, 6.264067998114892e+02,
        7.027491498229525e+02,  7.883956044626324e+02, 8.844800869450999e+02,
        9.922747156050262e+02,  1.113206646211529e+03, 1.248876966913255e+03,
        1.401081895975390e+03,  1.571836562957720e+03, 1.763401688186638e+03,
        1.978313513746104e+03,  2.219417382261360e+03, 2.489905408044465e+03,
        2.793358739351805e+03,  3.133794971288225e+03, 3.515721337084776e+03,
        3.944194381980310e+03,  4.424886909764171e+03, 4.964163088324202e+03,
        5.569162708566004e+03,  6.247895712256392e+03, 7.009348240300053e+03,
        7.863601605484239e+03,  8.821965764840819e+03, 9.897129058743927e+03,
        1.110332619922617e+04,  1.245652673160841e+04, 1.397464646459628e+04,
        1.567778466808916e+04,  1.758849017910182e+04, 1.973205993892924e+04,
        2.213687391406209e+04,  2.483477083506541e+04, 2.786146972804616e+04,
        3.125704281960959e+04,  3.506644607636869e+04, 3.934011440312676e+04,
        4.413462937990915e+04,  4.951346837840215e+04, 5.554784497578751e+04,
        6.231765179269986e+04,  6.991251823808729e+04, 7.843299716507383e+04,
        8.799189614865134e+04,  9.871577101076049e+04, 1.107466013664062e+05,
        1.242436703743393e+05,  1.393856735794148e+05, 1.563730847668183e+05,
        1.754308101510785e+05,  1.968111660402206e+05, 2.207972193980835e+05,
        2.477065355324534e+05,  2.778953825268291e+05, 3.117634480807426e+05,
        3.497591312076233e+05,  3.923854788500711e+05, 4.402068460108404e+05,
        4.938563675768826e+05,  5.540443407602682e+05, 6.215676291359995e+05,
        6.973202127822436e+05,  7.823050242024113e+05, 8.776472267317577e+05,
        9.846091112290357e+05])

    dlf.factor = np.array([1.1218734375719379])

    dlf.cos = np.array([
        2.573985433748149e-03, -2.072044432524625e-02, 8.438091711547871e-02,
        -2.334480374945598e-01,  4.971637805522022e-01, -8.750461405249571e-01,
        1.333415700757027e+00, -1.817857278999438e+00, 2.271742525614601e+00,
        -2.650839359834871e+00,  2.929816176240629e+00, -3.101605349462614e+00,
        3.172849598553826e+00, -3.158333463387740e+00, 3.076219518145157e+00,
        -2.944689843059625e+00,  2.780024568942243e+00, -2.595719779969896e+00,
        2.402350605158043e+00, -2.207779522612721e+00, 2.017594105093030e+00,
        -1.835527970829352e+00,  1.663920259620933e+00, -1.504039483923565e+00,
        1.356423126734103e+00, -1.221057462742312e+00, 1.097607689766934e+00,
        -9.854831639668219e-01,  8.840006973665168e-01, -7.923704689421897e-01,
        7.098300227775011e-01, -6.355744132962947e-01, 5.688875813158415e-01,
        -5.090282851725471e-01,  4.553774950936741e-01, -4.072828190689483e-01,
        3.642362670434036e-01, -3.256737103445609e-01, 2.911959791480738e-01,
        -2.603152640455789e-01,  2.327331570309761e-01, -2.080218702094741e-01,
        1.859749696667782e-01, -1.662071785063838e-01, 1.485969487672789e-01,
        -1.327839524784528e-01,  1.187272713875194e-01, -1.060740651133825e-01,
        9.486317265574648e-02, -8.473151012328659e-02, 7.580053967300120e-02,
        -6.767828163371538e-02,  6.057579031791280e-02, -5.405178236455697e-02,
        4.841846349893079e-02, -4.316208761750292e-02, 3.871284451741208e-02,
        -3.445743869419572e-02,  3.096708549327929e-02, -2.749651494490814e-02,
        2.478846014781735e-02, -2.192620197391235e-02, 1.986356477133265e-02,
        -1.746376930197379e-02,  1.594249816296365e-02, -1.388258927979823e-02,
        1.282624672523178e-02, -1.100069006189238e-02, 1.035664209512100e-02,
        -8.671560169935045e-03,  8.408376795833618e-03, -6.776738899031039e-03,
        6.882686516736744e-03, -5.219834049623621e-03, 5.702385679308191e-03,
        -3.921653713596426e-03,  4.807988471764018e-03, -2.816192636608797e-03,
        4.154713117406505e-03, -1.847264200719943e-03, 3.710210316571563e-03,
        -9.655983418104668e-04,  3.452896295365636e-03, -1.262640384889067e-04,
        3.370827726760556e-03,  7.136760975744594e-04, 3.461099737561205e-03,
        1.597372595722807e-03,  3.729726810304828e-03, 2.570391470398246e-03,
        4.191950937402854e-03,  3.683048200124032e-03, 4.872941625227778e-03,
        4.992964784095300e-03,  5.808908133499602e-03, 6.567853057094176e-03,
        7.048649360560634e-03,  8.488493292803347e-03, 8.655425482515024e-03,
        1.085182308666227e-02,  1.070880791806313e-02, 1.377379498928377e-02,
        1.330579956040371e-02,  1.739106597624804e-02, 1.655975905406871e-02,
        2.185942056374162e-02,  2.059407781725530e-02, 2.734455978483885e-02,
        2.552438621384559e-02,  3.399650586625764e-02, 3.141687500736497e-02,
        4.189054475997781e-02,  3.819877849864893e-02, 5.090221700189159e-02,
        4.547666713650442e-02,  6.045737507914792e-02, 5.218516934955635e-02,
        6.905902304597721e-02,  5.594605521455825e-02, 7.345454126722073e-02,
        5.200083839448903e-02,  6.734587378298894e-02, 3.173662493273055e-02,
        3.992917476294457e-02, -1.838550375569295e-02, -2.397773636392629e-02,
        -1.141550868458520e-01, -1.364180631421666e-01, -2.579978464039199e-01,
        -2.783350244258052e-01, -3.949234086091623e-01, -3.393321236784040e-01,
        -3.475708749988657e-01, -8.309968487457564e-02, 1.136203792647491e-01,
        5.634934316303769e-01,  6.583850354915292e-01, 6.518868979555039e-01,
        -1.501485600802549e-01, -7.813033407168875e-01, -1.093007847302057e+00,
        3.833853368779607e-01,  1.282013367267770e+00, 2.845449032623110e-01,
        -2.058783831648639e+00,  1.209750949676112e+00, 7.040614890578353e-01,
        -1.742049993519849e+00,  1.741132290073874e+00, -1.318424767807200e+00,
        9.007174373555373e-01, -6.085273047132523e-01, 4.259667647291260e-01,
        -3.134807813694764e-01,  2.417216785355954e-01, -1.934461726074489e-01,
        1.591659829008638e-01, -1.336457289506658e-01, 1.138988598115308e-01,
        -9.814397025931439e-02,  8.526827650704955e-02, -7.454576060490363e-02,
        6.548369826637619e-02, -5.773569245976477e-02, 5.105043114013155e-02,
        -4.524038427505326e-02,  4.016204946284081e-02, -3.570311893510748e-02,
        3.177392452843057e-02, -2.830160023880827e-02, 2.522601353974157e-02,
        -2.249687303433539e-02,  2.007163372774035e-02, -1.791395268505854e-02,
        1.599253061462437e-02, -1.428022817956371e-02, 1.275338071888894e-02,
        -1.139125817575050e-02,  1.017563265146541e-02, -9.090426866557829e-03,
        8.121424445062036e-03, -7.256028270711461e-03, 6.483056882115604e-03,
        -5.792571404381900e-03,  5.175727196581864e-03, -4.624645684779383e-03,
        4.132303011690729e-03, -3.692432980935121e-03, 3.299442142508096e-03,
        -2.948335089474538e-03,  2.634648489517598e-03, -2.354392904061866e-03,
        2.104001678817889e-03, -1.880286100963581e-03, 1.680395912528533e-03,
        -1.501784333722836e-03,  1.342176883640446e-03, -1.199543365542324e-03,
        1.072072487299258e-03, -9.581487548508404e-04, 8.563313760453976e-04,
        -7.653349620480948e-04,  6.840119836821465e-04, -6.113371472348392e-04,
        5.463938378836874e-04, -4.883625033668457e-04, 4.365105296790329e-04,
        -3.901830736548332e-04,  3.487945803629603e-04, -3.118210765665508e-04,
        2.787934276328260e-04, -2.492915523784893e-04, 2.229394184733318e-04,
        -1.994006680996778e-04,  1.783747993871292e-04, -1.595938065118272e-04,
        1.428191356382435e-04, -1.278388291932361e-04, 1.144647631077232e-04,
        -1.025299256786045e-04,  9.188578754406515e-05, -8.239993677764002e-05,
        7.395418447973129e-05, -6.644317144520790e-05, 5.977325048819278e-05,
        -5.386141578322173e-05,  4.863431532204160e-05, -4.402750207962412e-05,
        3.998490104646043e-05, -3.645833654257180e-05, 3.340701147193886e-05,
        -3.079679563171623e-05,  2.859897380035983e-05, -2.678774555509045e-05,
        2.533483928373590e-05, -2.419702895729874e-05, 2.328604080003924e-05,
        -2.239597546920266e-05,  2.103439404619766e-05, -1.807670762071730e-05,
        1.140396792815156e-05])

    dlf.sin = np.array([
        1.967005545865218e-10, -1.433611192275181e-09, 5.386453134154996e-09,
        -1.393836398944422e-08,  2.810158973085062e-08, -4.733984167038972e-08,
        6.976700567760687e-08, -9.291356831468874e-08, 1.145563482421100e-07,
        -1.331950589956367e-07,  1.481901450989667e-07, -1.595961894804095e-07,
        1.679579344961122e-07, -1.740358177568700e-07, 1.786885513317428e-07,
        -1.826971268722640e-07,  1.868003120723728e-07, -1.915696157142666e-07,
        1.975548111660444e-07, -2.051269356766739e-07, 2.147190183078524e-07,
        -2.265704864101776e-07,  2.410938637923347e-07, -2.584531304193038e-07,
        2.791285853082430e-07, -3.032349303994803e-07, 3.314044275843927e-07,
        -3.636972724516131e-07,  4.009929886338131e-07, -4.432582392200706e-07,
        4.917458331080907e-07, -5.462464678254600e-07, 6.085682267245114e-07,
        -6.781794860836133e-07,  7.577171014688372e-07, -8.460807541480724e-07,
        9.471544157135390e-07, -1.058863057051342e-06, 1.186983447614987e-06,
        -1.327804017767932e-06,  1.489996440779015e-06, -1.667141139261620e-06,
        1.872366272041974e-06, -2.094818002855587e-06, 2.354522121047758e-06,
        -2.633419506735300e-06,  2.962259516643178e-06, -3.311342165909682e-06,
        3.728149200842658e-06, -4.164255374239495e-06, 4.693327995051689e-06,
        -5.236921095650869e-06,  5.909779848521176e-06, -6.585452434703418e-06,
        7.443248817367435e-06, -8.280105308480091e-06, 9.376971248809820e-06,
        -1.040871066885904e-05,  1.181647694012281e-05, -1.308086489086373e-05,
        1.489579590400169e-05, -1.643299839198172e-05, 1.878552965850708e-05,
        -2.063442929000397e-05,  2.370342044306427e-05, -2.589446630933835e-05,
        2.992830392336809e-05, -3.247053052820925e-05, 3.781869997976733e-05,
        -4.067708157172121e-05,  4.783784240317701e-05, -5.089479936195305e-05,
        6.058776523010633e-05, -6.357888851585346e-05, 7.685629664242167e-05,
        -7.926437938452345e-05,  9.768268830663251e-05, -9.856464123672343e-05,
        1.244504845842736e-04, -1.221565888036176e-04, 1.590206705039449e-04,
        -1.507415982780862e-04,  2.039250063083074e-04, -1.849639094970543e-04,
        2.626501374752980e-04, -2.252566203418921e-04, 3.400597277405926e-04,
        -2.715666525558665e-04,  4.430278995600817e-04, -3.228801423595561e-04,
        5.813980073616601e-04, -3.764220699966869e-04, 7.694445136249034e-04,
        -4.263284243168565e-04,  1.028115335781203e-03, -4.614701220354354e-04,
        1.388487108626009e-03, -4.619214991216924e-04, 1.897105649343108e-03,
        -3.932775423791962e-04,  2.624250467245377e-03, -1.975847083236173e-04,
        3.676710851321636e-03,  2.209992359645100e-04, 5.217452274062335e-03,
        1.018611500160813e-03,  7.495613223401942e-03, 2.446097853450125e-03,
        1.089148041817982e-02,  4.900453419060485e-03, 1.598184015155261e-02,
        8.998208308647400e-03,  2.362945832701593e-02, 1.567112707431810e-02,
        3.509105555447196e-02,  2.626701074033223e-02, 5.210752870163974e-02,
        4.258698097657578e-02,  7.685703204761107e-02, 6.665955544060372e-02,
        1.114513473857200e-01,  9.975515861206709e-02, 1.562348056183209e-01,
        1.395790799893133e-01,  2.054534357433025e-01, 1.738733232648114e-01,
        2.384564666870356e-01,  1.692656404318401e-01, 2.076285145563062e-01,
        6.214100432471348e-02,  4.085773864115237e-02, -2.120030217966462e-01,
        -2.788970968243500e-01, -5.635171222319378e-01, -4.755507221458447e-01,
        -4.867419491287002e-01,  1.066910067501506e-01, 4.614580501271743e-01,
        1.034845302098941e+00,  3.764103855021101e-01, -4.329269419291238e-01,
        -1.455324167706572e+00,  2.071144418028701e-01, 1.349913786271064e+00,
        1.542337659378029e-01, -2.228987932739951e+00, 2.446506150864118e+00,
        -1.378605681876207e+00,  3.206812625344759e-01, 2.465832348751217e-01,
        -4.315945927966289e-01,  4.376711630002494e-01, -3.868161577159704e-01,
        3.284216175602037e-01, -2.770656717800635e-01, 2.349261553052497e-01,
        -2.007957619551110e-01,  1.729798893952139e-01, -1.500347821377614e-01,
        1.308631829780874e-01, -1.146558086495966e-01, 1.008159744956652e-01,
        -8.889859962303459e-02,  7.856617356272111e-02, -6.955808615176720e-02,
        6.166942945994992e-02, -5.473628487439387e-02, 4.862541225971129e-02,
        -4.322693473384397e-02,  3.844907736778588e-02, -3.421432802261911e-02,
        3.045659361641401e-02, -2.711906111916627e-02, 2.415256336052804e-02,
        -2.151431084609744e-02,  1.916689229057781e-02, -1.707747503528641e-02,
        1.521715619160838e-02, -1.356042905316614e-02, 1.208473893327246e-02,
        -1.077010938739636e-02,  9.598824634127222e-03, -8.555157475234823e-03,
        7.625134538354346e-03, -6.796332507333996e-03, 6.057700364251606e-03,
        -5.399403681543913e-03,  4.812687767094041e-03, -4.289757047100644e-03,
        3.823668521675937e-03, -3.408237481656769e-03, 3.037953955085763e-03,
        -2.707908575613410e-03,  2.413726747641181e-03, -2.151510135086631e-03,
        1.917784628102023e-03, -1.709454048915126e-03, 1.523758948382411e-03,
        -1.358239922491303e-03,  1.210704945231195e-03, -1.079200272616089e-03,
        9.619845235461060e-04, -8.575055877727934e-04, 7.643800504594847e-04,
        -6.813748575629618e-04,  6.073909770892527e-04, -5.414488385598928e-04,
        4.826753570758706e-04, -4.302923695079621e-04, 3.836063290315453e-04,
        -3.419991209582663e-04,  3.049198778902062e-04, -2.718776857047081e-04,
        2.424350838300435e-04, -2.162022738421805e-04, 1.928319597904253e-04,
        -1.720147520298727e-04,  1.534750738420537e-04, -1.369675168837808e-04,
        1.222735976157924e-04, -1.091988723621016e-04, 9.757037355552881e-05,
        -8.723433410673846e-05,  7.805417075146385e-05, -6.990870062381882e-05,
        6.269056792845232e-05, -5.630485890243343e-05, 5.066788212408145e-05,
        -4.570608516423470e-05,  4.135506222734868e-05, -3.755856918827694e-05,
        3.426737683044253e-05, -3.143760307989191e-05, 2.902774010545716e-05,
        -2.699262909691761e-05,  2.527042803006811e-05, -2.375357212497184e-05,
        2.222370982737331e-05, -2.021079897690936e-05, 1.673161082870818e-05,
        -1.011062914652405e-05])

    return dlf


def key_601_CosSin_2009():
    r"""Key 601 pt CosSin filter, as published in [Key09]_.

    Taken from file `FilterModules.f90` provided with 1DCSEM_.

    License: `Apache License, Version 2.0,
    <https://www.apache.org/licenses/LICENSE-2.0>`_.

    """

    dlf = DigitalFilter('Key 601 CosSin (2009)', 'key_601_CosSin_2009')

    dlf.base = np.array([
        4.193795658379545e-13, 4.611744532326466e-13,  5.071345712552199e-13,
        5.576750220213815e-13, 6.132522762484560e-13,  6.743682960028553e-13,
        7.415750683158400e-13, 8.154795906142572e-13,  8.967493529935761e-13,
        9.861183668479918e-13, 1.084393794306884e-12,  1.192463238353357e-12,
        1.311302759467621e-12, 1.441985691199862e-12,  1.585692334293061e-12,
        1.743720616910949e-12, 1.917497817252074e-12,  2.108593454426215e-12,
        2.318733464020732e-12, 2.549815786387618e-12,  2.803927508441469e-12,
        3.083363713789339e-12, 3.390648211442950e-12,  3.728556330330719e-12,
        4.100139985484672e-12, 4.508755242294825e-12,  4.958092627785737e-12,
        5.452210462680048e-12, 5.995571515297294e-12,  6.593083308338489e-12,
        7.250142442598540e-12, 7.972683337929069e-12,  8.767231831670027e-12,
        9.640964118639377e-12, 1.060177156501484e-11,  1.165833198149268e-11,
        1.282018799944826e-11, 1.409783325797471e-11,  1.550280718022315e-11,
        1.704779919504490e-11, 1.874676334524281e-11,  2.061504431754968e-11,
        2.266951603261053e-11, 2.492873404668339e-11,  2.741310314151893e-11,
        3.014506161605645e-11, 3.314928394441855e-11,  3.645290363056936e-11,
        4.008575827241402e-11, 4.408065904870462e-11,  4.847368706270256e-11,
        5.330451921911242e-11, 5.861678657753666e-11,  6.445846841902501e-11,
        7.088232558484702e-11, 7.794637700130874e-11,  8.571442369448666e-11,
        9.425662502767281e-11, 1.036501323659809e-10,  1.139797858912439e-10,
        1.253388808606835e-10, 1.378300102300461e-10,  1.515659912515919e-10,
        1.666708844157782e-10, 1.832811139395095e-10,  2.015466999209584e-10,
        2.216326132895250e-10, 2.437202657885651e-10,  2.680091484480837e-10,
        2.947186333457411e-10, 3.240899549293070e-10,  3.563883887954330e-10,
        3.919056476030219e-10, 4.309625157605925e-10,  4.739117466835593e-10,
        5.211412487888540e-10, 5.730775890021381e-10,  6.301898454205201e-10,
        6.929938439272241e-10, 7.620568170224476e-10,  8.380025269479489e-10,
        9.215168993763617e-10, 1.013354218547565e-09,  1.114343939805221e-09,
        1.225398181062923e-09, 1.347519960861261e-09,  1.481812257420264e-09,
        1.629487970506598e-09, 1.791880876089040e-09,  1.970457672722430e-09,
        2.166831228460398e-09, 2.382775147940245e-09,  2.620239791206898e-09,
        2.881369888954443e-09, 3.168523913282466e-09,  3.484295378919872e-09,
        3.831536267304207e-09, 4.213382785077881e-09,  4.633283689646769e-09,
        5.095031437631424e-09, 5.602796437537268e-09,  6.161164716006833e-09,
        6.775179337847431e-09, 7.450385953931326e-09,  8.192882888347710e-09,
        9.009376217182704e-09, 9.907240336387463e-09,  1.089458456577097e-08,
        1.198032639067020e-08, 1.317427200280147e-08,  1.448720486772051e-08,
        1.593098311881483e-08, 1.751864665746923e-08,  1.926453492671132e-08,
        2.118441642204417e-08, 2.329563110917976e-08,  2.561724703496062e-08,
        2.817023254594732e-08, 3.097764567011258e-08,  3.406484237209804e-08,
        3.745970556295245e-08, 4.119289693271730e-08,  4.529813388035942e-08,
        4.981249404222491e-08, 5.477675016944901e-08,  6.023573837886479e-08,
        6.623876310337785e-08, 7.284004239922980e-08,  8.009919763207375e-08,
        8.808179196460557e-08, 9.685992250925397e-08,  1.065128714941292e-07,
        1.171278223234277e-07, 1.288006469995941e-07,  1.416367720190751e-07,
        1.557521305622528e-07, 1.712742095775383e-07,  1.883432012166641e-07,
        2.071132690207019e-07, 2.277539402927298e-07,  2.504516372327622e-07,
        2.754113606638370e-07, 3.028585415563174e-07,  3.330410770729809e-07,
        3.662315695240386e-07, 4.027297884538403e-07,  4.428653780963272e-07,
        4.870008346523962e-07, 5.355347802793109e-07,  5.889055633621554e-07,
        6.475952175842209e-07, 7.121338155538044e-07,  7.831042563084790e-07,
        8.611475299365608e-07, 9.469685068647088e-07,  1.041342304099322e-06,
        1.145121285920239e-06, 1.259242762255579e-06,  1.384737454267802e-06,
        1.522738803610367e-06, 1.674493209434266e-06,  1.841371285602920e-06,
        2.024880239788189e-06, 2.226677486252946e-06,  2.448585615267984e-06,
        2.692608854363838e-06, 2.960951170092142e-06,  3.256036173787822e-06,
        3.580529011116689e-06, 3.937360433110378e-06,  4.329753266092978e-06,
        4.761251519570304e-06, 5.235752394978102e-06,  5.757541484385375e-06,
        6.331331477060257e-06, 6.962304723487946e-06,  7.656160041269817e-06,
        8.419164185644976e-06, 9.258208449505048e-06,  1.018087090410152e-05,
        1.119548484259094e-05, 1.231121404458520e-05,  1.353813554148094e-05,
        1.488733063008695e-05, 1.637098495656519e-05,  1.800249857462311e-05,
        1.979660697197936e-05, 2.176951415818827e-05,  2.393903901584477e-05,
        2.632477623698299e-05, 2.894827329821157e-05,  3.183322507298884e-05,
        3.500568783873448e-05, 3.849431461164446e-05,  4.233061393470499e-05,
        4.654923445622278e-05, 5.118827786912642e-05,  5.628964303744043e-05,
        6.189940441800879e-05, 6.806822819529511e-05,  7.485182988770060e-05,
        8.231147755840270e-05, 9.051454517561093e-05,  9.953512112007229e-05,
        1.094546773357366e-04, 1.203628051672132e-04,  1.323580245299439e-04,
        1.455486737213506e-04, 1.600538879095432e-04,  1.760046751370900e-04,
        1.935450995580938e-04, 2.128335825953212e-04,  2.340443337691579e-04,
        2.573689241213434e-04, 2.830180164443136e-04,  3.112232679432146e-04,
        3.422394225150394e-04, 3.763466115419071e-04,  4.138528839787616e-04,
        4.550969885867208e-04, 5.004514334406108e-04,  5.503258503435191e-04,
        6.051706945350531e-04, 6.654813131082850e-04,  7.318024188804728e-04,
        8.047330101246132e-04, 8.849317805958146e-04,  9.731230687146970e-04,
        1.070103399639604e-03, 1.176748679314238e-03,  1.294022105465848e-03,
        1.422982867004445e-03, 1.564795710394167e-03,  1.720741459398286e-03,
        1.892228583209938e-03, 2.080805917449529e-03,  2.288176652922169e-03,
        2.516213718478670e-03, 2.766976696914850e-03,  3.042730426689990e-03,
        3.345965457471272e-03, 3.679420544254346e-03,  4.046107383222199e-03,
        4.449337812751394e-03, 4.892753725239476e-03,  5.380359959910820e-03,
        5.916560473681857e-03, 6.506198116773342e-03,  7.154598372314579e-03,
        7.867617454986535e-03, 8.651695203120634e-03,  9.513913241963630e-03,
        1.046205894342679e-02, 1.150469575999052e-02,  1.265124056800531e-02,
        1.391204871893762e-02, 1.529850756672552e-02,  1.682313931596340e-02,
        1.849971411981924e-02, 2.034337444916063e-02,  2.237077185616560e-02,
        2.460021736763830e-02, 2.705184686635041e-02,  2.974780295409754e-02,
        3.271243493901982e-02, 3.597251875342965e-02,  3.955749878839872e-02,
        4.349975382930274e-02, 4.783488949419837e-02,  5.260205981627383e-02,
        5.784432087483846e-02, 6.360901966877180e-02,  6.994822174465536e-02,
        7.691918144183450e-02, 8.458485900156469e-02,  9.301448921066349e-02,
        1.022842067155374e-01, 1.124777336542896e-01,  1.236871358174548e-01,
        1.360136541668492e-01, 1.495686192226351e-01,  1.644744565771549e-01,
        1.808657926171221e-01, 1.988906704408197e-01,  2.187118869522148e-01,
        2.405084632083421e-01, 2.644772612998240e-01,  2.908347623678516e-01,
        3.198190218163038e-01, 3.516918193780669e-01,  3.867410234545012e-01,
        4.252831910822741e-01, 4.676664270099092e-01,  5.142735277066319e-01,
        5.655254386995371e-01, 6.218850564650201e-01,  6.838614092123558e-01,
        7.520142543193826e-01, 8.269591339433623e-01,  9.093729344682314e-01,
        1.000000000000000e+00, 1.099658855126103e+00,  1.209249597657251e+00,
        1.329762028121474e+00, 1.462284589434224e+00,  1.608014197485783e+00,
        1.768267051433735e+00, 1.944490521336831e+00,  2.138276220496818e+00,
        2.351374380574902e+00, 2.585709659315846e+00,  2.843398523651769e+00,
        3.126768365186156e+00, 3.438378520705125e+00,  3.781043387568781e+00,
        4.157857842756007e+00, 4.572225195142160e+00,  5.027887923468749e+00,
        5.528961477624004e+00, 6.079971448520338e+00,  6.685894442279269e+00,
        7.352203027890797e+00, 8.084915164305059e+00,  8.890648553371371e+00,
        9.776680409528907e+00, 1.075101318607635e+01,  1.182244685164636e+01,
        1.300065836967064e+01, 1.429628909867760e+01,  1.572104090280360e+01,
        1.728778184056764e+01, 1.901066238646844e+01,  2.090524323509275e+01,
        2.298863584203481e+01, 2.527965697096289e+01,  2.779899864266967e+01,
        3.056941502105021e+01, 3.361592792392278e+01,  3.696605281482250e+01,
        4.065004731687878e+01, 4.470118449330081e+01,  4.915605336268389e+01,
        5.405488936332660e+01, 5.944193775124387e+01,  6.536585321400990e+01,
        7.188013930965907e+01, 7.904363169956446e+01,  8.692102953975237e+01,
        9.558347983006632e+01, 1.051092199988997e+02,  1.155842845271877e+02,
        1.271032819937369e+02, 1.397702495600030e+02,  1.536995926118425e+02,
        1.690171180448872e+02, 1.858611705259540e+02,  2.043838819929681e+02,
        2.247525456786157e+02, 2.471511270676237e+02,  2.717819254343092e+02,
        2.988674009670603e+02, 3.286521839819514e+02,  3.614052843722859e+02,
        3.974225212493519e+02, 4.370291947183915e+02,  4.805830239207090e+02,
        5.284773778776877e+02, 5.811448283170228e+02,  6.390610565695528e+02,
        7.027491498229519e+02, 7.727843255351499e+02,  8.497991266773804e+02,
        9.344891347292100e+02, 1.027619252024106e+03,  1.130030610186371e+03,
        1.242648167054996e+03, 1.366489060708247e+03,  1.502671796040775e+03,
        1.652426346864483e+03, 1.817105264773206e+03,  1.998195895104117e+03,
        2.197333810327873e+03, 2.416317582195026e+03,  2.657125026057655e+03,
        2.921931064081479e+03, 3.213127368685231e+03,  3.533343963622751e+03,
        3.885472977804124e+03, 4.272694766395488e+03,  4.698506635117757e+03,
        5.166754427175994e+03, 5.681667258106072e+03,  6.247895712256392e+03,
        6.870553845887155e+03, 7.555265376250504e+03,  8.308214473821519e+03,
        9.136201616424696e+03, 1.004670500971882e+04,  1.104794812877709e+04,
        1.214897399078357e+04, 1.335972682966187e+04,  1.469114191030346e+04,
        1.615524429357940e+04, 1.776525744416004e+04,  1.953572266206552e+04,
        2.148263041662801e+04, 2.362356476904637e+04,  2.597786218792689e+04,
        2.856678619219933e+04, 3.141371939874610e+04,  3.454437470927775e+04,
        3.798702754385149e+04, 4.177277121851550e+04,  4.593579777359733e+04,
        5.051370678901826e+04, 5.554784497578751e+04,  6.108367961079667e+04,
        6.717120918769839e+04, 7.386541499278042e+04,  8.122675768437530e+04,
        8.932172336080558e+04, 9.822342404883401e+04,  1.080122580361064e+05,
        1.187766360115701e+05, 1.306137795722129e+05,  1.436305993080729e+05,
        1.579446603961915e+05, 1.736852444245569e+05,  1.909945170362057e+05,
        2.100288119393971e+05, 2.309600428807727e+05,  2.539772563341463e+05,
        2.792883389284763e+05, 3.071218950361589e+05,  3.377293114796219e+05,
        3.713870280042076e+05, 4.083990340237931e+05,  4.490996141892110e+05,
        4.938563675768817e+05, 5.430735277663299e+05,  5.971956137928166e+05,
        6.567114449497383e+05, 7.221585557016386e+05,  7.941280505823843e+05,
        8.732699429269477e+05, 9.602990256550852e+05,  1.056001327130584e+06,
        1.161241210404062e+06, 1.276969179958181e+06,  1.404230466464131e+06,
        1.544174467085141e+06, 1.698065126589807e+06,  1.867292353035306e+06,
        2.053384571124533e+06, 2.258022526616409e+06,  2.483054466467948e+06,
        2.730512831811902e+06, 3.002632614537411e+06,  3.301871543266503e+06,
        3.630932281041903e+06, 3.992786835210945e+06,  4.390703399970646e+06,
        4.828275874010011e+06, 5.309456319846827e+06,  5.838590658024817e+06,
        6.420457918553535e+06, 7.060313404101893e+06,  7.763936154786171e+06,
        8.537681143244324e+06, 9.388536671411762e+06,  1.032418748739408e+07,
        1.135308419249504e+07, 1.248451956526934e+07,  1.372871249194350e+07,
        1.509690026124605e+07, 1.660144005723478e+07,  1.825592056678340e+07,
        2.007528470974215e+07, 2.207596460024559e+07,  2.427602995811041e+07,
        2.669535131074274e+07, 2.935577945956044e+07,  3.228134283183457e+07,
        3.549846450038852e+07, 3.903620083123180e+07,  4.292650391454494e+07,
        4.720451014923478e+07, 5.190885758749597e+07,  5.708203490556967e+07,
        6.277076515252714e+07, 6.902642774301739e+07,  7.590552250533110e+07,
        8.347017997596124e+07, 9.178872254953519e+07,  1.009362815523093e+08,
        1.109954758124983e+08, 1.220571578561492e+08,  1.342212344680389e+08,
        1.475975690287357e+08, 1.623069737775358e+08,  1.784823009631873e+08,
        1.962696427374508e+08, 2.158296506286749e+08,  2.373389865125952e+08,
        2.609919181852297e+08, 2.870020739487360e+08,  3.156043720572838e+08,
        3.470571424493049e+08, 3.816444599291403e+08,  4.196787098708979e+08,
        4.615034096174309e+08, 5.074963110566983e+08,  5.580728123973287e+08,
        6.136897099578503e+08, 6.748493238549215e+08,  7.421040348529267e+08,
        8.160612733508301e+08, 8.973890055657258e+08,  9.868217664631569e+08,
        1.085167293922392e+09, 1.193313824054988e+09,  1.312238113566463e+09,
        1.443014261617332e+09, 1.586823410860753e+09,  1.744964415274437e+09,
        1.918865571136475e+09, 2.110097517096829e+09,  2.320387419855137e+09,
        2.551634573566909e+09, 2.805927553868766e+09,  3.085563081454121e+09,
        3.393066765571205e+09, 3.731215914794456e+09,  4.103064621091177e+09,
        4.511971343737535e+09, 4.961629242216197e+09,  5.456099532055670e+09,
        5.999848164874397e+09, 6.597786163916223e+09,  7.255313979378973e+09,
        7.978370264144282e+09, 8.773485510441036e+09,  9.647841031877066e+09,
        1.060933382355256e+10, 1.166664788605844e+10,  1.282933265754237e+10,
        1.410788926222501e+10, 1.551386535434417e+10,  1.705995941413861e+10,
        1.876013543784949e+10, 2.062974905759618e+10,  2.268568623021499e+10,
        2.494651574766827e+10, 2.743265694646616e+10,  3.016656413081807e+10,
        3.317292937518365e+10, 3.647890553789348e+10,  4.011435150005315e+10,
        4.411210184467462e+10, 4.850826341172090e+10,  5.334254140748837e+10,
        5.865859801367554e+10, 6.450444673502067e+10,  7.093288604717545e+10,
        7.800197626142746e+10, 8.577556391321468e+10,  9.432385841060141e+10,
        1.037240661508788e+11, 1.140610878324995e+11,  1.254282852603241e+11,
        1.379283245697981e+11, 1.516741034858860e+11,  1.667897709915673e+11,
        1.834118486153316e+11, 2.016904634648982e+11,  2.217907041436628e+11,
        2.438941117962321e+11, 2.682003197498430e+11,  2.949288565605668e+11,
        3.243211287490431e+11, 3.566426011333790e+11,  3.921851944515266e+11,
        4.312699219279733e+11, 4.742497885976400e+11,  5.215129795730765e+11,
        5.734863660507314e+11, 6.306393607217781e+11,  6.934881574087671e+11,
        7.626003932196346e+11, 8.386002753266213e+11,  9.221742186741061e+11,
        1.014077045533975e+12, 1.115138802901554e+12,  1.226272259305415e+12,
        1.348481148740691e+12, 1.482869236183318e+12,  1.630650286563071e+12,
        1.793159027232996e+12, 1.971863202946070e+12,  2.168376832216971e+12,
        2.384474784797678e+12])

    dlf.factor = np.array([1.0996588551261031])

    dlf.cos = np.array([
        9.297394385722251e-09, -9.331225377780101e-08, 4.720122900065709e-07,
        -1.613118471395310e-06,  4.212303970960659e-06, -9.009536457681384e-06,
        1.651602857948675e-05, -2.679596886705627e-05, 3.940422339703118e-05,
        -5.347974004860014e-05,  6.791971721071075e-05, -8.155130316929276e-05,
        9.326403165364431e-05, -1.021242360378208e-04, 1.075044839800603e-04,
        -1.092079615834928e-04,  1.075143562399657e-04, -1.030950944755197e-04,
        9.682549682891237e-05, -8.957830794965313e-05, 8.207447336214694e-05,
        -7.481894657743579e-05,  6.810717196176212e-05, -6.207122775612170e-05,
        5.673730708926640e-05, -5.207534712599374e-05, 4.803123832785169e-05,
        -4.454126647913836e-05,  4.153626254546913e-05, -3.894484241413979e-05,
        3.670018212582626e-05, -3.474760669238406e-05, 3.304789032391324e-05,
        -3.157405118447008e-05,  3.030417050465031e-05, -2.921456228037889e-05,
        2.827644329587734e-05, -2.745628846010215e-05, 2.671857445706401e-05,
        -2.602920102778078e-05,  2.535888695808226e-05, -2.468607273662796e-05,
        2.399924035665649e-05, -2.329770267206565e-05, 2.259026462846769e-05,
        -2.189149146666980e-05,  2.121716386042521e-05, -2.058041892479830e-05,
        1.999003726593088e-05, -1.945035041059098e-05, 1.896225718089684e-05,
        -1.852396425633805e-05,  1.813163216467783e-05, -1.777965071578391e-05,
        1.746165236340568e-05, -1.717168150623305e-05, 1.690577970679105e-05,
        -1.666244997183115e-05,  1.644257988471772e-05, -1.624809220129749e-05,
        1.608102355458193e-05, -1.594231869044314e-05, 1.583185544264933e-05,
        -1.574816191054980e-05,  1.568924212529655e-05, -1.565249673434141e-05,
        1.563547180372210e-05, -1.563521991811041e-05, 1.564873612274880e-05,
        -1.567205863642941e-05,  1.570100094942606e-05, -1.573043759313864e-05,
        1.575568901623209e-05, -1.577234363472105e-05, 1.577855848555620e-05,
        -1.577469454094538e-05,  1.576464133524747e-05, -1.575311215656368e-05,
        1.574557219403314e-05, -1.574501141478538e-05, 1.575350655562779e-05,
        -1.577039802560470e-05,  1.579548543552833e-05, -1.582697025278861e-05,
        1.586471505025963e-05, -1.590700358321762e-05, 1.595395944834813e-05,
        -1.600347227749585e-05,  1.605550830902334e-05, -1.610747055867073e-05,
        1.615951511314031e-05, -1.620878072415450e-05, 1.625562358329816e-05,
        -1.629669100596089e-05,  1.633276432502889e-05, -1.636065015813141e-05,
        1.638259866975109e-05, -1.639607578668321e-05, 1.640462724001378e-05,
        -1.640521769213474e-05,  1.640152929507142e-05, -1.638920820170809e-05,
        1.637250343735157e-05, -1.634655518077729e-05, 1.631719045165813e-05,
        -1.627909407619066e-05,  1.623957286491315e-05, -1.619224661741716e-05,
        1.614597203081240e-05, -1.609315597940615e-05, 1.604474827158578e-05,
        -1.599187711961352e-05,  1.594805230156663e-05, -1.590248899171649e-05,
        1.587071183503557e-05, -1.583788239180818e-05, 1.582031406961812e-05,
        -1.579770001003802e-05,  1.578819150667827e-05, -1.576669107885469e-05,
        1.575546447182241e-05, -1.572497066339458e-05, 1.570306689826559e-05,
        -1.565481680735495e-05,  1.561443687682504e-05, -1.554002127587872e-05,
        1.547327409568208e-05, -1.536401629466096e-05, 1.526346556783772e-05,
        -1.511208458166825e-05,  1.497331870364205e-05, -1.477700933020661e-05,
        1.460192305213648e-05, -1.436526806621614e-05, 1.416405436732868e-05,
        -1.389931142786357e-05,  1.368858420619113e-05, -1.341135768980989e-05,
        1.320792624324585e-05, -1.293001292298771e-05, 1.274452507005725e-05,
        -1.246972405316815e-05,  1.230566665969634e-05, -1.203159173372814e-05,
        1.188911230508952e-05, -1.161136505738162e-05, 1.149057302365972e-05,
        -1.120363603483696e-05,  1.110297393482285e-05, -1.079673875747435e-05,
        1.071049225922322e-05, -1.037103169734520e-05, 1.029506610873251e-05,
        -9.913414172350665e-06,  9.853838815402886e-06, -9.429895155274106e-06,
        9.401019443495364e-06, -8.936116237280648e-06, 8.951736859302795e-06,
        -8.440916793077236e-06,  8.510455779325827e-06, -7.942012711706528e-06,
        8.074254802299430e-06, -7.434696467270500e-06, 7.642816648842507e-06,
        -6.918274083303935e-06,  7.219396629313773e-06, -6.393381389861308e-06,
        6.808760999173057e-06, -5.862982745189440e-06, 6.420912162430656e-06,
        -5.335713799060760e-06,  6.071284246368857e-06, -4.821133024426985e-06,
        5.772463577807342e-06, -4.321022275907003e-06, 5.529510962731191e-06,
        -3.829927285455952e-06,  5.343778894081914e-06, -3.336693302950008e-06,
        5.209285671709436e-06, -2.816335852100177e-06, 5.106229736488585e-06,
        -2.229987964713504e-06,  5.010932531520813e-06, -1.543767484139696e-06,
        4.922735592097522e-06, -7.574543683946134e-07, 4.885997308252075e-06,
        9.309770668397733e-08,  4.968655041908437e-06, 9.775704093164090e-07,
        5.216977163380675e-06,  1.906257524947388e-06, 5.642346953332102e-06,
        2.923434481029140e-06,  6.244458469628350e-06, 4.079413856303055e-06,
        7.038520026498571e-06,  5.414680217443503e-06, 8.060494944331253e-06,
        6.968312913592730e-06,  9.355185156130350e-06, 8.792176223384357e-06,
        1.097045865548069e-05,  1.095322333207534e-05, 1.296417738420366e-05,
        1.352799587425616e-05,  1.541381200366568e-05, 1.660072384193364e-05,
        1.841997147882438e-05,  2.026912073003105e-05, 2.210551672304511e-05,
        2.465220942221532e-05,  2.661736864799657e-05, 2.989536173822840e-05,
        3.213232110580544e-05,  3.617482594482103e-05, 3.886373268637686e-05,
        4.370446462896611e-05,  4.706844302890467e-05, 5.274434571999963e-05,
        5.705562366374805e-05,  6.360999354022556e-05, 6.919856766912473e-05,
        7.668328514728211e-05,  8.394822291556238e-05, 9.242681616786029e-05,
        1.018487845955144e-04,  1.114011335437031e-04, 1.235573455065937e-04,
        1.342848581478068e-04,  1.498676619262224e-04, 1.618997849276840e-04,
        1.817379012620396e-04,  1.952416992037834e-04, 2.203245877889620e-04,
        2.355167181844915e-04,  2.670244391354883e-04, 2.841848882706381e-04,
        3.235251219161621e-04,  3.430123532795142e-04, 3.918676897535589e-04,
        4.141333163467392e-04,  4.745222262994887e-04, 5.001264810538308e-04,
        5.744760452366283e-04,  6.041111932887833e-04, 6.953377041337534e-04,
        7.298635797228439e-04,  8.414652456887652e-04, 8.819528243262603e-04,
        1.018124731652973e-03,  1.065902978573756e-03, 1.231683155383351e-03,
        1.288386810459016e-03,  1.489841519407945e-03, 1.557459803490981e-03,
        1.801912555421511e-03,  1.882845449381654e-03, 2.179149281364812e-03,
        2.276280583427097e-03,  2.635135332482890e-03, 2.751927858029521e-03,
        3.186249126577850e-03,  3.326862059512359e-03, 3.852213758365377e-03,
        4.021635411191684e-03,  4.656739764341880e-03, 4.860925667233624e-03,
        5.628257587483373e-03,  5.874261077316650e-03, 6.800723111882190e-03,
        7.096792112063695e-03,  8.214452072188272e-03, 8.570038585776058e-03,
        9.916881689099844e-03,  1.034246793647235e-02, 1.196305329516215e-02,
        1.246962114896129e-02,  1.441543192101280e-02, 1.501324787950850e-02,
        1.734235517342590e-02,  1.803848217679936e-02, 2.081381761344761e-02,
        2.160734309026994e-02,  2.489229977447262e-02, 2.576553958006069e-02,
        2.961467509491694e-02,  3.051737110266814e-02, 3.495844122438238e-02,
        3.577998856776111e-02,  4.078109826730666e-02, 4.130285745015720e-02,
        4.671501590165387e-02,  4.653076183321674e-02, 5.199196486468964e-02,
        5.038064846254973e-02,  5.516515485988877e-02, 5.090096052934653e-02,
        5.370474646971766e-02,  4.480843697750900e-02, 4.350085483263331e-02,
        2.700779636681336e-02,  1.850147060480727e-02, -9.481059185247232e-03,
        -2.878647686708867e-02, -7.201691801599090e-02, -1.041550388144556e-01,
        -1.627192708568376e-01, -2.025832442736199e-01, -2.648757436759084e-01,
        -2.896768123096005e-01, -3.204249235290653e-01, -2.793713989892394e-01,
        -2.177215915565538e-01, -4.997516720753165e-02, 1.364346619405615e-01,
        3.957582182746007e-01,  5.514593774068006e-01, 6.000355888483448e-01,
        3.186791318366248e-01, -1.567831386495452e-01, -7.479109158787856e-01,
        -8.488046630983653e-01, -2.726822402365101e-01, 8.543192655839413e-01,
        1.089835857141487e+00, -1.818874039618337e-01, -1.529241369785723e+00,
        5.251114541647035e-02,  1.745104311752220e+00, -8.932643503069300e-01,
        -1.200119921665596e+00,  2.345925576744023e+00, -2.204932102497117e+00,
        1.508712175144785e+00, -8.545543410651292e-01, 4.278678767318642e-01,
        -1.951651104274962e-01,  7.977247475727975e-02, -2.520612838163129e-02,
        2.784167518927987e-05,  1.128037363187448e-02, -1.602490295449761e-02,
        1.762362316839732e-02, -1.770082893251570e-02, 1.704836442363811e-02,
        -1.606262768958838e-02,  1.494636952733606e-02, -1.380448976967370e-02,
        1.269100072803114e-02, -1.163280779532750e-02, 1.064212005520605e-02,
        -9.723104079214704e-03,  8.875535347924807e-03, -8.096840465660830e-03,
        7.383254655423208e-03, -6.730481942741819e-03, 6.134071407476349e-03,
        -5.589629959028378e-03,  5.092940278405305e-03, -4.640022871665849e-03,
        4.227164339810403e-03, -3.850924696369492e-03, 3.508131500450960e-03,
        -3.195865960368183e-03,  2.911445078555397e-03, -2.652403279346110e-03,
        2.416475780004171e-03, -2.201584330908779e-03, 2.005824944280696e-03,
        -1.827457242918919e-03,  1.664895134294627e-03, -1.516697949517489e-03,
        1.381560747765788e-03, -1.258303255733349e-03, 1.145858434825516e-03,
        -1.043262154695598e-03,  9.496442040538239e-04, -8.642195933510369e-04,
        7.862796001679923e-04, -7.151834263282333e-04, 6.503513738625369e-04,
        -5.912587839952813e-04,  5.374288656004424e-04, -4.884237630780368e-04,
        4.438356860452133e-04, -4.032810043390997e-04, 3.663989461792944e-04,
        -3.328543851032324e-04,  3.023426261704347e-04, -2.745931892233651e-04,
        2.493695597793403e-04, -2.264636820007668e-04, 2.056871084893267e-04,
        -1.868627139954570e-04,  1.698200748261868e-04, -1.543950057071412e-04,
        1.404316008452683e-04, -1.277847843600985e-04, 1.163222988707842e-04,
        -1.059257934501936e-04,  9.649068017221013e-05, -8.792441846654285e-05,
        8.014346129696588e-05, -7.307009034780369e-05, 6.663106717837242e-05,
        -6.075938522170574e-05,  5.539822193643006e-05, -5.050427628106261e-05,
        4.604804125018462e-05, -4.201061458602852e-05, 3.837817486518200e-05,
        -3.513559082809176e-05,  3.226079373650442e-05, -2.972163988296620e-05,
        2.747604349368696e-05, -2.547449697103353e-05, 2.366358407874955e-05,
        -2.199022414951037e-05,  2.040722849984953e-05, -1.887986526429048e-05,
        1.739150593500115e-05, -1.594570919821283e-05, 1.456293955353814e-05,
        -1.327237711981726e-05,  1.210191094758502e-05, -1.107055329123273e-05,
        1.018599494527819e-05, -9.446732728569548e-06, 8.845647076873882e-06,
        -8.372216686790610e-06,  8.013159282610823e-06, -7.753133645178163e-06,
        7.576432722043398e-06, -7.468830093872313e-06, 7.418107311469552e-06,
        -7.412767416737676e-06,  7.440011386509294e-06, -7.484635044283664e-06,
        7.529425497140131e-06, -7.556246853467662e-06, 7.546978888488720e-06,
        -7.484443403466015e-06,  7.353829840238931e-06, -7.144458110830936e-06,
        6.851061543130811e-06, -6.474223369227419e-06, 6.020640004049169e-06,
        -5.503796462783514e-06,  4.944233105379788e-06, -4.367961630338038e-06,
        3.802847933285844e-06, -3.274212863452358e-06, 2.801082379781716e-06,
        -2.394304111474628e-06,  2.057660622354264e-06, -1.792014674462885e-06,
        1.600309790767040e-06, -1.490004264896332e-06, 1.470861438204534e-06,
        -1.549024862704097e-06,  1.720674705865420e-06, -1.968714833812948e-06,
        2.263913719747939e-06, -2.569436519621943e-06, 2.846821174913007e-06,
        -3.062142363423440e-06,  3.191506657749859e-06, -3.224581334134399e-06,
        3.165000349530554e-06, -3.027744051806499e-06, 2.834694189383818e-06,
        -2.609749350494529e-06,  2.374653030786580e-06, -2.146279135865408e-06,
        1.935330475083363e-06, -1.745828142696292e-06, 1.575236936968939e-06,
        -1.416013922329550e-06,  1.259318409502511e-06, -1.100113244233588e-06,
        9.410699739097717e-07, -7.924596036830598e-07, 6.674986354507494e-07,
        -5.759439576425730e-07,  5.201376470539767e-07, -4.958521149219669e-07,
        4.968861570696705e-07, -5.201376546764945e-07, 5.680874517337957e-07,
        -6.476758417845987e-07,  7.666958406725774e-07, -9.294975387600607e-07,
        1.133228200791028e-06, -1.365563006504285e-06, 1.605190106576952e-06,
        -1.825695639038146e-06,  2.001436016286443e-06, -2.112566732698138e-06,
        2.147590628351366e-06, -2.104276900366494e-06, 1.990524948310489e-06,
        -1.825121292689841e-06,  1.636841282531044e-06, -1.460884274849098e-06,
        1.333233799913604e-06, -1.284271861776232e-06, 1.332582358180179e-06,
        -1.479588446575088e-06,  1.706376667684837e-06, -1.974966486688615e-06,
        2.235381164425016e-06, -2.436739518420842e-06, 2.537929500304453e-06,
        -2.514003617946490e-06,  2.357475540929001e-06, -2.076220907296449e-06,
        1.690183203579621e-06, -1.228236424305994e-06, 7.254768690815435e-07,
        -2.204472290016658e-07, -2.484718379400915e-07, 6.478081445435230e-07,
        -9.540075584969148e-07,  1.156863204197333e-06, -1.259958368650068e-06,
        1.278195568002736e-06, -1.233378700121944e-06, 1.148676787526175e-06,
        -1.042802119635343e-06,  9.254711911091067e-07, -7.960649679105933e-07,
        6.462125216032012e-07, -4.649724778713449e-07, 2.441092104183003e-07,
        1.868493985474541e-08, -3.194628733775932e-07, 6.498918786796644e-07,
        -9.996705476756540e-07,  1.358252357379558e-06, -1.715918878739438e-06,
        2.064923358321897e-06, -2.400858984963648e-06, 2.723395365281677e-06,
        -3.035196331032546e-06,  3.338643928525930e-06, -3.631533371486772e-06,
        3.904130993548243e-06, -4.139831345008151e-06, 4.319796485239589e-06,
        -4.429412335736343e-06,  4.463099495294983e-06, -4.425185636429804e-06,
        4.327185563233206e-06, -4.183682112058815e-06, 4.009026274927233e-06,
        -3.816065958497831e-06,  3.617050761056201e-06, -3.425971229697643e-06,
        3.260772021627054e-06, -3.143311389863823e-06, 3.095486258178091e-06,
        -3.132275257276740e-06,  3.255188314712704e-06, -3.449944207123177e-06,
        3.689380419331001e-06, -3.939307652702181e-06, 4.164229695692462e-06,
        -4.331479577412883e-06,  4.414110342933081e-06, -4.393467845736048e-06,
        4.261939450428174e-06, -4.025227833561406e-06, 3.702213535114111e-06,
        -3.320616198107725e-06,  2.909132334933439e-06, -2.489740793035523e-06,
        2.074491786680098e-06, -1.668385069961250e-06, 1.275834595099470e-06,
        -9.059219147765586e-07,  5.729879334963921e-07, -2.927551120734085e-07,
        7.694258119528258e-08,  7.047419904575634e-08, -1.538374630535126e-07,
        1.836543837730242e-07, -1.734417502848936e-07, 1.373082222203108e-07,
        -8.884437236765175e-08,  4.052785480829440e-08, -2.506316590284845e-09,
        -1.938722282380230e-08,  2.515252321893277e-08, -2.002434723520940e-08,
        1.132162520918844e-08, -4.487826026291201e-09, 1.131156449938913e-09,
        -1.373736852774400e-10])

    dlf.sin = np.array([
        8.359261159716757e-12, -9.220758632982218e-11, 5.077803827675346e-10,
        -1.867552463969512e-09,  5.172013211121000e-09, -1.150520169188055e-08,
        2.133505606536172e-08, -3.356235702475845e-08, 4.458152579281038e-08,
        -4.764384505735926e-08,  3.285802030700915e-08, 1.170416876881884e-08,
        -9.728694589481136e-08,  2.302363895145795e-07, -4.079555605183399e-07,
        6.177404179101450e-07, -8.399063799040625e-07, 1.053635465227130e-06,
        -1.242201580787086e-06,  1.395188292393633e-06, -1.507739153919925e-06,
        1.578670342496271e-06, -1.609330769274755e-06, 1.603704787813562e-06,
        -1.568787772955808e-06,  1.514057996940974e-06, -1.449859436884266e-06,
        1.385471476675003e-06, -1.327714396862736e-06, 1.280365531766066e-06,
        -1.244157111494832e-06,  1.217066071485304e-06, -1.194813140091078e-06,
        1.171619350554489e-06, -1.141245957443412e-06, 1.098300017169870e-06,
        -1.039677652165878e-06,  9.657023972020309e-07, -8.802859741190706e-07,
        7.898002965067069e-07, -7.011502922371117e-07, 6.199819174236849e-07,
        -5.496364964808038e-07,  4.908282495265287e-07, -4.417340889517205e-07,
        3.983234968951235e-07, -3.549443284917371e-07, 3.052022769579055e-07,
        -2.431179353674635e-07,  1.644925081929142e-07, -6.826180446676947e-08,
        -4.267370024343952e-08,  1.617586507674555e-07, -2.803722965853880e-07,
        3.900333484620099e-07, -4.841854682231028e-07, 5.589285654245779e-07,
        -6.127907934111358e-07,  6.460770598217428e-07, -6.602760649580756e-07,
        6.576736756258753e-07, -6.410565760810279e-07, 6.134062518391666e-07,
        -5.776817085420256e-07,  5.368478628585665e-07, -4.940575236587126e-07,
        4.526353300864370e-07, -4.156542601539313e-07, 3.853255033044642e-07,
        -3.626302817958946e-07,  3.474120702880588e-07, -3.388116026167377e-07,
        3.357822909548376e-07, -3.374961146308590e-07, 3.435828525769017e-07,
        -3.542138677132927e-07,  3.700334924559362e-07, -3.919101945347247e-07,
        4.204777209363843e-07, -4.555173491852289e-07, 4.954154485992529e-07,
        -5.370620929924017e-07,  5.764057986537222e-07, -6.094701875560272e-07,
        6.332893162659519e-07, -6.462326093266543e-07, 6.475753791361509e-07,
        -6.366371781051364e-07,  6.120491507565731e-07, -5.716595544868212e-07,
        5.133516840288774e-07, -4.366225654466551e-07, 3.441541239853112e-07,
        -2.422778905407855e-07,  1.397790651787136e-07, -4.560549166906710e-08,
        -3.328991221128148e-08,  9.304779181335509e-08, -1.328140937569905e-07,
        1.539166770440108e-07, -1.588634375317458e-07, 1.505235965755548e-07,
        -1.316171002396016e-07,  1.044904524158723e-07, -7.109829024369389e-08,
        3.308802429746560e-08,  8.116007277443925e-09, -5.129436354707441e-08,
        9.547883951893789e-08, -1.400131020366568e-07, 1.845767969781941e-07,
        -2.291551421651494e-07,  2.739695597209863e-07, -3.193993375150602e-07,
        3.659197536698852e-07, -4.140702925835591e-07, 4.644569907056792e-07,
        -5.177839547050174e-07,  5.748940271338283e-07, -6.367794146115784e-07,
        7.045100842150862e-07, -7.790327824396196e-07, 8.608484724152268e-07,
        -9.496971114086013e-07,  1.044466202061779e-06, -1.143447813205447e-06,
        1.244838256819165e-06, -1.347224367354159e-06, 1.449857674959308e-06,
        -1.552692050090939e-06,  1.656284432144049e-06, -1.761657935230017e-06,
        1.870158450548530e-06, -1.983294246648629e-06, 2.102564089914738e-06,
        -2.229310757737318e-06,  2.364640578735712e-06, -2.509421909265687e-06,
        2.664342794306071e-06, -2.829989595555780e-06, 3.006913251480532e-06,
        -3.195669426252227e-06,  3.396840866305719e-06, -3.611055321786113e-06,
        3.839005622943003e-06, -4.081466199752818e-06, 4.339299875122917e-06,
        -4.613450428457505e-06,  4.904927295032445e-06, -5.214791189335373e-06,
        5.544154913168186e-06, -5.894199596087268e-06, 6.266202436970297e-06,
        -6.661558018637908e-06,  7.081789842986133e-06, -7.528541872478548e-06,
        8.003556903467527e-06, -8.508633786866912e-06, 9.045592479497848e-06,
        -9.616261203948259e-06,  1.022252155096364e-05, -1.086637486515759e-05,
        1.155001875448423e-05, -1.227587080100554e-05, 1.304657346345800e-05,
        -1.386495308148081e-05,  1.473401813880603e-05, -1.565694756839281e-05,
        1.663715077238186e-05, -1.767828071402171e-05, 1.878429914657035e-05,
        -1.995943733557077e-05,  2.120823415569920e-05, -2.253544922314025e-05,
        2.394611992736814e-05, -2.544545960035184e-05, 2.703898640450181e-05,
        -2.873242934495659e-05,  3.053195355337805e-05, -3.244401714882796e-05,
        3.447565672741261e-05, -3.663422383697410e-05, 3.892776251925333e-05,
        -4.136460370865096e-05,  4.395390868232029e-05, -4.670507008308927e-05,
        4.962849974172098e-05, -5.273474506074496e-05, 5.603563317932190e-05,
        -5.954297078623826e-05,  6.327020804601162e-05, -6.723052637483463e-05,
        7.143925882610647e-05, -7.591107995451901e-05, 8.066353015734433e-05,
        -8.571288970904039e-05,  9.107931287869562e-05, -9.678077219329197e-05,
        1.028405413088340e-04, -1.092783140000057e-04, 1.161211203441987e-04,
        -1.233903142846576e-04,  1.311175090992165e-04, -1.393255189054305e-04,
        1.480516242987183e-04, -1.573196597933747e-04, 1.671739880388771e-04,
        -1.776386459079633e-04,  1.887670083642166e-04, -2.005820496174674e-04,
        2.131489579144857e-04, -2.264877366442824e-04, 2.406793062382246e-04,
        -2.557376041179267e-04,  2.717646455644554e-04, -2.887634254819832e-04,
        3.068646043678929e-04, -3.260528651282183e-04, 3.464983356283099e-04,
        -3.681563775328206e-04,  3.912521416052410e-04, -4.156952553697350e-04,
        4.417882828039744e-04, -4.693707283323586e-04, 4.988549013178462e-04,
        -5.299740981465572e-04,  5.632972168037640e-04, -5.983981201542243e-04,
        6.360703008313139e-04, -6.756498273302946e-04, 7.182536629624953e-04,
        -7.628648911440259e-04,  8.110678884696581e-04, -8.613236288904599e-04,
        9.158936600338050e-04, -9.724687649776548e-04, 1.034293567545343e-03,
        -1.097925026760626e-03,  1.168037243661937e-03, -1.239520643257603e-03,
        1.319130531388254e-03, -1.399310723206821e-03, 1.489849562571212e-03,
        -1.579602309347674e-03,  1.682780854751798e-03, -1.782980640112731e-03,
        1.900868915922405e-03, -2.012335870346910e-03, 2.147473503496334e-03,
        -2.270889066841259e-03,  2.426439457570971e-03, -2.562215420051550e-03,
        2.742183034952833e-03, -2.890261392840365e-03, 3.099800300288151e-03,
        -3.259350857226141e-03,  3.505205495093838e-03, -3.674172710258315e-03,
        3.965310626574665e-03, -4.139738650325030e-03, 4.488262359241358e-03,
        -4.661294242426114e-03,  5.083759325365117e-03, -5.244158243862472e-03,
        5.763483157849691e-03, -5.893453245986144e-03, 6.541691318046123e-03,
        -6.613673304094213e-03,  7.436041154998661e-03, -7.408009065250789e-03,
        8.468745595771147e-03, -8.277314667970900e-03, 9.668205167903463e-03,
        -9.218548318547704e-03,  1.107132357506540e-02, -1.022244393130105e-02,
        1.272680215529608e-02, -1.127006800802479e-02, 1.469983053996089e-02,
        -1.232777617668802e-02,  1.707875299894973e-02, -1.333990201591076e-02,
        1.998449243486315e-02, -1.421829371168560e-02, 2.358373819470286e-02,
        -1.482760116032370e-02,  2.810708047451080e-02, -1.496512380422282e-02,
        3.387321442302536e-02, -1.433435153470303e-02, 4.131959123492098e-02,
        -1.251274602579928e-02,  5.103745787350833e-02, -8.918289519700926e-03,
        6.380297769558445e-02, -2.788897684625384e-03, 8.058181530268101e-02,
        6.790206156882212e-03,  1.024543404644486e-01, 2.073338204806540e-02,
        1.303497117692710e-01,  3.951881975960873e-02, 1.643745620689843e-01,
        6.218686679938456e-02,  2.023784581667486e-01, 8.434899410606132e-02,
        2.373088575358529e-01,  9.483819839368990e-02, 2.532746060255461e-01,
        7.159002104274344e-02,  2.222192506778304e-01, -1.905573270964483e-02,
        1.090141501356163e-01, -2.067328188554215e-01, -9.612246602135047e-02,
        -4.615266320685089e-01, -2.963866235183736e-01, -5.963313340250359e-01,
        -2.082472282663639e-01, -2.777908167617539e-01, 4.207178203208411e-01,
        4.291396414267995e-01,  9.433230333912225e-01, 2.207290574116785e-01,
        -9.583338573591282e-02, -1.232100887731011e+00, -5.002745008433491e-01,
        2.372029128753443e-01,  1.649761387946455e+00, -3.603783487843671e-01,
        -1.261037499322814e+00, -1.442344716329911e-01, 2.262432282201865e+00,
        -2.440529947584088e+00,  1.102479711406938e+00, 2.479466131565981e-01,
        -9.331576113883502e-01,  1.067534473089257e+00, -9.516666382066644e-01,
        7.803309151688617e-01, -6.296102325905817e-01, 5.147217959418201e-01,
        -4.305875729144710e-01,  3.687952586818823e-01, -3.224307395760154e-01,
        2.866508648266869e-01, -2.582226362549916e-01, 2.350036653885580e-01,
        -2.155613376127551e-01,  1.989230622019242e-01, -1.844173872024837e-01,
        1.715729150417765e-01, -1.600532114574932e-01, 1.496142179147893e-01,
        -1.400759347576565e-01,  1.313033137405831e-01, -1.231931979853633e-01,
        1.156652963349872e-01, -1.086558863590842e-01, 1.021133845398998e-01,
        -9.599520679521963e-02,  9.026552811021052e-02, -8.489367300426851e-02,
        7.985295115753441e-02, -7.511980873410520e-02, 7.067320463855671e-02,
        -6.649414780698869e-02,  6.256535036073049e-02, -5.887096455317380e-02,
        5.539638066390042e-02, -5.212806950810558e-02, 4.905345781230656e-02,
        -4.616082794260108e-02,  4.343923579804874e-02, -4.087844237105168e-02,
        3.846885569150211e-02, -3.620148075223817e-02, 3.406787567046667e-02,
        -3.206011282083423e-02,  3.017074400183465e-02, -2.839276891085451e-02,
        2.671960636084829e-02, -2.514506780695629e-02, 2.366333286889083e-02,
        -2.226892662563566e-02,  2.095669851005093e-02, -1.972180264944338e-02,
        1.855967952000187e-02, -1.746603882090155e-02, 1.643684349199548e-02,
        -1.546829477807993e-02,  1.455681821726215e-02, -1.369905043585772e-02,
        1.289182666256661e-02, -1.213216891026715e-02, 1.141727480509518e-02,
        -1.074450706385705e-02,  1.011138362578673e-02, -9.515568434992541e-03,
        8.954862851811404e-03, -8.427197646151262e-03, 7.930625498870823e-03,
        -7.463313926166862e-03,  7.023538562190149e-03, -6.609676775790488e-03,
        6.220201627706233e-03, -5.853676177845857e-03, 5.508748140447998e-03,
        -5.184144875714356e-03,  4.878668701343701e-03, -4.591192499582146e-03,
        4.320655587950222e-03, -4.066059823272036e-03, 3.826465918940025e-03,
        -3.600989965823915e-03,  3.388800151556466e-03, -3.189113672657683e-03,
        3.001193833090605e-03, -2.824347321567115e-03, 2.657921657084559e-03,
        -2.501302789453364e-03,  2.353912841312510e-03, -2.215207980404112e-03,
        2.084676414290291e-03, -1.961836502588783e-03, 1.846234982535470e-03,
        -1.737445301697302e-03,  1.635066049040852e-03, -1.538719475623248e-03,
        1.448050099380886e-03, -1.362723391425536e-03, 1.282424540880183e-03,
        -1.206857292918672e-03,  1.135742853839018e-03, -1.068818858721894e-03,
        1.005838399422039e-03, -9.465691112214001e-04, 8.907923154941191e-04,
        -8.383022145823728e-04,  7.889051346894440e-04, -7.424188130127174e-04,
        6.986717262292931e-04, -6.575024581926733e-04, 6.187591047945998e-04,
        -5.822987136686772e-04,  5.479867564542296e-04, -5.156966318555281e-04,
        4.853091982012345e-04, -4.567123341714300e-04, 4.298005259816778e-04,
        -4.044744790551956e-04,  3.806407522738502e-04, -3.582114132276843e-04,
        3.371037133127995e-04, -3.172397817468176e-04, 2.985463374264674e-04,
        -2.809544173688782e-04,  2.643991206956540e-04, -2.488193676787240e-04,
        2.341576737349979e-04, -2.203599380154558e-04, 2.073752455409625e-04,
        -1.951556812983906e-04,  1.836561548054073e-04, -1.728342342488136e-04,
        1.626499898310491e-04, -1.530658460737416e-04, 1.440464426955128e-04,
        -1.355585036161965e-04,  1.275707136819895e-04, -1.200536027285213e-04,
        1.129794365815728e-04, -1.063221146108472e-04, 1.000570734949858e-04,
        -9.416119686244207e-05,  8.861273043083695e-05, -8.339120221067994e-05,
        7.847734730539125e-05, -7.385303688505763e-05, 6.950121105100616e-05,
        -6.540581546107905e-05,  6.155174165890432e-05, -5.792477102610575e-05,
        5.451152221654109e-05, -5.129940188145374e-05, 4.827655843562163e-05,
        -4.543183855105308e-05,  4.275474607475218e-05, -4.023540320295300e-05,
        3.786451389972717e-05, -3.563332957269838e-05, 3.353361691763475e-05,
        -3.155762774924427e-05,  2.969807062617112e-05, -2.794808410995567e-05,
        2.630121151787229e-05, -2.475137704630406e-05, 2.329286318387569e-05,
        -2.192028939321493e-05,  2.062859207259327e-05, -1.941300578777201e-05,
        1.826904570832050e-05, -1.719249113666637e-05, 1.617937001166837e-05,
        -1.522594429630232e-05,  1.432869619293261e-05, -1.348431514822722e-05,
        1.268968561560221e-05, -1.194187554794177e-05, 1.123812559543480e-05,
        -1.057583897135925e-05,  9.952571928920077e-06, -9.366024786065420e-06,
        8.814033449847659e-06, -8.294561412310765e-06, 7.805692202122783e-06,
        -7.345622279264520e-06,  6.912654355788464e-06, -6.505191118069142e-06,
        6.121729323201997e-06, -5.760854248417804e-06, 5.421234478300124e-06,
        -5.101617009901882e-06,  4.800822646167135e-06, -4.517741648025717e-06,
        4.251329628341728e-06, -4.000603683148808e-06, 3.764638756304900e-06,
        -3.542564226345866e-06,  3.333560698953710e-06, -3.136856988686476e-06,
        2.951727276827061e-06, -2.777488435826684e-06, 2.613497513360022e-06,
        -2.459149368278866e-06,  2.313874447883890e-06, -2.177136694710908e-06,
        2.048431572667469e-06, -1.927284204758473e-06, 1.813247616245777e-06,
        -1.705901079034481e-06,  1.604848553527228e-06, -1.509717223666689e-06,
        1.420156119370289e-06, -1.335834820016856e-06, 1.256442234951006e-06,
        -1.181685456128025e-06,  1.111288676892987e-06, -1.044992172405241e-06,
        9.825513393481762e-07, -9.237357933971897e-07, 8.683285212104373e-07,
        -8.161250831545758e-07,  7.669328630553843e-07, -7.205703605348671e-07,
        6.768665218548227e-07, -6.356601082847431e-07, 5.967991030550851e-07,
        -5.601401552062655e-07,  5.255480561062212e-07, -4.928952460659735e-07,
        4.620613501444217e-07, -4.329327405101518e-07, 4.054021239180187e-07,
        -3.793681579064327e-07,  3.547350993344619e-07, -3.314124829852402e-07,
        3.093148271853663e-07, -2.883613713159475e-07, 2.684758528630320e-07,
        -2.495863260342058e-07,  2.316250244413231e-07, -2.145282802422723e-07,
        1.982365164330169e-07, -1.826943245023923e-07, 1.678506431892942e-07,
        -1.536590646985763e-07,  1.400782953273800e-07, -1.270727884794868e-07,
        1.146135666342606e-07, -1.026792513877522e-07, 9.125729806412655e-08,
        -8.034537960134519e-08,  6.995280306790059e-08, -6.010176816322779e-08,
        5.082816448601466e-08, -4.218145138533012e-08, 3.422301806844069e-08,
        -2.702234929053260e-08,  2.065039505532909e-08, -1.516986572337718e-08,
        1.062287966342394e-08, -7.017537133414707e-09, 4.316406487987238e-09,
        -2.431042249536996e-09,  1.226580375862224e-09, -5.382153970216142e-10,
        1.968534210247343e-10, -5.617276256001430e-11, 1.110014404476413e-11,
        -1.138077570686847e-12])

    return dlf


def key_101_CosSin_2012():
    r"""Key 101 pt CosSin filter, as published in [Key12]_.

    Taken from file `kk101CosSin.txt` provided with SEG-2012-003_.

    License: https://software.seg.org/disclaimer.txt.

    """

    dlf = DigitalFilter('Key 101 CosSin (2012)', 'key_101_CosSin_2012')

    dlf.base = np.array([
        2.0861919529505731e-04, 2.4715373649416861e-04, 2.9280608324001184e-04,
        3.4689098210084990e-04, 4.1096602956932233e-04, 4.8687652944195492e-04,
        5.7680863591051896e-04, 6.8335231283852345e-04, 8.0957592239343885e-04,
        9.5911459112023485e-04, 1.1362748983197658e-03, 1.3461589016632311e-03,
        1.5948110718689737e-03, 1.8893923680282996e-03, 2.2383864667932735e-03,
        2.6518440846418348e-03, 3.1416724294819243e-03, 3.7219781175407782e-03,
        4.4094734312376538e-03, 5.2239576179017510e-03, 6.1888870903059369e-03,
        7.3320509502793245e-03, 8.6863712898718453e-03, 1.0290851318025195e-02,
        1.2191698617946518e-02, 1.4443655883987865e-02, 1.7111577462058016e-02,
        2.0272297096513812e-02, 2.4016840672963606e-02, 2.8453047681987229e-02,
        3.3708676899572396e-02, 3.9935085725073552e-02, 4.7311589138320786e-02,
        5.6050623809926717e-02, 6.6403866086528832e-02, 7.8669480043444651e-02,
        9.3200704342144572e-02, 1.1041602518632210e-01, 1.3081122834855582e-01,
        1.5497367735510315e-01, 1.8359922902771786e-01, 2.1751227353489916e-01,
        2.5768947608804055e-01, 3.0528790402199785e-01, 3.6167834929472320e-01,
        4.2848480606401651e-01, 5.0763124026013273e-01, 6.0139699807591596e-01,
        7.1248244908919167e-01, 8.4408675448036252e-01, 1.0000000000000000e+00,
        1.1847123470331211e+00, 1.4035433452127262e+00, 1.6627951306696866e+00,
        1.9699339218909298e+00, 2.3338050401035639e+00, 2.7648876465788206e+00,
        3.2755965330612771e+00, 3.8806396566165793e+00, 4.5974417155800325e+00,
        5.4466459652127988e+00, 6.4527087249057331e+00, 7.6446036982041701e+00,
        9.0566563894375367e+00, 1.0729532647403058e+01, 1.2711409805273369e+01,
        1.5059364144505244e+01, 1.7841014640463239e+01, 2.1136470328155472e+01,
        2.5040637370464992e+01, 2.9665952270368855e+01, 3.5145619941201240e+01,
        4.1637449888474563e+01, 4.9328400981848674e+01, 5.8439965702596865e+01,
        6.9234548928058643e+01, 8.2023024956339754e+01, 9.7173690406781532e+01,
        1.1512287083168806e+02, 1.3638748650020000e+02, 1.6157993923759994e+02,
        1.9142574904764615e+02, 2.2678444843681012e+02, 2.6867433617818517e+02,
        3.1830180340122359e+02, 3.7709607657233846e+02, 4.4675037793299668e+02,
        5.2927068877893441e+02, 6.2703351991912803e+02, 7.4285435305182887e+02,
        8.8006872410780306e+02, 1.0426282836881999e+03, 1.2352146010513620e+03,
        1.4633739891011396e+03, 1.7336772332152309e+03, 2.0539088239603038e+03,
        2.4332911434260495e+03, 2.8827500615431823e+03, 3.4152295913206981e+03,
        4.0460646647905082e+03, 4.7934227652717464e+03
        ])

    dlf.factor = np.array([1.184712347033122])

    dlf.cos = np.array([
        2.8672864360507005e-02, -1.5449802884956709e-01,
        4.2925040863045694e-01, -8.2496971475989445e-01,
        1.2478359182403629e+00, -1.5992674211324920e+00,
        1.8243131715043821e+00, -1.9145396327623898e+00,
        1.8934809645883464e+00, -1.7937055008120979e+00,
        1.6483078327364769e+00, -1.4808155052910057e+00,
        1.3101325252212712e+00, -1.1449152844956805e+00,
        9.9343552640217170e-01, -8.5553234578004234e-01,
        7.3519828220268879e-01, -6.2796652499267325e-01,
        5.3738862198175896e-01, -4.5651464813036369e-01,
        3.9045436622456964e-01, -3.2999349735142486e-01,
        2.8305474305757988e-01, -2.3751637987170185e-01,
        2.0529432395051081e-01, -1.7012553670055422e-01,
        1.4940669628955736e-01, -1.2091379458447284e-01,
        1.0958698756544492e-01, -8.4691552456171854e-02,
        8.1619516737927278e-02, -5.7580267250933195e-02,
        6.2518018553384166e-02, -3.6651105073235317e-02,
        5.0237072673112150e-02, -1.9632218666205858e-02,
        4.3459101512968681e-02, -4.6794536371076211e-03,
        4.1441426761739050e-02, 9.7939747479241900e-03,
        4.3892151379272520e-02, 2.5210891626024919e-02,
        5.0808828645867020e-02, 4.2766233781071271e-02,
        6.2117512490275244e-02, 6.2962730570568531e-02,
        7.6696518689002652e-02, 8.4104539899597833e-02,
        8.9777495214153097e-02, 9.8272696765763598e-02,
        8.6678232290502866e-02, 8.2343990171152917e-02,
        3.0869547033063837e-02, -1.4909949071839479e-02,
        -1.4329751745033240e-01, -2.5692197105974507e-01,
        -4.5564109546179643e-01, -5.2642966384443990e-01,
        -5.1460104261798389e-01, -5.5502691731699465e-02,
        5.7834482024380740e-01, 1.1256064228777944e+00,
        2.1383340001419898e-01, -1.3697558643383851e+00,
        -5.4094053515695928e-01, 2.1676624940285554e+00,
        -1.9426841601818654e+00, 1.0545870497149261e+00,
        -4.1963490722345465e-01, 1.2372791785903402e-01,
        -1.2484679132220721e-02, -2.3788674938364895e-02,
        3.3114878230396952e-02, -3.3222597126475469e-02,
        3.0358352472771119e-02, -2.6736813112701912e-02,
        2.3150637102106384e-02, -1.9871405137636362e-02,
        1.6972847239886572e-02, -1.4452497064151562e-02,
        1.2279321682638183e-02, -1.0413813415050226e-02,
        8.8158262043057916e-03, -7.4478769417757051e-03,
        6.2764357354946447e-03, -5.2719066748527534e-03,
        4.4086512878557960e-03, -3.6646129972605474e-03,
        3.0209640546643227e-03, -2.4620010538490039e-03,
        1.9748775511266975e-03, -1.5496766879546100e-03,
        1.1795298139010500e-03, -8.6058717820595076e-04,
        5.9202977915967969e-04, -3.7525986704164700e-04,
        2.1195595608903056e-04, -1.0150060510905382e-04,
        3.8158120462333111e-05, -9.8633310205821847e-06,
        1.2936403297967877e-06
        ])

    dlf.sin = np.array([
        7.6040317586736167e-07, -3.5855391966011851e-06,
        9.0786150697210513e-06, -1.5826177772027392e-05,
        2.1667095104224339e-05, -2.4639596374368108e-05,
        2.4408848809863509e-05, -2.0820021816183944e-05,
        1.5067089085308662e-05, -7.0050769548280383e-06,
        -1.7958501740914329e-06, 1.2603332855969368e-05,
        -2.3184647095957185e-05, 3.6649243234792389e-05,
        -4.8925370942928653e-05, 6.6303764053919788e-05,
        -8.0743294791572212e-05, 1.0450895388320221e-04,
        -1.2159285962006951e-04, 1.5584791507883690e-04,
        -1.7545124243120011e-04, 2.2700988130497338e-04,
        -2.4734388406632239e-04, 3.2783247999524046e-04,
        -3.4332865428126714e-04, 4.7317261331939348e-04,
        -4.7006909503024977e-04, 6.8617937973613086e-04,
        -6.3341402749449334e-04, 1.0040019055114892e-03,
        -8.3487943140247750e-04, 1.4879378075097959e-03,
        -1.0636379989149227e-03, 2.2420753799874209e-03,
        -1.2790129815477512e-03, 3.4482247571698465e-03,
        -1.3737791982222189e-03, 5.4315927603032941e-03,
        -1.0998797914873670e-03, 8.7841427944231800e-03,
        7.8319393204598608e-05, 1.4595208448653587e-02,
        3.2659459359974446e-03, 2.4872512535509962e-02,
        1.0650179022920168e-02, 4.3259641676705020e-02,
        2.6322316164253033e-02, 7.6048270347343724e-02,
        5.7170057234916666e-02, 1.3274290157636107e-01,
        1.1202091605266952e-01, 2.2233535747440117e-01,
        1.9149401437051311e-01, 3.3181854408276518e-01,
        2.4683273185271720e-01, 3.5830743824321187e-01,
        8.6048124656303501e-02, 2.9088009098110983e-02,
        -5.6704064209706251e-01, -6.2443346412861278e-01,
        -8.6749964898300636e-01, 4.0884911445599137e-01,
        9.8650186504403581e-01, 9.1320791936604540e-01,
        -2.2349960681179670e+00, 9.8286263574585253e-01,
        5.5290844435608977e-01, -1.0935816133485650e+00,
        9.9396634297931441e-01, -7.5077030548862878e-01,
        5.4988381646049755e-01, -4.1131875464082862e-01,
        3.1720514147975598e-01, -2.5105433218659301e-01,
        2.0252402542560796e-01, -1.6558013292126614e-01,
        1.3663996383564952e-01, -1.1347727374114820e-01,
        9.4653222413267277e-02, -7.9178837982785980e-02,
        6.6352203986098701e-02, -5.5656395781301438e-02,
        4.6692213065925491e-02, -3.9151702109559969e-02,
        3.2786333094769089e-02, -2.7394200779191569e-02,
        2.2811690851216111e-02, -1.8900773546804892e-02,
        1.5548855145998195e-02, -1.2662527072754550e-02,
        1.0164652782169520e-02, -7.9964742791471654e-03,
        6.1136260652817258e-03, -4.4888557851635001e-03,
        3.1132725338383527e-03, -1.9897463347909677e-03,
        1.1299398381953670e-03, -5.4218278996456760e-04,
        2.0450525067138600e-04, -5.3541173044258997e-05,
        7.2320767464864696e-06
        ])

    return dlf


def key_201_CosSin_2012():
    r"""Key 201 pt CosSin filter, as published in [Key12]_.

    Taken from file `kk201CosSin.txt` provided with SEG-2012-003_.

    License: https://software.seg.org/disclaimer.txt.

    """

    dlf = DigitalFilter('Key 201 CosSin (2012)', 'key_201_CosSin_2012')

    dlf.base = np.array([
        9.1898135789795544e-07, 1.0560236258145801e-06, 1.2135021985967225e-06,
        1.3944646218148563e-06, 1.6024129035298632e-06, 1.8413712856029165e-06,
        2.1159641213409353e-06, 2.4315053665758195e-06, 2.7941014160203776e-06,
        3.2107692750032131e-06, 3.6895723534572921e-06, 4.2397765100647990e-06,
        4.8720293663446341e-06, 5.5985663607919089e-06, 6.4334475306554683e-06,
        7.3928296035848751e-06, 8.4952786646994725e-06, 9.7621294498526205e-06,
        1.1217898218180574e-05, 1.2890757193898812e-05, 1.4813079758803961e-05,
        1.7022066946117209e-05, 1.9560467359658073e-05, 2.2477404450316962e-05,
        2.5829327159387539e-05, 2.9681102325729859e-05, 3.4107269997169212e-05,
        3.9193485939077079e-05, 4.5038179255745865e-05, 5.1754457203059970e-05,
        5.9472294054641452e-05, 6.8341046380689574e-05, 7.8532343415514477e-05,
        9.0243408448527221e-05, 1.0370087551467150e-04, 1.1916517524538292e-04,
        1.3693557475562244e-04, 1.5735596909953090e-04, 1.8082153637169376e-04,
        2.0778638524439160e-04, 2.3877234293696414e-04, 2.7437905368319086e-04,
        3.1529558312353735e-04, 3.6231375319194681e-04, 4.1634346555562428e-04,
        4.7843031014898881e-04, 5.4977579956440249e-04, 6.3176062087820603e-04,
        7.2597135488438745e-04, 8.3423117980991972e-04, 9.5863515369401899e-04,
        1.1015907582204575e-03, 1.2658644886123726e-03, 1.4546353912032167e-03,
        1.6715565847497888e-03, 1.9208259560556996e-03, 2.2072673980160704e-03,
        2.5364241622125167e-03, 2.9146661326298548e-03, 3.3493130964692705e-03,
        3.8487763976105395e-03, 4.4227217140193290e-03, 5.0822561091888648e-03,
        5.8401429774594740e-03, 6.7110490428654943e-03, 7.7118281914628686e-03,
        8.8618476298971925e-03, 1.0183362682074693e-02, 1.1701947477049383e-02,
        1.3446989862853833e-02, 1.5452260123909515e-02, 1.7756564507909106e-02,
        2.0404496209307014e-02, 2.3447298342546899e-02, 2.6943855605395223e-02,
        3.0961833823176868e-02, 3.5578989426519304e-02, 4.0884674203786789e-02,
        4.6981564448367090e-02, 5.3987647963491189e-02, 6.2038507377358290e-02,
        7.1289943955573878e-02, 8.1920992687256264e-02, 9.4137386993145250e-02,
        1.0817554010518914e-01, 1.2430712016577938e-01, 1.4284430758454120e-01,
        1.6414583639372404e-01, 1.8862393651527715e-01, 2.1675231128725500e-01,
        2.4907530463166816e-01, 2.8621843526798946e-01, 3.2890050183176844e-01,
        3.7794749315816500e-01, 4.3430857292399505e-01, 4.9907444798513595e-01,
        5.7349847587571534e-01, 6.5902091994412060e-01, 7.5729682151433553e-01,
        8.7022802845825153e-01, 1.0000000000000000e+00, 1.1491241000036050e+00,
        1.3204861972090955e+00, 1.5174025129350848e+00, 1.7436837970197381e+00,
        2.0037090739411751e+00, 2.3025103862617100e+00, 2.6458701753619409e+00,
        3.0404331839891712e+00, 3.4938350461726517e+00, 4.0148500529942019e+00,
        4.6135609537963891e+00, 5.3015540788430497e+00, 6.0921435594709612e+00,
        7.0006289848698273e+00, 8.0445914816976902e+00, 9.2442339463025291e+00,
        1.0622772013767669e+01, 1.2206883329864255e+01, 1.4027223820279268e+01,
        1.6119020948027551e+01, 1.8522755439841418e+01, 2.1284944674394648e+01,
        2.4459042892590269e+01, 2.8106475650897377e+01, 3.2297828536610695e+01,
        3.7114213149203508e+01, 4.2648836782420439e+01, 4.9008806183799550e+01,
        5.6317200298209784e+01, 6.4715452107403038e+01, 7.4366085659245940e+01,
        8.5455861253972017e+01, 9.8199389653503559e+01, 1.1284328525648564e+02,
        1.2967093861180905e+02, 1.4900800062891784e+02, 1.7122868461604187e+02,
        1.9676300810421012e+02, 2.2610511460175258e+02, 2.5982283632295099e+02,
        2.9856868294999509e+02, 3.4309246908417492e+02, 3.9425582475436693e+02,
        4.5304886979204105e+02, 5.2060937475742980e+02, 5.9824477922157064e+02,
        6.8745749350484289e+02, 7.8997397351448706e+02, 9.0777813134110693e+02,
        1.0431497281803040e+03, 1.1987084925641964e+03, 1.3774648176845108e+03,
        1.5828780189083436e+03, 1.8189232788935387e+03, 2.0901685758341446e+03,
        2.4018630835612289e+03, 2.7600387542291814e+03, 3.1716270494286800e+03,
        3.6445930787218194e+03, 4.1880897414655765e+03, 4.8126348548959668e+03,
        5.5303146962783048e+03, 6.3550178980975252e+03, 7.3027042226581152e+03,
        8.3917134174545281e+03, 9.6431201283206228e+03, 1.1081141738683076e+04,
        1.2733607027476588e+04, 1.4632494715248606e+04, 1.6814552320467552e+04,
        1.9322007302220827e+04, 2.2203384251427578e+04, 2.5514443944955918e+04,
        2.9319262435339933e+04, 3.3691471058779483e+04, 3.8715681358217458e+04,
        4.4489122496788041e+04, 5.1123522849071662e+04, 5.8747272182953282e+04,
        6.7507906274902969e+04, 7.7574962041275547e+04, 8.9143258438494697e+04,
        1.0243666662452392e+05, 1.1771244234227551e+05, 1.3526620436579350e+05,
        1.5543765535274608e+05, 1.7861715581389511e+05, 2.0525327841984577e+05,
        2.3586148883699448e+05, 2.7103412108532194e+05, 3.1145184046243853e+05,
        3.5789681586586579e+05, 4.1126785642601945e+05, 4.7259780537596118e+05,
        5.4307352776633098e+05, 6.2405887883026747e+05, 7.1712109748508944e+05,
        8.2406113574115187e+05, 9.4694851095649926e+05, 1.0881613554026424e+06
        ])

    dlf.factor = np.array([1.149124100003605])

    dlf.cos = np.array([
        4.8963534801291350e-04, -3.2447354678906376e-03,
        1.0952238450063470e-02, -2.5330877509368774e-02,
        4.5603964620893764e-02, -6.8751965018998398e-02,
        9.1055170868309943e-02, -1.0955571014254684e-01,
        1.2271886285618731e-01, -1.3032033375840532e-01,
        1.3300188933965829e-01, -1.3178555117707785e-01,
        1.2775414321103981e-01, -1.2185305954486332e-01,
        1.1484395399555812e-01, -1.0727994394369871e-01,
        9.9560982839136489e-02, -9.1939517526987313e-02,
        8.4590736600088562e-02, -7.7600019560680286e-02,
        7.1031733586497212e-02, -6.4890821187381778e-02,
        5.9196016139512468e-02, -5.3915848643058928e-02,
        4.9055780940812956e-02, -4.4567058676679240e-02,
        4.0457261282427506e-02, -3.6666941211080724e-02,
        3.3213645139589902e-02, -3.0027761601860024e-02,
        2.7141909473138912e-02, -2.4473692903024750e-02,
        2.2075499247215292e-02, -1.9847468573710642e-02,
        1.7867265714909791e-02, -1.6011026177709484e-02,
        1.4388810260485966e-02, -1.2844078700958564e-02,
        1.1528724140086060e-02, -1.0242160186713188e-02,
        9.1906684982280187e-03, -8.1147007847875829e-03,
        7.2916108124280446e-03, -6.3832919299231621e-03,
        5.7602923199031935e-03, -4.9801561547513692e-03,
        4.5359177425499482e-03, -3.8467889983387665e-03,
        3.5670373081260649e-03, -2.9327308045782176e-03,
        2.8105929470807834e-03, -2.1944294961424278e-03,
        2.2311101317977000e-03, -1.5941611249750095e-03,
        1.8000275448786861e-03, -1.0989791755701920e-03,
        1.4951672246892860e-03, -6.7966542661269437e-04,
        1.3003578629701799e-03, -3.0965377717395037e-04,
        1.2052335925762474e-03, 3.6106423804137471e-05,
        1.2052407466848229e-03, 3.8238957499821198e-04,
        1.3018966393838437e-03, 7.5508986898323120e-04,
        1.5033588524742210e-03, 1.1828448709658736e-03,
        1.8253815899842722e-03, 1.6989785802421903e-03,
        2.2927582125052783e-03, 2.3438821839617966e-03,
        2.9413764369870584e-03, 3.1679810467363173e-03,
        3.8210431939779545e-03, 4.2354668717297693e-03,
        4.9992627735887547e-03, 5.6289926876045100e-03,
        6.5661551724297195e-03, 7.4555038861999377e-03,
        8.6406325432364806e-03, 9.8532339747283270e-03,
        1.1377691235455435e-02, 1.2999446350215346e-02,
        1.4975945575967134e-02, 1.7117326788278663e-02,
        1.9682681677977296e-02, 2.2477549053870567e-02,
        2.5789277329789900e-02, 2.9383250822409085e-02,
        3.3599553377482397e-02, 3.8111654527735343e-02,
        4.3330483453611145e-02, 4.8751291534594260e-02,
        5.4854616146235546e-02, 6.0801696080654326e-02,
        6.7092281900802533e-02, 7.2264672303858538e-02,
        7.6684426491380325e-02, 7.7748740946138262e-02,
        7.5380953371380677e-02, 6.5034557293637621e-02,
        4.5881721095444840e-02, 1.0745068396029680e-02,
        -4.1063368584660725e-02, -1.1762555750610600e-01,
        -2.1286042283609111e-01, -3.2473691853928877e-01,
        -4.1889667952194926e-01, -4.5782693031141991e-01,
        -3.5587281978710816e-01, -6.6666111275991896e-02,
        4.1398275206290663e-01, 8.2532976945066749e-01,
        7.4155935075441382e-01, -2.2362177591765958e-01,
        -1.2468909757950963e+00, -5.6409324151001772e-01,
        1.6313369576910255e+00, 1.3693670120559179e-01,
        -1.9009211808491244e+00, 2.1011582166679363e+00,
        -1.4539662980149537e+00, 8.1038085852727904e-01,
        -4.1770604353721019e-01, 2.1941406706811120e-01,
        -1.2468136830432722e-01, 7.8269403871874657e-02,
        -5.3876922626623532e-02, 3.9875972155938078e-02,
        -3.1099121841349150e-02, 2.5148873022191463e-02,
        -2.0846754019080382e-02, 1.7576952842787749e-02,
        -1.4997498936892097e-02, 1.2906795835689719e-02,
        -1.1178868977228673e-02, 9.7305065956261819e-03,
        -8.5038781802097181e-03, 7.4569676118292445e-03,
        -6.5580949021930646e-03, 5.7826495893214484e-03,
        -5.1110604027691945e-03, 4.5274795600497757e-03,
        -4.0188948307658011e-03, 3.5745070845087227e-03,
        -3.1852787667663190e-03, 2.8435964846781139e-03,
        -2.5430124541388067e-03, 2.2780422183555945e-03,
        -2.0440036868883257e-03, 1.8368872847487388e-03,
        -1.6532500255010323e-03, 1.4901283117434534e-03,
        -1.3449656159356862e-03, 1.2155521334433954e-03,
        -1.0999741706045096e-03, 9.9657152340340712e-04,
        -9.0390146341796690e-04, 8.2070821214142624e-04,
        -7.4589699233026084e-04, 6.7851191397762147e-04,
        -6.1771708408859378e-04, 5.6278043567672292e-04,
        -5.1305986066984307e-04, 4.6799130039041709e-04,
        -4.2707850066888483e-04, 3.8988418417160815e-04,
        -3.5602243103677837e-04, 3.2515209207027528e-04,
        -2.9697108796156711e-04, 2.7121146954596444e-04,
        -2.4763512919344816e-04, 2.2603006942123759e-04,
        -2.0620715318264257e-04, 1.8799727229799711e-04,
        -1.7124887513989931e-04, 1.5582580442025662e-04,
        -1.4160541245686331e-04, 1.2847693068685222e-04,
        -1.1634006683069419e-04, 1.0510380008256007e-04,
        -9.4685353393630364e-05, 8.5009334765365045e-05,
        -7.6007041962903359e-05, 6.7615916583788370e-05,
        -5.9779122420642347e-05, 5.2445208782622779e-05,
        -4.5567833380983288e-05, 3.9105728752033442e-05,
        -3.3023834075077836e-05, 2.7297600234901939e-05,
        -2.1922306831363493e-05, 1.6926318238944308e-05,
        -1.2382788632218085e-05, 8.4107903563777150e-06,
        -5.1548458966523642e-06, 2.7349863491621416e-06,
        -1.1775367755450835e-06, 3.6617415171458042e-07,
        -6.1920666642164617e-08])

    dlf.sin = np.array([
        -5.8602704468975832e-10, 4.8048608865691248e-09,
        -1.9771446411637653e-08, 5.5269671219669613e-08,
        -1.1943308954933238e-07, 2.1463221286467290e-07,
        -3.3618689168303137e-07, 4.7460375561491547e-07,
        -6.2010393774690030e-07, 7.6676687291580134e-07,
        -9.1382097135454670e-07, 1.0643058622431453e-06,
        -1.2227765259769829e-06, 1.3937672563527502e-06,
        -1.5810499935024131e-06, 1.7877429029287643e-06,
        -2.0163554732931835e-06, 2.2692361668040396e-06,
        -2.5484456753753459e-06, 2.8562454180065290e-06,
        -3.1946379472885138e-06, 3.5660627245815181e-06,
        -3.9725086879521357e-06, 4.4166880900049553e-06,
        -4.9004832484318760e-06, 5.4270061911978838e-06,
        -5.9978954164211254e-06, 6.6168927621162747e-06,
        -7.2851116079354270e-06, 8.0072717151050409e-06,
        -8.7834623074537534e-06, 9.6200272666869681e-06,
        -1.0515203893806995e-05, 1.1478097778214146e-05,
        -1.2503644637963446e-05, 1.3605711023452051e-05,
        -1.4773445946482213e-05, 1.6028992588028293e-05,
        -1.7351362255724533e-05, 1.8777046268328975e-05,
        -2.0267260460262344e-05, 2.1883367382220654e-05,
        -2.3555377916627267e-05, 2.5387811572432832e-05,
        -2.7256008695400006e-05, 2.9339419633021675e-05,
        -3.1417660496931868e-05, 3.3800354252141579e-05,
        -3.6099620853251731e-05, 3.8851364451479363e-05,
        -4.1374814496544362e-05, 4.4599458677506889e-05,
        -4.7332670414243943e-05, 5.1188885345556143e-05,
        -5.4081411276216648e-05, 5.8817223325521029e-05,
        -6.1748595688031771e-05, 6.7759540724914998e-05,
        -7.0477668174613142e-05, 7.8405578637928987e-05,
        -8.0416390943704734e-05, 9.1318327863896956e-05,
        -9.1689586953474345e-05, 1.0732811343723065e-04,
        -1.0434243728703050e-04, 1.2768616500544454e-04,
        -1.1822963938719640e-04, 1.5431866320930474e-04,
        -1.3280640984775213e-04, 1.9025153589598097e-04,
        -1.4674319163166278e-04, 2.4032678429752795e-04,
        -1.5722591368834836e-04, 3.1241839074880459e-04,
        -1.5869843609971545e-04, 4.1950672592888611e-04,
        -1.4061987157747521e-04, 5.8323092045218519e-04,
        -8.3489051752646859e-05, 8.3998754956665807e-04,
        4.8166642275560894e-05, 1.2514134018618409e-03,
        3.1809381315641577e-04, 1.9223958368384385e-03,
        8.3977376473052937e-04, 3.0319286374562394e-03,
        1.8135564849093052e-03, 4.8856220017935168e-03,
        3.5901028334771145e-03, 8.0038676866840078e-03,
        6.7764383041831900e-03, 1.3266048500372144e-02,
        1.2406252926137538e-02, 2.2134422870406390e-02,
        2.2191911862939541e-02, 3.6963935749189471e-02,
        3.8829525301567067e-02, 6.1308824849773652e-02,
        6.6140947825550744e-02, 9.9797901699075597e-02,
        1.0824048046010233e-01, 1.5616018092254505e-01,
        1.6533327222825650e-01, 2.2564430322919368e-01,
        2.2062815143611000e-01, 2.7477185256445674e-01,
        2.1146243753121693e-01, 2.0743923168031805e-01,
        2.9273136481935752e-03, -1.1875521395824359e-01,
        -4.8782254186329055e-01, -5.6383218637505039e-01,
        -7.0493717238569553e-01, -4.5783079000773212e-02,
        4.9572887642641605e-01, 1.2793243232592335e+00,
        7.6984445533323119e-04, -1.0809022247360809e+00,
        -9.7826142685914264e-01, 2.4412168561816161e+00,
        -1.5742192933752286e+00, 1.0474779812704657e-01,
        6.8117592802985916e-01, -8.1735467756558944e-01,
        6.8894183180797697e-01, -5.2159909808793825e-01,
        3.8460793473043292e-01, -2.8421923628404017e-01,
        2.1211949229810861e-01, -1.5993330303166020e-01,
        1.2163508069999608e-01, -9.3164674694277416e-02,
        7.1777818840364910e-02, -5.5578938642792224e-02,
        4.3227895023256115e-02, -3.3758552896064629e-02,
        2.6463811011765734e-02, -2.0820205680643536e-02,
        1.6436812641415149e-02, -1.3019659797407497e-02,
        1.0346416487448091e-02, -8.2481070073694342e-03,
        6.5957434607522691e-03, -5.2904671538314026e-03,
        4.2562264095930237e-03, -3.4343067314751316e-03,
        2.7792243881857633e-03, -2.2556298932160214e-03,
        1.8359634980842480e-03, -1.4986732799921144e-03,
        1.2268558973329326e-03, -1.0072161412856525e-03,
        8.2926783836384434e-04, -6.8471812797056032e-04,
        5.6699154764304634e-04, -4.7086106657177131e-04,
        3.9216119658083232e-04, -3.2756429146830335e-04,
        2.7440564055551230e-04, -2.3054635199842104e-04,
        1.9426558644629048e-04, -1.6417564894421281e-04,
        1.3915493002405042e-04, -1.1829482010512426e-04,
        1.0085758994667589e-04, -8.6242897629566575e-05,
        7.3961097310559017e-05, -6.3611923024107272e-05,
        5.4867429384991716e-05, -4.7458310810428852e-05,
        4.1162907677974120e-05, -3.5798353789658158e-05,
        3.1213433790926005e-05, -2.7282808824825689e-05,
        2.3902339183058118e-05, -2.0985288286859604e-05,
        1.8459236223596745e-05, -1.6263565819420051e-05,
        1.4347411857118876e-05, -1.2667986114133353e-05,
        1.1189208624850740e-05, -9.8805899457079486e-06,
        8.7163210286869178e-06, -7.6745372470670224e-06,
        6.7367316776180222e-06, -5.8873002689814021e-06,
        5.1132081051238435e-06, -4.4037712527232031e-06,
        3.7505516230790782e-06, -3.1473609977624753e-06,
        2.5903624227147574e-06, -2.0782393943789611e-06,
        1.6123579385552092e-06, -1.1967503151651560e-06,
        8.3762259929749212e-07, -5.4201864582008082e-07,
        3.1540800169690699e-07, -1.5849870094735947e-07,
        6.4517602920670347e-08, -1.8937067758737377e-08,
        3.0164202265996216e-09
        ])

    return dlf
