"""Module for mapping ISO 639-3 to Glottolog languages and language names.
The key is the ISO code and the value, being a ``Language`` object, contains
information from both the Glottolog and ISO data sets. The contents of this
module were generated by ``scripts/make_glottolog_languages.py``.

ISO 639-3 is an international standard for language languages with an
aim to cover all known natural languages. The extended language
coverage was based primarily on the language languages published by
SIL International, which is now the registration authority for
ISO 639-3. About: `<https://iso639-3.sil.org/>`_.

Glottolog is a project run by the Max Planck Institute for the
Science of Human History. The website contains languages for languages
as well as reconstructions of language families. About: `<http://glottolog.org/>`_.
Data of Glottolog 4.0 is published under the following license:
`<https://creativecommons.org/licenses/by/4.0/>`_.

Haspelmath, Martin & Forkel, Robert & Hammarström, Harald. 2019.
Glottolog 4.0.
Jena: Max Planck Institute for the Science of Human History.
(Available online at `<http://glottolog.org>`_, Accessed on 2019-10-02.)

>>> from cltk.languages.utils import get_lang
>>> akkadian = get_lang("akk")
>>> akkadian
Language(name='Akkadian', glottolog_id='akka1240', latitude=33.1, longitude=44.1, dates=[], family_id='afro1255', parent_id='east2678', level='language', iso_639_3_code='akk', type='a')
>>> akkadian.name
'Akkadian'
>>> akkadian.glottolog_id
'akka1240'
>>> akkadian.latitude
33.1
>>> akkadian.longitude
44.1
>>> akkadian.family_id
'afro1255'
>>> akkadian.parent_id
'east2678'
>>> len(LANGUAGES)
219
"""

import typing
from collections import OrderedDict

from cltk.core.data_types import Language

LANGUAGES = OrderedDict(
    [
        (
            "akk",
            Language(
                name="Akkadian",
                glottolog_id="akka1240",
                latitude=33.1,
                longitude=44.1,
                dates=[],
                family_id="afro1255",
                parent_id="east2678",
                level="language",
                iso_639_3_code="akk",
                type="a",
            ),
        ),
        (
            "ang",
            Language(
                name="Old English (ca. 450-1100)",
                glottolog_id="olde1238",
                latitude=51.06,
                longitude=-1.31,
                dates=[],
                family_id="indo1319",
                parent_id="angl1265",
                level="language",
                iso_639_3_code="ang",
                type="h",
            ),
        ),
        (
            "arc",
            Language(
                name="Official Aramaic (700-300 BCE)",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="arc",
                type="a",
            ),
        ),
        (
            "ave",
            Language(
                name="Avestan",
                glottolog_id="aves1237",
                latitude=31.541,
                longitude=53.633,
                dates=[],
                family_id="indo1319",
                parent_id="iran1269",
                level="language",
                iso_639_3_code="ave",
                type="a",
            ),
        ),
        (
            "axm",
            Language(
                name="Middle Armenian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="axm",
                type="h",
            ),
        ),
        (
            "chu",
            Language(
                name="Church Slavic",
                glottolog_id="chur1257",
                latitude=43.7171,
                longitude=22.8442,
                dates=[],
                family_id="indo1319",
                parent_id="east2269",
                level="language",
                iso_639_3_code="chu",
                type="a",
            ),
        ),
        (
            "cmg",
            Language(
                name="Classical Mongolian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="cmg",
                type="h",
            ),
        ),
        (
            "cms",
            Language(
                name="Messapic",
                glottolog_id="mess1244",
                latitude=40.538334,
                longitude=17.860344,
                dates=[],
                family_id="indo1319",
                parent_id="indo1319",
                level="language",
                iso_639_3_code="cms",
                type="a",
            ),
        ),
        (
            "cnx",
            Language(
                name="Middle Cornish",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="cnx",
                type="h",
            ),
        ),
        (
            "dum",
            Language(
                name="Middle Dutch",
                glottolog_id="midd1321",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="midd1347",
                level="language",
                iso_639_3_code="dum",
                type="h",
            ),
        ),
        (
            "ecr",
            Language(
                name="Eteocretan",
                glottolog_id="eteo1236",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="ecr",
                type="a",
            ),
        ),
        (
            "ecy",
            Language(
                name="Eteocypriot",
                glottolog_id="eteo1240",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="ecy",
                type="a",
            ),
        ),
        (
            "egy",
            Language(
                name="Egyptian (Ancient)",
                glottolog_id="egyp1246",
                latitude=28.381735,
                longitude=30.834959,
                dates=[],
                family_id="afro1255",
                parent_id="egyp1245",
                level="language",
                iso_639_3_code="egy",
                type="a",
            ),
        ),
        (
            "elx",
            Language(
                name="Elamite",
                glottolog_id="elam1244",
                latitude=31.1833,
                longitude=48.2333,
                dates=[],
                family_id="",
                parent_id="",
                level="language",
                iso_639_3_code="elx",
                type="a",
            ),
        ),
        (
            "emy",
            Language(
                name="Epigraphic Mayan",
                glottolog_id="epig1241",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="maya1287",
                parent_id="chol1287",
                level="language",
                iso_639_3_code="emy",
                type="a",
            ),
        ),
        (
            "enm",
            Language(
                name="Middle English",
                glottolog_id="midd1317",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="merc1242",
                level="language",
                iso_639_3_code="enm",
                type="h",
            ),
        ),
        (
            "ett",
            Language(
                name="Etruscan",
                glottolog_id="etru1241",
                latitude=41.9,
                longitude=12.4833,
                dates=[],
                family_id="",
                parent_id="",
                level="language",
                iso_639_3_code="ett",
                type="a",
            ),
        ),
        (
            "frk",
            Language(
                name="Old Frankish",
                glottolog_id="fran1264",
                latitude=52.896,
                longitude=9.98388,
                dates=[],
                family_id="indo1319",
                parent_id="oldd1237",
                level="dialect",
                iso_639_3_code="frk",
                type="h",
            ),
        ),
        (
            "frm",
            Language(
                name="Middle French",
                glottolog_id="midd1316",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="stan1290",
                level="dialect",
                iso_639_3_code="frm",
                type="h",
            ),
        ),
        (
            "fro",
            Language(
                name="Old French (842-ca. 1400)",
                glottolog_id="oldf1239",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="oila1234",
                level="language",
                iso_639_3_code="fro",
                type="h",
            ),
        ),
        (
            "gez",
            Language(
                name="Geez",
                glottolog_id="geez1241",
                latitude=10.1342,
                longitude=40.6821,
                dates=[],
                family_id="afro1255",
                parent_id="ethi1244",
                level="language",
                iso_639_3_code="gez",
                type="a",
            ),
        ),
        (
            "ghc",
            Language(
                name="Hiberno-Scottish Gaelic",
                glottolog_id="hibe1235",
                latitude=55.932,
                longitude=-5.222,
                dates=[],
                family_id="indo1319",
                parent_id="oldi1245",
                level="dialect",
                iso_639_3_code="ghc",
                type="h",
            ),
        ),
        (
            "gmh",
            Language(
                name="Middle High German",
                glottolog_id="midd1343",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="midd1349",
                level="language",
                iso_639_3_code="gmh",
                type="h",
            ),
        ),
        (
            "gml",
            Language(
                name="Middle Low German",
                glottolog_id="midd1318",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="midd1345",
                level="language",
                iso_639_3_code="gml",
                type="h",
            ),
        ),
        (
            "gmy",
            Language(
                name="Mycenaean Greek",
                glottolog_id="myce1241",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="east2798",
                level="language",
                iso_639_3_code="gmy",
                type="a",
            ),
        ),
        (
            "goh",
            Language(
                name="Old High German (ca. 750-1050)",
                glottolog_id="oldh1241",
                latitude=52.0,
                longitude=10.0,
                dates=[],
                family_id="indo1319",
                parent_id="high1286",
                level="language",
                iso_639_3_code="goh",
                type="h",
            ),
        ),
        (
            "got",
            Language(
                name="Gothic",
                glottolog_id="goth1244",
                latitude=46.9304,
                longitude=29.9786,
                dates=[],
                family_id="indo1319",
                parent_id="east2805",
                level="language",
                iso_639_3_code="got",
                type="a",
            ),
        ),
        (
            "grc",
            Language(
                name="Ancient Greek",
                glottolog_id="anci1242",
                latitude=39.8155,
                longitude=21.9129,
                dates=[],
                family_id="indo1319",
                parent_id="east2798",
                level="language",
                iso_639_3_code="grc",
                type="h",
            ),
        ),
        (
            "hbo",
            Language(
                name="Ancient Hebrew",
                glottolog_id="anci1244",
                latitude=31.7761,
                longitude=35.1725,
                dates=[],
                family_id="afro1255",
                parent_id="hebr1246",
                level="language",
                iso_639_3_code="hbo",
                type="h",
            ),
        ),
        (
            "hit",
            Language(
                name="Hittite",
                glottolog_id="hitt1242",
                latitude=40.0,
                longitude=35.0,
                dates=[],
                family_id="indo1319",
                parent_id="anat1257",
                level="language",
                iso_639_3_code="hit",
                type="a",
            ),
        ),
        (
            "hlu",
            Language(
                name="Hieroglyphic Luwian",
                glottolog_id="hier1240",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="luvi1235",
                level="language",
                iso_639_3_code="hlu",
                type="a",
            ),
        ),
        (
            "hmk",
            Language(
                name="Maek",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="hmk",
                type="a",
            ),
        ),
        (
            "htx",
            Language(
                name="Middle Hittite",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="htx",
                type="a",
            ),
        ),
        (
            "ims",
            Language(
                name="Marsian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="ims",
                type="a",
            ),
        ),
        (
            "imy",
            Language(
                name="Milyan",
                glottolog_id="mily1238",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="lyci1242",
                level="language",
                iso_639_3_code="imy",
                type="a",
            ),
        ),
        (
            "inm",
            Language(
                name="Minaean",
                glottolog_id="mina1279",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="afro1255",
                parent_id="sayh1236",
                level="language",
                iso_639_3_code="inm",
                type="a",
            ),
        ),
        (
            "jpa",
            Language(
                name="Palestinian Jewish Aramaic",
                glottolog_id="pale1261",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="afro1255",
                parent_id="west2815",
                level="language",
                iso_639_3_code="jpa",
                type="h",
            ),
        ),
        (
            "jut",
            Language(
                name="Jutish",
                glottolog_id="juti1236",
                latitude=54.911,
                longitude=9.38924,
                dates=[],
                family_id="indo1319",
                parent_id="sout3248",
                level="language",
                iso_639_3_code="jut",
                type="h",
            ),
        ),
        (
            "kaw",
            Language(
                name="Kawi",  # Old Javanese
                glottolog_id="kawi1241",
                latitude=-7.0,
                longitude=111.0,
                dates=[],
                family_id="aust1307",
                parent_id="java1253",
                level="language",
                iso_639_3_code="kaw",
                type="a",
            ),
        ),
        (
            "kho",
            Language(
                name="Khotanese",
                glottolog_id="khot1251",
                latitude=38.33,
                longitude=79.04,
                dates=[],
                family_id="indo1319",
                parent_id="saka1298",
                level="language",
                iso_639_3_code="kho",
                type="a",
            ),
        ),
        (
            "kjv",
            Language(
                name="Kajkavian",
                glottolog_id="kajk1237",
                latitude=45.649748,
                longitude=16.134109,
                dates=[],
                family_id="indo1319",
                parent_id="west2804",
                level="language",
                iso_639_3_code="kjv",
                type="h",
            ),
        ),
        (
            "lab",
            Language(
                name="Linear A",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="lab",
                type="a",
            ),
        ),
        (
            "lat",
            Language(
                name="Latin",
                glottolog_id="lati1261",
                latitude=41.9026,
                longitude=12.4502,
                dates=[],
                family_id="indo1319",
                parent_id="impe1234",
                level="language",
                iso_639_3_code="lat",
                type="a",
            ),
        ),
        (
            "lng",
            Language(
                name="Langobardic",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="lng",
                type="a",
            ),
        ),
        (
            "ltc",
            Language(
                name="Middle Chinese",
                glottolog_id="midd1344",
                latitude=34.27,
                longitude=108.9,
                dates=[],
                family_id="sino1245",
                parent_id="midd1354",
                level="language",
                iso_639_3_code="ltc",
                type="h",
            ),
        ),
        (
            "lzh",
            Language(
                name="Literary Chinese",  # or Classical Chinese
                glottolog_id="lite1248",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="sino1245",
                parent_id="clas1255",
                level="language",
                iso_639_3_code="lzh",
                type="h",
            ),
        ),
        (
            "mga",
            Language(
                name="Middle Irish (10-12th century)",
                glottolog_id="midd1360",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="oldi1245",
                level="dialect",
                iso_639_3_code="mga",
                type="h",
            ),
        ),
        (
            "mxi",
            Language(
                name="Mozarabic",
                glottolog_id="moza1249",
                latitude=37.8763,
                longitude=-1.79724,
                dates=[],
                family_id="indo1319",
                parent_id="unsh1234",
                level="language",
                iso_639_3_code="mxi",
                type="h",
            ),
        ),
        (
            "myz",
            Language(
                name="Classical Mandaic",
                glottolog_id="clas1253",
                latitude=36.1203,
                longitude=49.5659,
                dates=[],
                family_id="afro1255",
                parent_id="mand1468",
                level="language",
                iso_639_3_code="myz",
                type="h",
            ),
        ),
        (
            "nci",
            Language(
                name="Classical Nahuatl",
                glottolog_id="clas1250",
                latitude=19.72,
                longitude=-96.97,
                dates=[],
                family_id="utoa1244",
                parent_id="cent2258",
                level="language",
                iso_639_3_code="nci",
                type="h",
            ),
        ),
        (
            "ndf",
            Language(
                name="Nadruvian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="ndf",
                type="h",
            ),
        ),
        (
            "nei",
            Language(
                name="Neo-Hittite",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="nei",
                type="a",
            ),
        ),
        (
            "non",
            Language(
                name="Old Norse",
                glottolog_id="oldn1244",
                latitude=63.42,
                longitude=10.38,
                dates=[],
                family_id="indo1319",
                parent_id="west2805",
                level="language",
                iso_639_3_code="non",
                type="h",
            ),
        ),
        (
            "nrc",
            Language(
                name="Noric",
                glottolog_id="nori1240",
                latitude=46.419444,
                longitude=15.869444,
                dates=[],
                family_id="indo1319",
                parent_id="unun9887",
                level="language",
                iso_639_3_code="nrc",
                type="a",
            ),
        ),
        (
            "nrp",
            Language(
                name="North Picene",
                glottolog_id="nort1401",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="nrp",
                type="a",
            ),
        ),
        (
            "nwc",
            Language(
                name="Classical Newari",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="nwc",
                type="h",
            ),
        ),
        (
            "nwx",
            Language(
                name="Middle Newar",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="nwx",
                type="h",
            ),
        ),
        (
            "nxm",
            Language(
                name="Numidian",
                glottolog_id="numi1241",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="nxm",
                type="a",
            ),
        ),
        (
            "oar",
            Language(
                name="Old Aramaic (up to 700 BCE)",
                glottolog_id="olda1245",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="afro1255",
                parent_id="aram1259",
                level="language",
                iso_639_3_code="oar",
                type="a",
            ),
        ),
        (
            "oav",
            Language(
                name="Old Avar",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="oav",
                type="h",
            ),
        ),
        (
            "obm",
            Language(
                name="Moabite",
                glottolog_id="moab1234",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="book1242",
                parent_id="book1242",
                level="language",
                iso_639_3_code="obm",
                type="a",
            ),
        ),
        (
            "obr",
            Language(
                name="Old Burmese",
                glottolog_id="oldb1235",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="sino1245",
                parent_id="oldm1246",
                level="language",
                iso_639_3_code="obr",
                type="h",
            ),
        ),
        (
            "obt",
            Language(
                name="Old Breton",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="obt",
                type="h",
            ),
        ),
        (
            "och",
            Language(
                name="Old Chinese",
                glottolog_id="oldc1244",
                latitude=36.12,
                longitude=114.32,
                dates=[],
                family_id="sino1245",
                parent_id="sini1245",
                level="language",
                iso_639_3_code="och",
                type="a",
            ),
        ),
        (
            "oco",
            Language(
                name="Old Cornish",
                glottolog_id="oldc1252",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="corn1251",
                level="dialect",
                iso_639_3_code="oco",
                type="h",
            ),
        ),
        (
            "odt",
            Language(
                name="Old Dutch-Old Frankish",
                glottolog_id="oldd1237",
                latitude=52.16,
                longitude=5.2,
                dates=[],
                family_id="indo1319",
                parent_id="macr1270",
                level="language",
                iso_639_3_code="odt",
                type="h",
            ),
        ),
        (
            "ofs",
            Language(
                name="Old Frisian",
                glottolog_id="oldf1241",
                latitude=53.35,
                longitude=6.8,
                dates=[],
                family_id="indo1319",
                parent_id="fris1239",
                level="language",
                iso_639_3_code="ofs",
                type="h",
            ),
        ),
        (
            "oge",
            Language(
                name="Old Georgian",
                glottolog_id="oldg1234",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="kart1248",
                parent_id="geor1253",
                level="language",
                iso_639_3_code="oge",
                type="h",
            ),
        ),
        (
            "oht",
            Language(
                name="Old Hittite",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="oht",
                type="a",
            ),
        ),
        (
            "ohu",
            Language(
                name="Old Hungarian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="ohu",
                type="h",
            ),
        ),
        (
            "ojp",
            Language(
                name="Old Japanese",
                glottolog_id="oldj1239",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="japo1237",
                parent_id="japa1256",
                level="language",
                iso_639_3_code="ojp",
                type="h",
            ),
        ),
        (
            "okm",
            Language(
                name="Middle Korean (10th-16th cent.)",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="okm",
                type="h",
            ),
        ),
        (
            "oko",
            Language(
                name="Old Korean (3rd-9th cent.)",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="oko",
                type="h",
            ),
        ),
        (
            "olt",
            Language(
                name="Old Lithuanian",
                glottolog_id="oldl1240",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="lith1251",
                level="dialect",
                iso_639_3_code="olt",
                type="h",
            ),
        ),
        (
            "omn",
            Language(
                name="Minoan",
                glottolog_id="mino1236",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="omn",
                type="a",
            ),
        ),
        (
            "omp",
            Language(
                name="Old Manipuri",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="omp",
                type="h",
            ),
        ),
        (
            "omr",
            Language(
                name="Old Marathi",
                glottolog_id="oldm1244",
                latitude=18.0,
                longitude=75.0,
                dates=[],
                family_id="indo1319",
                parent_id="mara1416",
                level="language",
                iso_639_3_code="omr",
                type="h",
            ),
        ),
        (
            "omx",
            Language(
                name="Old Mon",
                glottolog_id="oldm1242",
                latitude=14.83,
                longitude=100.5,
                dates=[],
                family_id="aust1305",
                parent_id="moni1258",
                level="language",
                iso_639_3_code="omx",
                type="h",
            ),
        ),
        (
            "onw",
            Language(
                name="Old Nubian",
                glottolog_id="oldn1245",
                latitude=18.760713,
                longitude=31.748657,
                dates=[],
                family_id="nubi1251",
                parent_id="nobi1239",
                level="language",
                iso_639_3_code="onw",
                type="h",
            ),
        ),
        (
            "oos",
            Language(
                name="Old Ossetic",
                glottolog_id="oldo1234",
                latitude=48.48,
                longitude=39.46,
                dates=[],
                family_id="indo1319",
                parent_id="osse1245",
                level="language",
                iso_639_3_code="oos",
                type="a",
            ),
        ),
        (
            "orv",
            Language(
                name="Old Russian",
                glottolog_id="oldr1238",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="east1426",
                level="language",
                iso_639_3_code="orv",
                type="h",
            ),
        ),
        (
            "osc",
            Language(
                name="Oscan",
                glottolog_id="osca1245",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="sabe1249",
                level="language",
                iso_639_3_code="osc",
                type="a",
            ),
        ),
        (
            "osp",
            Language(
                name="Old Spanish",
                glottolog_id="olds1249",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="cast1243",
                level="language",
                iso_639_3_code="osp",
                type="h",
            ),
        ),
        (
            "osx",
            Language(
                name="Old Saxon",
                glottolog_id="olds1250",
                latitude=52.37,
                longitude=9.72,
                dates=[],
                family_id="indo1319",
                parent_id="alts1234",
                level="language",
                iso_639_3_code="osx",
                type="h",
            ),
        ),
        (
            "ota",
            Language(
                name="Ottoman Turkish (1500-1928)",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="ota",
                type="h",
            ),
        ),
        (
            "otb",
            Language(
                name="Old Tibetan",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="otb",
                type="h",
            ),
        ),
        (
            "otk",
            Language(
                name="Old Turkish",
                glottolog_id="oldt1247",
                latitude=37.0,
                longitude=59.0,
                dates=[],
                family_id="book1242",
                parent_id="book1242",
                level="language",
                iso_639_3_code="otk",
                type="h",
            ),
        ),
        (
            "oty",
            Language(
                name="Old Tamil",
                glottolog_id="oldt1248",
                latitude=11.0,
                longitude=78.0,
                dates=[],
                family_id="drav1251",
                parent_id="tami1299",
                level="language",
                iso_639_3_code="oty",
                type="a",
            ),
        ),
        (
            "oui",
            Language(
                name="Old Turkic",
                glottolog_id="oldu1238",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="turk1311",
                parent_id="karl1243",
                level="language",
                iso_639_3_code="oui",
                type="h",
            ),
        ),
        (
            "owl",
            Language(
                name="Old-Middle Welsh",
                glottolog_id="oldw1239",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="oldm1247",
                level="language",
                iso_639_3_code="owl",
                type="h",
            ),
        ),
        (
            "pal",
            Language(
                name="Pahlavi",
                glottolog_id="pahl1241",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="midd1352",
                level="language",
                iso_639_3_code="pal",
                type="a",
            ),
        ),
        (
            "peo",
            Language(
                name="Old Persian (ca. 600-400 B.C.)",
                glottolog_id="oldp1254",
                latitude=32.0,
                longitude=54.0,
                dates=[],
                family_id="indo1319",
                parent_id="sout3157",
                level="language",
                iso_639_3_code="peo",
                type="h",
            ),
        ),
        (
            "pgd",
            Language(
                name="Gāndhārī",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="pgd",
                type="h",
            ),
        ),
        (
            "pgl",
            Language(
                name="Primitive Irish",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="pgl",
                type="a",
            ),
        ),
        (
            "pgn",
            Language(
                name="Paelignian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="pgn",
                type="a",
            ),
        ),
        (
            "phn",
            Language(
                name="Phoenician",
                glottolog_id="phoe1239",
                latitude=34.12,
                longitude=35.65,
                dates=[],
                family_id="afro1255",
                parent_id="phoe1238",
                level="language",
                iso_639_3_code="phn",
                type="a",
            ),
        ),
        (
            "pka",
            Language(
                name="Ardhamāgadhī Prākrit",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="pka",
                type="h",
            ),
        ),
        (
            "pkc",
            Language(
                name="Paekche",
                glottolog_id="paek1234",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="pkc",
                type="a",
            ),
        ),
        (
            "pli",
            Language(
                name="Pali",
                glottolog_id="pali1273",
                latitude=24.5271,
                longitude=82.251,
                dates=[],
                family_id="indo1319",
                parent_id="biha1245",
                level="language",
                iso_639_3_code="pli",
                type="a",
            ),
        ),
        (
            "plq",
            Language(
                name="Palaic",
                glottolog_id="pala1331",
                latitude=41.334,
                longitude=33.499,
                dates=[],
                family_id="indo1319",
                parent_id="anat1257",
                level="language",
                iso_639_3_code="plq",
                type="a",
            ),
        ),
        (
            "pmh",
            Language(
                name="Maharastri Prakrit",
                glottolog_id="maha1305",
                latitude=19.18,
                longitude=74.71,
                dates=[],
                family_id="indo1319",
                parent_id="indo1325",
                level="language",
                iso_639_3_code="pmh",
                type="h",
            ),
        ),
        (
            "pro",
            Language(
                name="Old Provençal",
                glottolog_id="oldp1253",
                latitude=43.5,
                longitude=0.6,
                dates=[],
                family_id="indo1319",
                parent_id="occi1240",
                level="language",
                iso_639_3_code="pro",
                type="h",
            ),
        ),
        (
            "psu",
            Language(
                name="Sauraseni Prakrit",
                glottolog_id="saur1252",
                latitude=27.46,
                longitude=77.56,
                dates=[],
                family_id="indo1319",
                parent_id="indo1322",
                level="language",
                iso_639_3_code="psu",
                type="h",
            ),
        ),
        (
            "pyx",
            Language(
                name="Burma Pyu",
                glottolog_id="burm1262",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="sino1245",
                parent_id="uncl1508",
                level="language",
                iso_639_3_code="pyx",
                type="a",
            ),
        ),
        (
            "qwc",
            Language(
                name="Classical Quechua",
                glottolog_id="clas1251",
                latitude=-9.99999,
                longitude=-76.5,
                dates=[],
                family_id="quec1387",
                parent_id="quec1389",
                level="language",
                iso_639_3_code="qwc",
                type="h",
            ),
        ),
        (
            "san",
            Language(
                name="Sanskrit",
                glottolog_id="sans1269",
                latitude=20.0,
                longitude=77.0,
                dates=[],
                family_id="indo1319",
                parent_id="indo1321",
                level="language",
                iso_639_3_code="san",
                type="a",
            ),
        ),
        (
            "sbv",
            Language(
                name="Sabine",
                glottolog_id="sabi1245",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="sabe1248",
                level="dialect",
                iso_639_3_code="sbv",
                type="a",
            ),
        ),
        (
            "scx",
            Language(
                name="Sicula",
                glottolog_id="sicu1234",
                latitude=37.533968,
                longitude=14.84133,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="scx",
                type="a",
            ),
        ),
        (
            "sga",
            Language(
                name="Early Irish",
                glottolog_id="oldi1245",
                latitude=53.0,
                longitude=-8.0,
                dates=[],
                family_id="indo1319",
                parent_id="goid1240",
                level="language",
                iso_639_3_code="sga",
                type="h",
            ),
        ),
        (
            "sog",
            Language(
                name="Sogdian",
                glottolog_id="sogd1245",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="sogd1246",
                level="language",
                iso_639_3_code="sog",
                type="a",
            ),
        ),
        (
            "spx",
            Language(
                name="South Picene",
                glottolog_id="sout2618",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="sabe1248",
                level="dialect",
                iso_639_3_code="spx",
                type="a",
            ),
        ),
        (
            "sqr",
            Language(
                name="Siculo Arabic",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="sqr",
                type="h",
            ),
        ),
        (
            "sux",
            Language(
                name="Sumerian",
                glottolog_id="sume1241",
                latitude=31.3,
                longitude=45.6667,
                dates=[],
                family_id="",
                parent_id="",
                level="language",
                iso_639_3_code="sux",
                type="a",
            ),
        ),
        (
            "svx",
            Language(
                name="Skalvian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="svx",
                type="h",
            ),
        ),
        (
            "sxc",
            Language(
                name="Sicana",
                glottolog_id="sica1234",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="sxc",
                type="a",
            ),
        ),
        (
            "sxo",
            Language(
                name="Sorothaptic",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="sxo",
                type="a",
            ),
        ),
        (
            "syc",
            Language(
                name="Classical Syriac",
                glottolog_id="clas1252",
                latitude=37.1129,
                longitude=42.1432,
                dates=[],
                family_id="afro1255",
                parent_id="east2680",
                level="language",
                iso_639_3_code="syc",
                type="h",
            ),
        ),
        (
            "txb",
            Language(
                name="Tokharian B",
                glottolog_id="tokh1243",
                latitude=41.65,
                longitude=82.9,
                dates=[],
                family_id="indo1319",
                parent_id="tokh1241",
                level="language",
                iso_639_3_code="txb",
                type="a",
            ),
        ),
        (
            "txg",
            Language(
                name="Tangut",
                glottolog_id="tang1334",
                latitude=38.894904,
                longitude=102.246464,
                dates=[],
                family_id="sino1245",
                parent_id="qian1263",
                level="language",
                iso_639_3_code="txg",
                type="a",
            ),
        ),
        (
            "txh",
            Language(
                name="Thracian",
                glottolog_id="thra1250",
                latitude=42.115469,
                longitude=24.711977,
                dates=[],
                family_id="indo1319",
                parent_id="indo1319",
                level="language",
                iso_639_3_code="txh",
                type="a",
            ),
        ),
        (
            "txr",
            Language(
                name="Tartessian",
                glottolog_id="tart1237",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="txr",
                type="a",
            ),
        ),
        (
            "uga",
            Language(
                name="Ugaritic",
                glottolog_id="ugar1238",
                latitude=35.6,
                longitude=35.78,
                dates=[],
                family_id="afro1255",
                parent_id="nort3165",
                level="language",
                iso_639_3_code="uga",
                type="a",
            ),
        ),
        (
            "umc",
            Language(
                name="Marrucinian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="umc",
                type="a",
            ),
        ),
        (
            "wlm",
            Language(
                name="Middle Welsh",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="wlm",
                type="h",
            ),
        ),
        (
            "xaa",
            Language(
                name="Andalusian Arabic",
                glottolog_id="anda1287",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="afro1255",
                parent_id="moro1295",
                level="language",
                iso_639_3_code="xaa",
                type="h",
            ),
        ),
        (
            "xae",
            Language(
                name="Aequian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xae",
                type="a",
            ),
        ),
        (
            "xag",
            Language(
                name="Aghwan",
                glottolog_id="aghw1237",
                latitude=40.374444,
                longitude=47.126667,
                dates=[],
                family_id="nakh1245",
                parent_id="udia1235",
                level="language",
                iso_639_3_code="xag",
                type="a",
            ),
        ),
        (
            "xaq",
            Language(
                name="Aquitanian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xaq",
                type="a",
            ),
        ),
        (
            "xbc",
            Language(
                name="Bactrian",
                glottolog_id="bact1239",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="cent2318",
                level="language",
                iso_639_3_code="xbc",
                type="a",
            ),
        ),
        (
            "xbm",
            Language(
                name="Middle Breton",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xbm",
                type="h",
            ),
        ),
        (
            "xbo",
            Language(
                name="Bolgarian",
                glottolog_id="bolg1250",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="turk1311",
                parent_id="bolg1249",
                level="language",
                iso_639_3_code="xbo",
                type="h",
            ),
        ),
        (
            "xcb",
            Language(
                name="Cumbric",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xcb",
                type="h",
            ),
        ),
        (
            "xcc",
            Language(
                name="Camunic",
                glottolog_id="camu1235",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="xcc",
                type="a",
            ),
        ),
        (
            "xce",
            Language(
                name="Celtiberian",
                glottolog_id="celt1247",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="celt1248",
                level="language",
                iso_639_3_code="xce",
                type="a",
            ),
        ),
        (
            "xcg",
            Language(
                name="Cisalpine Gaulish",
                glottolog_id="cisa1237",
                latitude=45.2,
                longitude=8.93,
                dates=[],
                family_id="indo1319",
                parent_id="cisa1238",
                level="language",
                iso_639_3_code="xcg",
                type="a",
            ),
        ),
        (
            "xcl",
            Language(
                name="Classical Armenian",
                glottolog_id="clas1249",
                latitude=38.592316,
                longitude=41.346422,
                dates=[],
                family_id="indo1319",
                parent_id="arme1241",
                level="language",
                iso_639_3_code="xcl",
                type="h",
            ),
        ),
        (
            "xco",
            Language(
                name="Khwarezmian",
                glottolog_id="khwa1238",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="sogd1248",
                level="language",
                iso_639_3_code="xco",
                type="a",
            ),
        ),
        (
            "xcr",
            Language(
                name="Carian",
                glottolog_id="cari1274",
                latitude=37.203,
                longitude=28.182,
                dates=[],
                family_id="indo1319",
                parent_id="luvi1234",
                level="language",
                iso_639_3_code="xcr",
                type="a",
            ),
        ),
        (
            "xct",
            Language(
                name="Classical Tibetan",
                glottolog_id="clas1254",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="sino1245",
                parent_id="oldm1245",
                level="language",
                iso_639_3_code="xct",
                type="h",
            ),
        ),
        (
            "xcu",
            Language(
                name="Curonian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xcu",
                type="h",
            ),
        ),
        (
            "xdc",
            Language(
                name="Dacian",
                glottolog_id="daci1234",
                latitude=46.2,
                longitude=23.3,
                dates=[],
                family_id="indo1319",
                parent_id="indo1319",
                level="language",
                iso_639_3_code="xdc",
                type="a",
            ),
        ),
        (
            "xdm",
            Language(
                name="Edomite",
                glottolog_id="edom1234",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="book1242",
                parent_id="book1242",
                level="language",
                iso_639_3_code="xdm",
                type="a",
            ),
        ),
        (
            "xeb",
            Language(
                name="Eblaite",
                glottolog_id="ebla1238",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="afro1255",
                parent_id="east2678",
                level="language",
                iso_639_3_code="xeb",
                type="a",
            ),
        ),
        (
            "xep",
            Language(
                name="Epi-Olmec",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xep",
                type="a",
            ),
        ),
        (
            "xfa",
            Language(
                name="Faliscan",
                glottolog_id="fali1291",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="lati1262",
                level="language",
                iso_639_3_code="xfa",
                type="a",
            ),
        ),
        (
            "xga",
            Language(
                name="Galatian",
                glottolog_id="gala1252",
                latitude=39.34,
                longitude=32.84,
                dates=[],
                family_id="indo1319",
                parent_id="nucl1715",
                level="language",
                iso_639_3_code="xga",
                type="a",
            ),
        ),
        (
            "xgl",
            Language(
                name="Galindan",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xgl",
                type="h",
            ),
        ),
        (
            "xha",
            Language(
                name="Harami",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xha",
                type="a",
            ),
        ),
        (
            "xhc",
            Language(
                name="Hunnic",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xhc",
                type="a",
            ),
        ),
        (
            "xhd",
            Language(
                name="Hadrami",
                glottolog_id="hadr1235",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="afro1255",
                parent_id="sayh1236",
                level="language",
                iso_639_3_code="xhd",
                type="a",
            ),
        ),
        (
            "xhr",
            Language(
                name="Hernican",
                glottolog_id="hemi1234",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="osca1245",
                level="dialect",
                iso_639_3_code="xhr",
                type="a",
            ),
        ),
        (
            "xht",
            Language(
                name="Hattic",
                glottolog_id="hatt1246",
                latitude=33.9833,
                longitude=40.3333,
                dates=[],
                family_id="",
                parent_id="",
                level="language",
                iso_639_3_code="xht",
                type="a",
            ),
        ),
        (
            "xhu",
            Language(
                name="Hurrian",
                glottolog_id="hurr1240",
                latitude=40.7339,
                longitude=43.3154,
                dates=[],
                family_id="hurr1239",
                parent_id="hurr1239",
                level="language",
                iso_639_3_code="xhu",
                type="a",
            ),
        ),
        (
            "xib",
            Language(
                name="Iberian",
                glottolog_id="iber1250",
                latitude=40.624,
                longitude=0.212401,
                dates=[],
                family_id="",
                parent_id="",
                level="language",
                iso_639_3_code="xib",
                type="a",
            ),
        ),
        (
            "xil",
            Language(
                name="Illyrian",
                glottolog_id="illy1234",
                latitude=43.0,
                longitude=18.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="xil",
                type="a",
            ),
        ),
        (
            "xiv",
            Language(
                name="Harappan",
                glottolog_id="hara1272",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="unat1236",
                parent_id="unat1236",
                level="language",
                iso_639_3_code="xiv",
                type="a",
            ),
        ),
        (
            "xlc",
            Language(
                name="Lycian A",
                glottolog_id="lyci1241",
                latitude=36.517,
                longitude=29.764,
                dates=[],
                family_id="indo1319",
                parent_id="lyci1242",
                level="language",
                iso_639_3_code="xlc",
                type="a",
            ),
        ),
        (
            "xld",
            Language(
                name="Lydian",
                glottolog_id="lydi1241",
                latitude=40.287,
                longitude=28.599,
                dates=[],
                family_id="indo1319",
                parent_id="unun9888",
                level="language",
                iso_639_3_code="xld",
                type="a",
            ),
        ),
        (
            "xle",
            Language(
                name="Lemnian",
                glottolog_id="lemn1237",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="xle",
                type="a",
            ),
        ),
        (
            "xlg",
            Language(
                name="Ancient Ligurian",
                glottolog_id="anci1248",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="xlg",
                type="a",
            ),
        ),
        (
            "xli",
            Language(
                name="Liburnian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xli",
                type="a",
            ),
        ),
        (
            "xln",
            Language(
                name="Alanic",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xln",
                type="a",
            ),
        ),
        (
            "xlp",
            Language(
                name="Lepontic",
                glottolog_id="lepo1240",
                latitude=45.98,
                longitude=8.92,
                dates=[],
                family_id="indo1319",
                parent_id="cisa1238",
                level="language",
                iso_639_3_code="xlp",
                type="a",
            ),
        ),
        (
            "xls",
            Language(
                name="Lusitanian",
                glottolog_id="lusi1235",
                latitude=39.943975,
                longitude=-8.105283,
                dates=[],
                family_id="indo1319",
                parent_id="indo1319",
                level="language",
                iso_639_3_code="xls",
                type="a",
            ),
        ),
        (
            "xlu",
            Language(
                name="Cuneiform Luwian",
                glottolog_id="cune1239",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="luvi1235",
                level="language",
                iso_639_3_code="xlu",
                type="a",
            ),
        ),
        (
            "xly",
            Language(
                name="Elymian",
                glottolog_id="elym1237",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="xly",
                type="a",
            ),
        ),
        (
            "xme",
            Language(
                name="Median",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xme",
                type="a",
            ),
        ),
        (
            "xmk",
            Language(
                name="Ancient Macedonian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xmk",
                type="a",
            ),
        ),
        (
            "xmn",
            Language(
                name="Manichaean Middle Persian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xmn",
                type="h",
            ),
        ),
        (
            "xmr",
            Language(
                name="Meroitic",
                glottolog_id="mero1237",
                latitude=16.9333,
                longitude=33.7333,
                dates=[],
                family_id="",
                parent_id="",
                level="language",
                iso_639_3_code="xmr",
                type="a",
            ),
        ),
        (
            "xna",
            Language(
                name="Ancient North Arabian",
                glottolog_id="anci1245",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="afro1255",
                parent_id="arab1394",
                level="language",
                iso_639_3_code="xna",
                type="a",
            ),
        ),
        (
            "xng",
            Language(
                name="Middle Mongol",
                glottolog_id="midd1351",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="mong1329",
                parent_id="mong1329",
                level="language",
                iso_639_3_code="xng",
                type="h",
            ),
        ),
        (
            "xno",
            Language(
                name="Anglo-Norman",
                glottolog_id="angl1258",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="stan1290",
                level="dialect",
                iso_639_3_code="xno",
                type="h",
            ),
        ),
        (
            "xpc",
            Language(
                name="Pecheneg",
                glottolog_id="pech1242",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="turk1311",
                parent_id="unun9944",
                level="language",
                iso_639_3_code="xpc",
                type="h",
            ),
        ),
        (
            "xpg",
            Language(
                name="Phrygian",
                glottolog_id="phry1239",
                latitude=39.613,
                longitude=31.28,
                dates=[],
                family_id="indo1319",
                parent_id="grae1234",
                level="language",
                iso_639_3_code="xpg",
                type="a",
            ),
        ),
        (
            "xpi",
            Language(
                name="Pictish",
                glottolog_id="pict1238",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="xpi",
                type="h",
            ),
        ),
        (
            "xpp",
            Language(
                name="Puyo-Paekche",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xpp",
                type="a",
            ),
        ),
        (
            "xpr",
            Language(
                name="Parthian",
                glottolog_id="part1239",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="nort3177",
                level="language",
                iso_639_3_code="xpr",
                type="a",
            ),
        ),
        (
            "xps",
            Language(
                name="Pisidian",
                glottolog_id="pisi1234",
                latitude=37.273,
                longitude=31.28,
                dates=[],
                family_id="indo1319",
                parent_id="uncl1514",
                level="language",
                iso_639_3_code="xps",
                type="a",
            ),
        ),
        (
            "xpu",
            Language(
                name="Punic",
                glottolog_id="puni1241",
                latitude=36.527295,
                longitude=10.356444,
                dates=[],
                family_id="afro1255",
                parent_id="phoe1238",
                level="language",
                iso_639_3_code="xpu",
                type="a",
            ),
        ),
        (
            "xpy",
            Language(
                name="Puyo",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xpy",
                type="a",
            ),
        ),
        (
            "xqa",
            Language(
                name="Karakhanid",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xqa",
                type="h",
            ),
        ),
        (
            "xqt",
            Language(
                name="Qatabanian",
                glottolog_id="qata1238",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="afro1255",
                parent_id="sayh1236",
                level="language",
                iso_639_3_code="xqt",
                type="a",
            ),
        ),
        (
            "xrm",
            Language(
                name="Armazic",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xrm",
                type="a",
            ),
        ),
        (
            "xrr",
            Language(
                name="Raetic",
                glottolog_id="raet1238",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="xrr",
                type="a",
            ),
        ),
        (
            "xsa",
            Language(
                name="Sabaic",
                glottolog_id="saba1279",
                latitude=15.42,
                longitude=45.35,
                dates=[],
                family_id="afro1255",
                parent_id="sayh1236",
                level="language",
                iso_639_3_code="xsa",
                type="a",
            ),
        ),
        (
            "xsc",
            Language(
                name="Scythian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xsc",
                type="a",
            ),
        ),
        (
            "xsd",
            Language(
                name="Sidetic",
                glottolog_id="side1240",
                latitude=36.764,
                longitude=31.917,
                dates=[],
                family_id="indo1319",
                parent_id="unun9888",
                level="language",
                iso_639_3_code="xsd",
                type="a",
            ),
        ),
        (
            "xtg",
            Language(
                name="Transalpine Gaulish",
                glottolog_id="tran1282",
                latitude=43.97,
                longitude=3.3,
                dates=[],
                family_id="indo1319",
                parent_id="tgbc1234",
                level="language",
                iso_639_3_code="xtg",
                type="a",
            ),
        ),
        (
            "xto",
            Language(
                name="Tokharian A",
                glottolog_id="tokh1242",
                latitude=42.98,
                longitude=89.18,
                dates=[],
                family_id="indo1319",
                parent_id="tokh1241",
                level="language",
                iso_639_3_code="xto",
                type="a",
            ),
        ),
        (
            "xtq",
            Language(
                name="Tumshuqese",
                glottolog_id="tums1237",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="saka1298",
                level="language",
                iso_639_3_code="xtq",
                type="h",
            ),
        ),
        (
            "xtr",
            Language(
                name="Early Tripuri",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xtr",
                type="a",
            ),
        ),
        (
            "xum",
            Language(
                name="Umbrian",
                glottolog_id="umbr1253",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="sabe1249",
                level="language",
                iso_639_3_code="xum",
                type="a",
            ),
        ),
        (
            "xur",
            Language(
                name="Urartian",
                glottolog_id="urar1245",
                latitude=35.0333,
                longitude=44.3833,
                dates=[],
                family_id="hurr1239",
                parent_id="hurr1239",
                level="language",
                iso_639_3_code="xur",
                type="a",
            ),
        ),
        (
            "xve",
            Language(
                name="Venetic",
                glottolog_id="vene1257",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="indo1319",
                level="language",
                iso_639_3_code="xve",
                type="a",
            ),
        ),
        (
            "xvn",
            Language(
                name="Vandalic",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xvn",
                type="a",
            ),
        ),
        (
            "xvo",
            Language(
                name="Volscian",
                glottolog_id="vols1237",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="umbr1253",
                level="dialect",
                iso_639_3_code="xvo",
                type="a",
            ),
        ),
        (
            "xvs",
            Language(
                name="Vestinian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xvs",
                type="a",
            ),
        ),
        (
            "xzh",
            Language(
                name="Zhangzhung",
                glottolog_id="zhan1239",
                latitude=31.066667,
                longitude=81.3125,
                dates=[],
                family_id="sino1245",
                parent_id="unun9961",
                level="language",
                iso_639_3_code="xzh",
                type="a",
            ),
        ),
        (
            "xzp",
            Language(
                name="Ancient Zapotec",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="xzp",
                type="h",
            ),
        ),
        (
            "yms",
            Language(
                name="Mysian",
                glottolog_id="mysi1239",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="indo1319",
                parent_id="phry1239",
                level="dialect",
                iso_639_3_code="yms",
                type="a",
            ),
        ),
        (
            "zkg",
            Language(
                name="Koguryo",
                glottolog_id="kogu1234",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="uncl1493",
                parent_id="uncl1493",
                level="language",
                iso_639_3_code="zkg",
                type="a",
            ),
        ),
        (
            "zkh",
            Language(
                name="Khorezmian",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="zkh",
                type="h",
            ),
        ),
        (
            "zkt",
            Language(
                name="Kitan",
                glottolog_id="kita1247",
                latitude=44.3773,
                longitude=115.986,
                dates=[],
                family_id="mong1329",
                parent_id="unun9946",
                level="language",
                iso_639_3_code="zkt",
                type="h",
            ),
        ),
        (
            "zkz",
            Language(
                name="Khazar",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="zkz",
                type="h",
            ),
        ),
        (
            "zra",
            Language(
                name="Kara (Korea)",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="zra",
                type="a",
            ),
        ),
        (
            "zsk",
            Language(
                name="Kaskean",
                glottolog_id="",
                latitude=0.0,
                longitude=0.0,
                dates=[],
                family_id="",
                parent_id="",
                level="",
                iso_639_3_code="zsk",
                type="a",
            ),
        ),
    ]
)


# Extra languages in-scope for the CLTK but which are not labeled historical or ancient by Glottolog

LANGUAGES["arb"] = Language(
    name="Standard Arabic",
    glottolog_id="stan1318",
    latitude=27.9625,
    longitude=43.8525,
    dates=[],
    family_id="afro1255",
    parent_id="arab1395",
    level="language",
    iso_639_3_code="arb",
    type="",
)

LANGUAGES["ben"] = Language(
    name="Bengali",
    glottolog_id="beng1280",
    latitude=24.0,
    longitude=90.0,
    dates=[],
    family_id="indo1319",
    parent_id="gaud1238",
    level="language",
    iso_639_3_code="ben",
    type="",
)

LANGUAGES["guj"] = Language(
    name="Gujarati",
    glottolog_id="guja1252",
    latitude=22.687,
    longitude=71.0962,
    dates=[],
    family_id="indo1319",
    parent_id="guja1256",
    level="language",
    iso_639_3_code="guj",
    type="",
)

LANGUAGES["hin"] = Language(
    name="Hindi",
    glottolog_id="hind1269",
    latitude=25.0,
    longitude=77.0,
    dates=[],
    family_id="indo1319",
    parent_id="hind1270",
    level="language",
    iso_639_3_code="hin",
    type="",
)

LANGUAGES["kan"] = Language(
    name="Kannada",
    glottolog_id="nucl1305",
    latitude=13.5878,
    longitude=76.1198,
    dates=[],
    family_id="drav1251",
    parent_id="kann1255",
    level="language",
    iso_639_3_code="kan",
    type="",
)

LANGUAGES["ory"] = Language(
    name="Odia",
    glottolog_id="oriy1255",
    latitude=21.0,
    longitude=85.0,
    dates=[],
    family_id="indo1319",
    parent_id="macr1269",
    level="language",
    iso_639_3_code="ory",
    type="",
)

LANGUAGES["pes"] = Language(
    name="Western Farsi",
    glottolog_id="west2369",
    latitude=228.6593,
    longitude=55.1586,
    dates=[],
    family_id="indo1319",
    parent_id="fars1255",
    level="language",
    iso_639_3_code="pes",
    type="",
)

LANGUAGES["tel"] = Language(
    name="Telugu",
    glottolog_id="telu1262",
    latitude=16.4529,
    longitude=78.7024,
    dates=[],
    family_id="drav1251",
    parent_id="telu1265",
    level="language",
    iso_639_3_code="tel",
    type="",
)

LANGUAGES["urd"] = Language(
    name="Urdu",
    glottolog_id="urdu1245",
    latitude=25.0,
    longitude=67.0,
    dates=[],
    family_id="indo1319",
    parent_id="hind1270",
    level="language",
    iso_639_3_code="urd",
    type="",
)

# Note: Using Eastern Panjabi as a stand-in for both it and Western Panjabi (``west2386``, ``pnb``)
LANGUAGES["pan"] = Language(
    name="Eastern Panjabi",
    glottolog_id="panj125",
    latitude=30.0368,
    longitude=75.6702,
    dates=[],
    family_id="indo1319",
    parent_id="east2727",
    level="language",
    iso_639_3_code="pan",
    type="",
)

LANGUAGES["mal"] = Language(
    name="Malayalam",
    glottolog_id="panj125",
    latitude=9.59208,
    longitude=76.7651,
    dates=[],
    family_id="drav1251",
    parent_id="mala1541",
    level="language",
    iso_639_3_code="mal",
    type="",
)

LANGUAGES["cop"] = Language(
    name="Coptic",
    glottolog_id="copt1239",
    latitude=29.472,
    longitude=31.2053,
    dates=[],
    family_id="afro1255",
    parent_id="egyp1245",
    level="language",
    iso_639_3_code="cop",
    type="",
)


def _resort_languages_list(
    languages_list: typing.OrderedDict[str, Language]
) -> typing.OrderedDict[str, Language]:
    """Pick up the LANGUAGES global and return
    alphabetized according to a language's common name.

    >>> iso_dict_keys = _resort_languages_list(LANGUAGES)
    >>> list(iso_dict_keys)[:10]
    ['xae', 'xag', 'akk', 'xln', 'grc', 'hbo', 'xlg', 'xmk', 'xna', 'xzp']
    """
    name_sorted_langs = OrderedDict(
        sorted(languages_list.items(), key=lambda x: x[1].name)
    )
    return name_sorted_langs


LANGUAGES = _resort_languages_list(languages_list=LANGUAGES)
