"use strict";
// stolen from: https://github.com/aws/jsii/blob/main/packages/jsii-pacmak/lib/targets/version-utils.ts
Object.defineProperty(exports, "__esModule", { value: true });
exports.toReleaseVersion = exports.toPythonVersionRange = exports.toNuGetVersionRange = exports.toMavenVersionRange = exports.TargetName = void 0;
const util_1 = require("util");
const semver_1 = require("semver");
var TargetName;
(function (TargetName) {
    TargetName[TargetName["JAVA"] = 0] = "JAVA";
    TargetName[TargetName["DOTNET"] = 1] = "DOTNET";
    TargetName[TargetName["PYTHON"] = 2] = "PYTHON";
    TargetName[TargetName["GO"] = 3] = "GO";
    TargetName[TargetName["JAVASCRIPT"] = 4] = "JAVASCRIPT";
})(TargetName = exports.TargetName || (exports.TargetName = {}));
/**
 * Converts a SemVer range expression to a Maven version range expression.
 *
 * @param semverRange the SemVer range expression to convert.
 * @param suffix      the suffix to add to versions in the range.
 *
 * @see https://cwiki.apache.org/confluence/display/MAVENOLD/Dependency+Mediation+and+Conflict+Resolution
 */
function toMavenVersionRange(semverRange, suffix) {
    return toBracketNotation(semverRange, suffix, {
        semver: false,
        target: TargetName.JAVA,
    });
}
exports.toMavenVersionRange = toMavenVersionRange;
/**
 * Converts a SemVer range expression to a NuGet version range expression.
 *
 * @param semverRange the SemVer range expression to convert.
 *
 * @see https://docs.microsoft.com/en-us/nuget/concepts/package-versioning#version-ranges-and-wildcards
 */
function toNuGetVersionRange(semverRange) {
    return toBracketNotation(semverRange, undefined, {
        semver: false,
        target: TargetName.DOTNET,
    });
}
exports.toNuGetVersionRange = toNuGetVersionRange;
/**
 * Converts a SemVer range expression to a Python setuptools compatible version
 * constraint expression.
 *
 * @param semverRange the SemVer range expression to convert.
 */
function toPythonVersionRange(semverRange) {
    const range = new semver_1.Range(semverRange);
    return range.set
        .map((set) => set
        .map((comp) => {
        const versionId = toReleaseVersion(comp.semver.raw?.replace(/-0$/, "") ?? "0.0.0", TargetName.PYTHON);
        switch (comp.operator) {
            case "":
                // With ^0.0.0, somehow we get a left entry with an empty operator and value, we'll fix this up
                return comp.value === "" ? ">=0.0.0" : `==${versionId}`;
            case "=":
                return `==${versionId}`;
            default:
                // >, >=, <, <= are all valid expressions
                return `${comp.operator}${versionId}`;
        }
    })
        .join(", "))
        .join(", ");
}
exports.toPythonVersionRange = toPythonVersionRange;
/**
 * Converts an original version number from the NPM convention to the target
 * language's convention for expressing the same. For versions that do not
 * include a prerelease identifier, this always returns the assembly version
 * unmodified.
 *
 * @param assemblyVersion the assembly version being released
 * @param target          the target language for which the version is destined
 *
 * @returns the version that should be serialized
 */
function toReleaseVersion(assemblyVersion, target) {
    const version = semver_1.parse(assemblyVersion, { includePrerelease: true });
    if (version == null) {
        throw new Error(`Unable to parse the provided assembly version: "${assemblyVersion}"`);
    }
    if (version.prerelease.length === 0) {
        return assemblyVersion;
    }
    switch (target) {
        case TargetName.PYTHON:
            // Python supports a limited set of identifiers... And we have a mapping table...
            // https://packaging.python.org/guides/distributing-packages-using-setuptools/#pre-release-versioning
            const [label, sequence, ...rest] = version.prerelease;
            if (rest.filter((elt) => elt !== 0).length > 0 || sequence == null) {
                throw new Error(`Unable to map prerelease identifier (in: ${assemblyVersion}) components to python: ${util_1.inspect(version.prerelease)}. The format should be 'X.Y.Z-label.sequence', where sequence is a positive integer, and label is "dev", "pre", "alpha", beta", or "rc"`);
            }
            if (!Number.isInteger(sequence)) {
                throw new Error(`Unable to map prerelease identifier (in: ${assemblyVersion}) to python, as sequence ${util_1.inspect(sequence)} is not an integer`);
            }
            const baseVersion = `${version.major}.${version.minor}.${version.patch}`;
            // See PEP 440: https://www.python.org/dev/peps/pep-0440/#pre-releases
            switch (label) {
                case "dev":
                case "pre":
                    return `${baseVersion}.dev${sequence}`; // PEP 404 see developmental release as X.Y.devN
                case "alpha":
                    return `${baseVersion}a${sequence}`; // PEP 404 see alpha release as X.YaN
                case "beta":
                    return `${baseVersion}b${sequence}`; // PEP 404 see beta release as X.YbN
                case "rc":
                    return `${baseVersion}rc${sequence}`; // PEP 404 see release candidate as X.YrcN
                default:
                    throw new Error(`Unable to map prerelease identifier (in: ${assemblyVersion}) to python, as label ${util_1.inspect(label)} is not mapped (only "dev", "pre", "alpha", "beta" and "rc" are)`);
            }
        case TargetName.DOTNET:
        case TargetName.GO:
        case TargetName.JAVA:
        case TargetName.JAVASCRIPT:
            // Not touching - the NPM version number should be usable as-is
            break;
    }
    return assemblyVersion;
}
exports.toReleaseVersion = toReleaseVersion;
/**
 * Converts a semantic version range to the kind of bracket notation used by
 * Maven and NuGet. For example, this turns `^1.2.3` into `[1.2.3,2.0.0)`.
 *
 * @param semverRange The semantic version range to be converted.
 * @param suffix A version suffix to apply to versions in the resulting expression.
 * @param semver Whether the target supports full semantic versioning (including
 *               `-0` as the lowest possible prerelease identifier)
 *
 * @returns a bracket-notation version range.
 */
function toBracketNotation(semverRange, suffix, { semver = true, target = TargetName.JAVASCRIPT, } = {}) {
    if (semverRange === "*") {
        semverRange = ">=0.0.0";
    }
    const range = new semver_1.Range(semverRange);
    if (semverRange === range.range) {
        return semverRange;
    }
    return range.set
        .map((set) => {
        if (set.length === 1) {
            const version = set[0].semver.raw;
            if (!version && range.raw === ">=0.0.0") {
                // Case where version is '*'
                return "[0.0.0,)";
            }
            switch (set[0].operator || "=") {
                // "[version]" => means exactly version
                case "=":
                    return `[${addSuffix(version)}]`;
                // "(version,]" => means greater than version
                case ">":
                    return `(${addSuffix(version)},)`;
                // "[version,]" => means greater than or equal to that version
                case ">=":
                    return `[${addSuffix(version)},)`;
                // "[,version)" => means less than version
                case "<":
                    return `(,${addSuffix(version, !semver)})`;
                // "[,version]" => means less than or equal to version
                case "<=":
                    return `(,${addSuffix(version)}]`;
            }
        }
        else if (set.length === 2) {
            const nugetRange = toBracketRange(set[0], set[1]);
            if (nugetRange) {
                return nugetRange;
            }
        }
        throw new Error(`Unsupported SemVer range set in ${semverRange}: ${set
            .map((comp) => comp.value)
            .join(", ")}`);
    })
        .join(", ");
    function toBracketRange(left, right) {
        if (left.operator.startsWith("<") && right.operator.startsWith(">")) {
            // Order isn't ideal, swap around..
            [left, right] = [right, left];
        }
        // With ^0.0.0, somehow we get a left entry with an empty operator and value, we'll fix this up
        if (left.operator === "" && left.value === "") {
            left = new semver_1.Comparator(">=0.0.0", left.options);
        }
        if (!left.operator.startsWith(">") || !right.operator.startsWith("<")) {
            // We only support ranges defined like "> (or >=) left, < (or <=) right"
            return undefined;
        }
        const leftBrace = left.operator.endsWith("=") ? "[" : "(";
        const rightBrace = right.operator.endsWith("=") ? "]" : ")";
        return `${leftBrace}${addSuffix(left.semver.raw)},${addSuffix(right.semver.raw, right.operator === "<" && !semver)}${rightBrace}`;
    }
    function addSuffix(str, trimDashZero = false) {
        if (!str) {
            return "";
        }
        if (trimDashZero) {
            str = str.replace(/-0$/, "");
        }
        return suffix ? `${str}${suffix}` : toReleaseVersion(str, target);
    }
}
//# sourceMappingURL=data:application/json;base64,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