"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeTsconfigOptions = exports.TypeScriptLibraryProject = exports.TypeScriptAppProject = exports.TypeScriptProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const semver = require("semver");
const common_1 = require("../common");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const sample_file_1 = require("../sample-file");
const textfile_1 = require("../textfile");
const typescript_1 = require("../typescript");
/**
 * TypeScript project
 * @pjid typescript
 */
class TypeScriptProject extends javascript_1.NodeProject {
    constructor(options) {
        super({
            ...options,
            // disable .projenrc.js if typescript is enabled
            projenrcJs: options.projenrcTs ? false : options.projenrcJs,
            jestOptions: {
                ...options.jestOptions,
                jestConfig: {
                    ...options.jestOptions?.jestConfig,
                    testMatch: [],
                },
            },
        });
        this.srcdir = options.srcdir ?? "src";
        this.libdir = options.libdir ?? "lib";
        this.docgen = options.docgen;
        this.docsDirectory = options.docsDirectory ?? "docs/";
        this.compileTask.exec("tsc --build");
        this.watchTask = this.addTask("watch", {
            description: "Watch & compile in the background",
            exec: "tsc --build -w",
        });
        this.testdir = options.testdir ?? "test";
        this.gitignore.include(`/${this.testdir}/`);
        this.npmignore?.exclude(`/${this.testdir}/`);
        // if the test directory is under `src/`, then we will run our tests against
        // the javascript files and not let jest compile it for us.
        const compiledTests = this.testdir.startsWith(this.srcdir + path.posix.sep);
        if (options.entrypointTypes || this.entrypoint !== "") {
            const entrypointTypes = options.entrypointTypes ??
                `${path
                    .join(path.dirname(this.entrypoint), path.basename(this.entrypoint, ".js"))
                    .replace(/\\/g, "/")}.d.ts`;
            this.package.addField("types", entrypointTypes);
        }
        const compilerOptionDefaults = {
            alwaysStrict: true,
            declaration: true,
            esModuleInterop: true,
            experimentalDecorators: true,
            inlineSourceMap: true,
            inlineSources: true,
            lib: ["es2019"],
            module: "CommonJS",
            noEmitOnError: false,
            noFallthroughCasesInSwitch: true,
            noImplicitAny: true,
            noImplicitReturns: true,
            noImplicitThis: true,
            noUnusedLocals: true,
            noUnusedParameters: true,
            resolveJsonModule: true,
            strict: true,
            strictNullChecks: true,
            strictPropertyInitialization: true,
            stripInternal: true,
            target: "ES2019",
        };
        if (!options.disableTsconfig) {
            this.tsconfig = new javascript_1.TypescriptConfig(this, mergeTsconfigOptions({
                include: [`${this.srcdir}/**/*.ts`],
                // exclude: ['node_modules'], // TODO: shouldn't we exclude node_modules?
                compilerOptions: {
                    rootDir: this.srcdir,
                    outDir: this.libdir,
                    ...compilerOptionDefaults,
                },
            }, options.tsconfig));
        }
        const tsconfigDevFile = options.tsconfigDevFile ?? "tsconfig.dev.json";
        this.tsconfigDev = new javascript_1.TypescriptConfig(this, mergeTsconfigOptions({
            fileName: tsconfigDevFile,
            include: [
                common_1.PROJEN_RC,
                `${this.srcdir}/**/*.ts`,
                `${this.testdir}/**/*.ts`,
            ],
            exclude: ["node_modules"],
            compilerOptions: compilerOptionDefaults,
        }, options.tsconfig, options.tsconfigDev));
        this.gitignore.include(`/${this.srcdir}/`);
        this.npmignore?.exclude(`/${this.srcdir}/`);
        if (this.srcdir !== this.libdir) {
            // separated, can ignore the entire libdir
            this.gitignore.exclude(`/${this.libdir}`);
        }
        else {
            // collocated, can only ignore the compiled output
            this.gitignore.exclude(`/${this.libdir}/**/*.js`);
            this.gitignore.exclude(`/${this.libdir}/**/*.d.ts`);
            this.gitignore.exclude(`/${this.libdir}/**/*.d.ts.map`);
        }
        this.npmignore?.include(`/${this.libdir}/`);
        this.npmignore?.include(`/${this.libdir}/**/*.js`);
        this.npmignore?.include(`/${this.libdir}/**/*.d.ts`);
        this.gitignore.exclude("/dist/");
        this.npmignore?.exclude("dist"); // jsii-pacmak expects this to be "dist" and not "/dist". otherwise it will tamper with it
        this.npmignore?.exclude("/tsconfig.json");
        this.npmignore?.exclude("/.github/");
        this.npmignore?.exclude("/.vscode/");
        this.npmignore?.exclude("/.idea/");
        this.npmignore?.exclude("/.projenrc.js");
        this.npmignore?.exclude("tsconfig.tsbuildinfo");
        if (this.jest) {
            if (compiledTests) {
                this.addJestCompiled(this.jest);
            }
            else {
                this.addJestNoCompile(this.jest);
            }
        }
        const projenrcTypeScript = options.projenrcTs ?? false;
        const projenRcFilename = projenrcTypeScript
            ? options.projenrcTsOptions?.filename ?? ".projenrc.ts"
            : undefined;
        if (options.eslint ?? true) {
            const devdirs = [this.testdir, "build-tools"];
            if (projenrcTypeScript) {
                devdirs.push(options.projenrcTsOptions?.projenCodeDir ?? "projenrc");
            }
            this.eslint = new javascript_1.Eslint(this, {
                tsconfigPath: `./${this.tsconfigDev.fileName}`,
                dirs: [this.srcdir],
                devdirs,
                fileExtensions: [".ts", ".tsx"],
                lintProjenRcFile: projenRcFilename,
                ...options.eslintOptions,
            });
            this.tsconfigEslint = this.tsconfigDev;
        }
        if (projenrcTypeScript) {
            new typescript_1.Projenrc(this, options.projenrcTsOptions);
        }
        const tsver = options.typescriptVersion
            ? `@${options.typescriptVersion}`
            : "";
        this.addDevDeps(`typescript${tsver}`, 
        // @types/node versions numbers match the node runtime versions' major.minor, however, new
        // releases are only created when API changes are included in a node release... We might for
        // example have dependencies that require `node >= 12.22`, but as 12.21 and 12.22 did not
        // include API changes, `@types/node@12.20.x` is the "correct" version to use. As it is not
        // possible to easily determine the correct version to use, we pick up the latest version.
        //
        // Additionally, we default to tracking the 12.x line, as the current earliest LTS release of
        // node is 12.x, so this is what corresponds to the broadest compatibility with supported node
        // runtimes.
        `@types/node@^${semver.major(this.package.minNodeVersion ?? "14.0.0")}`);
        // generate sample code in `src` and `lib` if these directories are empty or non-existent.
        if (options.sampleCode ?? true) {
            new SampleCode(this);
        }
        if (this.docgen) {
            new typescript_1.TypedocDocgen(this);
        }
    }
    /**
     * Tests are compiled to `lib/TESTDIR`, so we don't need jest to compile them
     * for us. just run them directly from javascript.
     */
    addJestCompiled(jest) {
        this.addDevDeps(`@types/jest${jest.jestVersion}`);
        const testout = path.posix.relative(this.srcdir, this.testdir);
        const libtest = path.posix.join(this.libdir, testout);
        const srctest = this.testdir;
        this.npmignore?.exclude(`/${libtest}/`);
        jest.addTestMatch(`**/${libtest}/**/?(*.)+(spec|test).js?(x)`);
        jest.addWatchIgnorePattern(`/${this.srcdir}/`);
        const resolveSnapshotPath = (test, ext) => {
            const fullpath = test.replace(libtest, srctest);
            return path.join(path.dirname(fullpath), "__snapshots__", path.basename(fullpath, ".js") + ".ts" + ext);
        };
        const resolveTestPath = (snap, ext) => {
            const filename = path.basename(snap, ".ts" + ext) + ".js";
            const dir = path.dirname(path.dirname(snap)).replace(srctest, libtest);
            return path.join(dir, filename);
        };
        const resolver = new textfile_1.TextFile(this, path.posix.join(common_1.PROJEN_DIR, "jest-snapshot-resolver.js"));
        if (!resolver.marker) {
            resolver.addLine(`// ${resolver.marker}`);
        }
        resolver.addLine('const path = require("path");');
        resolver.addLine(`const libtest = "${libtest}";`);
        resolver.addLine(`const srctest= "${srctest}";`);
        resolver.addLine("module.exports = {");
        resolver.addLine(`  resolveSnapshotPath: ${resolveSnapshotPath.toString()},`);
        resolver.addLine(`  resolveTestPath: ${resolveTestPath.toString()},`);
        resolver.addLine("  testPathForConsistencyCheck: path.join('some', '__tests__', 'example.test.js')");
        resolver.addLine("};");
        jest.addSnapshotResolver(`./${resolver.path}`);
    }
    addJestNoCompile(jest) {
        this.addDevDeps(`@types/jest${jest.jestVersion}`, `ts-jest${jest.jestVersion}`);
        jest.addTestMatch(`<rootDir>/${this.srcdir}/**/__tests__/**/*.ts?(x)`);
        jest.addTestMatch(`<rootDir>/(${this.testdir}|${this.srcdir})/**/*(*.)@(spec|test).ts?(x)`);
        // add relevant deps
        jest.config.preset = "ts-jest";
        jest.config.globals = {
            "ts-jest": {
                tsconfig: this.tsconfigDev.fileName,
            },
        };
    }
}
exports.TypeScriptProject = TypeScriptProject;
_a = JSII_RTTI_SYMBOL_1;
TypeScriptProject[_a] = { fqn: "projen.typescript.TypeScriptProject", version: "0.60.16" };
class SampleCode extends component_1.Component {
    constructor(project) {
        super(project);
        const srcCode = [
            "export class Hello {",
            "  public sayHello() {",
            "    return 'hello, world!';",
            "  }",
            "}",
        ].join("\n");
        const testCode = [
            "import { Hello } from '../src';",
            "",
            "test('hello', () => {",
            "  expect(new Hello().sayHello()).toBe('hello, world!');",
            "});",
        ].join("\n");
        new sample_file_1.SampleDir(project, project.srcdir, {
            files: {
                "index.ts": srcCode,
            },
        });
        if (project.jest) {
            new sample_file_1.SampleDir(project, project.testdir, {
                files: {
                    "hello.test.ts": testCode,
                },
            });
        }
    }
}
/**
 * TypeScript app.
 *
 * @pjid typescript-app
 */
class TypeScriptAppProject extends TypeScriptProject {
    constructor(options) {
        super({
            allowLibraryDependencies: false,
            releaseWorkflow: false,
            entrypoint: "",
            package: false,
            ...options,
        });
    }
}
exports.TypeScriptAppProject = TypeScriptAppProject;
_b = JSII_RTTI_SYMBOL_1;
TypeScriptAppProject[_b] = { fqn: "projen.typescript.TypeScriptAppProject", version: "0.60.16" };
/**
 * @deprecated use `TypeScriptProject`
 */
class TypeScriptLibraryProject extends TypeScriptProject {
}
exports.TypeScriptLibraryProject = TypeScriptLibraryProject;
_c = JSII_RTTI_SYMBOL_1;
TypeScriptLibraryProject[_c] = { fqn: "projen.typescript.TypeScriptLibraryProject", version: "0.60.16" };
/**
 * @internal
 */
function mergeTsconfigOptions(...options) {
    const definedOptions = options.filter(Boolean);
    return definedOptions.reduce((previous, current) => ({
        ...previous,
        ...current,
        include: [...(previous.include ?? []), ...(current.include ?? [])],
        exclude: [...(previous.exclude ?? []), ...(current.exclude ?? [])],
        compilerOptions: {
            ...previous.compilerOptions,
            ...current.compilerOptions,
        },
    }), { compilerOptions: {} });
}
exports.mergeTsconfigOptions = mergeTsconfigOptions;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidHlwZXNjcmlwdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy90eXBlc2NyaXB0L3R5cGVzY3JpcHQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2QkFBNkI7QUFDN0IsaUNBQWlDO0FBQ2pDLHNDQUFrRDtBQUNsRCw0Q0FBeUM7QUFDekMsOENBU3VCO0FBQ3ZCLGdEQUEyQztBQUUzQywwQ0FBdUM7QUFDdkMsOENBSXVCO0FBc0h2Qjs7O0dBR0c7QUFDSCxNQUFhLGlCQUFrQixTQUFRLHdCQUFXO0lBZ0NoRCxZQUFZLE9BQWlDO1FBQzNDLEtBQUssQ0FBQztZQUNKLEdBQUcsT0FBTztZQUVWLGdEQUFnRDtZQUNoRCxVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUUzRCxXQUFXLEVBQUU7Z0JBQ1gsR0FBRyxPQUFPLENBQUMsV0FBVztnQkFDdEIsVUFBVSxFQUFFO29CQUNWLEdBQUcsT0FBTyxDQUFDLFdBQVcsRUFBRSxVQUFVO29CQUNsQyxTQUFTLEVBQUUsRUFBRTtpQkFDZDthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLE1BQU0sR0FBRyxPQUFPLENBQUMsTUFBTSxJQUFJLEtBQUssQ0FBQztRQUN0QyxJQUFJLENBQUMsTUFBTSxHQUFHLE9BQU8sQ0FBQyxNQUFNLElBQUksS0FBSyxDQUFDO1FBRXRDLElBQUksQ0FBQyxNQUFNLEdBQUcsT0FBTyxDQUFDLE1BQU0sQ0FBQztRQUM3QixJQUFJLENBQUMsYUFBYSxHQUFHLE9BQU8sQ0FBQyxhQUFhLElBQUksT0FBTyxDQUFDO1FBRXRELElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRXJDLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUU7WUFDckMsV0FBVyxFQUFFLG1DQUFtQztZQUNoRCxJQUFJLEVBQUUsZ0JBQWdCO1NBQ3ZCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxPQUFPLEdBQUcsT0FBTyxDQUFDLE9BQU8sSUFBSSxNQUFNLENBQUM7UUFDekMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSSxJQUFJLENBQUMsT0FBTyxHQUFHLENBQUMsQ0FBQztRQUM1QyxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxJQUFJLElBQUksQ0FBQyxPQUFPLEdBQUcsQ0FBQyxDQUFDO1FBRTdDLDRFQUE0RTtRQUM1RSwyREFBMkQ7UUFDM0QsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRTVFLElBQUksT0FBTyxDQUFDLGVBQWUsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLEVBQUUsRUFBRTtZQUNyRCxNQUFNLGVBQWUsR0FDbkIsT0FBTyxDQUFDLGVBQWU7Z0JBQ3ZCLEdBQUcsSUFBSTtxQkFDSixJQUFJLENBQ0gsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQzdCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FDdEM7cUJBQ0EsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsT0FBTyxDQUFDO1lBQ2hDLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxlQUFlLENBQUMsQ0FBQztTQUNqRDtRQUVELE1BQU0sc0JBQXNCLEdBQThCO1lBQ3hELFlBQVksRUFBRSxJQUFJO1lBQ2xCLFdBQVcsRUFBRSxJQUFJO1lBQ2pCLGVBQWUsRUFBRSxJQUFJO1lBQ3JCLHNCQUFzQixFQUFFLElBQUk7WUFDNUIsZUFBZSxFQUFFLElBQUk7WUFDckIsYUFBYSxFQUFFLElBQUk7WUFDbkIsR0FBRyxFQUFFLENBQUMsUUFBUSxDQUFDO1lBQ2YsTUFBTSxFQUFFLFVBQVU7WUFDbEIsYUFBYSxFQUFFLEtBQUs7WUFDcEIsMEJBQTBCLEVBQUUsSUFBSTtZQUNoQyxhQUFhLEVBQUUsSUFBSTtZQUNuQixpQkFBaUIsRUFBRSxJQUFJO1lBQ3ZCLGNBQWMsRUFBRSxJQUFJO1lBQ3BCLGNBQWMsRUFBRSxJQUFJO1lBQ3BCLGtCQUFrQixFQUFFLElBQUk7WUFDeEIsaUJBQWlCLEVBQUUsSUFBSTtZQUN2QixNQUFNLEVBQUUsSUFBSTtZQUNaLGdCQUFnQixFQUFFLElBQUk7WUFDdEIsNEJBQTRCLEVBQUUsSUFBSTtZQUNsQyxhQUFhLEVBQUUsSUFBSTtZQUNuQixNQUFNLEVBQUUsUUFBUTtTQUNqQixDQUFDO1FBRUYsSUFBSSxDQUFDLE9BQU8sQ0FBQyxlQUFlLEVBQUU7WUFDNUIsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLDZCQUFnQixDQUNsQyxJQUFJLEVBQ0osb0JBQW9CLENBQ2xCO2dCQUNFLE9BQU8sRUFBRSxDQUFDLEdBQUcsSUFBSSxDQUFDLE1BQU0sVUFBVSxDQUFDO2dCQUNuQyx5RUFBeUU7Z0JBQ3pFLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUUsSUFBSSxDQUFDLE1BQU07b0JBQ3BCLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTtvQkFDbkIsR0FBRyxzQkFBc0I7aUJBQzFCO2FBQ0YsRUFDRCxPQUFPLENBQUMsUUFBUSxDQUNqQixDQUNGLENBQUM7U0FDSDtRQUVELE1BQU0sZUFBZSxHQUFHLE9BQU8sQ0FBQyxlQUFlLElBQUksbUJBQW1CLENBQUM7UUFDdkUsSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLDZCQUFnQixDQUNyQyxJQUFJLEVBQ0osb0JBQW9CLENBQ2xCO1lBQ0UsUUFBUSxFQUFFLGVBQWU7WUFDekIsT0FBTyxFQUFFO2dCQUNQLGtCQUFTO2dCQUNULEdBQUcsSUFBSSxDQUFDLE1BQU0sVUFBVTtnQkFDeEIsR0FBRyxJQUFJLENBQUMsT0FBTyxVQUFVO2FBQzFCO1lBQ0QsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO1lBQ3pCLGVBQWUsRUFBRSxzQkFBc0I7U0FDeEMsRUFDRCxPQUFPLENBQUMsUUFBUSxFQUNoQixPQUFPLENBQUMsV0FBVyxDQUNwQixDQUNGLENBQUM7UUFFRixJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxJQUFJLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLElBQUksSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7UUFFNUMsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDL0IsMENBQTBDO1lBQzFDLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLElBQUksSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7U0FDM0M7YUFBTTtZQUNMLGtEQUFrRDtZQUNsRCxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxJQUFJLElBQUksQ0FBQyxNQUFNLFVBQVUsQ0FBQyxDQUFDO1lBQ2xELElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLElBQUksSUFBSSxDQUFDLE1BQU0sWUFBWSxDQUFDLENBQUM7WUFDcEQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSSxJQUFJLENBQUMsTUFBTSxnQkFBZ0IsQ0FBQyxDQUFDO1NBQ3pEO1FBRUQsSUFBSSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsSUFBSSxJQUFJLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztRQUU1QyxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxJQUFJLElBQUksQ0FBQyxNQUFNLFVBQVUsQ0FBQyxDQUFDO1FBQ25ELElBQUksQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLElBQUksSUFBSSxDQUFDLE1BQU0sWUFBWSxDQUFDLENBQUM7UUFFckQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDakMsSUFBSSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQywwRkFBMEY7UUFFM0gsSUFBSSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUMxQyxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUNyQyxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUNyQyxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNuQyxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUN6QyxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBRWhELElBQUksSUFBSSxDQUFDLElBQUksRUFBRTtZQUNiLElBQUksYUFBYSxFQUFFO2dCQUNqQixJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUNqQztpQkFBTTtnQkFDTCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQ2xDO1NBQ0Y7UUFFRCxNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDO1FBRXZELE1BQU0sZ0JBQWdCLEdBQUcsa0JBQWtCO1lBQ3pDLENBQUMsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLEVBQUUsUUFBUSxJQUFJLGNBQWM7WUFDdkQsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVkLElBQUksT0FBTyxDQUFDLE1BQU0sSUFBSSxJQUFJLEVBQUU7WUFDMUIsTUFBTSxPQUFPLEdBQUcsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLGFBQWEsQ0FBQyxDQUFDO1lBQzlDLElBQUksa0JBQWtCLEVBQUU7Z0JBQ3RCLE9BQU8sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLGFBQWEsSUFBSSxVQUFVLENBQUMsQ0FBQzthQUN0RTtZQUVELElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxtQkFBTSxDQUFDLElBQUksRUFBRTtnQkFDN0IsWUFBWSxFQUFFLEtBQUssSUFBSSxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUU7Z0JBQzlDLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUM7Z0JBQ25CLE9BQU87Z0JBQ1AsY0FBYyxFQUFFLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQztnQkFDL0IsZ0JBQWdCLEVBQUUsZ0JBQWdCO2dCQUNsQyxHQUFHLE9BQU8sQ0FBQyxhQUFhO2FBQ3pCLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQztTQUN4QztRQUVELElBQUksa0JBQWtCLEVBQUU7WUFDdEIsSUFBSSxxQkFBVSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQztTQUNqRDtRQUVELE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxpQkFBaUI7WUFDckMsQ0FBQyxDQUFDLElBQUksT0FBTyxDQUFDLGlCQUFpQixFQUFFO1lBQ2pDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFFUCxJQUFJLENBQUMsVUFBVSxDQUNiLGFBQWEsS0FBSyxFQUFFO1FBQ3BCLDBGQUEwRjtRQUMxRiw0RkFBNEY7UUFDNUYseUZBQXlGO1FBQ3pGLDJGQUEyRjtRQUMzRiwwRkFBMEY7UUFDMUYsRUFBRTtRQUNGLDZGQUE2RjtRQUM3Riw4RkFBOEY7UUFDOUYsWUFBWTtRQUNaLGdCQUFnQixNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxJQUFJLFFBQVEsQ0FBQyxFQUFFLENBQ3hFLENBQUM7UUFFRiwwRkFBMEY7UUFDMUYsSUFBSSxPQUFPLENBQUMsVUFBVSxJQUFJLElBQUksRUFBRTtZQUM5QixJQUFJLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUN0QjtRQUVELElBQUksSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNmLElBQUksMEJBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUN6QjtJQUNILENBQUM7SUFFRDs7O09BR0c7SUFDSyxlQUFlLENBQUMsSUFBVTtRQUNoQyxJQUFJLENBQUMsVUFBVSxDQUFDLGNBQWMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7UUFFbEQsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDL0QsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN0RCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDO1FBRTdCLElBQUksQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLElBQUksT0FBTyxHQUFHLENBQUMsQ0FBQztRQUN4QyxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sT0FBTyw4QkFBOEIsQ0FBQyxDQUFDO1FBQy9ELElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO1FBRS9DLE1BQU0sbUJBQW1CLEdBQUcsQ0FBQyxJQUFZLEVBQUUsR0FBVyxFQUFFLEVBQUU7WUFDeEQsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDaEQsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUNkLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEVBQ3RCLGVBQWUsRUFDZixJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsR0FBRyxLQUFLLEdBQUcsR0FBRyxDQUM3QyxDQUFDO1FBQ0osQ0FBQyxDQUFDO1FBRUYsTUFBTSxlQUFlLEdBQUcsQ0FBQyxJQUFZLEVBQUUsR0FBVyxFQUFFLEVBQUU7WUFDcEQsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsS0FBSyxHQUFHLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztZQUMxRCxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ3ZFLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDbEMsQ0FBQyxDQUFDO1FBRUYsTUFBTSxRQUFRLEdBQUcsSUFBSSxtQkFBUSxDQUMzQixJQUFJLEVBQ0osSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsbUJBQVUsRUFBRSwyQkFBMkIsQ0FBQyxDQUN6RCxDQUFDO1FBQ0YsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUU7WUFDcEIsUUFBUSxDQUFDLE9BQU8sQ0FBQyxNQUFNLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1NBQzNDO1FBQ0QsUUFBUSxDQUFDLE9BQU8sQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQ2xELFFBQVEsQ0FBQyxPQUFPLENBQUMsb0JBQW9CLE9BQU8sSUFBSSxDQUFDLENBQUM7UUFDbEQsUUFBUSxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsT0FBTyxJQUFJLENBQUMsQ0FBQztRQUNqRCxRQUFRLENBQUMsT0FBTyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDdkMsUUFBUSxDQUFDLE9BQU8sQ0FDZCwwQkFBMEIsbUJBQW1CLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FDNUQsQ0FBQztRQUNGLFFBQVEsQ0FBQyxPQUFPLENBQUMsc0JBQXNCLGVBQWUsQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDdEUsUUFBUSxDQUFDLE9BQU8sQ0FDZCxrRkFBa0YsQ0FDbkYsQ0FBQztRQUNGLFFBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFdkIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssUUFBUSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVPLGdCQUFnQixDQUFDLElBQVU7UUFDakMsSUFBSSxDQUFDLFVBQVUsQ0FDYixjQUFjLElBQUksQ0FBQyxXQUFXLEVBQUUsRUFDaEMsVUFBVSxJQUFJLENBQUMsV0FBVyxFQUFFLENBQzdCLENBQUM7UUFFRixJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsSUFBSSxDQUFDLE1BQU0sMkJBQTJCLENBQUMsQ0FBQztRQUN2RSxJQUFJLENBQUMsWUFBWSxDQUNmLGNBQWMsSUFBSSxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUMsTUFBTSwrQkFBK0IsQ0FDekUsQ0FBQztRQUVGLG9CQUFvQjtRQUNwQixJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxTQUFTLENBQUM7UUFDL0IsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLEdBQUc7WUFDcEIsU0FBUyxFQUFFO2dCQUNULFFBQVEsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVE7YUFDcEM7U0FDRixDQUFDO0lBQ0osQ0FBQzs7QUFqVEgsOENBa1RDOzs7QUFFRCxNQUFNLFVBQVcsU0FBUSxxQkFBUztJQUNoQyxZQUFZLE9BQTBCO1FBQ3BDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNmLE1BQU0sT0FBTyxHQUFHO1lBQ2Qsc0JBQXNCO1lBQ3RCLHVCQUF1QjtZQUN2Qiw2QkFBNkI7WUFDN0IsS0FBSztZQUNMLEdBQUc7U0FDSixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUViLE1BQU0sUUFBUSxHQUFHO1lBQ2YsaUNBQWlDO1lBQ2pDLEVBQUU7WUFDRix1QkFBdUI7WUFDdkIseURBQXlEO1lBQ3pELEtBQUs7U0FDTixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUViLElBQUksdUJBQVMsQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLE1BQU0sRUFBRTtZQUNyQyxLQUFLLEVBQUU7Z0JBQ0wsVUFBVSxFQUFFLE9BQU87YUFDcEI7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLE9BQU8sQ0FBQyxJQUFJLEVBQUU7WUFDaEIsSUFBSSx1QkFBUyxDQUFDLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTyxFQUFFO2dCQUN0QyxLQUFLLEVBQUU7b0JBQ0wsZUFBZSxFQUFFLFFBQVE7aUJBQzFCO2FBQ0YsQ0FBQyxDQUFDO1NBQ0o7SUFDSCxDQUFDO0NBQ0Y7QUFFRDs7OztHQUlHO0FBQ0gsTUFBYSxvQkFBcUIsU0FBUSxpQkFBaUI7SUFDekQsWUFBWSxPQUFpQztRQUMzQyxLQUFLLENBQUM7WUFDSix3QkFBd0IsRUFBRSxLQUFLO1lBQy9CLGVBQWUsRUFBRSxLQUFLO1lBQ3RCLFVBQVUsRUFBRSxFQUFFO1lBQ2QsT0FBTyxFQUFFLEtBQUs7WUFDZCxHQUFHLE9BQU87U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDOztBQVRILG9EQVVDOzs7QUFFRDs7R0FFRztBQUNILE1BQWEsd0JBQXlCLFNBQVEsaUJBQWlCOztBQUEvRCw0REFBa0U7OztBQVFsRTs7R0FFRztBQUNILFNBQWdCLG9CQUFvQixDQUNsQyxHQUFHLE9BQWdEO0lBRW5ELE1BQU0sY0FBYyxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUE4QixDQUFDO0lBQzVFLE9BQU8sY0FBYyxDQUFDLE1BQU0sQ0FDMUIsQ0FBQyxRQUFRLEVBQUUsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3RCLEdBQUcsUUFBUTtRQUNYLEdBQUcsT0FBTztRQUNWLE9BQU8sRUFBRSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ2xFLE9BQU8sRUFBRSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ2xFLGVBQWUsRUFBRTtZQUNmLEdBQUcsUUFBUSxDQUFDLGVBQWU7WUFDM0IsR0FBRyxPQUFPLENBQUMsZUFBZTtTQUMzQjtLQUNGLENBQUMsRUFDRixFQUFFLGVBQWUsRUFBRSxFQUFFLEVBQUUsQ0FDeEIsQ0FBQztBQUNKLENBQUM7QUFqQkQsb0RBaUJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0ICogYXMgc2VtdmVyIGZyb20gXCJzZW12ZXJcIjtcbmltcG9ydCB7IFBST0pFTl9ESVIsIFBST0pFTl9SQyB9IGZyb20gXCIuLi9jb21tb25cIjtcbmltcG9ydCB7IENvbXBvbmVudCB9IGZyb20gXCIuLi9jb21wb25lbnRcIjtcbmltcG9ydCB7XG4gIEVzbGludCxcbiAgRXNsaW50T3B0aW9ucyxcbiAgSmVzdCxcbiAgTm9kZVByb2plY3QsXG4gIE5vZGVQcm9qZWN0T3B0aW9ucyxcbiAgVHlwZVNjcmlwdENvbXBpbGVyT3B0aW9ucyxcbiAgVHlwZXNjcmlwdENvbmZpZyxcbiAgVHlwZXNjcmlwdENvbmZpZ09wdGlvbnMsXG59IGZyb20gXCIuLi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQgeyBTYW1wbGVEaXIgfSBmcm9tIFwiLi4vc2FtcGxlLWZpbGVcIjtcbmltcG9ydCB7IFRhc2sgfSBmcm9tIFwiLi4vdGFza1wiO1xuaW1wb3J0IHsgVGV4dEZpbGUgfSBmcm9tIFwiLi4vdGV4dGZpbGVcIjtcbmltcG9ydCB7XG4gIFByb2plbnJjIGFzIFByb2plbnJjVHMsXG4gIFByb2plbnJjT3B0aW9ucyBhcyBQcm9qZW5yY1RzT3B0aW9ucyxcbiAgVHlwZWRvY0RvY2dlbixcbn0gZnJvbSBcIi4uL3R5cGVzY3JpcHRcIjtcblxuZXhwb3J0IGludGVyZmFjZSBUeXBlU2NyaXB0UHJvamVjdE9wdGlvbnMgZXh0ZW5kcyBOb2RlUHJvamVjdE9wdGlvbnMge1xuICAvKipcbiAgICogVHlwZXNjcmlwdCAgYXJ0aWZhY3RzIG91dHB1dCBkaXJlY3RvcnlcbiAgICpcbiAgICogQGRlZmF1bHQgXCJsaWJcIlxuICAgKi9cbiAgcmVhZG9ubHkgbGliZGlyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUeXBlc2NyaXB0IHNvdXJjZXMgZGlyZWN0b3J5LlxuICAgKlxuICAgKiBAZGVmYXVsdCBcInNyY1wiXG4gICAqL1xuICByZWFkb25seSBzcmNkaXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEplc3QgdGVzdHMgZGlyZWN0b3J5LiBUZXN0cyBmaWxlcyBzaG91bGQgYmUgbmFtZWQgYHh4eC50ZXN0LnRzYC5cbiAgICpcbiAgICogSWYgdGhpcyBkaXJlY3RvcnkgaXMgdW5kZXIgYHNyY2RpcmAgKGUuZy4gYHNyYy90ZXN0YCwgYHNyYy9fX3Rlc3RzX19gKSxcbiAgICogdGhlbiB0ZXN0cyBhcmUgZ29pbmcgdG8gYmUgY29tcGlsZWQgaW50byBgbGliL2AgYW5kIGV4ZWN1dGVkIGFzIGphdmFzY3JpcHQuXG4gICAqIElmIHRoZSB0ZXN0IGRpcmVjdG9yeSBpcyBvdXRzaWRlIG9mIGBzcmNgLCB0aGVuIHdlIGNvbmZpZ3VyZSBqZXN0IHRvXG4gICAqIGNvbXBpbGUgdGhlIGNvZGUgaW4tbWVtb3J5LlxuICAgKlxuICAgKiBAZGVmYXVsdCBcInRlc3RcIlxuICAgKi9cbiAgcmVhZG9ubHkgdGVzdGRpcj86IHN0cmluZztcblxuICAvKipcbiAgICogU2V0dXAgZXNsaW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBlc2xpbnQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBFc2xpbnQgb3B0aW9uc1xuICAgKiBAZGVmYXVsdCAtIG9waW5pb25hdGVkIGRlZmF1bHQgb3B0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgZXNsaW50T3B0aW9ucz86IEVzbGludE9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIFR5cGVTY3JpcHQgdmVyc2lvbiB0byB1c2UuXG4gICAqXG4gICAqIE5PVEU6IFR5cGVzY3JpcHQgaXMgbm90IHNlbWFudGljYWxseSB2ZXJzaW9uZWQgYW5kIHNob3VsZCByZW1haW4gb24gdGhlXG4gICAqIHNhbWUgbWlub3IsIHNvIHdlIHJlY29tbWVuZCB1c2luZyBhIGB+YCBkZXBlbmRlbmN5IChlLmcuIGB+MS4yLjNgKS5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJsYXRlc3RcIlxuICAgKi9cbiAgcmVhZG9ubHkgdHlwZXNjcmlwdFZlcnNpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERvY2dlbiBieSBUeXBlZG9jXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBkb2NnZW4/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBEb2NzIGRpcmVjdG9yeVxuICAgKlxuICAgKiBAZGVmYXVsdCBcImRvY3NcIlxuICAgKi9cbiAgcmVhZG9ubHkgZG9jc0RpcmVjdG9yeT86IHN0cmluZztcblxuICAvKipcbiAgICogQ3VzdG9tIFRTQ29uZmlnXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSB0c2NvbmZpZz86IFR5cGVzY3JpcHRDb25maWdPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBDdXN0b20gdHNjb25maWcgb3B0aW9ucyBmb3IgdGhlIGRldmVsb3BtZW50IHRzY29uZmlnLmpzb24gZmlsZSAodXNlZCBmb3IgdGVzdGluZykuXG4gICAqIEBkZWZhdWx0IC0gdXNlIHRoZSBwcm9kdWN0aW9uIHRzY29uZmlnIG9wdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IHRzY29uZmlnRGV2PzogVHlwZXNjcmlwdENvbmZpZ09wdGlvbnM7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBkZXZlbG9wbWVudCB0c2NvbmZpZy5qc29uIGZpbGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwidHNjb25maWcuZGV2Lmpzb25cIlxuICAgKi9cbiAgcmVhZG9ubHkgdHNjb25maWdEZXZGaWxlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEbyBub3QgZ2VuZXJhdGUgYSBgdHNjb25maWcuanNvbmAgZmlsZSAodXNlZCBieSBqc2lpIHByb2plY3RzIHNpbmNlXG4gICAqIHRzY29uZmlnLmpzb24gaXMgZ2VuZXJhdGVkIGJ5IHRoZSBqc2lpIGNvbXBpbGVyKS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGRpc2FibGVUc2NvbmZpZz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIG9uZS10aW1lIHNhbXBsZSBpbiBgc3JjL2AgYW5kIGB0ZXN0L2AgaWYgdGhlcmUgYXJlIG5vIGZpbGVzIHRoZXJlLlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBzYW1wbGVDb2RlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIC5kLnRzIGZpbGUgdGhhdCBpbmNsdWRlcyB0aGUgdHlwZSBkZWNsYXJhdGlvbnMgZm9yIHRoaXMgbW9kdWxlLlxuICAgKiBAZGVmYXVsdCAtIC5kLnRzIGZpbGUgZGVyaXZlZCBmcm9tIHRoZSBwcm9qZWN0J3MgZW50cnlwb2ludCAodXN1YWxseSBsaWIvaW5kZXguZC50cylcbiAgICovXG4gIHJlYWRvbmx5IGVudHJ5cG9pbnRUeXBlcz86IHN0cmluZztcblxuICAvKipcbiAgICogVXNlIFR5cGVTY3JpcHQgZm9yIHlvdXIgcHJvamVucmMgZmlsZSAoYC5wcm9qZW5yYy50c2ApLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVucmNUcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIC5wcm9qZW5yYy50c1xuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVucmNUc09wdGlvbnM/OiBQcm9qZW5yY1RzT3B0aW9ucztcbn1cblxuLyoqXG4gKiBUeXBlU2NyaXB0IHByb2plY3RcbiAqIEBwamlkIHR5cGVzY3JpcHRcbiAqL1xuZXhwb3J0IGNsYXNzIFR5cGVTY3JpcHRQcm9qZWN0IGV4dGVuZHMgTm9kZVByb2plY3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgZG9jZ2VuPzogYm9vbGVhbjtcbiAgcHVibGljIHJlYWRvbmx5IGRvY3NEaXJlY3Rvcnk6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGVzbGludD86IEVzbGludDtcbiAgcHVibGljIHJlYWRvbmx5IHRzY29uZmlnRXNsaW50PzogVHlwZXNjcmlwdENvbmZpZztcbiAgcHVibGljIHJlYWRvbmx5IHRzY29uZmlnPzogVHlwZXNjcmlwdENvbmZpZztcblxuICAvKipcbiAgICogQSB0eXBlc2NyaXB0IGNvbmZpZ3VyYXRpb24gZmlsZSB3aGljaCBjb3ZlcnMgYWxsIGZpbGVzIChzb3VyY2VzLCB0ZXN0cywgcHJvamVuKS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB0c2NvbmZpZ0RldjogVHlwZXNjcmlwdENvbmZpZztcblxuICAvKipcbiAgICogVGhlIGRpcmVjdG9yeSBpbiB3aGljaCB0aGUgLnRzIHNvdXJjZXMgcmVzaWRlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNyY2Rpcjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZGlyZWN0b3J5IGluIHdoaWNoIGNvbXBpbGVkIC5qcyBmaWxlcyByZXNpZGUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbGliZGlyOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBkaXJlY3RvcnkgaW4gd2hpY2ggdGVzdHMgcmVzaWRlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHRlc3RkaXI6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIFwid2F0Y2hcIiB0YXNrLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHdhdGNoVGFzazogVGFzaztcblxuICBjb25zdHJ1Y3RvcihvcHRpb25zOiBUeXBlU2NyaXB0UHJvamVjdE9wdGlvbnMpIHtcbiAgICBzdXBlcih7XG4gICAgICAuLi5vcHRpb25zLFxuXG4gICAgICAvLyBkaXNhYmxlIC5wcm9qZW5yYy5qcyBpZiB0eXBlc2NyaXB0IGlzIGVuYWJsZWRcbiAgICAgIHByb2plbnJjSnM6IG9wdGlvbnMucHJvamVucmNUcyA/IGZhbHNlIDogb3B0aW9ucy5wcm9qZW5yY0pzLFxuXG4gICAgICBqZXN0T3B0aW9uczoge1xuICAgICAgICAuLi5vcHRpb25zLmplc3RPcHRpb25zLFxuICAgICAgICBqZXN0Q29uZmlnOiB7XG4gICAgICAgICAgLi4ub3B0aW9ucy5qZXN0T3B0aW9ucz8uamVzdENvbmZpZyxcbiAgICAgICAgICB0ZXN0TWF0Y2g6IFtdLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMuc3JjZGlyID0gb3B0aW9ucy5zcmNkaXIgPz8gXCJzcmNcIjtcbiAgICB0aGlzLmxpYmRpciA9IG9wdGlvbnMubGliZGlyID8/IFwibGliXCI7XG5cbiAgICB0aGlzLmRvY2dlbiA9IG9wdGlvbnMuZG9jZ2VuO1xuICAgIHRoaXMuZG9jc0RpcmVjdG9yeSA9IG9wdGlvbnMuZG9jc0RpcmVjdG9yeSA/PyBcImRvY3MvXCI7XG5cbiAgICB0aGlzLmNvbXBpbGVUYXNrLmV4ZWMoXCJ0c2MgLS1idWlsZFwiKTtcblxuICAgIHRoaXMud2F0Y2hUYXNrID0gdGhpcy5hZGRUYXNrKFwid2F0Y2hcIiwge1xuICAgICAgZGVzY3JpcHRpb246IFwiV2F0Y2ggJiBjb21waWxlIGluIHRoZSBiYWNrZ3JvdW5kXCIsXG4gICAgICBleGVjOiBcInRzYyAtLWJ1aWxkIC13XCIsXG4gICAgfSk7XG5cbiAgICB0aGlzLnRlc3RkaXIgPSBvcHRpb25zLnRlc3RkaXIgPz8gXCJ0ZXN0XCI7XG4gICAgdGhpcy5naXRpZ25vcmUuaW5jbHVkZShgLyR7dGhpcy50ZXN0ZGlyfS9gKTtcbiAgICB0aGlzLm5wbWlnbm9yZT8uZXhjbHVkZShgLyR7dGhpcy50ZXN0ZGlyfS9gKTtcblxuICAgIC8vIGlmIHRoZSB0ZXN0IGRpcmVjdG9yeSBpcyB1bmRlciBgc3JjL2AsIHRoZW4gd2Ugd2lsbCBydW4gb3VyIHRlc3RzIGFnYWluc3RcbiAgICAvLyB0aGUgamF2YXNjcmlwdCBmaWxlcyBhbmQgbm90IGxldCBqZXN0IGNvbXBpbGUgaXQgZm9yIHVzLlxuICAgIGNvbnN0IGNvbXBpbGVkVGVzdHMgPSB0aGlzLnRlc3RkaXIuc3RhcnRzV2l0aCh0aGlzLnNyY2RpciArIHBhdGgucG9zaXguc2VwKTtcblxuICAgIGlmIChvcHRpb25zLmVudHJ5cG9pbnRUeXBlcyB8fCB0aGlzLmVudHJ5cG9pbnQgIT09IFwiXCIpIHtcbiAgICAgIGNvbnN0IGVudHJ5cG9pbnRUeXBlcyA9XG4gICAgICAgIG9wdGlvbnMuZW50cnlwb2ludFR5cGVzID8/XG4gICAgICAgIGAke3BhdGhcbiAgICAgICAgICAuam9pbihcbiAgICAgICAgICAgIHBhdGguZGlybmFtZSh0aGlzLmVudHJ5cG9pbnQpLFxuICAgICAgICAgICAgcGF0aC5iYXNlbmFtZSh0aGlzLmVudHJ5cG9pbnQsIFwiLmpzXCIpXG4gICAgICAgICAgKVxuICAgICAgICAgIC5yZXBsYWNlKC9cXFxcL2csIFwiL1wiKX0uZC50c2A7XG4gICAgICB0aGlzLnBhY2thZ2UuYWRkRmllbGQoXCJ0eXBlc1wiLCBlbnRyeXBvaW50VHlwZXMpO1xuICAgIH1cblxuICAgIGNvbnN0IGNvbXBpbGVyT3B0aW9uRGVmYXVsdHM6IFR5cGVTY3JpcHRDb21waWxlck9wdGlvbnMgPSB7XG4gICAgICBhbHdheXNTdHJpY3Q6IHRydWUsXG4gICAgICBkZWNsYXJhdGlvbjogdHJ1ZSxcbiAgICAgIGVzTW9kdWxlSW50ZXJvcDogdHJ1ZSxcbiAgICAgIGV4cGVyaW1lbnRhbERlY29yYXRvcnM6IHRydWUsXG4gICAgICBpbmxpbmVTb3VyY2VNYXA6IHRydWUsXG4gICAgICBpbmxpbmVTb3VyY2VzOiB0cnVlLFxuICAgICAgbGliOiBbXCJlczIwMTlcIl0sXG4gICAgICBtb2R1bGU6IFwiQ29tbW9uSlNcIixcbiAgICAgIG5vRW1pdE9uRXJyb3I6IGZhbHNlLFxuICAgICAgbm9GYWxsdGhyb3VnaENhc2VzSW5Td2l0Y2g6IHRydWUsXG4gICAgICBub0ltcGxpY2l0QW55OiB0cnVlLFxuICAgICAgbm9JbXBsaWNpdFJldHVybnM6IHRydWUsXG4gICAgICBub0ltcGxpY2l0VGhpczogdHJ1ZSxcbiAgICAgIG5vVW51c2VkTG9jYWxzOiB0cnVlLFxuICAgICAgbm9VbnVzZWRQYXJhbWV0ZXJzOiB0cnVlLFxuICAgICAgcmVzb2x2ZUpzb25Nb2R1bGU6IHRydWUsXG4gICAgICBzdHJpY3Q6IHRydWUsXG4gICAgICBzdHJpY3ROdWxsQ2hlY2tzOiB0cnVlLFxuICAgICAgc3RyaWN0UHJvcGVydHlJbml0aWFsaXphdGlvbjogdHJ1ZSxcbiAgICAgIHN0cmlwSW50ZXJuYWw6IHRydWUsXG4gICAgICB0YXJnZXQ6IFwiRVMyMDE5XCIsXG4gICAgfTtcblxuICAgIGlmICghb3B0aW9ucy5kaXNhYmxlVHNjb25maWcpIHtcbiAgICAgIHRoaXMudHNjb25maWcgPSBuZXcgVHlwZXNjcmlwdENvbmZpZyhcbiAgICAgICAgdGhpcyxcbiAgICAgICAgbWVyZ2VUc2NvbmZpZ09wdGlvbnMoXG4gICAgICAgICAge1xuICAgICAgICAgICAgaW5jbHVkZTogW2Ake3RoaXMuc3JjZGlyfS8qKi8qLnRzYF0sXG4gICAgICAgICAgICAvLyBleGNsdWRlOiBbJ25vZGVfbW9kdWxlcyddLCAvLyBUT0RPOiBzaG91bGRuJ3Qgd2UgZXhjbHVkZSBub2RlX21vZHVsZXM/XG4gICAgICAgICAgICBjb21waWxlck9wdGlvbnM6IHtcbiAgICAgICAgICAgICAgcm9vdERpcjogdGhpcy5zcmNkaXIsXG4gICAgICAgICAgICAgIG91dERpcjogdGhpcy5saWJkaXIsXG4gICAgICAgICAgICAgIC4uLmNvbXBpbGVyT3B0aW9uRGVmYXVsdHMsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgb3B0aW9ucy50c2NvbmZpZ1xuICAgICAgICApXG4gICAgICApO1xuICAgIH1cblxuICAgIGNvbnN0IHRzY29uZmlnRGV2RmlsZSA9IG9wdGlvbnMudHNjb25maWdEZXZGaWxlID8/IFwidHNjb25maWcuZGV2Lmpzb25cIjtcbiAgICB0aGlzLnRzY29uZmlnRGV2ID0gbmV3IFR5cGVzY3JpcHRDb25maWcoXG4gICAgICB0aGlzLFxuICAgICAgbWVyZ2VUc2NvbmZpZ09wdGlvbnMoXG4gICAgICAgIHtcbiAgICAgICAgICBmaWxlTmFtZTogdHNjb25maWdEZXZGaWxlLFxuICAgICAgICAgIGluY2x1ZGU6IFtcbiAgICAgICAgICAgIFBST0pFTl9SQyxcbiAgICAgICAgICAgIGAke3RoaXMuc3JjZGlyfS8qKi8qLnRzYCxcbiAgICAgICAgICAgIGAke3RoaXMudGVzdGRpcn0vKiovKi50c2AsXG4gICAgICAgICAgXSxcbiAgICAgICAgICBleGNsdWRlOiBbXCJub2RlX21vZHVsZXNcIl0sXG4gICAgICAgICAgY29tcGlsZXJPcHRpb25zOiBjb21waWxlck9wdGlvbkRlZmF1bHRzLFxuICAgICAgICB9LFxuICAgICAgICBvcHRpb25zLnRzY29uZmlnLFxuICAgICAgICBvcHRpb25zLnRzY29uZmlnRGV2XG4gICAgICApXG4gICAgKTtcblxuICAgIHRoaXMuZ2l0aWdub3JlLmluY2x1ZGUoYC8ke3RoaXMuc3JjZGlyfS9gKTtcbiAgICB0aGlzLm5wbWlnbm9yZT8uZXhjbHVkZShgLyR7dGhpcy5zcmNkaXJ9L2ApO1xuXG4gICAgaWYgKHRoaXMuc3JjZGlyICE9PSB0aGlzLmxpYmRpcikge1xuICAgICAgLy8gc2VwYXJhdGVkLCBjYW4gaWdub3JlIHRoZSBlbnRpcmUgbGliZGlyXG4gICAgICB0aGlzLmdpdGlnbm9yZS5leGNsdWRlKGAvJHt0aGlzLmxpYmRpcn1gKTtcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gY29sbG9jYXRlZCwgY2FuIG9ubHkgaWdub3JlIHRoZSBjb21waWxlZCBvdXRwdXRcbiAgICAgIHRoaXMuZ2l0aWdub3JlLmV4Y2x1ZGUoYC8ke3RoaXMubGliZGlyfS8qKi8qLmpzYCk7XG4gICAgICB0aGlzLmdpdGlnbm9yZS5leGNsdWRlKGAvJHt0aGlzLmxpYmRpcn0vKiovKi5kLnRzYCk7XG4gICAgICB0aGlzLmdpdGlnbm9yZS5leGNsdWRlKGAvJHt0aGlzLmxpYmRpcn0vKiovKi5kLnRzLm1hcGApO1xuICAgIH1cblxuICAgIHRoaXMubnBtaWdub3JlPy5pbmNsdWRlKGAvJHt0aGlzLmxpYmRpcn0vYCk7XG5cbiAgICB0aGlzLm5wbWlnbm9yZT8uaW5jbHVkZShgLyR7dGhpcy5saWJkaXJ9LyoqLyouanNgKTtcbiAgICB0aGlzLm5wbWlnbm9yZT8uaW5jbHVkZShgLyR7dGhpcy5saWJkaXJ9LyoqLyouZC50c2ApO1xuXG4gICAgdGhpcy5naXRpZ25vcmUuZXhjbHVkZShcIi9kaXN0L1wiKTtcbiAgICB0aGlzLm5wbWlnbm9yZT8uZXhjbHVkZShcImRpc3RcIik7IC8vIGpzaWktcGFjbWFrIGV4cGVjdHMgdGhpcyB0byBiZSBcImRpc3RcIiBhbmQgbm90IFwiL2Rpc3RcIi4gb3RoZXJ3aXNlIGl0IHdpbGwgdGFtcGVyIHdpdGggaXRcblxuICAgIHRoaXMubnBtaWdub3JlPy5leGNsdWRlKFwiL3RzY29uZmlnLmpzb25cIik7XG4gICAgdGhpcy5ucG1pZ25vcmU/LmV4Y2x1ZGUoXCIvLmdpdGh1Yi9cIik7XG4gICAgdGhpcy5ucG1pZ25vcmU/LmV4Y2x1ZGUoXCIvLnZzY29kZS9cIik7XG4gICAgdGhpcy5ucG1pZ25vcmU/LmV4Y2x1ZGUoXCIvLmlkZWEvXCIpO1xuICAgIHRoaXMubnBtaWdub3JlPy5leGNsdWRlKFwiLy5wcm9qZW5yYy5qc1wiKTtcbiAgICB0aGlzLm5wbWlnbm9yZT8uZXhjbHVkZShcInRzY29uZmlnLnRzYnVpbGRpbmZvXCIpO1xuXG4gICAgaWYgKHRoaXMuamVzdCkge1xuICAgICAgaWYgKGNvbXBpbGVkVGVzdHMpIHtcbiAgICAgICAgdGhpcy5hZGRKZXN0Q29tcGlsZWQodGhpcy5qZXN0KTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRoaXMuYWRkSmVzdE5vQ29tcGlsZSh0aGlzLmplc3QpO1xuICAgICAgfVxuICAgIH1cblxuICAgIGNvbnN0IHByb2plbnJjVHlwZVNjcmlwdCA9IG9wdGlvbnMucHJvamVucmNUcyA/PyBmYWxzZTtcblxuICAgIGNvbnN0IHByb2plblJjRmlsZW5hbWUgPSBwcm9qZW5yY1R5cGVTY3JpcHRcbiAgICAgID8gb3B0aW9ucy5wcm9qZW5yY1RzT3B0aW9ucz8uZmlsZW5hbWUgPz8gXCIucHJvamVucmMudHNcIlxuICAgICAgOiB1bmRlZmluZWQ7XG5cbiAgICBpZiAob3B0aW9ucy5lc2xpbnQgPz8gdHJ1ZSkge1xuICAgICAgY29uc3QgZGV2ZGlycyA9IFt0aGlzLnRlc3RkaXIsIFwiYnVpbGQtdG9vbHNcIl07XG4gICAgICBpZiAocHJvamVucmNUeXBlU2NyaXB0KSB7XG4gICAgICAgIGRldmRpcnMucHVzaChvcHRpb25zLnByb2plbnJjVHNPcHRpb25zPy5wcm9qZW5Db2RlRGlyID8/IFwicHJvamVucmNcIik7XG4gICAgICB9XG5cbiAgICAgIHRoaXMuZXNsaW50ID0gbmV3IEVzbGludCh0aGlzLCB7XG4gICAgICAgIHRzY29uZmlnUGF0aDogYC4vJHt0aGlzLnRzY29uZmlnRGV2LmZpbGVOYW1lfWAsXG4gICAgICAgIGRpcnM6IFt0aGlzLnNyY2Rpcl0sXG4gICAgICAgIGRldmRpcnMsXG4gICAgICAgIGZpbGVFeHRlbnNpb25zOiBbXCIudHNcIiwgXCIudHN4XCJdLFxuICAgICAgICBsaW50UHJvamVuUmNGaWxlOiBwcm9qZW5SY0ZpbGVuYW1lLFxuICAgICAgICAuLi5vcHRpb25zLmVzbGludE9wdGlvbnMsXG4gICAgICB9KTtcblxuICAgICAgdGhpcy50c2NvbmZpZ0VzbGludCA9IHRoaXMudHNjb25maWdEZXY7XG4gICAgfVxuXG4gICAgaWYgKHByb2plbnJjVHlwZVNjcmlwdCkge1xuICAgICAgbmV3IFByb2plbnJjVHModGhpcywgb3B0aW9ucy5wcm9qZW5yY1RzT3B0aW9ucyk7XG4gICAgfVxuXG4gICAgY29uc3QgdHN2ZXIgPSBvcHRpb25zLnR5cGVzY3JpcHRWZXJzaW9uXG4gICAgICA/IGBAJHtvcHRpb25zLnR5cGVzY3JpcHRWZXJzaW9ufWBcbiAgICAgIDogXCJcIjtcblxuICAgIHRoaXMuYWRkRGV2RGVwcyhcbiAgICAgIGB0eXBlc2NyaXB0JHt0c3Zlcn1gLFxuICAgICAgLy8gQHR5cGVzL25vZGUgdmVyc2lvbnMgbnVtYmVycyBtYXRjaCB0aGUgbm9kZSBydW50aW1lIHZlcnNpb25zJyBtYWpvci5taW5vciwgaG93ZXZlciwgbmV3XG4gICAgICAvLyByZWxlYXNlcyBhcmUgb25seSBjcmVhdGVkIHdoZW4gQVBJIGNoYW5nZXMgYXJlIGluY2x1ZGVkIGluIGEgbm9kZSByZWxlYXNlLi4uIFdlIG1pZ2h0IGZvclxuICAgICAgLy8gZXhhbXBsZSBoYXZlIGRlcGVuZGVuY2llcyB0aGF0IHJlcXVpcmUgYG5vZGUgPj0gMTIuMjJgLCBidXQgYXMgMTIuMjEgYW5kIDEyLjIyIGRpZCBub3RcbiAgICAgIC8vIGluY2x1ZGUgQVBJIGNoYW5nZXMsIGBAdHlwZXMvbm9kZUAxMi4yMC54YCBpcyB0aGUgXCJjb3JyZWN0XCIgdmVyc2lvbiB0byB1c2UuIEFzIGl0IGlzIG5vdFxuICAgICAgLy8gcG9zc2libGUgdG8gZWFzaWx5IGRldGVybWluZSB0aGUgY29ycmVjdCB2ZXJzaW9uIHRvIHVzZSwgd2UgcGljayB1cCB0aGUgbGF0ZXN0IHZlcnNpb24uXG4gICAgICAvL1xuICAgICAgLy8gQWRkaXRpb25hbGx5LCB3ZSBkZWZhdWx0IHRvIHRyYWNraW5nIHRoZSAxMi54IGxpbmUsIGFzIHRoZSBjdXJyZW50IGVhcmxpZXN0IExUUyByZWxlYXNlIG9mXG4gICAgICAvLyBub2RlIGlzIDEyLngsIHNvIHRoaXMgaXMgd2hhdCBjb3JyZXNwb25kcyB0byB0aGUgYnJvYWRlc3QgY29tcGF0aWJpbGl0eSB3aXRoIHN1cHBvcnRlZCBub2RlXG4gICAgICAvLyBydW50aW1lcy5cbiAgICAgIGBAdHlwZXMvbm9kZUBeJHtzZW12ZXIubWFqb3IodGhpcy5wYWNrYWdlLm1pbk5vZGVWZXJzaW9uID8/IFwiMTQuMC4wXCIpfWBcbiAgICApO1xuXG4gICAgLy8gZ2VuZXJhdGUgc2FtcGxlIGNvZGUgaW4gYHNyY2AgYW5kIGBsaWJgIGlmIHRoZXNlIGRpcmVjdG9yaWVzIGFyZSBlbXB0eSBvciBub24tZXhpc3RlbnQuXG4gICAgaWYgKG9wdGlvbnMuc2FtcGxlQ29kZSA/PyB0cnVlKSB7XG4gICAgICBuZXcgU2FtcGxlQ29kZSh0aGlzKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5kb2NnZW4pIHtcbiAgICAgIG5ldyBUeXBlZG9jRG9jZ2VuKHRoaXMpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBUZXN0cyBhcmUgY29tcGlsZWQgdG8gYGxpYi9URVNURElSYCwgc28gd2UgZG9uJ3QgbmVlZCBqZXN0IHRvIGNvbXBpbGUgdGhlbVxuICAgKiBmb3IgdXMuIGp1c3QgcnVuIHRoZW0gZGlyZWN0bHkgZnJvbSBqYXZhc2NyaXB0LlxuICAgKi9cbiAgcHJpdmF0ZSBhZGRKZXN0Q29tcGlsZWQoamVzdDogSmVzdCkge1xuICAgIHRoaXMuYWRkRGV2RGVwcyhgQHR5cGVzL2plc3Qke2plc3QuamVzdFZlcnNpb259YCk7XG5cbiAgICBjb25zdCB0ZXN0b3V0ID0gcGF0aC5wb3NpeC5yZWxhdGl2ZSh0aGlzLnNyY2RpciwgdGhpcy50ZXN0ZGlyKTtcbiAgICBjb25zdCBsaWJ0ZXN0ID0gcGF0aC5wb3NpeC5qb2luKHRoaXMubGliZGlyLCB0ZXN0b3V0KTtcbiAgICBjb25zdCBzcmN0ZXN0ID0gdGhpcy50ZXN0ZGlyO1xuXG4gICAgdGhpcy5ucG1pZ25vcmU/LmV4Y2x1ZGUoYC8ke2xpYnRlc3R9L2ApO1xuICAgIGplc3QuYWRkVGVzdE1hdGNoKGAqKi8ke2xpYnRlc3R9LyoqLz8oKi4pKyhzcGVjfHRlc3QpLmpzPyh4KWApO1xuICAgIGplc3QuYWRkV2F0Y2hJZ25vcmVQYXR0ZXJuKGAvJHt0aGlzLnNyY2Rpcn0vYCk7XG5cbiAgICBjb25zdCByZXNvbHZlU25hcHNob3RQYXRoID0gKHRlc3Q6IHN0cmluZywgZXh0OiBzdHJpbmcpID0+IHtcbiAgICAgIGNvbnN0IGZ1bGxwYXRoID0gdGVzdC5yZXBsYWNlKGxpYnRlc3QsIHNyY3Rlc3QpO1xuICAgICAgcmV0dXJuIHBhdGguam9pbihcbiAgICAgICAgcGF0aC5kaXJuYW1lKGZ1bGxwYXRoKSxcbiAgICAgICAgXCJfX3NuYXBzaG90c19fXCIsXG4gICAgICAgIHBhdGguYmFzZW5hbWUoZnVsbHBhdGgsIFwiLmpzXCIpICsgXCIudHNcIiArIGV4dFxuICAgICAgKTtcbiAgICB9O1xuXG4gICAgY29uc3QgcmVzb2x2ZVRlc3RQYXRoID0gKHNuYXA6IHN0cmluZywgZXh0OiBzdHJpbmcpID0+IHtcbiAgICAgIGNvbnN0IGZpbGVuYW1lID0gcGF0aC5iYXNlbmFtZShzbmFwLCBcIi50c1wiICsgZXh0KSArIFwiLmpzXCI7XG4gICAgICBjb25zdCBkaXIgPSBwYXRoLmRpcm5hbWUocGF0aC5kaXJuYW1lKHNuYXApKS5yZXBsYWNlKHNyY3Rlc3QsIGxpYnRlc3QpO1xuICAgICAgcmV0dXJuIHBhdGguam9pbihkaXIsIGZpbGVuYW1lKTtcbiAgICB9O1xuXG4gICAgY29uc3QgcmVzb2x2ZXIgPSBuZXcgVGV4dEZpbGUoXG4gICAgICB0aGlzLFxuICAgICAgcGF0aC5wb3NpeC5qb2luKFBST0pFTl9ESVIsIFwiamVzdC1zbmFwc2hvdC1yZXNvbHZlci5qc1wiKVxuICAgICk7XG4gICAgaWYgKCFyZXNvbHZlci5tYXJrZXIpIHtcbiAgICAgIHJlc29sdmVyLmFkZExpbmUoYC8vICR7cmVzb2x2ZXIubWFya2VyfWApO1xuICAgIH1cbiAgICByZXNvbHZlci5hZGRMaW5lKCdjb25zdCBwYXRoID0gcmVxdWlyZShcInBhdGhcIik7Jyk7XG4gICAgcmVzb2x2ZXIuYWRkTGluZShgY29uc3QgbGlidGVzdCA9IFwiJHtsaWJ0ZXN0fVwiO2ApO1xuICAgIHJlc29sdmVyLmFkZExpbmUoYGNvbnN0IHNyY3Rlc3Q9IFwiJHtzcmN0ZXN0fVwiO2ApO1xuICAgIHJlc29sdmVyLmFkZExpbmUoXCJtb2R1bGUuZXhwb3J0cyA9IHtcIik7XG4gICAgcmVzb2x2ZXIuYWRkTGluZShcbiAgICAgIGAgIHJlc29sdmVTbmFwc2hvdFBhdGg6ICR7cmVzb2x2ZVNuYXBzaG90UGF0aC50b1N0cmluZygpfSxgXG4gICAgKTtcbiAgICByZXNvbHZlci5hZGRMaW5lKGAgIHJlc29sdmVUZXN0UGF0aDogJHtyZXNvbHZlVGVzdFBhdGgudG9TdHJpbmcoKX0sYCk7XG4gICAgcmVzb2x2ZXIuYWRkTGluZShcbiAgICAgIFwiICB0ZXN0UGF0aEZvckNvbnNpc3RlbmN5Q2hlY2s6IHBhdGguam9pbignc29tZScsICdfX3Rlc3RzX18nLCAnZXhhbXBsZS50ZXN0LmpzJylcIlxuICAgICk7XG4gICAgcmVzb2x2ZXIuYWRkTGluZShcIn07XCIpO1xuXG4gICAgamVzdC5hZGRTbmFwc2hvdFJlc29sdmVyKGAuLyR7cmVzb2x2ZXIucGF0aH1gKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkSmVzdE5vQ29tcGlsZShqZXN0OiBKZXN0KSB7XG4gICAgdGhpcy5hZGREZXZEZXBzKFxuICAgICAgYEB0eXBlcy9qZXN0JHtqZXN0Lmplc3RWZXJzaW9ufWAsXG4gICAgICBgdHMtamVzdCR7amVzdC5qZXN0VmVyc2lvbn1gXG4gICAgKTtcblxuICAgIGplc3QuYWRkVGVzdE1hdGNoKGA8cm9vdERpcj4vJHt0aGlzLnNyY2Rpcn0vKiovX190ZXN0c19fLyoqLyoudHM/KHgpYCk7XG4gICAgamVzdC5hZGRUZXN0TWF0Y2goXG4gICAgICBgPHJvb3REaXI+Lygke3RoaXMudGVzdGRpcn18JHt0aGlzLnNyY2Rpcn0pLyoqLyooKi4pQChzcGVjfHRlc3QpLnRzPyh4KWBcbiAgICApO1xuXG4gICAgLy8gYWRkIHJlbGV2YW50IGRlcHNcbiAgICBqZXN0LmNvbmZpZy5wcmVzZXQgPSBcInRzLWplc3RcIjtcbiAgICBqZXN0LmNvbmZpZy5nbG9iYWxzID0ge1xuICAgICAgXCJ0cy1qZXN0XCI6IHtcbiAgICAgICAgdHNjb25maWc6IHRoaXMudHNjb25maWdEZXYuZmlsZU5hbWUsXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuY2xhc3MgU2FtcGxlQ29kZSBleHRlbmRzIENvbXBvbmVudCB7XG4gIGNvbnN0cnVjdG9yKHByb2plY3Q6IFR5cGVTY3JpcHRQcm9qZWN0KSB7XG4gICAgc3VwZXIocHJvamVjdCk7XG4gICAgY29uc3Qgc3JjQ29kZSA9IFtcbiAgICAgIFwiZXhwb3J0IGNsYXNzIEhlbGxvIHtcIixcbiAgICAgIFwiICBwdWJsaWMgc2F5SGVsbG8oKSB7XCIsXG4gICAgICBcIiAgICByZXR1cm4gJ2hlbGxvLCB3b3JsZCEnO1wiLFxuICAgICAgXCIgIH1cIixcbiAgICAgIFwifVwiLFxuICAgIF0uam9pbihcIlxcblwiKTtcblxuICAgIGNvbnN0IHRlc3RDb2RlID0gW1xuICAgICAgXCJpbXBvcnQgeyBIZWxsbyB9IGZyb20gJy4uL3NyYyc7XCIsXG4gICAgICBcIlwiLFxuICAgICAgXCJ0ZXN0KCdoZWxsbycsICgpID0+IHtcIixcbiAgICAgIFwiICBleHBlY3QobmV3IEhlbGxvKCkuc2F5SGVsbG8oKSkudG9CZSgnaGVsbG8sIHdvcmxkIScpO1wiLFxuICAgICAgXCJ9KTtcIixcbiAgICBdLmpvaW4oXCJcXG5cIik7XG5cbiAgICBuZXcgU2FtcGxlRGlyKHByb2plY3QsIHByb2plY3Quc3JjZGlyLCB7XG4gICAgICBmaWxlczoge1xuICAgICAgICBcImluZGV4LnRzXCI6IHNyY0NvZGUsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgaWYgKHByb2plY3QuamVzdCkge1xuICAgICAgbmV3IFNhbXBsZURpcihwcm9qZWN0LCBwcm9qZWN0LnRlc3RkaXIsIHtcbiAgICAgICAgZmlsZXM6IHtcbiAgICAgICAgICBcImhlbGxvLnRlc3QudHNcIjogdGVzdENvZGUsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBUeXBlU2NyaXB0IGFwcC5cbiAqXG4gKiBAcGppZCB0eXBlc2NyaXB0LWFwcFxuICovXG5leHBvcnQgY2xhc3MgVHlwZVNjcmlwdEFwcFByb2plY3QgZXh0ZW5kcyBUeXBlU2NyaXB0UHJvamVjdCB7XG4gIGNvbnN0cnVjdG9yKG9wdGlvbnM6IFR5cGVTY3JpcHRQcm9qZWN0T3B0aW9ucykge1xuICAgIHN1cGVyKHtcbiAgICAgIGFsbG93TGlicmFyeURlcGVuZGVuY2llczogZmFsc2UsXG4gICAgICByZWxlYXNlV29ya2Zsb3c6IGZhbHNlLFxuICAgICAgZW50cnlwb2ludDogXCJcIiwgLy8gXCJtYWluXCIgaXMgbm90IG5lZWRlZCBpbiB0eXBlc2NyaXB0IGFwcHNcbiAgICAgIHBhY2thZ2U6IGZhbHNlLFxuICAgICAgLi4ub3B0aW9ucyxcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIEBkZXByZWNhdGVkIHVzZSBgVHlwZVNjcmlwdFByb2plY3RgXG4gKi9cbmV4cG9ydCBjbGFzcyBUeXBlU2NyaXB0TGlicmFyeVByb2plY3QgZXh0ZW5kcyBUeXBlU2NyaXB0UHJvamVjdCB7fVxuXG4vKipcbiAqIEBkZXByZWNhdGVkIHVzZSBUeXBlU2NyaXB0UHJvamVjdE9wdGlvbnNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBUeXBlU2NyaXB0TGlicmFyeVByb2plY3RPcHRpb25zXG4gIGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3RPcHRpb25zIHt9XG5cbi8qKlxuICogQGludGVybmFsXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBtZXJnZVRzY29uZmlnT3B0aW9ucyhcbiAgLi4ub3B0aW9uczogKFR5cGVzY3JpcHRDb25maWdPcHRpb25zIHwgdW5kZWZpbmVkKVtdXG4pOiBUeXBlc2NyaXB0Q29uZmlnT3B0aW9ucyB7XG4gIGNvbnN0IGRlZmluZWRPcHRpb25zID0gb3B0aW9ucy5maWx0ZXIoQm9vbGVhbikgYXMgVHlwZXNjcmlwdENvbmZpZ09wdGlvbnNbXTtcbiAgcmV0dXJuIGRlZmluZWRPcHRpb25zLnJlZHVjZTxUeXBlc2NyaXB0Q29uZmlnT3B0aW9ucz4oXG4gICAgKHByZXZpb3VzLCBjdXJyZW50KSA9PiAoe1xuICAgICAgLi4ucHJldmlvdXMsXG4gICAgICAuLi5jdXJyZW50LFxuICAgICAgaW5jbHVkZTogWy4uLihwcmV2aW91cy5pbmNsdWRlID8/IFtdKSwgLi4uKGN1cnJlbnQuaW5jbHVkZSA/PyBbXSldLFxuICAgICAgZXhjbHVkZTogWy4uLihwcmV2aW91cy5leGNsdWRlID8/IFtdKSwgLi4uKGN1cnJlbnQuZXhjbHVkZSA/PyBbXSldLFxuICAgICAgY29tcGlsZXJPcHRpb25zOiB7XG4gICAgICAgIC4uLnByZXZpb3VzLmNvbXBpbGVyT3B0aW9ucyxcbiAgICAgICAgLi4uY3VycmVudC5jb21waWxlck9wdGlvbnMsXG4gICAgICB9LFxuICAgIH0pLFxuICAgIHsgY29tcGlsZXJPcHRpb25zOiB7fSB9XG4gICk7XG59XG4iXX0=