import { TaskCommonOptions, TaskSpec, TaskStep, TaskStepOptions } from "./task-model";
export interface TaskOptions extends TaskCommonOptions {
    /**
     * Shell command to execute as the first command of the task.
     * @default - add steps using `task.exec(command)` or `task.spawn(subtask)`
     */
    readonly exec?: string;
    /**
     * List of task steps to run.
     */
    readonly steps?: TaskStep[];
}
/**
 * A task that can be performed on the project. Modeled as a series of shell
 * commands and subtasks.
 */
export declare class Task {
    /**
     * Task name.
     */
    readonly name: string;
    /**
     * A command to execute which determines if the task should be skipped. If it
     * returns a zero exit code, the task will not be executed.
     */
    readonly condition?: string;
    private readonly _steps;
    private readonly _env;
    private readonly cwd?;
    private readonly requiredEnv?;
    private _locked;
    private _description?;
    constructor(name: string, props?: TaskOptions);
    /**
     * Forbid additional changes to this task.
     */
    lock(): void;
    /**
     * Returns the description of this task.
     */
    get description(): string | undefined;
    /**
     * Sets the description of this task.
     */
    set description(desc: string | undefined);
    /**
     * Reset the task so it no longer has any commands.
     * @param command the first command to add to the task after it was cleared.
     */
    reset(command?: string, options?: TaskStepOptions): void;
    /**
     * Executes a shell command
     * @param command Shell command
     * @param options Options
     */
    exec(command: string, options?: TaskStepOptions): void;
    /**
     * Execute a builtin task.
     *
     * Builtin tasks are programs bundled as part of projen itself and used as
     * helpers for various components.
     *
     * In the future we should support built-in tasks from external modules.
     *
     * @param name The name of the builtin task to execute (e.g.
     * `release/resolve-version`).
     */
    builtin(name: string): void;
    /**
     * Say something.
     * @param message Your message
     * @param options Options
     */
    say(message: string, options?: TaskStepOptions): void;
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     *
     * @deprecated use `prependExec()`
     */
    prepend(shell: string, options?: TaskStepOptions): void;
    /**
     * Spawns a sub-task.
     * @param subtask The subtask to execute.
     */
    spawn(subtask: Task, options?: TaskStepOptions): void;
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     */
    prependExec(shell: string, options?: TaskStepOptions): void;
    /**
     * Adds a spawn instruction at the beginning of the task.
     * @param subtask The subtask to execute.
     */
    prependSpawn(subtask: Task, options?: TaskStepOptions): void;
    /**
     * Says something at the beginning of the task.
     * @param message Your message
     */
    prependSay(message: string, options?: TaskStepOptions): void;
    /**
     * Adds an environment variable to this task.
     * @param name The name of the variable
     * @param value The value. If the value is surrounded by `$()`, we will
     * evaluate it within a subshell and use the result as the value of the
     * environment variable.
     */
    env(name: string, value: string): void;
    /**
     * Returns an immutable copy of all the step specifications of the task.
     */
    get steps(): TaskStep[];
    /**
     * Renders a task spec into the manifest.
     *
     * @internal
     */
    _renderSpec(): TaskSpec;
    private assertUnlocked;
}
