"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PoetryPyproject = exports.Poetry = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const task_runtime_1 = require("../task-runtime");
const toml_1 = require("../toml");
const util_1 = require("../util");
/**
 * Manage project dependencies, virtual environments, and packaging through the
 * poetry CLI tool.
 */
class Poetry extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.installTask = project.addTask("install", {
            description: "Install and upgrade dependencies",
            exec: "poetry update",
        });
        this.project.tasks.addEnvironment("VIRTUAL_ENV", "$(poetry env info -p)");
        this.project.tasks.addEnvironment("PATH", "$(echo $(poetry env info -p)/bin:$PATH)");
        project.packageTask.exec("poetry build");
        this.publishTestTask = project.addTask("publish:test", {
            description: "Uploads the package against a test PyPI endpoint.",
            exec: "poetry publish -r testpypi",
        });
        this.publishTask = project.addTask("publish", {
            description: "Uploads the package to PyPI.",
            exec: "poetry publish",
        });
        new PoetryPyproject(project, {
            name: project.name,
            version: options.version,
            description: options.description ?? "",
            license: options.license,
            authors: [`${options.authorName} <${options.authorEmail}>`],
            homepage: options.homepage,
            classifiers: options.classifiers,
            ...options.poetryOptions,
            dependencies: () => this.synthDependencies(),
            devDependencies: () => this.synthDevDependencies(),
        });
        new toml_1.TomlFile(project, "poetry.toml", {
            committed: false,
            obj: {
                repositories: {
                    testpypi: {
                        url: "https://test.pypi.org/legacy/",
                    },
                },
            },
        });
    }
    synthDependencies() {
        const dependencies = {};
        let pythonDefined = false;
        for (const pkg of this.project.deps.all) {
            if (pkg.name === "python") {
                pythonDefined = true;
            }
            if (pkg.type === dependencies_1.DependencyType.RUNTIME) {
                dependencies[pkg.name] = pkg.version;
            }
        }
        if (!pythonDefined) {
            // Python version must be defined for poetry projects. Default to ^3.6.
            dependencies.python = "^3.6";
        }
        return dependencies;
    }
    synthDevDependencies() {
        const dependencies = {};
        for (const pkg of this.project.deps.all) {
            if ([dependencies_1.DependencyType.DEVENV].includes(pkg.type)) {
                dependencies[pkg.name] = pkg.version;
            }
        }
        return dependencies;
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.RUNTIME);
    }
    /**
     * Adds a dev dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDevDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.DEVENV);
    }
    /**
     * Initializes the virtual environment if it doesn't exist (called during post-synthesis).
     */
    setupEnvironment() {
        const result = util_1.execOrUndefined("which poetry", {
            cwd: this.project.outdir,
        });
        if (!result) {
            this.project.logger.info("Unable to setup an environment since poetry is not installed. Please install poetry (https://python-poetry.org/docs/) or use a different component for managing environments such as 'venv'.");
        }
        let envPath = util_1.execOrUndefined("poetry env info -p", {
            cwd: this.project.outdir,
        });
        if (!envPath) {
            this.project.logger.info("Setting up a virtual environment...");
            util_1.exec("poetry env use python", { cwd: this.project.outdir });
            envPath = util_1.execOrUndefined("poetry env info -p", {
                cwd: this.project.outdir,
            });
            this.project.logger.info(`Environment successfully created (located in ${envPath}}).`);
        }
    }
    /**
     * Installs dependencies (called during post-synthesis).
     */
    installDependencies() {
        this.project.logger.info("Installing dependencies...");
        const runtime = new task_runtime_1.TaskRuntime(this.project.outdir);
        runtime.runTask(this.installTask.name);
    }
}
exports.Poetry = Poetry;
_a = JSII_RTTI_SYMBOL_1;
Poetry[_a] = { fqn: "projen.python.Poetry", version: "0.60.16" };
/**
 * Represents configuration of a pyproject.toml file for a Poetry project.
 *
 * @see https://python-poetry.org/docs/pyproject/
 */
class PoetryPyproject extends component_1.Component {
    constructor(project, options) {
        super(project);
        const decamelisedOptions = util_1.decamelizeKeysRecursively(options, {
            separator: "-",
        });
        this.file = new toml_1.TomlFile(project, "pyproject.toml", {
            omitEmpty: false,
            obj: {
                "build-system": {
                    requires: ["poetry_core>=1.0.0"],
                    "build-backend": "poetry.core.masonry.api",
                },
                tool: {
                    poetry: {
                        ...decamelisedOptions,
                    },
                },
            },
        });
    }
}
exports.PoetryPyproject = PoetryPyproject;
_b = JSII_RTTI_SYMBOL_1;
PoetryPyproject[_b] = { fqn: "projen.python.PoetryPyproject", version: "0.60.16" };
//# sourceMappingURL=data:application/json;base64,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