"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpgradeDependenciesSchedule = exports.UpgradeDependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const release_1 = require("../release");
function context(value) {
    return `\${{ ${value} }}`;
}
const REPO = context("github.repository");
const RUN_ID = context("github.run_id");
const RUN_URL = `https://github.com/${REPO}/actions/runs/${RUN_ID}`;
const CREATE_PATCH_STEP_ID = "create_patch";
const PATCH_CREATED_OUTPUT = "patch_created";
/**
 * Upgrade node project dependencies.
 */
class UpgradeDependencies extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        /**
         * The workflows that execute the upgrades. One workflow per branch.
         */
        this.workflows = [];
        this._project = project;
        this.options = options;
        this.pullRequestTitle = options.pullRequestTitle ?? "upgrade dependencies";
        this.gitIdentity =
            options.workflowOptions?.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.postBuildSteps = [];
        this.containerOptions = options.workflowOptions?.container;
        project.addDevDeps("npm-check-updates@^15");
        this.postUpgradeTask =
            project.tasks.tryFind("post-upgrade") ??
                project.tasks.addTask("post-upgrade", {
                    description: "Runs after upgrading dependencies",
                });
        this.upgradeTask = project.addTask(options.taskName ?? "upgrade", {
            // this task should not run in CI mode because its designed to
            // update package.json and lock files.
            env: { CI: "0" },
            description: this.pullRequestTitle,
            steps: { toJSON: () => this.renderTaskSteps() },
        });
        this.upgradeTask.lock(); // this task is a lazy value, so make it readonly
        if (this.upgradeTask && project.github && (options.workflow ?? true)) {
            if (options.workflowOptions?.branches) {
                for (const branch of options.workflowOptions.branches) {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                }
            }
            else if (release_1.Release.of(project)) {
                const release = release_1.Release.of(project);
                release._forEachBranch((branch) => {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                });
            }
            else {
                // represents the default repository branch.
                // just like not specifying anything.
                const defaultBranch = undefined;
                this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, defaultBranch));
            }
        }
    }
    /**
     * Add steps to execute a successful build.
     * @param steps workflow steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    renderTaskSteps() {
        const exclude = this.options.exclude ?? [];
        // exclude depedencies that has already version pinned (fully or with patch version) by Projen with ncu (but not package manager upgrade)
        // Getting only unique values through set
        const ncuExcludes = [
            ...new Set(this.project.deps.all
                .filter((dep) => dep.version && dep.version[0] !== "^")
                .map((dep) => dep.name)
                .concat(exclude)),
        ];
        const ncuIncludes = this.options.include?.filter((item) => !ncuExcludes.includes(item));
        const includeLength = this.options.include?.length ?? 0;
        const ncuIncludesLength = ncuIncludes?.length ?? 0;
        // If all explicit includes already have version pinned, don't add task.
        // Note that without explicit includes task gets added
        if (includeLength > 0 && ncuIncludesLength === 0) {
            return [{ exec: "echo No dependencies to upgrade." }];
        }
        const steps = new Array();
        // update npm-check-updates before everything else, in case there is a bug
        // in it or one of its dependencies. This will make upgrade workflows
        // slightly more stable and resilient to upstream changes.
        steps.push({
            exec: this._project.package.renderUpgradePackagesCommand([], ["npm-check-updates"]),
        });
        for (const dep of ["dev", "optional", "peer", "prod", "bundle"]) {
            const ncuCommand = [
                "npm-check-updates",
                "--dep",
                dep,
                "--upgrade",
                "--target=minor",
            ];
            // Don't add includes and excludes same time
            if (ncuIncludes) {
                ncuCommand.push(`--filter='${ncuIncludes.join(",")}'`);
            }
            else if (ncuExcludes.length > 0) {
                ncuCommand.push(`--reject='${ncuExcludes.join(",")}'`);
            }
            steps.push({ exec: ncuCommand.join(" ") });
        }
        // run "yarn/npm install" to update the lockfile and install any deps (such as projen)
        steps.push({ exec: this._project.package.installAndUpdateLockfileCommand });
        // run upgrade command to upgrade transitive deps as well
        steps.push({
            exec: this._project.package.renderUpgradePackagesCommand(exclude, this.options.include),
        });
        // run "projen" to give projen a chance to update dependencies (it will also run "yarn install")
        steps.push({ exec: this._project.projenCommand });
        steps.push({ spawn: this.postUpgradeTask.name });
        return steps;
    }
    createWorkflow(task, github, branch) {
        const schedule = this.options.workflowOptions?.schedule ??
            UpgradeDependenciesSchedule.DAILY;
        const workflowName = `${task.name}${branch ? `-${branch.replace(/\//g, "-")}` : ""}`;
        const workflow = github.addWorkflow(workflowName);
        const triggers = {
            workflowDispatch: {},
            schedule: schedule.cron.length > 0
                ? schedule.cron.map((e) => ({ cron: e }))
                : undefined,
        };
        workflow.on(triggers);
        const upgrade = this.createUpgrade(task, branch);
        const pr = this.createPr(workflow, upgrade);
        const jobs = {};
        jobs[upgrade.jobId] = upgrade.job;
        jobs[pr.jobId] = pr.job;
        workflow.addJobs(jobs);
        return workflow;
    }
    createUpgrade(task, branch) {
        const runsOn = this.options.workflowOptions?.runsOn ?? ["ubuntu-latest"];
        // thats all we should need at this stage since all we do is clone.
        // note that this also prevents new code that is introduced in the upgrade
        // to have write access to anything, in case its somehow executed. (for example during build)
        const permissions = {
            contents: github_1.workflows.JobPermission.READ,
        };
        const steps = [
            {
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: branch ? { ref: branch } : undefined,
            },
            ...this._project.renderWorkflowSetup({ mutable: false }),
            {
                name: "Upgrade dependencies",
                run: this._project.runTaskCommand(task),
            },
        ];
        steps.push(...this.postBuildSteps);
        steps.push(...workflow_actions_1.WorkflowActions.createUploadGitPatch({
            stepId: CREATE_PATCH_STEP_ID,
            outputName: PATCH_CREATED_OUTPUT,
        }));
        return {
            job: {
                name: "Upgrade",
                container: this.containerOptions,
                permissions: permissions,
                runsOn: runsOn ?? ["ubuntu-latest"],
                steps: steps,
                outputs: {
                    [PATCH_CREATED_OUTPUT]: {
                        stepId: CREATE_PATCH_STEP_ID,
                        outputName: PATCH_CREATED_OUTPUT,
                    },
                },
            },
            jobId: "upgrade",
            ref: branch,
        };
    }
    createPr(workflow, upgrade) {
        // default to API access method used by all GitHub workflows, unless a
        // custom one is specified
        const apiAccess = this.options.workflowOptions?.projenCredentials ??
            workflow.projenCredentials;
        const token = apiAccess.tokenRef;
        const runsOn = this.options.workflowOptions?.runsOn ?? ["ubuntu-latest"];
        const workflowName = workflow.name;
        const branchName = `github-actions/${workflowName}`;
        const prStepId = "create-pr";
        const title = `chore(deps): ${this.pullRequestTitle}`;
        const description = [
            "Upgrades project dependencies. See details in [workflow run].",
            "",
            `[Workflow Run]: ${RUN_URL}`,
            "",
            "------",
            "",
            `*Automatically created by projen via the "${workflow.name}" workflow*`,
        ].join("\n");
        const committer = `${this.gitIdentity.name} <${this.gitIdentity.email}>`;
        const steps = [
            ...apiAccess.setupSteps,
            ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                token: apiAccess.tokenRef,
                ref: upgrade.ref,
            }),
            ...workflow_actions_1.WorkflowActions.setGitIdentity(this.gitIdentity),
            {
                name: "Create Pull Request",
                id: prStepId,
                uses: "peter-evans/create-pull-request@v3",
                with: {
                    // the pr can modify workflow files, so we need to use the custom
                    // secret if one is configured.
                    token: token,
                    "commit-message": `${title}\n\n${description}`,
                    branch: branchName,
                    title: title,
                    labels: this.options.workflowOptions?.labels?.join(",") || undefined,
                    assignees: this.options.workflowOptions?.assignees?.join(",") || undefined,
                    body: description,
                    author: committer,
                    committer: committer,
                    signoff: this.options.signoff ?? true,
                },
            },
        ];
        return {
            job: {
                name: "Create Pull Request",
                if: `\${{ needs.${upgrade.jobId}.outputs.${PATCH_CREATED_OUTPUT} }}`,
                needs: [upgrade.jobId],
                permissions: {
                    contents: github_1.workflows.JobPermission.WRITE,
                    pullRequests: github_1.workflows.JobPermission.WRITE,
                },
                runsOn: runsOn ?? ["ubuntu-latest"],
                steps: steps,
            },
            jobId: "pr",
        };
    }
}
exports.UpgradeDependencies = UpgradeDependencies;
_a = JSII_RTTI_SYMBOL_1;
UpgradeDependencies[_a] = { fqn: "projen.javascript.UpgradeDependencies", version: "0.60.16" };
/**
 * How often to check for new versions and raise pull requests for version upgrades.
 */
class UpgradeDependenciesSchedule {
    constructor(cron) {
        this.cron = cron;
    }
    /**
     * Create a schedule from a raw cron expression.
     */
    static expressions(cron) {
        return new UpgradeDependenciesSchedule(cron);
    }
}
exports.UpgradeDependenciesSchedule = UpgradeDependenciesSchedule;
_b = JSII_RTTI_SYMBOL_1;
UpgradeDependenciesSchedule[_b] = { fqn: "projen.javascript.UpgradeDependenciesSchedule", version: "0.60.16" };
/**
 * Disables automatic upgrades.
 */
UpgradeDependenciesSchedule.NEVER = new UpgradeDependenciesSchedule([]);
/**
 * At 00:00.
 */
UpgradeDependenciesSchedule.DAILY = new UpgradeDependenciesSchedule(["0 0 * * *"]);
/**
 * At 00:00 on every day-of-week from Monday through Friday.
 */
UpgradeDependenciesSchedule.WEEKDAY = new UpgradeDependenciesSchedule([
    "0 0 * * 1-5",
]);
/**
 * At 00:00 on Monday.
 */
UpgradeDependenciesSchedule.WEEKLY = new UpgradeDependenciesSchedule([
    "0 0 * * 1",
]);
/**
 * At 00:00 on day-of-month 1.
 */
UpgradeDependenciesSchedule.MONTHLY = new UpgradeDependenciesSchedule([
    "0 0 1 * *",
]);
//# sourceMappingURL=data:application/json;base64,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