"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultNpmToken = exports.NpmAccess = exports.NodePackageManager = exports.NodePackage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const url_1 = require("url");
const fs_extra_1 = require("fs-extra");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const json_1 = require("../json");
const release_1 = require("../release");
const util_1 = require("../util");
const util_2 = require("./util");
const UNLICENSED = "UNLICENSED";
const DEFAULT_NPM_REGISTRY_URL = "https://registry.npmjs.org/";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const DEFAULT_NPM_TOKEN_SECRET = "NPM_TOKEN";
const DEFAULT_GITHUB_TOKEN_SECRET = "GITHUB_TOKEN";
/**
 * Represents the npm `package.json` file.
 */
class NodePackage extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        this.keywords = new Set();
        this.bin = {};
        this.engines = {};
        this.packageName = options.packageName ?? project.name;
        this.peerDependencyOptions = {
            pinnedDevDependency: true,
            ...options.peerDependencyOptions,
        };
        this.allowLibraryDependencies = options.allowLibraryDependencies ?? true;
        this.packageManager = options.packageManager ?? NodePackageManager.YARN;
        this.entrypoint = options.entrypoint ?? "lib/index.js";
        this.lockFile = determineLockfile(this.packageManager);
        this.project.annotateGenerated(`/${this.lockFile}`);
        const { npmAccess, npmRegistry, npmRegistryUrl, npmTokenSecret, codeArtifactOptions, scopedPackagesOptions, } = this.parseNpmOptions(options);
        this.npmAccess = npmAccess;
        this.npmRegistry = npmRegistry;
        this.npmRegistryUrl = npmRegistryUrl;
        this.npmTokenSecret = npmTokenSecret;
        this.codeArtifactOptions = codeArtifactOptions;
        this.scopedPackagesOptions = scopedPackagesOptions;
        this.processDeps(options);
        this.addCodeArtifactLoginScript();
        const prev = this.readPackageJson() ?? {};
        // empty objects are here to preserve order for backwards compatibility
        this.manifest = {
            name: this.packageName,
            description: options.description,
            repository: !options.repository
                ? undefined
                : {
                    type: "git",
                    url: options.repository,
                    directory: options.repositoryDirectory,
                },
            bin: () => this.renderBin(),
            scripts: () => this.renderScripts(),
            author: this.renderAuthor(options),
            devDependencies: {},
            peerDependencies: {},
            dependencies: {},
            bundledDependencies: [],
            keywords: () => this.renderKeywords(),
            engines: () => this.renderEngines(),
            main: this.entrypoint !== "" ? this.entrypoint : undefined,
            license: () => this.license ?? UNLICENSED,
            homepage: options.homepage,
            publishConfig: () => this.renderPublishConfig(),
            // in release CI builds we bump the version before we run "build" so we want
            // to preserve the version number. otherwise, we always set it to 0.0.0
            version: this.determineVersion(prev?.version),
            bugs: options.bugsEmail || options.bugsUrl
                ? {
                    email: options.bugsEmail,
                    url: options.bugsUrl,
                }
                : undefined,
        };
        // override any scripts from options (if specified)
        for (const [cmdname, shell] of Object.entries(options.scripts ?? {})) {
            project.addTask(cmdname, { exec: shell });
        }
        this.file = new json_1.JsonFile(this.project, "package.json", {
            obj: this.manifest,
            readonly: false,
            newline: false,
        });
        this.addKeywords(...(options.keywords ?? []));
        this.addBin(options.bin ?? {});
        // automatically add all executable files under "bin"
        if (options.autoDetectBin ?? true) {
            this.autoDiscoverBinaries();
        }
        // node version
        this.minNodeVersion = options.minNodeVersion;
        this.maxNodeVersion = options.maxNodeVersion;
        this.addNodeEngine();
        // license
        if (options.licensed ?? true) {
            this.license = options.license ?? "Apache-2.0";
        }
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.RUNTIME);
        }
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        if (Object.keys(deps).length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add peer dependencies to an APP project: ${Object.keys(deps).join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.PEER);
        }
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        if (deps.length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add bundled dependencies to an APP project: ${deps.join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUNDLED);
        }
    }
    /**
     * Adds an `engines` requirement to your package.
     * @param engine The engine (e.g. `node`)
     * @param version The semantic version requirement (e.g. `^10`)
     */
    addEngine(engine, version) {
        this.engines[engine] = version;
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        for (const k of keywords) {
            this.keywords.add(k);
        }
    }
    addBin(bins) {
        for (const [k, v] of Object.entries(bins)) {
            this.bin[k] = v;
        }
    }
    /**
     * Override the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.file.addOverride(`scripts.${name}`, command);
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.file.addDeletionOverride(`scripts.${name}`);
    }
    /**
     * Indicates if a script by the given name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.project.tasks.tryFind(name) !== undefined;
    }
    /**
     * Directly set fields in `package.json`.
     * @escape
     * @param name field name
     * @param value field value
     */
    addField(name, value) {
        this.manifest[name] = value;
    }
    /**
     * Sets the package version.
     * @param version Package version.
     */
    addVersion(version) {
        this.manifest.version = version;
    }
    /**
     * Defines resolutions for dependencies to change the normally resolved
     * version of a dependency to something else.
     *
     * @param resolutions Names resolutions to be added. Specify a version or
     * range with this syntax:
     * `module@^7`.
     */
    addPackageResolutions(...resolutions) {
        const fieldName = util_2.packageResolutionsFieldName(this.packageManager);
        for (const resolution of resolutions) {
            const { name, version = "*" } = dependencies_1.Dependencies.parseDependency(resolution);
            this.file.addOverride(`${fieldName}.${name}`, version);
        }
    }
    /**
     * Returns the command to execute in order to install all dependencies (always frozen).
     */
    get installCommand() {
        return this.renderInstallCommand(true);
    }
    /**
     * Renders `yarn install` or `npm install` with lockfile update (not frozen)
     */
    get installAndUpdateLockfileCommand() {
        return this.renderInstallCommand(false);
    }
    /**
     * Render a package manager specific command to upgrade all requested dependencies.
     */
    renderUpgradePackagesCommand(exclude, include) {
        const project = this.project;
        function upgradePackages(command) {
            return () => {
                if (exclude.length === 0 && !include) {
                    // request to upgrade all packages
                    // separated for asthetic reasons.
                    return command;
                }
                // filter by exclude and include.
                return `${command} ${project.deps.all
                    .map((d) => d.name)
                    .filter((d) => (include ? include.includes(d) : true))
                    .filter((d) => !exclude.includes(d))
                    .join(" ")}`;
            };
        }
        let lazy = undefined;
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                lazy = upgradePackages("yarn upgrade");
                break;
            case NodePackageManager.NPM:
                lazy = upgradePackages("npm update");
                break;
            case NodePackageManager.PNPM:
                lazy = upgradePackages("pnpm update");
                break;
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
        // return a lazy function so that dependencies include ones that were
        // added post project instantiation (i.e using project.addDeps)
        return lazy;
    }
    // ---------------------------------------------------------------------------------------
    preSynthesize() {
        super.preSynthesize();
        this._renderedDeps = this.renderDependencies();
    }
    postSynthesize() {
        super.postSynthesize();
        // only run "install" if package.json has changed or if we don't have a
        // `node_modules` directory.
        if (this.file.changed ||
            !fs_extra_1.existsSync(path_1.join(this.project.outdir, "node_modules"))) {
            this.installDependencies();
        }
        // resolve "*" deps in package.json and update it. if it was changed,
        // install deps again so that lockfile is updated.
        if (this.resolveDepsAndWritePackageJson()) {
            this.installDependencies();
        }
    }
    /**
     * The command which executes "projen".
     */
    get projenCommand() {
        return this.project.projenCommand;
    }
    /**
     * Returns `true` if we are running within a CI build.
     */
    get isAutomatedBuild() {
        return util_1.isTruthy(process.env.CI);
    }
    determineVersion(currVersion) {
        if (!this.isReleaseBuild) {
            return "0.0.0";
        }
        return currVersion ?? "0.0.0";
    }
    /**
     * Returns `true` if this is a CI release build.
     */
    get isReleaseBuild() {
        return util_1.isTruthy(process.env.RELEASE);
    }
    // -------------------------------------------------------------------------------------------
    parseNpmOptions(options) {
        let npmRegistryUrl = options.npmRegistryUrl;
        if (options.npmRegistry) {
            if (npmRegistryUrl) {
                throw new Error('cannot use the deprecated "npmRegistry" together with "npmRegistryUrl". please use "npmRegistryUrl" instead.');
            }
            npmRegistryUrl = `https://${options.npmRegistry}`;
        }
        const npmr = url_1.parse(npmRegistryUrl ?? DEFAULT_NPM_REGISTRY_URL);
        if (!npmr || !npmr.hostname || !npmr.href) {
            throw new Error(`unable to determine npm registry host from url ${npmRegistryUrl}. Is this really a URL?`);
        }
        const npmAccess = options.npmAccess ?? defaultNpmAccess(this.packageName);
        if (!isScoped(this.packageName) && npmAccess === NpmAccess.RESTRICTED) {
            throw new Error(`"npmAccess" cannot be RESTRICTED for non-scoped npm package "${this.packageName}"`);
        }
        const isAwsCodeArtifact = release_1.isAwsCodeArtifactRegistry(npmRegistryUrl);
        const hasScopedPackage = options.scopedPackagesOptions &&
            options.scopedPackagesOptions.length !== 0;
        if (isAwsCodeArtifact) {
            if (options.npmTokenSecret) {
                throw new Error('"npmTokenSecret" must not be specified when publishing AWS CodeArtifact.');
            }
        }
        else {
            if ((options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret ||
                options.codeArtifactOptions?.roleToAssume) &&
                !hasScopedPackage) {
                throw new Error("codeArtifactOptions must only be specified when publishing AWS CodeArtifact or used in scoped packages.");
            }
        }
        // apply defaults for AWS CodeArtifact
        let codeArtifactOptions;
        if (isAwsCodeArtifact || hasScopedPackage) {
            codeArtifactOptions = {
                accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID",
                secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret ??
                    "AWS_SECRET_ACCESS_KEY",
                roleToAssume: options.codeArtifactOptions?.roleToAssume,
            };
        }
        return {
            npmAccess,
            npmRegistry: npmr.hostname + this.renderNpmRegistryPath(npmr.pathname),
            npmRegistryUrl: npmr.href,
            npmTokenSecret: defaultNpmToken(options.npmTokenSecret, npmr.hostname),
            codeArtifactOptions,
            scopedPackagesOptions: this.parseScopedPackagesOptions(options.scopedPackagesOptions),
        };
    }
    parseScopedPackagesOptions(scopedPackagesOptions) {
        if (!scopedPackagesOptions) {
            return undefined;
        }
        return scopedPackagesOptions.map((option) => {
            if (!isScoped(option.scope)) {
                throw new Error(`Scope must start with "@" in options, found ${option.scope}`);
            }
            if (!release_1.isAwsCodeArtifactRegistry(option.registryUrl)) {
                throw new Error(`Only AWS Code artifact scoped registry is supported for now, found ${option.registryUrl}`);
            }
            const result = {
                registryUrl: option.registryUrl,
                scope: option.scope,
            };
            return result;
        });
    }
    addCodeArtifactLoginScript() {
        if (!this.scopedPackagesOptions ||
            this.scopedPackagesOptions.length === 0) {
            return;
        }
        this.project.addTask("ca:login", {
            requiredEnv: ["AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY"],
            steps: [
                { exec: "which aws" },
                ...this.scopedPackagesOptions.map((scopedPackagesOption) => {
                    const { registryUrl, scope } = scopedPackagesOption;
                    const { domain, region, accountId, registry } = util_2.extractCodeArtifactDetails(registryUrl);
                    // reference: https://docs.aws.amazon.com/codeartifact/latest/ug/npm-auth.html
                    const commands = [
                        `npm config set ${scope}:registry ${registryUrl}`,
                        `CODEARTIFACT_AUTH_TOKEN=$(aws codeartifact get-authorization-token --domain ${domain} --region ${region} --domain-owner ${accountId} --query authorizationToken --output text)`,
                        `npm config set //${registry}:_authToken=$CODEARTIFACT_AUTH_TOKEN`,
                        `npm config set //${registry}:always-auth=true`,
                    ];
                    return {
                        exec: commands.join("; "),
                    };
                }),
            ],
        });
    }
    addNodeEngine() {
        if (!this.minNodeVersion && !this.maxNodeVersion) {
            return;
        }
        let nodeVersion = "";
        if (this.minNodeVersion) {
            nodeVersion += `>= ${this.minNodeVersion}`;
        }
        if (this.maxNodeVersion) {
            nodeVersion += ` <= ${this.maxNodeVersion}`;
        }
        this.addEngine("node", nodeVersion);
    }
    renderNpmRegistryPath(path) {
        if (!path || path == "/") {
            return "";
        }
        else {
            return path;
        }
    }
    renderInstallCommand(frozen) {
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                return [
                    "yarn install",
                    "--check-files",
                    ...(frozen ? ["--frozen-lockfile"] : []),
                ].join(" ");
            case NodePackageManager.NPM:
                return frozen ? "npm ci" : "npm install";
            case NodePackageManager.PNPM:
                return frozen
                    ? "pnpm i --frozen-lockfile"
                    : "pnpm i --no-frozen-lockfile";
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
    }
    processDeps(options) {
        this.addDeps(...(options.deps ?? []));
        this.addDevDeps(...(options.devDeps ?? []));
        this.addPeerDeps(...(options.peerDeps ?? []));
        this.addBundledDeps(...(options.bundledDeps ?? []));
    }
    renderDependencies() {
        const devDependencies = {};
        const peerDependencies = {};
        const dependencies = {};
        const bundledDependencies = new Array();
        // synthetic dependencies: add a pinned build dependency to ensure we are
        // testing against the minimum requirement of the peer.
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const dep of this.project.deps.all.filter((d) => d.type === dependencies_1.DependencyType.PEER)) {
                let req = dep.name;
                // skip if we already have a runtime dependency on this peer
                if (this.project.deps.tryGetDependency(dep.name, dependencies_1.DependencyType.RUNTIME)) {
                    continue;
                }
                if (dep.version) {
                    const ver = util_2.minVersion(dep.version);
                    if (!ver) {
                        throw new Error(`unable to determine minimum semver for peer dependency ${dep.name}@${dep.version}`);
                    }
                    req += "@" + ver;
                }
                this.addDevDeps(req);
            }
        }
        for (const dep of this.project.deps.all) {
            let version = dep.version ?? "*";
            let name = dep.name;
            if (name.startsWith("file:")) {
                const localDependencyPath = name.substring(5);
                const depPackageJson = path_1.resolve(this.project.outdir, localDependencyPath, "package.json");
                const pkgFile = fs_1.readFileSync(depPackageJson, "utf8");
                const pkg = JSON.parse(pkgFile);
                version = localDependencyPath;
                name = pkg.name;
            }
            switch (dep.type) {
                case dependencies_1.DependencyType.BUNDLED:
                    bundledDependencies.push(name);
                    if (this.project.deps.all.find((d) => d.name === name && d.type === dependencies_1.DependencyType.PEER)) {
                        throw new Error(`unable to bundle "${name}". it cannot appear as a peer dependency`);
                    }
                    // also add as a runtime dependency
                    dependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.PEER:
                    peerDependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.TEST:
                case dependencies_1.DependencyType.DEVENV:
                case dependencies_1.DependencyType.BUILD:
                    devDependencies[name] = version;
                    break;
            }
        }
        // returns a lazy value to normalize during synthesis
        const normalize = (obj) => () => util_1.sorted(obj);
        // update the manifest we are about to save into `package.json`
        this.manifest.devDependencies = normalize(devDependencies);
        this.manifest.peerDependencies = normalize(peerDependencies);
        this.manifest.dependencies = normalize(dependencies);
        this.manifest.bundledDependencies = util_1.sorted(bundledDependencies);
        // nothing further to do if package.json file does not exist
        const pkg = this.readPackageJson();
        if (!pkg) {
            return { devDependencies, peerDependencies, dependencies };
        }
        const readDeps = (user, current = {}) => {
            for (const [name, userVersion] of Object.entries(user)) {
                const currentVersion = current[name];
                // respect user version if it's not '*' or if current version is undefined
                if (userVersion !== "*" || !currentVersion || currentVersion === "*") {
                    continue;
                }
                // memoize current version in memory so it is preserved when saving
                user[name] = currentVersion;
            }
            // report removals
            for (const name of Object.keys(current ?? {})) {
                if (!user[name]) {
                    this.project.logger.verbose(`${name}: removed`);
                }
            }
        };
        readDeps(devDependencies, pkg.devDependencies);
        readDeps(dependencies, pkg.dependencies);
        readDeps(peerDependencies, pkg.peerDependencies);
        return { devDependencies, dependencies, peerDependencies };
    }
    /**
     * Resolves any deps that do not have a specified version (e.g. `*`) and
     * update `package.json` if needed.
     *
     * @returns `true` if package.json was updated or `false` if not.
     */
    resolveDepsAndWritePackageJson() {
        const outdir = this.project.outdir;
        const rootPackageJson = path_1.join(outdir, "package.json");
        const original = fs_1.readFileSync(rootPackageJson, "utf8");
        const pkg = JSON.parse(original);
        const resolveDeps = (current, user) => {
            const result = {};
            current = current ?? {};
            user = user ?? {};
            for (const [name, currentDefinition] of Object.entries(user)) {
                // find actual version from node_modules
                let desiredVersion = currentDefinition;
                if (currentDefinition === "*") {
                    try {
                        const modulePath = require.resolve(`${name}/package.json`, {
                            paths: [outdir],
                        });
                        const module = fs_extra_1.readJsonSync(modulePath);
                        desiredVersion = `^${module.version}`;
                    }
                    catch (e) { }
                    if (!desiredVersion) {
                        this.project.logger.warn(`unable to resolve version for ${name} from installed modules`);
                        continue;
                    }
                }
                if (currentDefinition !== desiredVersion) {
                    this.project.logger.verbose(`${name}: ${currentDefinition} => ${desiredVersion}`);
                }
                result[name] = desiredVersion;
            }
            // print removed packages
            for (const name of Object.keys(current)) {
                if (!result[name]) {
                    this.project.logger.verbose(`${name} removed`);
                }
            }
            return result;
        };
        const rendered = this._renderedDeps;
        if (!rendered) {
            throw new Error("assertion failed");
        }
        const deps = resolveDeps(pkg.dependencies, rendered.dependencies);
        const devDeps = resolveDeps(pkg.devDependencies, rendered.devDependencies);
        const peerDeps = resolveDeps(pkg.peerDependencies, rendered.peerDependencies);
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const [name, version] of Object.entries(peerDeps)) {
                // Skip if we already have a runtime dependency on this peer
                // or if devDependency version is already set.
                // Relies on the "*" devDependency added in the presynth step
                if (deps[name] || rendered.devDependencies[name] !== "*") {
                    continue;
                }
                // Take version and pin as dev dependency
                const ver = util_2.minVersion(version);
                if (!ver) {
                    throw new Error(`unable to determine minimum semver for peer dependency ${name}@${version}`);
                }
                devDeps[name] = ver;
            }
        }
        pkg.dependencies = util_1.sorted(deps);
        pkg.devDependencies = util_1.sorted(devDeps);
        pkg.peerDependencies = util_1.sorted(peerDeps);
        const jsonContent = JSON.stringify(pkg, undefined, 2);
        const updated = this.packageManager === NodePackageManager.NPM
            ? `${jsonContent}\n`
            : jsonContent;
        if (original === updated) {
            return false;
        }
        util_1.writeFile(rootPackageJson, updated);
        return true;
    }
    renderPublishConfig() {
        // omit values if they are the same as the npm defaults
        return _resolve_1.resolve({
            registry: this.npmRegistryUrl !== DEFAULT_NPM_REGISTRY_URL
                ? this.npmRegistryUrl
                : undefined,
            access: this.npmAccess !== defaultNpmAccess(this.packageName)
                ? this.npmAccess
                : undefined,
        }, { omitEmpty: true });
    }
    renderKeywords() {
        const kwds = Array.from(this.keywords);
        return util_1.sorted(kwds.sort());
    }
    renderEngines() {
        return util_1.sorted(this.engines);
    }
    autoDiscoverBinaries() {
        const binrel = "bin";
        const bindir = path_1.join(this.project.outdir, binrel);
        if (fs_extra_1.existsSync(bindir)) {
            for (const file of fs_extra_1.readdirSync(bindir)) {
                try {
                    fs_extra_1.accessSync(path_1.join(bindir, file), fs_extra_1.constants.X_OK);
                    this.bin[file] = path_1.join(binrel, file).replace(/\\/g, "/");
                }
                catch (e) {
                    // not executable, skip
                }
            }
        }
    }
    renderAuthor(options) {
        let author;
        if (options.authorName) {
            author = {
                name: options.authorName,
                email: options.authorEmail,
                url: options.authorUrl,
                organization: options.authorOrganization ?? false,
            };
        }
        else {
            if (options.authorEmail ||
                options.authorUrl ||
                options.authorOrganization !== undefined) {
                throw new Error('"authorName" is required if specifying "authorEmail" or "authorUrl"');
            }
        }
        return author;
    }
    renderBin() {
        return util_1.sorted(this.bin);
    }
    renderScripts() {
        const result = {};
        for (const task of this.project.tasks.all.sort((x, y) => x.name.localeCompare(y.name))) {
            result[task.name] = this.npmScriptForTask(task);
        }
        return result;
    }
    npmScriptForTask(task) {
        return `${this.projenCommand} ${task.name}`;
    }
    readPackageJson() {
        const file = path_1.join(this.project.outdir, "package.json");
        if (!fs_extra_1.existsSync(file)) {
            return undefined;
        }
        return fs_extra_1.readJsonSync(file);
    }
    installDependencies() {
        util_1.exec(this.renderInstallCommand(this.isAutomatedBuild), {
            cwd: this.project.outdir,
        });
    }
}
exports.NodePackage = NodePackage;
_a = JSII_RTTI_SYMBOL_1;
NodePackage[_a] = { fqn: "projen.javascript.NodePackage", version: "0.60.16" };
/**
 * The node package manager to use.
 */
var NodePackageManager;
(function (NodePackageManager) {
    /**
     * Use `yarn` as the package manager.
     */
    NodePackageManager["YARN"] = "yarn";
    /**
     * Use `npm` as the package manager.
     */
    NodePackageManager["NPM"] = "npm";
    /**
     * Use `pnpm` as the package manager.
     */
    NodePackageManager["PNPM"] = "pnpm";
})(NodePackageManager = exports.NodePackageManager || (exports.NodePackageManager = {}));
/**
 * Npm package access level
 */
var NpmAccess;
(function (NpmAccess) {
    /**
     * Package is public.
     */
    NpmAccess["PUBLIC"] = "public";
    /**
     * Package can only be accessed with credentials.
     */
    NpmAccess["RESTRICTED"] = "restricted";
})(NpmAccess = exports.NpmAccess || (exports.NpmAccess = {}));
/**
 * Determines if an npm package is "scoped" (i.e. it starts with "xxx@").
 */
function isScoped(packageName) {
    return packageName.includes("@");
}
function defaultNpmAccess(packageName) {
    return isScoped(packageName) ? NpmAccess.RESTRICTED : NpmAccess.PUBLIC;
}
function defaultNpmToken(npmToken, registry) {
    // if we are publishing to AWS CdodeArtifact, no NPM_TOKEN used (will be requested using AWS CLI later).
    if (release_1.isAwsCodeArtifactRegistry(registry)) {
        return undefined;
    }
    // if we are publishing to GitHub Packages, default to GITHUB_TOKEN.
    const isGitHubPackages = registry === GITHUB_PACKAGES_REGISTRY;
    return (npmToken ??
        (isGitHubPackages ? DEFAULT_GITHUB_TOKEN_SECRET : DEFAULT_NPM_TOKEN_SECRET));
}
exports.defaultNpmToken = defaultNpmToken;
function determineLockfile(packageManager) {
    if (packageManager === NodePackageManager.YARN) {
        return "yarn.lock";
    }
    else if (packageManager === NodePackageManager.NPM) {
        return "package-lock.json";
    }
    else if (packageManager === NodePackageManager.PNPM) {
        return "pnpm-lock.yaml";
    }
    throw new Error(`unsupported package manager ${packageManager}`);
}
//# sourceMappingURL=data:application/json;base64,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