"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const util_1 = require("./util");
/**
 * Adds support for bundling JavaScript applications and dependencies into a
 * single file. In the future, this will also supports bundling websites.
 */
class Bundler extends component_1.Component {
    /**
     * Creates a `Bundler`.
     */
    constructor(project, options = {}) {
        super(project);
        this.esbuildVersion = options.esbuildVersion;
        this.bundledir = options.assetsDir ?? "assets";
        this.addToPreCompile = options.addToPreCompile ?? true;
    }
    /**
     * Returns the `Bundler` instance associated with a project or `undefined` if
     * there is no Bundler.
     * @param project The project
     * @returns A bundler
     */
    static of(project) {
        const isBundler = (o) => o instanceof Bundler;
        return project.components.find(isBundler);
    }
    /**
     * Gets or creates the singleton "bundle" task of the project.
     *
     * If the project doesn't have a "bundle" task, it will be created and spawned
     * during the pre-compile phase.
     */
    get bundleTask() {
        if (!this._task) {
            this.addBundlingSupport();
            this._task = this.project.tasks.addTask("bundle", {
                description: "Prepare assets",
            });
            // install the bundle task into the pre-compile phase.
            if (this.addToPreCompile) {
                this.project.preCompileTask.spawn(this._task);
            }
        }
        return this._task;
    }
    /**
     * Adds a task to the project which bundles a specific entrypoint and all of
     * its dependencies into a single javascript output file.
     *
     * @param entrypoint The relative path of the artifact within the project
     * @param options Bundling options
     */
    addBundle(entrypoint, options) {
        const name = util_1.renderBundleName(entrypoint);
        const outdir = path_1.join(this.bundledir, name);
        const outfile = path_1.join(outdir, options.outfile ?? "index.js");
        const args = [
            "esbuild",
            "--bundle",
            entrypoint,
            `--target="${options.target}"`,
            `--platform="${options.platform}"`,
            `--outfile="${outfile}"`,
        ];
        for (const x of options.externals ?? []) {
            args.push(`--external:${x}`);
        }
        const sourcemap = options.sourcemap ?? false;
        if (sourcemap) {
            args.push("--sourcemap");
        }
        const bundleTask = this.project.addTask(`bundle:${name}`, {
            description: `Create a JavaScript bundle from ${entrypoint}`,
            exec: args.join(" "),
        });
        this.bundleTask.spawn(bundleTask);
        if (options.executable ?? false) {
            bundleTask.exec(`chmod +x ${outfile}`);
        }
        let watchTask;
        const watch = options.watchTask ?? true;
        if (watch) {
            watchTask = this.project.addTask(`bundle:${name}:watch`, {
                description: `Continuously update the JavaScript bundle from ${entrypoint}`,
                exec: `${args.join(" ")} --watch`,
            });
        }
        return {
            bundleTask: bundleTask,
            watchTask: watchTask,
            outdir: outdir,
            outfile: outfile,
        };
    }
    /**
     * Add bundling support to a project. This is called implicitly when
     * `bundleTask` is referenced first. It adds the dependency on `esbuild`,
     * gitignore/npmignore, etc.
     */
    addBundlingSupport() {
        const ignoreEntry = `/${this.bundledir}/`;
        this.project.addGitIgnore(ignoreEntry);
        this.project.addPackageIgnore(`!${ignoreEntry}`); // include in tarball
        const dep = this.esbuildVersion
            ? `esbuild@${this.esbuildVersion}`
            : "esbuild";
        this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
    }
}
exports.Bundler = Bundler;
_a = JSII_RTTI_SYMBOL_1;
Bundler[_a] = { fqn: "projen.javascript.Bundler", version: "0.60.16" };
//# sourceMappingURL=data:application/json;base64,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