"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pom = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const semver_1 = require("../util/semver");
const xmlfile_1 = require("../xmlfile");
const POM_XML_ATTRS = {
    "@xsi:schemaLocation": "http://maven.apache.org/POM/4.0.0 http://maven.apache.org/xsd/maven-4.0.0.xsd",
    "@xmlns": "http://maven.apache.org/POM/4.0.0",
    "@xmlns:xsi": "http://www.w3.org/2001/XMLSchema-instance",
};
/**
 * A Project Object Model or POM is the fundamental unit of work in Maven. It is
 * an XML file that contains information about the project and configuration
 * details used by Maven to build the project.
 */
class Pom extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.properties = {};
        this.repositories = [];
        this.fileName = "pom.xml";
        this.groupId = options.groupId;
        this.artifactId = options.artifactId;
        this.version = options.version;
        this.packaging = options.packaging ?? "jar";
        this.name = project.name;
        this.description = options.description;
        this.url = options.url;
        new xmlfile_1.XmlFile(project, this.fileName, { obj: () => this.synthPom() });
    }
    /**
     * Adds a key/value property to the pom.
     * @param key the key
     * @param value the value
     */
    addProperty(key, value) {
        this.properties[key] = value;
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.RUNTIME);
    }
    /**
     * Adds a test dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`
     */
    addTestDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.TEST);
    }
    /**
     * Adds a build plugin to the pom.
     *
     * The plug in is also added as a BUILD dep to the project.
     *
     * @param spec dependency spec (`group/artifact@version`)
     * @param options plugin options
     */
    addPlugin(spec, options = {}) {
        for (const dep of options.dependencies ?? []) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
        return this.project.deps.addDependency(spec, dependencies_1.DependencyType.BUILD, options);
    }
    /**
     * Adds a repository to the pom
     * @param repository the repository to add
     */
    addRepository(repository) {
        this.repositories.push(repository);
    }
    synthPom() {
        return _resolve_1.resolve({
            project: {
                ...POM_XML_ATTRS,
                modelVersion: "4.0.0",
                groupId: this.groupId,
                artifactId: this.artifactId,
                version: this.version,
                packaging: this.packaging,
                name: this.name,
                description: this.description,
                url: this.url,
                properties: this.properties,
                ...this.synthRepositories(),
                ...this.synthDependencies(),
            },
        }, { omitEmpty: true });
    }
    synthDependencies() {
        const deps = this.project.deps.all;
        if (deps.length === 0) {
            return;
        }
        const dependencies = [];
        const plugins = [];
        for (const dep of deps) {
            switch (dep.type) {
                case dependencies_1.DependencyType.PEER:
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies.push(mavenCoords(dep));
                    break;
                case dependencies_1.DependencyType.TEST:
                    dependencies.push({
                        ...mavenCoords(dep),
                        scope: "test",
                    });
                    break;
                // build maps to plugins
                case dependencies_1.DependencyType.BUILD:
                    plugins.push({
                        ...mavenCoords(dep),
                        ...pluginConfig(dep.metadata),
                    });
                    break;
                default:
                    throw new Error(`unsupported dependency type: ${dep.type}`);
            }
        }
        return {
            build: { plugins: { plugin: plugins } },
            dependencies: { dependency: dependencies },
        };
    }
    synthRepositories() {
        if (this.repositories.length === 0) {
            return;
        }
        return {
            repositories: { repository: this.repositories },
        };
    }
}
exports.Pom = Pom;
_a = JSII_RTTI_SYMBOL_1;
Pom[_a] = { fqn: "projen.java.Pom", version: "0.60.16" };
/**
 * Parses maven groupId and artifactId from a dependency name.
 *
 *     name    <=> <groupId>/<artifactId>
 *     version <=> <version>
 */
function mavenCoords(dep) {
    const name = dep.name;
    const parts = name.split("/");
    if (parts.length !== 2) {
        throw new Error(`invalid maven coordinates in dependency named "${name}". format is "<groupId>/<artifactId>". For example "org.junit.jupiter/junit-jupiter-engine"`);
    }
    return {
        groupId: parts[0],
        artifactId: parts[1],
        version: dep.version ? semver_1.toMavenVersionRange(dep.version) : undefined,
    };
}
function pluginConfig(options = {}) {
    return {
        configuration: options.configuration,
        dependencies: options.dependencies && options.dependencies.length > 0
            ? {
                dependency: options.dependencies?.map((d) => mavenCoords(dependencies_1.Dependencies.parseDependency(d))),
            }
            : undefined,
        executions: options.executions?.map((e) => ({
            execution: {
                id: e.id,
                goals: e.goals.map((goal) => ({ goal })),
            },
        })),
    };
}
//# sourceMappingURL=data:application/json;base64,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