"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.resolveProjectType = exports.discover = void 0;
const path = require("path");
const case_1 = require("case");
const fs = require("fs-extra");
const PROJEN_MODULE_ROOT = path.join(__dirname, "..");
const PROJECT_BASE_FQN = "projen.Project";
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = discoverJsiiTypes(...moduleDirs);
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        if (isProjectType(jsii, fqn)) {
            const p = toProjectType(jsii, fqn);
            result.push(p);
        }
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
/**
 * Resolve all jsii types from @modulesDirs.
 * When a jsii module is found it will recusively list the types from the dependant module as well
 *
 * @param moduleDirs
 * @returns
 */
function discoverJsiiTypes(...moduleDirs) {
    const jsii = {};
    const discoveredManifests = [];
    const discoverJsii = (dir) => {
        const jsiiFile = path.join(dir, ".jsii");
        if (!fs.existsSync(jsiiFile)) {
            return;
        } // no jsii manifest
        const manifest = fs.readJsonSync(jsiiFile);
        if (discoveredManifests.includes(manifest.fingerprint)) {
            return;
        }
        discoveredManifests.push(manifest.fingerprint);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = {
                ...type,
            };
        }
        // Also search recursively in nested project dependencies. If the requested module is an external module
        // this will also end-up in the projen module and add the projen types
        if (manifest.dependencies) {
            for (const dependency of Object.keys(manifest.dependencies)) {
                const nestedDependencyFolder = path.dirname(require.resolve(`${dependency}/package.json`, {
                    paths: [dir],
                }));
                if (fs.existsSync(nestedDependencyFolder)) {
                    discoverJsii(nestedDependencyFolder);
                }
            }
        }
    };
    // read all .jsii manifests from all requested modules and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        // Read from scoped packages
        if (dir.includes("@") && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map((file) => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    return jsii;
}
function resolveProjectType(projectFqn) {
    let [moduleName] = projectFqn.split(".");
    if (moduleName === "projen") {
        moduleName = PROJEN_MODULE_ROOT;
    }
    // try picking the manifest. We only need the base folder but this is directly a nice check if we request from a valid jsii package
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`, {
        paths: [process.cwd()],
    });
    const moduleFolder = path.dirname(jsiiManifestFile);
    // Read all jsii types that can be loaded from this project type
    const jsii = discoverJsiiTypes(moduleFolder);
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    if (!isProjectType(jsii, fqn)) {
        throw new Error(`Fully qualified name "${fqn}" is not a valid project type.`);
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf(".") + 1);
    const docsurl = `https://projen.io/api/API.html#projen-${typename
        .toLocaleLowerCase()
        .replace(/\./g, "-")}`;
    let pjid = typeinfo.docs?.custom?.pjid ?? case_1.snake(typename).replace(/_project$/, "");
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn).sort((o1, o2) => o1.name.localeCompare(o2.name)),
        docs: typeinfo.docs?.summary,
        docsurl,
    };
}
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split(".");
    if (moduleName === "projen") {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return fs.readJsonSync(jsiiManifestFile);
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    const options = {};
    const params = jsii[fqn]?.initializer?.parameters ?? [];
    const optionsParam = params[0];
    const optionsTypeFqn = optionsParam?.type?.fqn;
    if (params.length > 1 ||
        (params.length === 1 && optionsParam?.name !== "options")) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.switch.localeCompare(b.switch));
    function addOptions(ofqn, basePath = [], optional = false) {
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of struct.properties ?? []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let jsiiKind;
            if (prop.type?.fqn) {
                jsiiKind = jsii[prop.type?.fqn].kind; // e.g. 'class', 'interface', 'enum'
            }
            const isOptional = optional || prop.optional;
            let defaultValue = prop.docs?.default;
            if (defaultValue === "undefined") {
                defaultValue = undefined;
            }
            // if this is a mandatory option and we have a default value, it has to be JSON-parsable to the correct type
            if (!isOptional && defaultValue) {
                if (!prop.type?.primitive) {
                    throw new Error(`required option "${prop.name}" with a @default must use primitive types (string, number or boolean). type found is: ${JSON.stringify(prop.type)}`);
                }
                checkDefaultIsParsable(prop.name, defaultValue, prop.type?.primitive);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                fqn: prop.type?.fqn,
                docs: prop.docs.summary,
                simpleType: prop.type ? getSimpleTypeName(prop.type) : "unknown",
                fullType: prop.type,
                kind: jsiiKind,
                jsonLike: prop.type ? isJsonLike(jsii, prop.type) : undefined,
                switch: propPath.map((p) => case_1.snake(p).replace(/_/g, "-")).join("-"),
                default: defaultValue,
                optional: isOptional,
                featured: prop.docs?.custom?.featured === "true",
                deprecated: prop.docs.stability === "deprecated" ? true : undefined,
            });
        }
        for (const ifc of struct.interfaces ?? []) {
            addOptions(ifc);
        }
    }
}
function getSimpleTypeName(type) {
    if (type?.primitive) {
        return type.primitive; // e.g. 'string', 'boolean', 'number'
    }
    else if (type?.fqn) {
        return type.fqn.split(".").pop(); // projen.NodeProjectOptions -> NodeProjectOptions
    }
    else {
        // any other types such as collection types
        return "unknown";
    }
}
/**
 * Whether a value of this type is serializable into JSON.
 */
function isJsonLike(jsii, type) {
    if (type.primitive) {
        // string, boolean, number, any
        return true;
    }
    else if (type.fqn) {
        const kind = jsii[type.fqn].kind;
        if (["interface", "enum"].includes(kind)) {
            // not 'class'
            return true;
        }
    }
    else if (type.collection) {
        return isJsonLike(jsii, type.collection.elementtype);
    }
    return false;
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    const type = jsii[fqn];
    if (!type) {
        throw new Error(`Could not find project type with fqn "${fqn}" in  .jsii file.`);
    }
    if (type.kind !== "class") {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if (type.docs?.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function checkDefaultIsParsable(prop, value, type) {
    // macros are pass-through
    if (value.startsWith("$")) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        if (typeof parsed !== type) {
            throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
        }
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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