"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Gitpod = exports.GitpodPortVisibility = exports.GitpodOnOpen = exports.GitpodOpenIn = exports.GitpodOpenMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("./component");
const yaml_1 = require("./yaml");
/**
 * Relative path of Gitpod file to generate
 */
const GITPOD_FILE = ".gitpod.yml";
/**
 * https://www.gitpod.io/docs/configuration/
 * https://www.gitpod.io/docs/config-start-tasks/
 * https://www.gitpod.io/docs/prebuilds/#configure-the-github-app
 * https://www.gitpod.io/docs/vscode-extensions/
 */
/**
 * Configure how the terminal should be opened relative to the previous task.
 */
var GitpodOpenMode;
(function (GitpodOpenMode) {
    /**
     * Opens in the same tab group right after the previous tab
     */
    GitpodOpenMode["TAB_AFTER"] = "tab-after";
    /**
     * Opens in the same tab group left before the previous tab
     */
    GitpodOpenMode["TAB_BEFORE"] = "tab-before";
    /**
     * Splits and adds the terminal to the right
     */
    GitpodOpenMode["SPLIT_RIGHT"] = "split-right";
    /**
     * Splits and adds the terminal to the left
     */
    GitpodOpenMode["SPLIT_LEFT"] = "split-left";
    /**
     * Splits and adds the terminal to the top
     */
    GitpodOpenMode["SPLIT_TOP"] = "split-top";
    /**
     * Splits and adds the terminal to the bottom
     */
    GitpodOpenMode["SPLIT_BOTTOM"] = "split-bottom";
})(GitpodOpenMode = exports.GitpodOpenMode || (exports.GitpodOpenMode = {}));
/**
 * Configure where in the IDE the terminal should be opened.
 */
var GitpodOpenIn;
(function (GitpodOpenIn) {
    /**
     * the bottom panel (default)
     */
    GitpodOpenIn["BOTTOM"] = "bottom";
    /**
     * the left panel
     */
    GitpodOpenIn["LEFT"] = "left";
    /**
     * the right panel
     */
    GitpodOpenIn["RIGHT"] = "right";
    /**
     * the main editor area
     */
    GitpodOpenIn["MAIN"] = "main";
})(GitpodOpenIn = exports.GitpodOpenIn || (exports.GitpodOpenIn = {}));
/**
 * What to do when a service on a port is detected.
 */
var GitpodOnOpen;
(function (GitpodOnOpen) {
    /**
     * Open a new browser tab
     */
    GitpodOnOpen["OPEN_BROWSER"] = "open-browser";
    /**
     * Open a preview on the right side of the IDE
     */
    GitpodOnOpen["OPEN_PREVIEW"] = "open-preview";
    /**
     * Show a notification asking the user what to do (default)
     */
    GitpodOnOpen["NOTIFY"] = "notify";
    /**
     * Do nothing.
     */
    GitpodOnOpen["IGNORE"] = "ignore";
})(GitpodOnOpen = exports.GitpodOnOpen || (exports.GitpodOnOpen = {}));
/**
 * Whether the port visibility should be private or public
 */
var GitpodPortVisibility;
(function (GitpodPortVisibility) {
    /**
     * Allows everyone with the port URL to access the port (default)
     */
    GitpodPortVisibility["PUBLIC"] = "public";
    /**
     * Only allows users with workspace access to access the port
     */
    GitpodPortVisibility["PRIVATE"] = "private";
})(GitpodPortVisibility = exports.GitpodPortVisibility || (exports.GitpodPortVisibility = {}));
/**
 * The Gitpod component which emits .gitpod.yml
 */
class Gitpod extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        this.tasks = new Array();
        this.ports = new Array();
        this.vscodeExtensions = new Array();
        this.dockerImage = options?.dockerImage;
        if (options?.tasks) {
            for (const task of options.tasks) {
                this.addTasks(task);
            }
        }
        if (options?.prebuilds) {
            this.addPrebuilds(options.prebuilds);
        }
        if (options?.ports) {
            this.addPorts(...options.ports);
        }
        if (options?.vscodeExtensions) {
            this.addVscodeExtensions(...options.vscodeExtensions);
        }
        this.config = {
            image: () => this.renderDockerImage(),
            tasks: this.tasks,
            github: () => this.renderPrebuilds(),
            ports: this.ports,
            vscode: {
                extensions: this.vscodeExtensions,
            },
        };
        new yaml_1.YamlFile(this.project, GITPOD_FILE, {
            obj: this.config,
            omitEmpty: true,
        });
    }
    /**
     * Add a custom Docker image or Dockerfile for the container.
     *
     * @param image The Docker image
     */
    addDockerImage(image) {
        if (this.dockerImage) {
            throw new Error("dockerImage cannot be redefined.");
        }
        this.dockerImage = image;
    }
    /**
     * Add tasks to run when gitpod starts.
     *
     * By default, all tasks will be run in parallel. To run tasks in sequence,
     * create a new `Task` and specify the other tasks as subtasks.
     *
     * @param tasks The new tasks
     */
    addTasks(...tasks) {
        this.tasks.push(...tasks.map((task) => ({
            name: task.name,
            command: `npx projen ${task.name}`,
        })));
    }
    /**
     * Add a prebuilds configuration for the Gitpod App
     * @param config The configuration
     */
    addPrebuilds(config) {
        this.prebuilds = config;
    }
    /**
     * Add a task with more granular options.
     *
     * By default, all tasks will be run in parallel. To run tasks in sequence,
     * create a new `Task` and set the other tasks as subtasks.
     *
     * @param options The task parameters
     */
    addCustomTask(options) {
        this.tasks.push({
            name: options.name,
            command: options.command,
            openMode: options.openMode,
            openIn: options.openIn,
            before: options.before,
            init: options.init,
            prebuild: options.prebuild,
        });
    }
    /**
     * Add ports that should be exposed (forwarded) from the container.
     *
     * @param ports The new ports
     */
    addPorts(...ports) {
        this.ports.push(...ports.map((port) => ({ port: port })));
    }
    /**
     * Add a list of VSCode extensions that should be automatically installed
     * in the container.
     *
     * These must be in the format defined in the Open VSX registry.
     * @example 'scala-lang.scala@0.3.9:O5XmjwY5Gz+0oDZAmqneJw=='
     * @see https://www.gitpod.io/docs/vscode-extensions/
     *
     * @param extensions The extension IDs
     */
    addVscodeExtensions(...extensions) {
        this.vscodeExtensions.push(...extensions);
    }
    renderDockerImage() {
        if (this.dockerImage?.image) {
            return this.dockerImage.image;
        }
        else if (this.dockerImage?.dockerFile) {
            return {
                file: this.dockerImage.dockerFile,
            };
        }
        else {
            return undefined;
        }
    }
    renderPrebuilds() {
        if (this.prebuilds) {
            return {
                prebuilds: this.prebuilds,
            };
        }
        else {
            return undefined;
        }
    }
}
exports.Gitpod = Gitpod;
_a = JSII_RTTI_SYMBOL_1;
Gitpod[_a] = { fqn: "projen.Gitpod", version: "0.60.16" };
//# sourceMappingURL=data:application/json;base64,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