"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaExtension = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
const internal_1 = require("./internal");
const lambda_function_1 = require("./lambda-function");
/**
 * Create a Lambda Extension
 */
class LambdaExtension extends component_1.Component {
    constructor(project, options) {
        super(project);
        const basePath = path_1.join(path_1.dirname(options.entrypoint), path_1.basename(options.entrypoint, internal_1.TYPESCRIPT_LAMBDA_EXTENSION_EXT));
        const name = options.name ?? path_1.basename(basePath);
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const compatibleRuntimes = options.compatibleRuntimes ?? [
            lambda_function_1.LambdaRuntime.NODEJS_16_X,
            lambda_function_1.LambdaRuntime.NODEJS_14_X,
            lambda_function_1.LambdaRuntime.NODEJS_12_X,
        ];
        if (compatibleRuntimes.length === 0) {
            throw new Error("Compatible runtimes must include at least one runtime");
        }
        // Use the lowest runtime version to bundle
        const [bundlerRuntime] = compatibleRuntimes.sort((a, b) => a.functionRuntime.localeCompare(b.functionRuntime));
        // Allow extension code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(options.entrypoint);
        const bundle = bundler.addBundle(options.entrypoint, {
            platform: bundlerRuntime.esbuildPlatform,
            target: bundlerRuntime.esbuildTarget,
            externals: ["aws-sdk"],
            outfile: `extensions/${name}`,
            // Make the output executable because Lambda expects to run
            // extensions as stand-alone programs alongside the main lambda
            // process.
            executable: true,
            ...options.bundlingOptions,
        });
        const constructFile = options.constructFile ?? `${basePath}-layer-version.ts`;
        new LambdaLayerConstruct(project, {
            constructFile: constructFile,
            constructName: options.constructName,
            assetDir: bundle.outdir,
            compatibleRuntimes: compatibleRuntimes,
            description: `Provides a Lambda Extension \`${name}\` from ${internal_1.convertToPosixPath(options.entrypoint)}`,
            cdkDeps: options.cdkDeps,
        });
    }
}
exports.LambdaExtension = LambdaExtension;
_a = JSII_RTTI_SYMBOL_1;
LambdaExtension[_a] = { fqn: "projen.awscdk.LambdaExtension", version: "0.60.16" };
class LambdaLayerConstruct extends source_code_1.SourceCode {
    constructor(project, options) {
        super(project, options.constructFile);
        const src = this;
        const cdkDeps = options.cdkDeps;
        const constructName = options.constructName ?? case_1.pascal(path_1.basename(options.constructFile, ".ts"));
        const propsType = `${constructName}Props`;
        const assetDir = path_1.relative(path_1.dirname(options.constructFile), options.assetDir);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        src.open(`export interface ${propsType} extends lambda.LayerVersionOptions {`);
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * ${options.description}`);
        src.line(" */");
        src.open(`export class ${constructName} extends lambda.LayerVersion {`);
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: ${encodeCodeString(options.description)},`);
        src.line("...props,");
        src.open("compatibleRuntimes: [");
        for (const runtime of options.compatibleRuntimes) {
            src.line(`new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        }
        src.close("],");
        src.open(`code: lambda.Code.fromAsset(path.join(__dirname,`);
        src.line(`${encodeCodeString(internal_1.convertToPosixPath(assetDir))})),`);
        src.close();
        src.close("});");
        src.close("}");
        src.close("}");
    }
}
/**
 * Encodes a string for embedding in source code.
 */
function encodeCodeString(value) {
    const json = JSON.stringify(value);
    const escapedString = json.substring(1, json.length - 1).replace(/'/g, "\\'");
    return `'${escapedString}'`;
}
//# sourceMappingURL=data:application/json;base64,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