# ------------------------------------------------------------------------------
#
# Project: pyows <http://eoxserver.org>
# Authors: Fabian Schindler <fabian.schindler@eox.at>
#
# ------------------------------------------------------------------------------
# Copyright (C) 2020 EOX IT Services GmbH
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies of this Software or works derived from this Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
# ------------------------------------------------------------------------------

from datetime import datetime, date # noqa

from ows.util import Version, year, month
from ..types import GetMapRequest, GetFeatureInfoRequest, BoundingBox, Range
from .decoders import kvp_decode_getmap, kvp_decode_getfeatureinfo


# ------------------------------------------------------------------------------
# GetMap
# ------------------------------------------------------------------------------

def test_kvp_decode_getmap():
    # simplest form
    request = "service=WMS&version=1.3.0&request=GetMap&layers=a,b,c&styles=s1,s2,&crs=EPSG:4326&bbox=0,0,10,10&width=256&height=256&format=image/jpeg" # noqa
    assert kvp_decode_getmap(request) == GetMapRequest(
        Version(1, 3, 0),
        layers=['a', 'b', 'c'],
        styles=['s1', 's2', None],
        bounding_box=BoundingBox('EPSG:4326', [0, 0, 10, 10]),
        width=256,
        height=256,
        format='image/jpeg',
        dimensions={}
    )

    # multi-dim
    request = "service=WMS&version=1.3.0&request=GetMap&layers=a,b,c&styles=s1,s2,&crs=EPSG:4326&bbox=0,0,10,10&width=256&height=256&format=image/jpeg&time=2012/2013&elevation=1000&dim_wavelength=2456.2&dim_pressure=123,234&dim_range=0/1,2/4" # noqa
    assert kvp_decode_getmap(request) == GetMapRequest(
        Version(1, 3, 0),
        layers=['a', 'b', 'c'],
        styles=['s1', 's2', None],
        bounding_box=BoundingBox('EPSG:4326', [0, 0, 10, 10]),
        width=256,
        height=256,
        format='image/jpeg',
        time=Range(year(2012), year(2013)),
        elevation=1000,
        dimensions={
            'wavelength': '2456.2',
            'pressure': ['123', '234'],
            'range': [Range('0', '1'), Range('2', '4')]
        }
    )

    # elevation-range
    request = "service=WMS&version=1.3.0&request=GetMap&layers=a,b,c&styles=s1,s2,&crs=EPSG:4326&bbox=0,0,10,10&width=256&height=256&format=image/jpeg&elevation=1000/2000/10" # noqa
    assert kvp_decode_getmap(request) == GetMapRequest(
        Version(1, 3, 0),
        layers=['a', 'b', 'c'],
        styles=['s1', 's2', None],
        bounding_box=BoundingBox('EPSG:4326', [0, 0, 10, 10]),
        width=256,
        height=256,
        format='image/jpeg',
        elevation=Range(1000, 2000, 10),
        dimensions={}
    )

    # month to date
    request = "service=WMS&version=1.3.0&request=GetMap&layers=a,b,c&styles=s1,s2,&crs=EPSG:4326&bbox=0,0,10,10&width=256&height=256&format=image/jpeg&time=2012-02/2012-03-15" # noqa
    assert kvp_decode_getmap(request) == GetMapRequest(
        Version(1, 3, 0),
        layers=['a', 'b', 'c'],
        styles=['s1', 's2', None],
        bounding_box=BoundingBox('EPSG:4326', [0, 0, 10, 10]),
        width=256,
        height=256,
        format='image/jpeg',
        time=Range(month(2012, 2), date(2012, 3, 15)),
        dimensions={}
    )


def test_kvp_decode_getfeatureinfo():
    # minimal form
    request = "service=WMS&version=1.3.0&request=GetFeatureInfo&layers=a,b,c&styles=s1,s2,&crs=EPSG:4326&bbox=0,0,10,10&width=256&height=256&format=image/jpeg&query_layers=a,b&info_format=text/xml&i=12&j=12" # noqa
    assert kvp_decode_getfeatureinfo(request) == GetFeatureInfoRequest(
        Version(1, 3, 0),
        layers=['a', 'b', 'c'],
        styles=['s1', 's2', None],
        bounding_box=BoundingBox('EPSG:4326', [0, 0, 10, 10]),
        width=256,
        height=256,
        format='image/jpeg',
        dimensions={},
        query_layers=['a', 'b'],
        info_format='text/xml',
        i=12,
        j=12,
        # feature_count=15,
    )

    # full
    request = "service=WMS&version=1.3.0&request=GetFeatureInfo&layers=a,b,c&styles=s1,s2,&crs=EPSG:4326&bbox=0,0,10,10&width=256&height=256&format=image/jpeg&query_layers=a,b&info_format=text/xml&i=12&j=12&feature_count=15" # noqa
    assert kvp_decode_getfeatureinfo(request) == GetFeatureInfoRequest(
        Version(1, 3, 0),
        layers=['a', 'b', 'c'],
        styles=['s1', 's2', None],
        bounding_box=BoundingBox('EPSG:4326', [0, 0, 10, 10]),
        width=256,
        height=256,
        format='image/jpeg',
        dimensions={},
        query_layers=['a', 'b'],
        info_format='text/xml',
        i=12,
        j=12,
        feature_count=15,
    )
