#
# Lockstep Platform SDK for Python
#
# (c) 2021-2022 Lockstep, Inc.
#
# For the full copyright and license information, please view the LICENSE
# file that was distributed with this source code.
#
# @author     Lockstep Network <support@lockstep.io>
# @copyright  2021-2022 Lockstep, Inc.
# @link       https://github.com/Lockstep-Network/lockstep-sdk-python
#

from lockstep.lockstep_response import LockstepResponse
from lockstep.error_result import ErrorResult
from lockstep.action_result_model import ActionResultModel
from lockstep.fetch_result import FetchResult
from lockstep.models.paymentdetailheadermodel import PaymentDetailHeaderModel
from lockstep.models.paymentdetailmodel import PaymentDetailModel
from lockstep.models.paymentmodel import PaymentModel
from lockstep.models.paymentsummarymodel import PaymentSummaryModel
from requests.models import Response

class PaymentsClient:
    """
    API methods related to Payments
    """
    from lockstep.lockstep_api import LockstepApi

    def __init__(self, client: LockstepApi):
        self.client = client

    def retrieve_payment(self, id: str, include: str) -> LockstepResponse[PaymentModel]:
        """
        Retrieves the Payment specified by this unique identifier,
        optionally including nested data sets.

        A Payment represents money sent from one company to another. A
        single payment may contain payments for one or more invoices; it
        is also possible for payments to be made in advance of an
        invoice, for example, as a deposit. The creator of the Payment
        is identified by the `CustomerId` field, and the recipient of
        the Payment is identified by the `CompanyId` field. Most
        Payments are uniquely identified both by a Lockstep Platform ID
        number and a customer ERP "key" that was generated by the system
        that originated the Payment. Payments that have not been fully
        applied have a nonzero `UnappliedAmount` value, which represents
        a deposit that has been paid and not yet applied to an Invoice.

        Parameters
        ----------
        id : str
            The unique Lockstep Platform ID number of this Payment; NOT
            the customer's ERP key
        include : str
            To fetch additional data on this object, specify the list of
            elements to retrieve. Available collections: Applications,
            Notes, Attachments, CustomFields
        """
        path = f"/api/v1/Payments/{id}"
        result = self.client.send_request("GET", path, None, {"include": include}, None)
        if result.status_code >= 200 and result.status_code < 300:
            return LockstepResponse(True, result.status_code, PaymentModel(**result.json()), None)
        else:
            return LockstepResponse(False, result.status_code, None, ErrorResult(**result.json()))

    def update_payment(self, id: str, body: object) -> LockstepResponse[PaymentModel]:
        """
        Updates an existing Payment with the information supplied to
        this PATCH call.

        The PATCH method allows you to change specific values on the
        object while leaving other values alone. As input you should
        supply a list of field names and new values. If you do not
        provide the name of a field, that field will remain unchanged.
        This allows you to ensure that you are only updating the
        specific fields desired.

        A Payment represents money sent from one company to another. A
        single payment may contain payments for one or more invoices; it
        is also possible for payments to be made in advance of an
        invoice, for example, as a deposit. The creator of the Payment
        is identified by the `CustomerId` field, and the recipient of
        the Payment is identified by the `CompanyId` field. Most
        Payments are uniquely identified both by a Lockstep Platform ID
        number and a customer ERP "key" that was generated by the system
        that originated the Payment. Payments that have not been fully
        applied have a nonzero `UnappliedAmount` value, which represents
        a deposit that has been paid and not yet applied to an Invoice.

        Parameters
        ----------
        id : str
            The unique Lockstep Platform ID number of the Payment to
            update; NOT the customer's ERP key
        body : object
            A list of changes to apply to this Payment
        """
        path = f"/api/v1/Payments/{id}"
        result = self.client.send_request("PATCH", path, body, {}, None)
        if result.status_code >= 200 and result.status_code < 300:
            return LockstepResponse(True, result.status_code, PaymentModel(**result.json()), None)
        else:
            return LockstepResponse(False, result.status_code, None, ErrorResult(**result.json()))

    def delete_payment(self, id: str) -> LockstepResponse[ActionResultModel]:
        """
        Deletes the Payment referred to by this unique identifier.

        A Payment represents money sent from one company to another. A
        single payment may contain payments for one or more invoices; it
        is also possible for payments to be made in advance of an
        invoice, for example, as a deposit. The creator of the Payment
        is identified by the `CustomerId` field, and the recipient of
        the Payment is identified by the `CompanyId` field. Most
        Payments are uniquely identified both by a Lockstep Platform ID
        number and a customer ERP "key" that was generated by the system
        that originated the Payment. Payments that have not been fully
        applied have a nonzero `UnappliedAmount` value, which represents
        a deposit that has been paid and not yet applied to an Invoice.

        Parameters
        ----------
        id : str
            The unique Lockstep Platform ID number of the Payment to
            delete; NOT the customer's ERP key
        """
        path = f"/api/v1/Payments/{id}"
        result = self.client.send_request("DELETE", path, None, {}, None)
        if result.status_code >= 200 and result.status_code < 300:
            return LockstepResponse(True, result.status_code, ActionResultModel(**result.json()), None)
        else:
            return LockstepResponse(False, result.status_code, None, ErrorResult(**result.json()))

    def create_payments(self, body: list[PaymentModel]) -> LockstepResponse[list[PaymentModel]]:
        """
        Creates one or more Payments within this account and returns the
        records as created.

        A Payment represents money sent from one company to another. A
        single payment may contain payments for one or more invoices; it
        is also possible for payments to be made in advance of an
        invoice, for example, as a deposit. The creator of the Payment
        is identified by the `CustomerId` field, and the recipient of
        the Payment is identified by the `CompanyId` field. Most
        Payments are uniquely identified both by a Lockstep Platform ID
        number and a customer ERP "key" that was generated by the system
        that originated the Payment. Payments that have not been fully
        applied have a nonzero `UnappliedAmount` value, which represents
        a deposit that has been paid and not yet applied to an Invoice.

        Parameters
        ----------
        body : list[PaymentModel]
            The Payments to create
        """
        path = "/api/v1/Payments"
        result = self.client.send_request("POST", path, body, {}, None)
        if result.status_code >= 200 and result.status_code < 300:
            return LockstepResponse(True, result.status_code, list[PaymentModel](**result.json()), None)
        else:
            return LockstepResponse(False, result.status_code, None, ErrorResult(**result.json()))

    def query_payments(self, filter: str, include: str, order: str, pageSize: int, pageNumber: int) -> LockstepResponse[FetchResult[PaymentModel]]:
        """
        Queries Payments for this account using the specified filtering,
        sorting, nested fetch, and pagination rules requested.

        More information on querying can be found on the [Searchlight
        Query Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        page on the Lockstep Developer website.

        A Payment represents money sent from one company to another. A
        single payment may contain payments for one or more invoices; it
        is also possible for payments to be made in advance of an
        invoice, for example, as a deposit. The creator of the Payment
        is identified by the `CustomerId` field, and the recipient of
        the Payment is identified by the `CompanyId` field. Most
        Payments are uniquely identified both by a Lockstep Platform ID
        number and a customer ERP "key" that was generated by the system
        that originated the Payment. Payments that have not been fully
        applied have a nonzero `UnappliedAmount` value, which represents
        a deposit that has been paid and not yet applied to an Invoice.

        Parameters
        ----------
        filter : str
            The filter for this query. See [Searchlight Query
            Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        include : str
            To fetch additional data on this object, specify the list of
            elements to retrieve. Available collections: Applications,
            Notes, Attachments, CustomFields
        order : str
            The sort order for this query. See See [Searchlight Query
            Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        pageSize : int
            The page size for results (default 200). See [Searchlight
            Query Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        pageNumber : int
            The page number for results (default 0). See [Searchlight
            Query Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        """
        path = "/api/v1/Payments/query"
        result = self.client.send_request("GET", path, None, {"filter": filter, "include": include, "order": order, "pageSize": pageSize, "pageNumber": pageNumber}, None)
        if result.status_code >= 200 and result.status_code < 300:
            return LockstepResponse(True, result.status_code, FetchResult[PaymentModel](**result.json()), None)
        else:
            return LockstepResponse(False, result.status_code, None, ErrorResult(**result.json()))

    def retrieve_payment_pdf(self, id: str) -> Response:
        """
        Retrieves a PDF file for this payment if it has been synced
        using an app enrollment to one of the supported apps.

        Supported apps: Quickbooks Online

        Parameters
        ----------
        id : str
            The unique Lockstep Platform ID number of this payment; NOT
            the customer's ERP key
        """
        path = f"/api/v1/Payments/{id}/pdf"
        result = self.client.send_request("GET", path, None, {}, None)
        return result

    def query_payment_summary_view(self, filter: str, include: str, order: str, pageSize: int, pageNumber: int) -> LockstepResponse[FetchResult[PaymentSummaryModel]]:
        """
        Queries Payments for this account using the specified filtering,
        sorting, nested fetch, and pagination rules requested. This
        query endpoint provides extra data about the summary of payment
        information.

        More information on querying can be found on the [Searchlight
        Query Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        page on the Lockstep Developer website.

        A Payment represents money sent from one company to another. A
        single payment may contain payments for one or more invoices; it
        is also possible for payments to be made in advance of an
        invoice, for example, as a deposit. The creator of the Payment
        is identified by the `CustomerId` field, and the recipient of
        the Payment is identified by the `CompanyId` field. Most
        Payments are uniquely identified both by a Lockstep Platform ID
        number and a customer ERP "key" that was generated by the system
        that originated the Payment. Payments that have not been fully
        applied have a nonzero `UnappliedAmount` value, which represents
        a deposit that has been paid and not yet applied to an Invoice.

        Parameters
        ----------
        filter : str
            The filter for this query. See [Searchlight Query
            Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        include : str
            To fetch additional data on this object, specify the list of
            elements to retrieve.

            No collections are currently available but may be offered in
            the future
        order : str
            The sort order for this query. See See [Searchlight Query
            Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        pageSize : int
            The page size for results (default 200). See [Searchlight
            Query Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        pageNumber : int
            The page number for results (default 0). See [Searchlight
            Query Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        """
        path = "/api/v1/Payments/views/summary"
        result = self.client.send_request("GET", path, None, {"filter": filter, "include": include, "order": order, "pageSize": pageSize, "pageNumber": pageNumber}, None)
        if result.status_code >= 200 and result.status_code < 300:
            return LockstepResponse(True, result.status_code, FetchResult[PaymentSummaryModel](**result.json()), None)
        else:
            return LockstepResponse(False, result.status_code, None, ErrorResult(**result.json()))

    def retrieve_payment_detail_header(self, ) -> LockstepResponse[PaymentDetailHeaderModel]:
        """
        Retrieves aggregated payment data from your account.

        Parameters
        ----------
        """
        path = "/api/v1/Payments/views/detail-header"
        result = self.client.send_request("GET", path, None, None, None)
        if result.status_code >= 200 and result.status_code < 300:
            return LockstepResponse(True, result.status_code, PaymentDetailHeaderModel(**result.json()), None)
        else:
            return LockstepResponse(False, result.status_code, None, ErrorResult(**result.json()))

    def query_payment_detail_view(self, filter: str, include: str, order: str, pageSize: int, pageNumber: int) -> LockstepResponse[FetchResult[PaymentDetailModel]]:
        """
        Queries Payments within the Lockstep platform using the
        specified filtering, sorting, nested fetch, and pagination rules
        requested.

        More information on querying can be found on the [Searchlight
        Query Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        page on the Lockstep Developer website. A Payment represents
        money sent from one company to another. A single payment may
        contain payments for one or more invoices; it is also possible
        for payments to be made in advance of an invoice, for example,
        as a deposit. The creator of the Payment is identified by the
        CustomerId field, and the recipient of the Payment is identified
        by the CompanyId field. Most Payments are uniquely identified
        both by a Lockstep Platform ID number and a customer ERP "key"
        that was generated by the system that originated the Payment.
        Payments that have not been fully applied have a nonzero
        UnappliedAmount value, which represents a deposit that has been
        paid and not yet applied to an Invoice.

        Parameters
        ----------
        filter : str
            The filter for this query. See [Searchlight Query
            Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        include : str
            To fetch additional data on this object, specify the list of
            elements to retrieve.

            No collections are currently available but may be offered in
            the future
        order : str
            The sort order for this query. See See [Searchlight Query
            Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        pageSize : int
            The page size for results (default 200). See [Searchlight
            Query Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        pageNumber : int
            The page number for results (default 0). See [Searchlight
            Query Language](https://developer.lockstep.io/docs/querying-with-searchlight)
        """
        path = "/api/v1/Payments/views/detail"
        result = self.client.send_request("GET", path, None, {"filter": filter, "include": include, "order": order, "pageSize": pageSize, "pageNumber": pageNumber}, None)
        if result.status_code >= 200 and result.status_code < 300:
            return LockstepResponse(True, result.status_code, FetchResult[PaymentDetailModel](**result.json()), None)
        else:
            return LockstepResponse(False, result.status_code, None, ErrorResult(**result.json()))
