# This file is part of the Data Cleaning Library (openclean).
#
# Copyright (C) 2018-2021 New York University.
#
# openclean is released under the Revised BSD License. See file LICENSE for
# full license details.

"""String tokenizer that is a wrapper around the string split method."""

from typing import Callable, List, Optional, Tuple

import re

from openclean.data.types import Scalar
from openclean.function.token.base import Tokenizer, Token

import openclean.function.token.base as TT

# Default token classifier.
DEFAULT_CLASSIFIER = [(str.isalpha, TT.ALPHA), (str.isdigit, TT.DIGIT)]


class ChartypeSplit(Tokenizer):
    """Split values basesd of a list of character type functions. That is, a
    value that contains characters of different types, e.g., W35ST, will be
    split into tokens with homogeneous character type, e.g., ['W', '35', 'ST'].

    The type of a character is determined by a classifier that is given as a
    list of Boolean predicates, i.e., callables that accept a single character
    and that return True if the charcter belongs to the type that the function
    represents or False otherwise. With each classifier a token type label is
    associated that is assigned to the generated token. If a token does not
    match any of the given classifier the default token type is returned.
    """
    def __init__(self, chartypes: Optional[List[Tuple[Callable, str]]] = None):
        """Initialize the character type classifier.

        Parameters
        ----------
        chartypes: list of tuple of callable and string, default=None
            List of functions that are used to determine the type of a character.
            The functions are applied in the given order. The first function that
            returns True for given character defines the character type. By
            default, we only distinguish between letters and digits. With each
            funciton a token type label is associated that will be assigned to
            the generated tokens.
        """
        self.chartypes = chartypes if chartypes is not None else DEFAULT_CLASSIFIER

    def get_type(self, c: str) -> str:
        """The type of a character is the label that is associated with the
        first type predicate that returns True.

        If no predicate evaluates to True for a given value None is returned.

        Parameters
        ----------
        c: string
            Expects a single character string.

        Returns
        -------
        int
        """
        for f, label in self.chartypes:
            if f(c):
                return label
        return None

    def tokens(self, value: Scalar, rowidx: Optional[int] = None) -> List[Token]:
        """Convert a given scalar values into a list of string tokens. If a
        given value cannot be converted into tokens None should be returned.

        The order of tokens in the returned list not necissarily corresponds to
        their order in the original value. This is implementation dependent.

        Parameters
        ----------
        value: scalar
            Value that is converted into a list of tokens.
        rowidx: int, default=None
            Optional index of the dataset row that the value originates from.

        Returns
        -------
        list of openclean.function.token.base.Token
        """
        # Ensure that the value is a string.
        value = str(value) if not isinstance(value, str) else value
        # If the string is empty an empty list is returned.
        if not value:
            return list()
        # Split value into list of homogeneous tokens.
        tokens = list()
        # Get the type of the first character.
        prev_type = self.get_type(value[0])
        start = 0
        for i in range(1, len(value)):
            next_type = self.get_type(value[i])
            if prev_type != next_type:
                # Add homogeneous token from start to previous postion.
                tokens.append(Token(value[start:i], token_type=prev_type, rowidx=rowidx))
                start = i
                prev_type = next_type
        # Ensure to add the homogenous suffix if necessary.
        tokens.append(Token(value[start:], token_type=prev_type, rowidx=rowidx))
        return tokens


class Split(Tokenizer):
    """String tokenizer that is a wrapper around the regular expression split
    method. Defines a extra parameters to (a) pre-process a given value and
    (b) modify the generated token lists.

    The split operator allows to further split tokens that are generated by
    the standard split function using a nested string tokenizer.
    """
    def __init__(
        self, pattern: str, sort: Optional[bool] = False,
        reverse: Optional[bool] = False, unique: Optional[bool] = False,
        preproc: Optional[Callable] = None, subtokens: Optional[Tokenizer] = None
    ):
        """Initialize the tokenizer parameters.

        Parameters
        ----------
        pattern: string
            Regular expression defining the split criteria.
        sort: bool, default=False
            Return a sorted token list if True. Tokens are sorted in ascending
            order by default.
        reverse: bool, default=False
            Reverse token lists before returning them.
        unique: bool, default=True
            Remove duplicate tokens from the generated token lists.
        preproc: callable, default=None
            Optional pre-processor that is evaluated on each value before
            tokenization.
        subtokens: openclean.function.token.base.Tokenizer, default=None
            Tokenizer that is used to split generated tokens into sub-tokens.
        """
        self.pattern = pattern
        self.sort = sort
        self.reverse = reverse
        self.unique = unique
        self.preproc = preproc
        self.subtokens = subtokens

    def tokens(self, value: Scalar, rowidx: Optional[int] = None) -> List[Token]:
        """Convert a given scalar values into a list of string tokens. If a
        given value cannot be converted into tokens None should be returned.

        The order of tokens in the returned list not necissarily corresponds to
        their order in the original value. This is implementation dependent.

        Parameters
        ----------
        value: scalar
            Value that is converted into a list of tokens.
        rowidx: int, default=None
            Optional index of the dataset row that the value originates from.

        Returns
        -------
        list of openclean.function.token.base.Token
        """
        # Convert value to string if necessary
        if not isinstance(value, str):
            value = str(value)
        # Apply pre-processing if defined.
        if self.preproc is not None:
            value = self.preproc(value)
        # Use split and the defined pattern to generate initial token list.
        tokens = [Token(value=t, rowidx=rowidx) for t in filter(None, re.split(self.pattern, value))]
        # Remove duplicates if the unique flag is True.
        if self.unique:
            tokens = list(set(tokens))
        # Split tokens into sub-tokens if the optional tokenizer is given.
        if self.subtokens is not None:
            stokens = list()
            for t in tokens:
                stokens.extend(self.subtokens.tokens(t))
            tokens = stokens
            # Need to remove duplicates again.
            if self.unique:
                tokens = list(set(tokens))
        # Sort tokens and reverse the token list if the respective flags are
        # set to True.
        if self.sort:
            # Can take care of sorting and reverse in one statement here.
            tokens = sorted(tokens, reverse=self.reverse)
        elif self.reverse:
            # Reverse list even if sort is False.
            tokens = tokens[::-1]
        return tokens
