# This file is part of the Data Cleaning Library (openclean).
#
# Copyright (C) 2018-2021 New York University.
#
# openclean is released under the Revised BSD License. See file LICENSE for
# full license details.

"""Interface for data stores that maintain serialized objects in a key-value
store. Provides a simple abstraction for different types of key-value stores
that we may want to connect to in the future.
"""

from abc import ABCMeta, abstractmethod
from typing import Any, Optional


class DataStore(metaclass=ABCMeta):
    """Interface for key-value stores that maintain serialized objects.

    Each object has a unique identifier that is generated by the store when the
    object is created.

    There are no constraints on the contents of the object serializations defined
    by this class. However, depending on the implementation, there may be some
    restrictions on the data that can be stored (e.g., Json serialized).
    """
    @abstractmethod
    def delete_object(self, object_id: str):
        """Remove the object with the given identifier. If not object with the
        given identifier exist, a KeyError is raised.

        Parameters
        ----------
        object_id: string
            Unique object identifier.

        Raises
        ------
        KeyError
        """
        raise NotImplementedError()  # pargma: no cover

    @abstractmethod
    def read_object(self, object_id: str) -> Any:
        """Get the serialized object that is identified by the given object id.
        Raises a KeyError if the referenced object does not exist.

        Parameters
        ----------
        object_id: string
            Unique object identifier.

        Returns
        -------
        any

        Raises
        ------
        KeyError
        """
        raise NotImplementedError()  # pargma: no cover

    @abstractmethod
    def write_object(self, object: Any, object_id: Optional[str] = None) -> str:
        """Store an object in the repository. Creates a new object identifier
        if no identifier is given. If an indentifier is given an existing object
        under this identifier will be replaced.

        Returns unique object identifier.

        Parameters
        ----------
        object: any
            Serialization for an object.
        object_id: str, default=None
            Optional identifier for the stored object. If not given, a unique
            identifier will be generated.

        Returns
        -------
        string
        """
        raise NotImplementedError()  # pargma: no cover
