# Copyright 2018 The KaiJIN Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
from bisect import bisect_right
import torch


# FIXME ideally this would be achieved with a CombinedLRScheduler,
# separating MultiStepLR with WarmupLR
# but the current LRScheduler design doesn't allow it
class WarmupMultiStepLR(torch.optim.lr_scheduler._LRScheduler):
  def __init__(self,
               optimizer,
               milestones,
               gamma=0.1,
               warmup_factor=1.0 / 3,
               warmup_iters=500,
               warmup_method="linear",
               last_epoch=-1):
    if not list(milestones) == sorted(milestones):
      raise ValueError(
          "Milestones should be a list of increasing integers. Got {}", milestones)
    if warmup_method not in ("constant", "linear"):
      raise ValueError("Only 'constant' or 'linear' warmup_method accepted"
                       "got {}".format(warmup_method))
    self.milestones = milestones
    self.gamma = gamma
    self.warmup_factor = warmup_factor
    self.warmup_iters = warmup_iters
    self.warmup_method = warmup_method
    super(WarmupMultiStepLR, self).__init__(optimizer, last_epoch)

  def get_lr(self):
    warmup_factor = 1
    if self.last_epoch < self.warmup_iters:
      if self.warmup_method == "constant":
        warmup_factor = self.warmup_factor
      elif self.warmup_method == "linear":
        alpha = float(self.last_epoch) / self.warmup_iters
        warmup_factor = self.warmup_factor * (1 - alpha) + alpha
    return [base_lr
            * warmup_factor
            * self.gamma
            ** bisect_right(self.milestones, self.last_epoch)
            for base_lr in self.base_lrs]
