"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatSecurityChanges = exports.formatDifferences = void 0;
const util_1 = require("util");
const chalk = require("chalk");
const diff_template_1 = require("./diff-template");
const util_2 = require("./diff/util");
const format_table_1 = require("./format-table");
// from cx-api
const PATH_METADATA_KEY = 'aws:cdk:path';
/* eslint-disable @typescript-eslint/no-require-imports */
const { structuredPatch } = require('diff');
/**
 * Renders template differences to the process' console.
 *
 * @param stream           The IO stream where to output the rendered diff.
 * @param templateDiff     TemplateDiff to be rendered to the console.
 * @param logicalToPathMap A map from logical ID to construct path. Useful in
 *                         case there is no aws:cdk:path metadata in the template.
 * @param context          the number of context lines to use in arbitrary JSON diff (defaults to 3).
 */
function formatDifferences(stream, templateDiff, logicalToPathMap = {}, context = 3) {
    const formatter = new Formatter(stream, logicalToPathMap, templateDiff, context);
    if (templateDiff.awsTemplateFormatVersion || templateDiff.transform || templateDiff.description) {
        formatter.printSectionHeader('Template');
        formatter.formatDifference('AWSTemplateFormatVersion', 'AWSTemplateFormatVersion', templateDiff.awsTemplateFormatVersion);
        formatter.formatDifference('Transform', 'Transform', templateDiff.transform);
        formatter.formatDifference('Description', 'Description', templateDiff.description);
        formatter.printSectionFooter();
    }
    formatSecurityChangesWithBanner(formatter, templateDiff);
    formatter.formatSection('Parameters', 'Parameter', templateDiff.parameters);
    formatter.formatSection('Metadata', 'Metadata', templateDiff.metadata);
    formatter.formatSection('Mappings', 'Mapping', templateDiff.mappings);
    formatter.formatSection('Conditions', 'Condition', templateDiff.conditions);
    formatter.formatSection('Resources', 'Resource', templateDiff.resources, formatter.formatResourceDifference.bind(formatter));
    formatter.formatSection('Outputs', 'Output', templateDiff.outputs);
    formatter.formatSection('Other Changes', 'Unknown', templateDiff.unknown);
}
exports.formatDifferences = formatDifferences;
/**
 * Renders a diff of security changes to the given stream
 */
function formatSecurityChanges(stream, templateDiff, logicalToPathMap = {}, context) {
    const formatter = new Formatter(stream, logicalToPathMap, templateDiff, context);
    formatSecurityChangesWithBanner(formatter, templateDiff);
}
exports.formatSecurityChanges = formatSecurityChanges;
function formatSecurityChangesWithBanner(formatter, templateDiff) {
    if (!templateDiff.iamChanges.hasChanges && !templateDiff.securityGroupChanges.hasChanges) {
        return;
    }
    formatter.formatIamChanges(templateDiff.iamChanges);
    formatter.formatSecurityGroupChanges(templateDiff.securityGroupChanges);
    formatter.warning('(NOTE: There may be security-related changes not in this list. See https://github.com/aws/aws-cdk/issues/1299)');
    formatter.printSectionFooter();
}
const ADDITION = chalk.green('[+]');
const CONTEXT = chalk.grey('[ ]');
const UPDATE = chalk.yellow('[~]');
const REMOVAL = chalk.red('[-]');
class Formatter {
    constructor(stream, logicalToPathMap, diff, context = 3) {
        this.stream = stream;
        this.logicalToPathMap = logicalToPathMap;
        this.context = context;
        // Read additional construct paths from the diff if it is supplied
        if (diff) {
            this.readConstructPathsFrom(diff);
        }
    }
    print(fmt, ...args) {
        this.stream.write(chalk.white(util_1.format(fmt, ...args)) + '\n');
    }
    warning(fmt, ...args) {
        this.stream.write(chalk.yellow(util_1.format(fmt, ...args)) + '\n');
    }
    formatSection(title, entryType, collection, formatter = this.formatDifference.bind(this)) {
        if (collection.differenceCount === 0) {
            return;
        }
        this.printSectionHeader(title);
        collection.forEachDifference((id, diff) => formatter(entryType, id, diff));
        this.printSectionFooter();
    }
    printSectionHeader(title) {
        this.print(chalk.underline(chalk.bold(title)));
    }
    printSectionFooter() {
        this.print('');
    }
    /**
     * Print a simple difference for a given named entity.
     *
     * @param logicalId the name of the entity that is different.
     * @param diff the difference to be rendered.
     */
    formatDifference(type, logicalId, diff) {
        if (!diff || !diff.isDifferent) {
            return;
        }
        let value;
        const oldValue = this.formatValue(diff.oldValue, chalk.red);
        const newValue = this.formatValue(diff.newValue, chalk.green);
        if (diff.isAddition) {
            value = newValue;
        }
        else if (diff.isUpdate) {
            value = `${oldValue} to ${newValue}`;
        }
        else if (diff.isRemoval) {
            value = oldValue;
        }
        this.print(`${this.formatPrefix(diff)} ${chalk.cyan(type)} ${this.formatLogicalId(logicalId)}: ${value}`);
    }
    /**
     * Print a resource difference for a given logical ID.
     *
     * @param logicalId the logical ID of the resource that changed.
     * @param diff      the change to be rendered.
     */
    formatResourceDifference(_type, logicalId, diff) {
        if (!diff.isDifferent) {
            return;
        }
        const resourceType = diff.isRemoval ? diff.oldResourceType : diff.newResourceType;
        // eslint-disable-next-line max-len
        this.print(`${this.formatPrefix(diff)} ${this.formatValue(resourceType, chalk.cyan)} ${this.formatLogicalId(logicalId)} ${this.formatImpact(diff.changeImpact)}`);
        if (diff.isUpdate) {
            const differenceCount = diff.differenceCount;
            let processedCount = 0;
            diff.forEachDifference((_, name, values) => {
                processedCount += 1;
                this.formatTreeDiff(name, values, processedCount === differenceCount);
            });
        }
    }
    formatPrefix(diff) {
        if (diff.isAddition) {
            return ADDITION;
        }
        if (diff.isUpdate) {
            return UPDATE;
        }
        if (diff.isRemoval) {
            return REMOVAL;
        }
        return chalk.white('[?]');
    }
    /**
     * @param value the value to be formatted.
     * @param color the color to be used.
     *
     * @returns the formatted string, with color applied.
     */
    formatValue(value, color) {
        if (value == null) {
            return undefined;
        }
        if (typeof value === 'string') {
            return color(value);
        }
        return color(JSON.stringify(value));
    }
    /**
     * @param impact the impact to be formatted
     * @returns a user-friendly, colored string representing the impact.
     */
    formatImpact(impact) {
        switch (impact) {
            case diff_template_1.ResourceImpact.MAY_REPLACE:
                return chalk.italic(chalk.yellow('may be replaced'));
            case diff_template_1.ResourceImpact.WILL_REPLACE:
                return chalk.italic(chalk.bold(chalk.red('replace')));
            case diff_template_1.ResourceImpact.WILL_DESTROY:
                return chalk.italic(chalk.bold(chalk.red('destroy')));
            case diff_template_1.ResourceImpact.WILL_ORPHAN:
                return chalk.italic(chalk.yellow('orphan'));
            case diff_template_1.ResourceImpact.WILL_UPDATE:
            case diff_template_1.ResourceImpact.WILL_CREATE:
            case diff_template_1.ResourceImpact.NO_CHANGE:
                return ''; // no extra info is gained here
        }
    }
    /**
     * Renders a tree of differences under a particular name.
     * @param name    the name of the root of the tree.
     * @param diff    the difference on the tree.
     * @param last    whether this is the last node of a parent tree.
     */
    formatTreeDiff(name, diff, last) {
        let additionalInfo = '';
        if (diff_template_1.isPropertyDifference(diff)) {
            if (diff.changeImpact === diff_template_1.ResourceImpact.MAY_REPLACE) {
                additionalInfo = ' (may cause replacement)';
            }
            else if (diff.changeImpact === diff_template_1.ResourceImpact.WILL_REPLACE) {
                additionalInfo = ' (requires replacement)';
            }
        }
        this.print(' %s─ %s %s%s', last ? '└' : '├', this.changeTag(diff.oldValue, diff.newValue), name, additionalInfo);
        return this.formatObjectDiff(diff.oldValue, diff.newValue, ` ${last ? ' ' : '│'}`);
    }
    /**
     * Renders the difference between two objects, looking for the differences as deep as possible,
     * and rendering a tree graph of the path until the difference is found.
     *
     * @param oldObject  the old object.
     * @param newObject  the new object.
     * @param linePrefix a prefix (indent-like) to be used on every line.
     */
    formatObjectDiff(oldObject, newObject, linePrefix) {
        if ((typeof oldObject !== typeof newObject) || Array.isArray(oldObject) || typeof oldObject === 'string' || typeof oldObject === 'number') {
            if (oldObject !== undefined && newObject !== undefined) {
                if (typeof oldObject === 'object' || typeof newObject === 'object') {
                    const oldStr = JSON.stringify(oldObject, null, 2);
                    const newStr = JSON.stringify(newObject, null, 2);
                    const diff = _diffStrings(oldStr, newStr, this.context);
                    for (let i = 0; i < diff.length; i++) {
                        this.print('%s   %s %s', linePrefix, i === 0 ? '└─' : '  ', diff[i]);
                    }
                }
                else {
                    this.print('%s   ├─ %s %s', linePrefix, REMOVAL, this.formatValue(oldObject, chalk.red));
                    this.print('%s   └─ %s %s', linePrefix, ADDITION, this.formatValue(newObject, chalk.green));
                }
            }
            else if (oldObject !== undefined /* && newObject === undefined */) {
                this.print('%s   └─ %s', linePrefix, this.formatValue(oldObject, chalk.red));
            }
            else /* if (oldObject === undefined && newObject !== undefined) */ {
                this.print('%s   └─ %s', linePrefix, this.formatValue(newObject, chalk.green));
            }
            return;
        }
        const keySet = new Set(Object.keys(oldObject));
        Object.keys(newObject).forEach(k => keySet.add(k));
        const keys = new Array(...keySet).filter(k => !util_2.deepEqual(oldObject[k], newObject[k])).sort();
        const lastKey = keys[keys.length - 1];
        for (const key of keys) {
            const oldValue = oldObject[key];
            const newValue = newObject[key];
            const treePrefix = key === lastKey ? '└' : '├';
            if (oldValue !== undefined && newValue !== undefined) {
                this.print('%s   %s─ %s %s:', linePrefix, treePrefix, this.changeTag(oldValue, newValue), chalk.blue(`.${key}`));
                this.formatObjectDiff(oldValue, newValue, `${linePrefix}   ${key === lastKey ? ' ' : '│'}`);
            }
            else if (oldValue !== undefined /* && newValue === undefined */) {
                this.print('%s   %s─ %s Removed: %s', linePrefix, treePrefix, REMOVAL, chalk.blue(`.${key}`));
            }
            else /* if (oldValue === undefined && newValue !== undefined */ {
                this.print('%s   %s─ %s Added: %s', linePrefix, treePrefix, ADDITION, chalk.blue(`.${key}`));
            }
        }
    }
    /**
     * @param oldValue the old value of a difference.
     * @param newValue the new value of a difference.
     *
     * @returns a tag to be rendered in the diff, reflecting whether the difference
     *      was an ADDITION, UPDATE or REMOVAL.
     */
    changeTag(oldValue, newValue) {
        if (oldValue !== undefined && newValue !== undefined) {
            return UPDATE;
        }
        else if (oldValue !== undefined /* && newValue === undefined*/) {
            return REMOVAL;
        }
        else /* if (oldValue === undefined && newValue !== undefined) */ {
            return ADDITION;
        }
    }
    /**
     * Find 'aws:cdk:path' metadata in the diff and add it to the logicalToPathMap
     *
     * There are multiple sources of logicalID -> path mappings: synth metadata
     * and resource metadata, and we combine all sources into a single map.
     */
    readConstructPathsFrom(templateDiff) {
        for (const [logicalId, resourceDiff] of Object.entries(templateDiff.resources)) {
            if (!resourceDiff) {
                continue;
            }
            const oldPathMetadata = resourceDiff.oldValue && resourceDiff.oldValue.Metadata && resourceDiff.oldValue.Metadata[PATH_METADATA_KEY];
            if (oldPathMetadata && !(logicalId in this.logicalToPathMap)) {
                this.logicalToPathMap[logicalId] = oldPathMetadata;
            }
            const newPathMetadata = resourceDiff.newValue && resourceDiff.newValue.Metadata && resourceDiff.newValue.Metadata[PATH_METADATA_KEY];
            if (newPathMetadata && !(logicalId in this.logicalToPathMap)) {
                this.logicalToPathMap[logicalId] = newPathMetadata;
            }
        }
    }
    formatLogicalId(logicalId) {
        // if we have a path in the map, return it
        const normalized = this.normalizedLogicalIdPath(logicalId);
        if (normalized) {
            return `${normalized} ${chalk.gray(logicalId)}`;
        }
        return logicalId;
    }
    normalizedLogicalIdPath(logicalId) {
        // if we have a path in the map, return it
        const path = this.logicalToPathMap[logicalId];
        return path ? normalizePath(path) : undefined;
        /**
         * Path is supposed to start with "/stack-name". If this is the case (i.e. path has more than
         * two components, we remove the first part. Otherwise, we just use the full path.
         * @param p
         */
        function normalizePath(p) {
            if (p.startsWith('/')) {
                p = p.slice(1);
            }
            let parts = p.split('/');
            if (parts.length > 1) {
                parts = parts.slice(1);
                // remove the last component if it's "Resource" or "Default" (if we have more than a single component)
                if (parts.length > 1) {
                    const last = parts[parts.length - 1];
                    if (last === 'Resource' || last === 'Default') {
                        parts = parts.slice(0, parts.length - 1);
                    }
                }
                p = parts.join('/');
            }
            return p;
        }
    }
    formatIamChanges(changes) {
        if (!changes.hasChanges) {
            return;
        }
        if (changes.statements.hasChanges) {
            this.printSectionHeader('IAM Statement Changes');
            this.print(format_table_1.formatTable(this.deepSubstituteBracedLogicalIds(changes.summarizeStatements()), this.stream.columns));
        }
        if (changes.managedPolicies.hasChanges) {
            this.printSectionHeader('IAM Policy Changes');
            this.print(format_table_1.formatTable(this.deepSubstituteBracedLogicalIds(changes.summarizeManagedPolicies()), this.stream.columns));
        }
    }
    formatSecurityGroupChanges(changes) {
        if (!changes.hasChanges) {
            return;
        }
        this.printSectionHeader('Security Group Changes');
        this.print(format_table_1.formatTable(this.deepSubstituteBracedLogicalIds(changes.summarize()), this.stream.columns));
    }
    deepSubstituteBracedLogicalIds(rows) {
        return rows.map(row => row.map(this.substituteBracedLogicalIds.bind(this)));
    }
    /**
     * Substitute all strings like ${LogId.xxx} with the path instead of the logical ID
     */
    substituteBracedLogicalIds(source) {
        return source.replace(/\$\{([^.}]+)(.[^}]+)?\}/ig, (_match, logId, suffix) => {
            return '${' + (this.normalizedLogicalIdPath(logId) || logId) + (suffix || '') + '}';
        });
    }
}
/**
 * Creates a unified diff of two strings.
 *
 * @param oldStr  the "old" version of the string.
 * @param newStr  the "new" version of the string.
 * @param context the number of context lines to use in arbitrary JSON diff.
 *
 * @returns an array of diff lines.
 */
function _diffStrings(oldStr, newStr, context) {
    const patch = structuredPatch(null, null, oldStr, newStr, null, null, { context });
    const result = new Array();
    for (const hunk of patch.hunks) {
        result.push(chalk.magenta(`@@ -${hunk.oldStart},${hunk.oldLines} +${hunk.newStart},${hunk.newLines} @@`));
        const baseIndent = _findIndent(hunk.lines);
        for (const line of hunk.lines) {
            // Don't care about termination newline.
            if (line === '\\ No newline at end of file') {
                continue;
            }
            const marker = line.charAt(0);
            const text = line.slice(1 + baseIndent);
            switch (marker) {
                case ' ':
                    result.push(`${CONTEXT} ${text}`);
                    break;
                case '+':
                    result.push(chalk.bold(`${ADDITION} ${chalk.green(text)}`));
                    break;
                case '-':
                    result.push(chalk.bold(`${REMOVAL} ${chalk.red(text)}`));
                    break;
                default:
                    throw new Error(`Unexpected diff marker: ${marker} (full line: ${line})`);
            }
        }
    }
    return result;
    function _findIndent(lines) {
        let indent = Number.MAX_SAFE_INTEGER;
        for (const line of lines) {
            for (let i = 1; i < line.length; i++) {
                if (line.charAt(i) !== ' ') {
                    indent = indent > i - 1 ? i - 1 : indent;
                    break;
                }
            }
        }
        return indent;
    }
}
//# sourceMappingURL=data:application/json;base64,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