"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isTagPropertyStringMap = exports.isTagPropertyJson = exports.isTagPropertyAutoScalingGroup = exports.isTagPropertyStandard = exports.isTagProperty = exports.isTagPropertyName = exports.isPropertyScrutinyType = exports.PropertyScrutinyType = exports.isUnionProperty = exports.isMapOfListsOfPrimitivesProperty = exports.isMapOfStructsProperty = exports.isPrimitiveMapProperty = exports.isMapProperty = exports.isComplexListProperty = exports.isPrimitiveListProperty = exports.isListProperty = exports.isCollectionProperty = exports.isComplexProperty = exports.isPrimitiveProperty = exports.isScalarProperty = exports.isUpdateType = exports.UpdateType = void 0;
const base_types_1 = require("./base-types");
var UpdateType;
(function (UpdateType) {
    UpdateType["Conditional"] = "Conditional";
    UpdateType["Immutable"] = "Immutable";
    UpdateType["Mutable"] = "Mutable";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
function isUpdateType(str) {
    switch (str) {
        case UpdateType.Conditional:
        case UpdateType.Immutable:
        case UpdateType.Mutable:
            return true;
        default:
            return false;
    }
}
exports.isUpdateType = isUpdateType;
function isScalarProperty(prop) {
    return isPrimitiveProperty(prop)
        || isComplexProperty(prop)
        // A UnionProperty is only Scalar if it defines Types or PrimitiveTypes
        || (isUnionProperty(prop) && !!(prop.Types || prop.PrimitiveTypes));
}
exports.isScalarProperty = isScalarProperty;
function isPrimitiveProperty(prop) {
    return !!prop.PrimitiveType;
}
exports.isPrimitiveProperty = isPrimitiveProperty;
function isComplexProperty(prop) {
    const propType = prop.Type;
    return propType != null && propType !== 'Map' && propType !== 'List';
}
exports.isComplexProperty = isComplexProperty;
function isCollectionProperty(prop) {
    return isListProperty(prop)
        || isMapProperty(prop)
        // A UnionProperty is only Collection if it defines ItemTypes or PrimitiveItemTypes
        || (isUnionProperty(prop) && !!(prop.ItemTypes || prop.PrimitiveItemTypes || prop.InclusiveItemTypes || prop.InclusivePrimitiveItemTypes));
}
exports.isCollectionProperty = isCollectionProperty;
function isListProperty(prop) {
    return prop.Type === 'List';
}
exports.isListProperty = isListProperty;
function isPrimitiveListProperty(prop) {
    return isListProperty(prop) && !!prop.PrimitiveItemType;
}
exports.isPrimitiveListProperty = isPrimitiveListProperty;
function isComplexListProperty(prop) {
    return isListProperty(prop) && !!prop.ItemType;
}
exports.isComplexListProperty = isComplexListProperty;
function isMapProperty(prop) {
    return prop.Type === 'Map';
}
exports.isMapProperty = isMapProperty;
function isPrimitiveMapProperty(prop) {
    return isMapProperty(prop) && !!prop.PrimitiveItemType;
}
exports.isPrimitiveMapProperty = isPrimitiveMapProperty;
function isMapOfStructsProperty(prop) {
    return isMapProperty(prop) &&
        !isPrimitiveMapProperty(prop) &&
        !isMapOfListsOfPrimitivesProperty(prop);
}
exports.isMapOfStructsProperty = isMapOfStructsProperty;
// note: this (and the MapOfListsOfPrimitives type) are not actually valid in the CFN spec!
// they are only here to support our patch of the CFN spec
// to alleviate https://github.com/aws/aws-cdk/issues/3092
function isMapOfListsOfPrimitivesProperty(prop) {
    return isMapProperty(prop) && prop.ItemType === 'List';
}
exports.isMapOfListsOfPrimitivesProperty = isMapOfListsOfPrimitivesProperty;
function isUnionProperty(prop) {
    const castProp = prop;
    return !!(castProp.ItemTypes ||
        castProp.PrimitiveTypes ||
        castProp.Types ||
        castProp.PrimitiveItemTypes ||
        castProp.InclusiveItemTypes ||
        castProp.InclusivePrimitiveItemTypes);
}
exports.isUnionProperty = isUnionProperty;
var PropertyScrutinyType;
(function (PropertyScrutinyType) {
    /**
     * No additional scrutiny
     */
    PropertyScrutinyType["None"] = "None";
    /**
     * This is an IAM policy directly on a resource
     */
    PropertyScrutinyType["InlineResourcePolicy"] = "InlineResourcePolicy";
    /**
     * Either an AssumeRolePolicyDocument or a dictionary of policy documents
     */
    PropertyScrutinyType["InlineIdentityPolicies"] = "InlineIdentityPolicies";
    /**
     * A list of managed policies (on an identity resource)
     */
    PropertyScrutinyType["ManagedPolicies"] = "ManagedPolicies";
    /**
     * A set of ingress rules (on a security group)
     */
    PropertyScrutinyType["IngressRules"] = "IngressRules";
    /**
     * A set of egress rules (on a security group)
     */
    PropertyScrutinyType["EgressRules"] = "EgressRules";
})(PropertyScrutinyType = exports.PropertyScrutinyType || (exports.PropertyScrutinyType = {}));
function isPropertyScrutinyType(str) {
    return PropertyScrutinyType[str] !== undefined;
}
exports.isPropertyScrutinyType = isPropertyScrutinyType;
const tagPropertyNames = {
    FileSystemTags: '',
    HostedZoneTags: '',
    Tags: '',
    UserPoolTags: '',
};
function isTagPropertyName(name) {
    if (undefined === name) {
        return false;
    }
    return tagPropertyNames.hasOwnProperty(name);
}
exports.isTagPropertyName = isTagPropertyName;
/**
 * This function validates that the property **can** be a Tag Property
 *
 * The property is only a Tag if the name of this property is Tags, which is
 * validated using `ResourceType.isTaggable(resource)`.
 */
function isTagProperty(prop) {
    return (isTagPropertyStandard(prop) ||
        isTagPropertyAutoScalingGroup(prop) ||
        isTagPropertyJson(prop) ||
        isTagPropertyStringMap(prop));
}
exports.isTagProperty = isTagProperty;
function isTagPropertyStandard(prop) {
    return (prop.ItemType === 'Tag' ||
        prop.ItemType === 'TagsEntry' ||
        prop.Type === 'Tags' ||
        prop.ItemType === 'TagRef' ||
        prop.ItemType === 'ElasticFileSystemTag' ||
        prop.ItemType === 'HostedZoneTag');
}
exports.isTagPropertyStandard = isTagPropertyStandard;
function isTagPropertyAutoScalingGroup(prop) {
    return prop.ItemType === 'TagProperty';
}
exports.isTagPropertyAutoScalingGroup = isTagPropertyAutoScalingGroup;
function isTagPropertyJson(prop) {
    return prop.PrimitiveType === base_types_1.PrimitiveType.Json;
}
exports.isTagPropertyJson = isTagPropertyJson;
function isTagPropertyStringMap(prop) {
    return prop.PrimitiveItemType === 'String';
}
exports.isTagPropertyStringMap = isTagPropertyStringMap;
//# sourceMappingURL=data:application/json;base64,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