"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
/**
 * Split the given CloudFormation resource specification up into a patch set, by service
 *
 * After splitting, only write out those fragments that are valid.(*)
 *
 * The reason for this is that some service updates sometimes contain mistakes,
 * and we want to be able to take partial upgrades. By splitting the spec, we can
 * take updates to some services while leaving updates to other services behind.
 *
 * (*) If `NO_VALIDATE` is set, all pieces are written out. In normal operation this
 *     should not be used.
 */
const path = require("path");
const fs = require("fs-extra");
const patch_set_1 = require("./patch-set");
const validate_cfn_1 = require("./validate-cfn");
async function main(args) {
    if (args.length < 2) {
        throw new Error('Usage: split-spec-by-service <SPECFILE> <DIRECTORY>');
    }
    const [specFile, outDir] = args;
    log(`Loading specification: ${specFile}`);
    const spec = await fs.readJson(specFile);
    // Split
    log('Splitting');
    const byService = {};
    for (const [propTypeName, propType] of Object.entries(spec.PropertyTypes)) {
        const svcName = serviceName(propTypeName);
        serviceSpec(svcName).PropertyTypes[propTypeName] = propType;
    }
    for (const [resTypeName, resType] of Object.entries(spec.ResourceTypes)) {
        const svcName = serviceName(resTypeName);
        serviceSpec(svcName).ResourceTypes[resTypeName] = resType;
    }
    // Write out
    log('Writing');
    for (const [svcName, svcSpec] of Object.entries(byService)) {
        const successTarget = path.join(outDir, `000_${svcName}.json`);
        const rejectedTarget = path.join(outDir, `.000_${svcName}.rejected.json`);
        const errors = !process.env.NO_VALIDATE ? validate_cfn_1.CfnSpecValidator.validate(svcSpec) : [];
        if (errors.length === 0) {
            // Change 'ResourceSpecificationVersion' to '$version', otherwise they will all conflict
            const toWrite = {
                PropertyTypes: svcSpec.PropertyTypes,
                ResourceTypes: svcSpec.ResourceTypes,
                $version: svcSpec.ResourceSpecificationVersion,
            };
            await patch_set_1.writeSorted(successTarget, toWrite);
            await ensureGone(rejectedTarget);
        }
        else {
            console.warn('='.repeat(70));
            console.warn(' '.repeat(Math.floor(35 - svcName.length / 2)) + svcName);
            console.warn('='.repeat(70));
            for (const error of errors) {
                console.warn(validate_cfn_1.formatErrorInContext(error));
            }
            await patch_set_1.writeSorted(rejectedTarget, svcSpec);
            // Make sure that the success file exists. If not, the initial import of a
            // new service failed.
            if (!await fs.pathExists(successTarget)) {
                await patch_set_1.writeSorted(successTarget, {
                    PropertyTypes: {},
                    ResourceTypes: {},
                    $version: '0.0.0',
                });
            }
        }
    }
    await fs.writeJson(path.join(outDir, '001_Version.json'), {
        ResourceSpecificationVersion: spec.ResourceSpecificationVersion,
    }, { spaces: 2 });
    function serviceSpec(svcName) {
        if (!(svcName in byService)) {
            byService[svcName] = {
                PropertyTypes: {},
                ResourceTypes: {},
                ResourceSpecificationVersion: spec.ResourceSpecificationVersion,
            };
        }
        return byService[svcName];
    }
}
async function ensureGone(fileName) {
    try {
        await fs.unlink(fileName);
    }
    catch (e) {
        if (e.code === 'ENOENT') {
            return;
        }
        throw e;
    }
}
function serviceName(x) {
    return x.split('::').slice(0, 2).join('_');
}
function log(x) {
    // eslint-disable-next-line no-console
    console.log(x);
}
main(process.argv.slice(2)).catch(e => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exitCode = 1;
});
//# sourceMappingURL=data:application/json;base64,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