"use strict";
/* eslint-disable no-console */
/**
 * Report on spec fragment files that are being held back.
 *
 * Report formats:
 *
 * - 'outdated'/'changelog': print for changelog format
 * - 'rejected': print validation errors, exit with error code 1 if there are any
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const validate_cfn_1 = require("./validate-cfn");
async function main(args) {
    if (args.length < 2) {
        throw new Error('Usage: report-issues <DIR> <FORMAT>');
    }
    const [dir, format] = args;
    const officialVersion = (await fs.readJson(path.join(dir, '001_Version.json'))).ResourceSpecificationVersion;
    let headerPrinted = false;
    for (const file of await fs.readdir(dir)) {
        if (!file.startsWith('000_')) {
            continue;
        }
        const json = await fs.readJson(path.join(dir, file));
        const fragmentVersion = json.$version;
        const serviceName = file.replace(/^000_/, '').replace(/\.json$/, '').replace('_', '::');
        switch (format) {
            case 'outdated':
            case 'changelog':
                if (fragmentVersion !== officialVersion) {
                    if (!headerPrinted) {
                        console.log('## Unapplied changes');
                        console.log('');
                        headerPrinted = true;
                    }
                    console.log(`* ${serviceName} is at ${fragmentVersion}`);
                }
                break;
            case 'rejected':
                if (fragmentVersion !== officialVersion) {
                    // Read the 'rejected' file, parse it (which we expect to fail)
                    // and print the failures.
                    const rejectedFileName = `.${file.replace(/.json$/, '.rejected.json')}`;
                    const rejectedPath = path.join(dir, rejectedFileName);
                    if (!await fs.pathExists(rejectedPath)) {
                        // If for whatever reason the file doesn't exist, ignore
                        continue;
                    }
                    const rejectedSpec = await fs.readJson(rejectedPath);
                    const errors = validate_cfn_1.CfnSpecValidator.validate(rejectedSpec);
                    console.warn('='.repeat(70));
                    console.warn(' '.repeat(Math.floor(35 - serviceName.length / 2)) + serviceName);
                    console.warn('='.repeat(70));
                    for (const error of errors) {
                        console.warn(validate_cfn_1.formatErrorInContext(error));
                        process.exitCode = 1;
                    }
                }
                break;
            default:
                throw new Error(`Unknown format type requested: ${format}`);
        }
    }
}
main(process.argv.slice(2)).catch(e => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exitCode = 1;
});
//# sourceMappingURL=data:application/json;base64,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