"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Provider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const runner = require("./runner");
const gitLabClusterRoleBinding = {
    apiVersion: 'rbac.authorization.k8s.io/v1beta1',
    kind: 'ClusterRoleBinding',
    metadata: { name: 'gitlab-admin' },
    roleRef: {
        apiGroup: 'rbac.authorization.k8s.io',
        kind: 'ClusterRole',
        name: 'cluster-admin',
    },
    subjects: [
        {
            kind: 'ServiceAccount',
            name: 'gitlab',
            namespace: 'kube-system',
        },
    ],
};
/**
 * The Provider to create GitLab Integrations with AWS
 */
class Provider extends constructs_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        const stack = cdk.Stack.of(this);
        this.vpc = props.vpc ?? new aws_cdk_lib_1.aws_ec2.Vpc(this, 'Vpc', { natGateways: 1 });
        new cdk.CfnOutput(this, 'Region', { value: stack.region });
    }
    createFargateRunner(executor) {
        const token = this.node.tryGetContext('GITLAB_REGISTRATION_TOKEN') || process.env.GITLAB_REGISTRATION_TOKEN;
        if (!token) {
            throw new Error('missing GITLAB_REGISTRATION_TOKEN in the context variable');
        }
        new runner.FargateRunner(this, 'FargateRunner', {
            vpc: this.vpc,
            registrationToken: this.node.tryGetContext('GITLAB_REGISTRATION_TOKEN'),
            executor,
        });
    }
    createEksCluster(scope, id, props) {
        const cluster = new aws_cdk_lib_1.aws_eks.Cluster(scope, id, props.clusterOptions);
        if (props.rbac != false) {
            cluster.addServiceAccount('gitlab');
            cluster.addManifest('ClusterRoleBinding', gitLabClusterRoleBinding);
        }
        return cluster;
    }
    createFargateEksCluster(scope, id, props) {
        const cluster = new aws_cdk_lib_1.aws_eks.FargateCluster(scope, id, props.clusterOptions);
        if (props.rbac != false) {
            cluster.addServiceAccount('gitlab');
            cluster.addManifest('ClusterRoleBinding', gitLabClusterRoleBinding);
        }
        const registrationToken = props.helmRunnerOptions?.registrationToken ?? (this.node.tryGetContext('GITLAB_REGISTRATION_TOKEN') || process.env.GITLAB_REGISTRATION_TOKEN);
        if (!registrationToken) {
            throw new Error('missing GITLAB_REGISTRATION_TOKEN in the context variable');
        }
        cluster.addHelmChart('helmrunner', {
            chart: 'gitlab-runner',
            repository: 'https://charts.gitlab.io',
            namespace: 'default',
            values: {
                // use fargate run job ,always need to pull.
                imagePullPolicy: 'Always',
                terminationGracePeriodSeconds: 3600,
                concurrent: props.helmRunnerOptions?.concurrent ?? 10,
                checkInterval: 30,
                gitlabUrl: props.helmRunnerOptions?.gitlabURL ?? 'https://gitlab.com/',
                runnerRegistrationToken: registrationToken,
                unregisterRunners: true,
                rbac: {
                    create: true,
                    clusterWideAccess: false,
                },
                metrics: {
                    enabled: true,
                },
                securityContext: {
                    fsGroup: 65533,
                    runAsUser: 100,
                },
                // runners default image when job start not set "image" in gitlab-ci.yaml.
                runners: {
                    image: props.helmRunnerOptions?.jobDefaultImage ?? 'alpine:3.11',
                    tags: this.synthesizeTags(props.helmRunnerOptions?.tags ?? ['eks', 'fargate', 'runner']),
                    privileged: false,
                },
                envVars: [
                    {
                        name: 'RUNNER_EXECUTOR',
                        value: 'kubernetes',
                    },
                ],
            },
        });
        return cluster;
    }
    createSecurityGroup() {
        const sg = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'GitlabEksSecurityGroup', {
            vpc: this.vpc,
        });
        new cdk.CfnOutput(this, 'SecurityGroup', { value: sg.securityGroupId });
        return sg;
    }
    createEksServiceRole() {
        return new aws_cdk_lib_1.aws_iam.Role(this, 'GitlabEksServiceRole', {
            managedPolicies: [aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy')],
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('eks.amazonaws.com'),
        });
    }
    // this.gitlabEksRole = new iam.Role(this, 'GitlabManagedEksRole', {
    //   assumedBy: new iam.AccountPrincipal(props.accountId),
    //   externalIds: [props.externalId],
    //   inlinePolicies: {
    //     default: policy,
    //   },
    // });
    //   new cdk.CfnOutput(this, 'RoleArn', { value: this.gitlabEksRole.roleArn });
    // }
    createGitlabManagedEksRole(props) {
        const policy = new aws_cdk_lib_1.aws_iam.PolicyDocument({
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: [
                        'autoscaling:CreateAutoScalingGroup',
                        'autoscaling:DescribeAutoScalingGroups',
                        'autoscaling:DescribeScalingActivities',
                        'autoscaling:UpdateAutoScalingGroup',
                        'autoscaling:CreateLaunchConfiguration',
                        'autoscaling:DescribeLaunchConfigurations',
                        'cloudformation:CreateStack',
                        'cloudformation:DescribeStacks',
                        'ec2:AuthorizeSecurityGroupEgress',
                        'ec2:AuthorizeSecurityGroupIngress',
                        'ec2:RevokeSecurityGroupEgress',
                        'ec2:RevokeSecurityGroupIngress',
                        'ec2:CreateSecurityGroup',
                        'ec2:createTags',
                        'ec2:DescribeImages',
                        'ec2:DescribeKeyPairs',
                        'ec2:DescribeRegions',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeVpcs',
                        'eks:CreateCluster',
                        'eks:DescribeCluster',
                        'iam:AddRoleToInstanceProfile',
                        'iam:AttachRolePolicy',
                        'iam:CreateRole',
                        'iam:CreateInstanceProfile',
                        'iam:CreateServiceLinkedRole',
                        'iam:GetRole',
                        'iam:ListRoles',
                        'iam:PassRole',
                        'ssm:GetParameters',
                    ],
                    resources: ['*'],
                }),
            ],
        });
        this.gitlabEksRole = new aws_cdk_lib_1.aws_iam.Role(this, 'GitlabManagedEksRole', {
            assumedBy: new aws_cdk_lib_1.aws_iam.AccountPrincipal(props.accountId),
            externalIds: [props.externalId],
            inlinePolicies: {
                default: policy,
            },
        });
        const eksAdminRole = this.createEksServiceRole();
        this.createSecurityGroup();
        const subnetIds = this.vpc.selectSubnets({
            subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE,
        }).subnetIds;
        new cdk.CfnOutput(this, 'EksClusterSubnetIds', {
            value: subnetIds.join(','),
        });
        new cdk.CfnOutput(this, 'RoleArn', { value: this.gitlabEksRole.roleArn });
        new cdk.CfnOutput(this, 'EksAdminRole', { value: eksAdminRole.roleName });
    }
    synthesizeTags(tags) {
        return tags.join(',');
    }
}
exports.Provider = Provider;
_a = JSII_RTTI_SYMBOL_1;
Provider[_a] = { fqn: "cdk-gitlab.Provider", version: "2.0.165" };
//# sourceMappingURL=data:application/json;base64,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