// Copyright 2022 The TensorStore Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <stddef.h>

#include <cmath>
#include <vector>

#include <gmock/gmock.h>
#include <gtest/gtest.h>
#include "absl/strings/cord.h"
#include "riegeli/bytes/cord_reader.h"
#include "riegeli/bytes/cord_writer.h"
#include "riegeli/bytes/string_reader.h"
#include "tensorstore/internal/image/jpeg_reader.h"
#include "tensorstore/internal/image/jpeg_writer.h"
#include "tensorstore/util/result.h"
#include "tensorstore/util/span.h"
#include "tensorstore/util/status_testutil.h"

namespace {

using ::tensorstore::internal_image::ImageInfo;
using ::tensorstore::internal_image::JpegReader;
using ::tensorstore::internal_image::JpegWriter;

TEST(JpegTest, Decode) {
  // Started the same as the png image, but very much the worse for wear after
  // jpeg minimization.

  // hexdump -e \"\"\ 16/1\ \"\ 0x%02x,\"\ \"\\n\" image.jpg
  static constexpr unsigned char data[] = {
      0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01,
      0x01, 0x00, 0x00, 0x48, 0x00, 0x48, 0x00, 0x00, 0xff, 0xdb, 0x00, 0x84,
      0x00, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x30, 0x1c, 0x1c, 0x30, 0x44,
      0x30, 0x30, 0x30, 0x44, 0x5c, 0x44, 0x44, 0x44, 0x44, 0x5c, 0x74, 0x5c,
      0x5c, 0x5c, 0x5c, 0x5c, 0x74, 0x8c, 0x74, 0x74, 0x74, 0x74, 0x74, 0x74,
      0x8c, 0x8c, 0x8c, 0x8c, 0x8c, 0x8c, 0x8c, 0x8c, 0xa8, 0xa8, 0xa8, 0xa8,
      0xa8, 0xa8, 0xc4, 0xc4, 0xc4, 0xc4, 0xc4, 0xdc, 0xdc, 0xdc, 0xdc, 0xdc,
      0xdc, 0xdc, 0xdc, 0xdc, 0xdc, 0x01, 0x22, 0x24, 0x24, 0x38, 0x34, 0x38,
      0x60, 0x34, 0x34, 0x60, 0xe6, 0x9c, 0x80, 0x9c, 0xe6, 0xe6, 0xe6, 0xe6,
      0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6,
      0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6,
      0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6,
      0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xe6, 0xff, 0xc2,
      0x00, 0x11, 0x08, 0x01, 0x00, 0x01, 0x00, 0x03, 0x01, 0x22, 0x00, 0x02,
      0x11, 0x01, 0x03, 0x11, 0x01, 0xff, 0xc4, 0x00, 0x17, 0x00, 0x01, 0x01,
      0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x01, 0x00, 0x05, 0x06, 0xff, 0xda, 0x00, 0x08, 0x01, 0x01,
      0x00, 0x00, 0x00, 0x00, 0xe6, 0xd6, 0x59, 0x59, 0x65, 0x96, 0x59, 0x65,
      0x95, 0xcb, 0x96, 0x56, 0x59, 0x65, 0x96, 0x59, 0x65, 0x67, 0x2d, 0x96,
      0x56, 0x59, 0x65, 0x96, 0x59, 0x65, 0x6c, 0xb5, 0x95, 0x96, 0x59, 0x65,
      0x96, 0x59, 0x59, 0x72, 0xe5, 0x95, 0x96, 0x59, 0x65, 0x96, 0x59, 0x59,
      0xcb, 0x65, 0x95, 0x96, 0x59, 0x65, 0x96, 0x59, 0x5b, 0x2d, 0x65, 0x65,
      0x96, 0x59, 0x65, 0x96, 0x56, 0x5c, 0xb9, 0x65, 0x65, 0x96, 0x59, 0x65,
      0x96, 0x56, 0x72, 0xd9, 0x59, 0x65, 0x96, 0x59, 0x65, 0x95, 0x96, 0xcb,
      0x59, 0x59, 0x65, 0x96, 0x59, 0x65, 0x95, 0x97, 0x2e, 0x56, 0x59, 0x65,
      0x96, 0x59, 0x65, 0x65, 0x9c, 0xb6, 0x56, 0x59, 0x65, 0x96, 0x59, 0x65,
      0x65, 0xb2, 0xd6, 0x56, 0x59, 0x65, 0x96, 0x59, 0x59, 0x65, 0xcb, 0x95,
      0x96, 0x59, 0x65, 0x96, 0x59, 0x59, 0x67, 0x2d, 0x95, 0x96, 0x59, 0x65,
      0x96, 0x56, 0x59, 0x6c, 0xb5, 0x65, 0x96, 0x59, 0x65, 0x96, 0x56, 0x59,
      0x72, 0xe5, 0x65, 0x96, 0x59, 0x65, 0x95, 0x96, 0x59, 0xcb, 0x59, 0x65,
      0x96, 0x59, 0x65, 0x95, 0x96, 0x5b, 0x2d, 0x59, 0x65, 0x96, 0x59, 0x65,
      0x95, 0x96, 0x5c, 0xb6, 0x59, 0x65, 0x96, 0x59, 0x65, 0x65, 0x96, 0x72,
      0xd6, 0x59, 0x65, 0x96, 0x59, 0x65, 0x65, 0x96, 0xcc, 0x96, 0x59, 0x65,
      0x96, 0x59, 0x59, 0x65, 0x97, 0x2d, 0x96, 0x59, 0x65, 0x96, 0x59, 0x59,
      0x65, 0x9c, 0xb5, 0x96, 0x59, 0x65, 0x96, 0x56, 0x59, 0x65, 0xb3, 0x25,
      0x96, 0x59, 0x65, 0x96, 0x56, 0x59, 0x65, 0xcb, 0x65, 0x96, 0x59, 0x65,
      0x95, 0x96, 0x59, 0x67, 0x2d, 0x65, 0x96, 0x59, 0x65, 0x95, 0x96, 0x59,
      0x6c, 0xc9, 0x65, 0x96, 0x59, 0x65, 0x65, 0x96, 0x59, 0x72, 0xd9, 0x65,
      0x96, 0x59, 0x65, 0x65, 0x96, 0x59, 0xcb, 0x59, 0x65, 0x96, 0x59, 0x65,
      0x65, 0x96, 0x5b, 0x32, 0x59, 0x65, 0x96, 0x59, 0x59, 0x65, 0x96, 0x5c,
      0xb6, 0x59, 0x65, 0x96, 0x59, 0x59, 0x65, 0x96, 0x7f, 0xff, 0xc4, 0x00,
      0x16, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0xff, 0xda, 0x00,
      0x08, 0x01, 0x02, 0x10, 0x00, 0x00, 0x00, 0x08, 0x82, 0x08, 0x82, 0xd0,
      0x41, 0x04, 0x41, 0x06, 0x90, 0x82, 0x20, 0x82, 0x34, 0x44, 0x10, 0x41,
      0x11, 0xa2, 0x08, 0x20, 0x88, 0x2d, 0x04, 0x10, 0x44, 0x10, 0x69, 0x08,
      0x82, 0x08, 0x23, 0x44, 0x41, 0x04, 0x11, 0x1a, 0x20, 0x82, 0x08, 0x82,
      0xd0, 0x41, 0x10, 0x41, 0x06, 0x90, 0x88, 0x20, 0x82, 0x34, 0x84, 0x10,
      0x44, 0x11, 0xa2, 0x08, 0x22, 0x08, 0x2d, 0x04, 0x11, 0x04, 0x10, 0x69,
      0x20, 0x82, 0x08, 0x83, 0x48, 0x41, 0x04, 0x41, 0x1f, 0xff, 0xc4, 0x00,
      0x17, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x01, 0x05, 0x00, 0xff, 0xda,
      0x00, 0x08, 0x01, 0x03, 0x10, 0x00, 0x00, 0x00, 0xe4, 0x92, 0x55, 0x24,
      0x92, 0x55, 0x61, 0x54, 0x92, 0x49, 0x54, 0x92, 0x4b, 0x05, 0x24, 0xaa,
      0x49, 0x24, 0xaa, 0x58, 0x29, 0x24, 0x92, 0xa9, 0x24, 0x92, 0xc1, 0x4a,
      0xa4, 0x92, 0x4a, 0xa4, 0x96, 0x0a, 0x49, 0x25, 0x52, 0x49, 0x25, 0x70,
      0x95, 0x49, 0x24, 0x95, 0x49, 0x25, 0x82, 0x92, 0x4a, 0xa4, 0x92, 0x4a,
      0xac, 0x2a, 0x92, 0x49, 0x2a, 0x92, 0x49, 0x60, 0xa4, 0x95, 0x49, 0x24,
      0x95, 0x4b, 0x05, 0x24, 0x92, 0x55, 0x24, 0x92, 0x58, 0x29, 0x2a, 0x92,
      0x49, 0x2a, 0x92, 0xc1, 0x49, 0x24, 0xaa, 0x49, 0x24, 0xae, 0x12, 0xa9,
      0x24, 0x92, 0xa9, 0x24, 0xb0, 0x52, 0x49, 0x54, 0x92, 0x49, 0x55, 0x85,
      0x52, 0x49, 0x25, 0x52, 0x49, 0x2f, 0xff, 0xc4, 0x00, 0x14, 0x10, 0x01,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0xa0, 0xff, 0xda, 0x00, 0x08, 0x01, 0x01, 0x00, 0x01,
      0x3f, 0x00, 0x00, 0x1f, 0xff, 0xc4, 0x00, 0x14, 0x11, 0x01, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x80, 0xff, 0xda, 0x00, 0x08, 0x01, 0x02, 0x01, 0x01, 0x3f, 0x00,
      0x00, 0x7f, 0xff, 0xc4, 0x00, 0x15, 0x11, 0x01, 0x01, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x01, 0xff, 0xda, 0x00, 0x08, 0x01, 0x03, 0x01, 0x01, 0x3f, 0x00, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
      0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0xff, 0xd9,
  };

  riegeli::StringReader string_reader(reinterpret_cast<const char*>(data),
                                      sizeof(data));

  JpegReader decoder;
  ASSERT_THAT(decoder.Initialize(&string_reader), ::tensorstore::IsOk());

  const auto info = decoder.GetImageInfo();
  EXPECT_EQ(256, info.width);
  EXPECT_EQ(256, info.height);
  EXPECT_EQ(3, info.num_components);

  std::vector<unsigned char> pixel(256 * 256 * 3);
  ASSERT_THAT(decoder.Decode(pixel), tensorstore::IsOk());

  EXPECT_EQ(0, pixel[0]);
  EXPECT_EQ(8, pixel[1]);
  EXPECT_EQ(0, pixel[2]);
}

// Tests resistance to libjpeg-turbo vulnerability LJT-01-003 DoS via
// progressive, arithmetic image decoding.
//
// https://cure53.de/pentest-report_libjpeg-turbo.pdf
TEST(JpegTest, LJT_01_003) {
  static constexpr unsigned char initial[] = {
      /*SOI*/ 0xFF, 0xD8,
      /*SOF10*/ 0xFF, 0xCA, 0x00, 0x0B, 0x08,
      /*dimension*/ 0x20, 0x00,
      /*dimension*/ 0x20, 0x00, 0x01, 0x00, 0x11, 0x00,
      /*DQT*/ 0xFF, 0xDB, 0x00, 0x43, 0x00,
      /*quanttab*/
      16, 11, 10, 16, 24, 40, 51, 61,      //
      12, 12, 14, 19, 26, 58, 60, 55,      //
      14, 13, 16, 24, 40, 57, 69, 56,      //
      14, 17, 22, 29, 51, 87, 80, 62,      //
      18, 22, 37, 56, 68, 109, 103, 77,    //
      24, 35, 55, 64, 81, 104, 113, 92,    //
      49, 64, 78, 87, 103, 121, 120, 101,  //
      72, 92, 95, 98, 112, 100, 103, 99,   //
  };
  std::string encoded(std::begin(initial), std::end(initial));
  static constexpr unsigned char sos[] = {0xFF, 0xDA, 0x00, 0x08, 0x01,
                                          0x00, 0x00, 0x00, 0x00, 0x10};
  encoded.reserve(8000 * 1024);
  while (encoded.size() + std::size(sos) < 8000 * 1024) {
    encoded.append(std::begin(sos), std::end(sos));
  }

  // libjpeg-turbo reports a warning, and we treat all warnings as errors and
  // abort, which prevents the DoS.
  JpegReader decoder;
  riegeli::StringReader string_reader(encoded);
  EXPECT_THAT(decoder.Initialize(&string_reader), ::tensorstore::IsOk());

  const auto info = decoder.GetImageInfo();
  EXPECT_EQ(8192, info.width);
  EXPECT_EQ(8192, info.height);
  EXPECT_EQ(1, info.num_components);

  std::vector<unsigned char> raw(ImageRequiredBytes(info));
  ASSERT_THAT(decoder.Decode(raw),
              ::tensorstore::MatchesStatus(absl::StatusCode::kDataLoss,
                                           "Inconsistent progression .*"));
}

TEST(JpegTest, EncodeDecode) {
  static constexpr unsigned char raw[] = {
      0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01,
      0x01, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0xff, 0xdb, 0x00, 0x43,
      0x00, 0x08, 0x06, 0x06, 0x07, 0x06, 0x05, 0x08, 0x07, 0x07, 0x07, 0x09,
      0x09, 0x08, 0x0a, 0x0c, 0x14, 0x0d, 0x0c, 0x0b, 0x0b, 0x0c, 0x19, 0x12,
      0x13, 0x0f, 0x14, 0x1d, 0x1a, 0x1f, 0x1e, 0x1d, 0x1a, 0x1c, 0x1c, 0x20,
      0x24, 0x2e, 0x27, 0x20, 0x22, 0x2c, 0x23, 0x1c, 0x1c, 0x28, 0x37, 0x29,
      0x2c, 0x30, 0x31, 0x34, 0x34, 0x34, 0x1f, 0x27, 0x39, 0x3d, 0x38, 0x32,
      0x3c, 0x2e, 0x33, 0x34, 0x32, 0xff, 0xc0, 0x00, 0x0b, 0x08, 0x00, 0x01,
      0x00, 0x01, 0x01, 0x01, 0x11, 0x00, 0xff, 0xc4, 0x00, 0x1f, 0x00, 0x00,
      0x01, 0x05, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
      0x09, 0x0a, 0x0b, 0xff, 0xc4, 0x00, 0xb5, 0x10, 0x00, 0x02, 0x01, 0x03,
      0x03, 0x02, 0x04, 0x03, 0x05, 0x05, 0x04, 0x04, 0x00, 0x00, 0x01, 0x7d,
      0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05, 0x12, 0x21, 0x31, 0x41, 0x06,
      0x13, 0x51, 0x61, 0x07, 0x22, 0x71, 0x14, 0x32, 0x81, 0x91, 0xa1, 0x08,
      0x23, 0x42, 0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0, 0x24, 0x33, 0x62, 0x72,
      0x82, 0x09, 0x0a, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x25, 0x26, 0x27, 0x28,
      0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x43, 0x44, 0x45,
      0x46, 0x47, 0x48, 0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
      0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x73, 0x74, 0x75,
      0x76, 0x77, 0x78, 0x79, 0x7a, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
      0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0xa2, 0xa3,
      0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6,
      0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9,
      0xca, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xe1, 0xe2,
      0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xf1, 0xf2, 0xf3, 0xf4,
      0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xff, 0xda, 0x00, 0x08, 0x01, 0x01,
      0x00, 0x00, 0x3f, 0x00, 0xf9, 0xfe, 0xbf, 0xff, 0xd9,
  };

  uint8_t pixels[1] = {};
  absl::Cord encoded;

  {
    JpegWriter encoder;
    riegeli::CordWriter cord_writer(&encoded);
    ASSERT_THAT(encoder.Initialize(&cord_writer), ::tensorstore::IsOk());

    ASSERT_THAT(encoder.Encode(ImageInfo{1, 1, 1}, pixels),
                ::tensorstore::IsOk());
    ASSERT_THAT(encoder.Done(), ::tensorstore::IsOk());
  }
  EXPECT_THAT(encoded, ::testing::StrEq(std::string_view(
                           reinterpret_cast<const char*>(raw), sizeof(raw))));

  {
    JpegReader decoder;
    riegeli::CordReader cord_reader(&encoded);
    ASSERT_THAT(decoder.Initialize(&cord_reader), ::tensorstore::IsOk());

    const auto& info = decoder.GetImageInfo();
    EXPECT_EQ(1, info.width);
    EXPECT_EQ(1, info.height);
    EXPECT_EQ(1, info.num_components);

    uint8_t new_pixel[1] = {};
    ASSERT_THAT(
        decoder.Decode(tensorstore::span(
            reinterpret_cast<unsigned char*>(&new_pixel), sizeof(new_pixel))),
        tensorstore::IsOk());
  }
}

TEST(JpegTest, NotAJpeg) {
  static constexpr unsigned char data[] = {
      0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a,  // sig
      0x00, 0x00, 0x00, 0x0d, 0x49, 0x48, 0x44, 0x52, 0x00, 0x00, 0x00, 0x01,
      0x00, 0x00, 0x00, 0x01, 0x08, 0x02, 0x00, 0x00, 0x00, 0x90, 0x77, 0x53,
      0xde,  // ihdr
  };

  riegeli::StringReader string_reader(reinterpret_cast<const char*>(data),
                                      sizeof(data));

  JpegReader decoder;
  EXPECT_THAT(decoder.Initialize(&string_reader),
              tensorstore::MatchesStatus(absl::StatusCode::kInvalidArgument));
}

TEST(JpegTest, CorruptData) {
  static constexpr unsigned char data[] = {
      0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

  riegeli::StringReader string_reader(reinterpret_cast<const char*>(data),
                                      sizeof(data));

  JpegReader decoder;
  EXPECT_THAT(decoder.Initialize(&string_reader),
              tensorstore::MatchesStatus(absl::StatusCode::kDataLoss));
}

TEST(JpegTest, InvalidNumComponents) {
  absl::Cord encoded;

  for (int num_components : {2, 5}) {
    std::vector<uint8_t> pixels(num_components);
    JpegWriter encoder;
    riegeli::CordWriter cord_writer(&encoded);
    EXPECT_THAT(encoder.Initialize(&cord_writer), ::tensorstore::IsOk());

    EXPECT_THAT(
        encoder.Encode(ImageInfo{1, 1, num_components}, pixels),
        ::tensorstore::MatchesStatus(absl::StatusCode::kInvalidArgument));
  }
}

}  // namespace
