// Copyright 2022 The TensorStore Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <stddef.h>

#include <cmath>
#include <vector>

#include <gmock/gmock.h>
#include <gtest/gtest.h>
#include "absl/strings/str_format.h"
#include "riegeli/bytes/string_reader.h"
#include "tensorstore/internal/image/bmp_reader.h"
#include "tensorstore/util/result.h"
#include "tensorstore/util/span.h"
#include "tensorstore/util/status_testutil.h"

namespace {

using ::tensorstore::internal_image::BmpReader;

TEST(BmpTest, Decode) {
  // hexdump -e \"\"\ 16/1\ \"\ 0x%02x,\"\ \"\\n\" image.bmp #83426dff
  static constexpr unsigned char data[] = {
      0x42, 0x4d, 0x8e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8a, 0x00,
      0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
      0x00, 0x00, 0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0xff,
      0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x42, 0x47,
      0x52, 0x73, 0x8f, 0xc2, 0xf5, 0x28, 0x51, 0xb8, 0x1e, 0x15, 0x1e, 0x85,
      0xeb, 0x01, 0x33, 0x33, 0x33, 0x13, 0x66, 0x66, 0x66, 0x26, 0x66, 0x66,
      0x66, 0x06, 0x99, 0x99, 0x99, 0x09, 0x3d, 0x0a, 0xd7, 0x03, 0x28, 0x5c,
      0x8f, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6d, 0x42, 0x83, 0x00,
  };

  riegeli::StringReader string_reader(reinterpret_cast<const char*>(data),
                                      sizeof(data));

  BmpReader decoder;
  ASSERT_THAT(decoder.Initialize(&string_reader), ::tensorstore::IsOk());

  const auto info = decoder.GetImageInfo();
  EXPECT_EQ(1, info.width);
  EXPECT_EQ(1, info.height);
  EXPECT_EQ(3, info.num_components);

  uint32_t pixel = 0;
  ASSERT_THAT(decoder.Decode(tensorstore::span<unsigned char>(
                  reinterpret_cast<unsigned char*>(&pixel), sizeof(pixel))),
              tensorstore::IsOk());

  EXPECT_EQ(0x6d4283, pixel) << absl::StrFormat(" %x", pixel);
}

TEST(BmpTest, CorruptData) {
  static constexpr unsigned char data[] = {
      0x42, 0x4d, 0x8e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8a, 0x00,
      0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

  riegeli::StringReader string_reader(reinterpret_cast<const char*>(data),
                                      sizeof(data));

  BmpReader decoder;
  EXPECT_THAT(decoder.Initialize(&string_reader),
              tensorstore::MatchesStatus(absl::StatusCode::kInvalidArgument));
}

}  // namespace
