"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const semver_1 = require("semver");
const aws_lambda_shared_1 = require("../shared/aws.lambda-shared");
const constants_1 = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
async function handler({ describe, widgetContext }) {
    var _a, _b;
    console.log(`Event: ${JSON.stringify({ describe, widgetContext }, null, 2)}`);
    if (describe) {
        // Description is naked markdown, and nothing else. The first YAML block it
        // contains is going to be used to pre-populate parameters in the widget
        // creation GUI.
        return [
            '### Missing Documentation widget',
            '',
            'This widget will render the contents of the missing documentations object',
            'from S3. It requires the following parameters:',
            '',
            '- `language`: the DocumentationLanguage for which missing documentation should',
            '  be listed.',
            '',
            'Example:',
            '```yaml',
            `language: csharp # One of: ${language_1.DocumentationLanguage.ALL.map(({ name }) => name).join(' | ')}`,
            '```',
        ].join('\n');
    }
    try {
        const language = language_1.DocumentationLanguage.fromString(widgetContext.params.language);
        const bucketName = env_lambda_shared_1.requireEnv('BUCKET_NAME');
        const key = constants_1.missingDocumentationKey(language);
        let { Body, ContentEncoding, LastModified } = await aws_lambda_shared_1.s3().getObject({
            Bucket: bucketName,
            Key: key,
        }).promise();
        // If it was compressed, de-compress it now...
        if (ContentEncoding === 'gzip') {
            Body = zlib_1.gunzipSync(Buffer.from(Body));
        }
        const list = Array.from(JSON.parse(Body.toString('utf-8'))
            .reduce((map, entry) => {
            // Split on the @ that is not at the beginning of the string
            const [name, version] = entry.split(/(?!^)@/);
            if (!map.has(name)) {
                map.set(name, []);
            }
            map.get(name).push(version);
            return map;
        }, new Map())
            .entries())
            .sort(([l], [r]) => l.localeCompare(r));
        // Trying to ensure we don't cause the dashboard to hang due to large DOM.
        const maxCount = 100;
        const objectUrl = `${widgetContext.domain}/s3/object/${bucketName}?prefix=${key}`;
        return {
            markdown: [
                'This widget shows the name and version(s) of all packages tracked by',
                'this ConstructHub instance, and for which at least one documentation',
                `object is missing for _${language.name}_.`,
                '',
                ...(list.length > maxCount
                    ? [
                        `Showing only the first ${maxCount} packages.`,
                        `The complete list can be obtained [from S3](${objectUrl}).`,
                        '',
                    ]
                    : []),
                'Id | Package Name | Count | Versions',
                '--:|--------------|-------|---------',
                ...list.slice(0, maxCount).map(([name, versions], index) => {
                    versions = semver_1.sort(versions).reverse();
                    return `${index + 1} | \`${name}\` | ${versions.length} | ${versions.map((v) => `\`${v}\``).join(', ')}`;
                }),
                '',
                `Last updated: \`${(_a = LastModified === null || LastModified === void 0 ? void 0 : LastModified.toISOString()) !== null && _a !== void 0 ? _a : 'N/A'}\``,
            ].join('\n'),
        };
    }
    catch (error) {
        return {
            markdown: [
                '**⚠️ An error occurred**',
                `- **name:** \`${error.name}\``,
                `- **message:** ${error.message}`,
                '- **stack:**',
                '  ```',
                (_b = error.stack) === null || _b === void 0 ? void 0 : _b.replace(/^/g, '  '),
                '  ```',
            ].join('\n'),
        };
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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