"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parsePrefixList = exports.createRestrictedSecurityGroups = void 0;
const crypto_1 = require("crypto");
const fs = require("fs");
const path_1 = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const s3_1 = require("./s3");
/**
   * Creates SecurityGroups where "sensitive" operations should be listed,
   * which only allows DNS requests to be issued within the VPC (to the local
   * Route53 resolver), as well as HTTPS (port 443) traffic to:
   * - allow-listed IP ranges
   * - endpoints within the same SecurityGroup.
   *
   * This returns MULTIPLE security groups in order to avoid hitting the maximum
   * count of rules per security group, which is relatively low, and prefix
   * lists count as their expansions.
   *
   * There is also a limit of how many security groups can be bound to a network
   * interface (defaults to 5), so there is only so much we can do here.
   *
   * @param scope the scope in which to attach new constructs.
   * @param vpc the VPC in which a SecurityGroup is to be added.
   */
function createRestrictedSecurityGroups(scope, vpc) {
    const securityGroups = new Array();
    securityGroups.push(createInternalTrafficSecurityGroup(scope, vpc));
    const ALLOW_LIST_DIR = path_1.resolve(__dirname, '..', 'resources', 'vpc-allow-lists');
    for (const file of fs.readdirSync(ALLOW_LIST_DIR)) {
        const matches = /^(.+)-(IPv4|IPv6)\.txt$/.exec(file);
        if (matches == null) {
            throw new Error(`Allow-list file ${file} in ${ALLOW_LIST_DIR} is invalid: file name must end in IPv4.txt or IPv6.txt`);
        }
        const [, namespace, ipLabel] = matches;
        const entries = parsePrefixList(path_1.join(ALLOW_LIST_DIR, file));
        if (entries.length === 0) {
            continue;
        }
        // We use a SHA-1 digest of the list of prefixes to be sure we create a
        // whole new prefix list whenever it changes, so we are never bothered by
        // the maxEntries being what it is.
        const hash = entries.reduce((h, { cidr }) => h.update(cidr).update('\0'), crypto_1.createHash('SHA1')).digest('hex');
        // Note - the `prefixListName` is NOT a physical ID, and so updating it
        // will NOT cause a replacement. Additionally, it needs not be unique in
        // any way.
        const pl = new aws_ec2_1.CfnPrefixList(scope, `${namespace}.${ipLabel}#${hash}`, {
            addressFamily: ipLabel,
            prefixListName: `${namespace}.${ipLabel}`,
            entries,
            maxEntries: entries.length,
        });
        const id = `${namespace}-${ipLabel}`;
        const sg = new aws_ec2_1.SecurityGroup(scope, id, {
            allowAllOutbound: false,
            description: `${scope.node.path}/${id}`,
            vpc,
        });
        // We intentionally ONLY allow HTTPS though there...
        sg.connections.allowTo(NamedPeer.from(aws_ec2_1.Peer.prefixList(pl.attrPrefixListId), pl.node.path), aws_ec2_1.Port.tcp(443), `to ${namespace} (${ipLabel})`);
        core_1.Tags.of(sg).add('Name', `${namespace}.${ipLabel}`);
        securityGroups.push(sg);
    }
    return securityGroups;
}
exports.createRestrictedSecurityGroups = createRestrictedSecurityGroups;
/**
 * Creates a SecurityGroup that allows traffic to flow freely between
 * endpoints within itself on port 443, to the local Route53 resolver on DNS
 * ports, and to the region's AW S3 prefix list on port 443.
 *
 * @param scope the scope in which to attach the new Security Group.
 * @param vpc the VPC in which the SecurityGroup will be created.
 */
function createInternalTrafficSecurityGroup(scope, vpc) {
    const sg = new aws_ec2_1.SecurityGroup(scope, 'InternalTraffic', {
        allowAllOutbound: false,
        description: `${scope.node.path}/SG`,
        vpc,
    });
    // Allow all traffic within the security group on port 443
    sg.connections.allowInternally(aws_ec2_1.Port.tcp(443), 'Traffic within this SecurityGroup');
    // Allow access to S3. This is needed for the S3 Gateway endpoint to work.
    sg.connections.allowTo(NamedPeer.from(aws_ec2_1.Peer.prefixList(new s3_1.S3PrefixList(scope, 'S3-PrefixList').prefixListId), 'AWS S3'), aws_ec2_1.Port.tcp(443), 'to AWS S3');
    // Allow making DNS requests, there should be a Route53 resolver wihtin the VPC.
    sg.connections.allowTo(aws_ec2_1.Peer.ipv4(vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(53), 'to Route53 DNS resolver');
    sg.connections.allowTo(aws_ec2_1.Peer.ipv4(vpc.vpcCidrBlock), aws_ec2_1.Port.udp(53), 'to Route53 DNS resolver');
    return sg;
}
/**
 * Parses the PrefixList in the designated path.
 *
 * @param filePath the file containing the prefix list.
 */
function parsePrefixList(filePath) {
    return fs.readFileSync(filePath, 'utf8')
        .split(/\n/)
        .map((line) => {
        const match = /^\s*([^\s]+)?\s*(?:#.*)?$/.exec(line);
        if (!match) {
            throw new Error(`Invalid line in allow list ${filePath}: ${line}`);
        }
        const [, cidr] = match;
        return cidr;
    })
        // Remove empty lines.
        .filter((cidr) => !!cidr)
        .sort()
        .map((cidr) => ({ cidr }));
}
exports.parsePrefixList = parsePrefixList;
/**
 * This is to work around an issue where the peer's `uniqueId` is a token for
 * our PrefixList values, and this causes the VPC construct to "de-duplicate"
 * all of them (it considers they are identical).
 *
 * There is a fix in the latest EC2 library, however that fix isn't great
 * either, as it addresses the problem at the wrong location (in the in/egress
 * rule, instead of in the peer).
 *
 * Basically, this ensures the `uniqueId` is some string we control, so we
 * remain faithful to the declaraiton intent.
 */
class NamedPeer {
    constructor(peer, uniqueId) {
        this.peer = peer;
        this.uniqueId = uniqueId;
        this.connections = new aws_ec2_1.Connections({ peer: this });
    }
    static from(peer, name) {
        return new NamedPeer(peer, name);
    }
    get canInlineRule() {
        return this.peer.canInlineRule;
    }
    toIngressRuleConfig() {
        return this.peer.toIngressRuleConfig();
    }
    toEgressRuleConfig() {
        return this.peer.toEgressRuleConfig();
    }
}
//# sourceMappingURL=data:application/json;base64,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