"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const os_1 = require("os");
const path = require("path");
const glob = require("glob");
const cleanup_1 = require("./cleanup");
const common_1 = require("./common");
const dependencies_1 = require("./dependencies");
const file_1 = require("./file");
const gitattributes_1 = require("./gitattributes");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const project_build_1 = require("./project-build");
const projenrc_json_1 = require("./projenrc-json");
const renovatebot_1 = require("./renovatebot");
const tasks_1 = require("./tasks");
const util_1 = require("./util");
/**
 * Base project
 */
class Project {
    constructor(options) {
        this._components = new Array();
        this.subprojects = new Array();
        this.tips = new Array();
        this.initProject = render_options_1.resolveInitProject(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this._ejected = util_1.isTruthy(process.env.PROJEN_EJECTING);
        if (this.ejected) {
            this.projenCommand = "scripts/run-task";
        }
        else {
            this.projenCommand = options.projenCommand ?? "npx projen";
        }
        this.outdir = this.determineOutdir(options.outdir);
        this.root = this.parent ? this.parent.root : this;
        // must happen after this.outdir, this.parent and this.root are initialized
        this.parent?._addSubProject(this);
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this);
        this.annotateGenerated("/.projen/**"); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        this.gitignore = new ignore_file_1.IgnoreFile(this, ".gitignore");
        this.gitignore.exclude("node_modules/"); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injectionn but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        if (!this.ejected) {
            this.defaultTask = this.tasks.addTask(Project.DEFAULT_TASK, {
                description: "Synthesize project files",
            });
            this.ejectTask = this.tasks.addTask("eject", {
                description: "Remove projen from the project",
                env: {
                    PROJEN_EJECTING: "true",
                },
            });
            this.ejectTask.spawn(this.defaultTask);
        }
        this.projectBuild = new project_build_1.ProjectBuild(this);
        this.deps = new dependencies_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        const projenrcJson = options.projenrcJson ?? false;
        if (projenrcJson) {
            new projenrc_json_1.Projenrc(this, options.projenrcJsonOptions);
        }
        if (options.renovatebot) {
            new renovatebot_1.Renovatebot(this, options.renovatebotOptions);
        }
        this.commitGenerated = options.commitGenerated ?? true;
        if (!this.ejected) {
            new json_1.JsonFile(this, cleanup_1.FILE_MANIFEST, {
                omitEmpty: true,
                obj: () => ({
                    // replace `\` with `/` to ensure paths match across platforms
                    files: this.files
                        .filter((f) => f.readonly)
                        .map((f) => f.path.replace(/\\/g, "/")),
                }),
                // This file is used by projen to track the generated files, so must be committed.
                committed: true,
            });
        }
    }
    /**
     * Returns all the components within this project.
     */
    get components() {
        return [...this._components];
    }
    /**
     * All files in this project.
     */
    get files() {
        const isFile = (c) => c instanceof file_1.FileBase;
        return this._components
            .filter(isFile)
            .sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * Adds a new task to this project. This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add
     * @param props Task properties
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    get buildTask() {
        return this.projectBuild.buildTask;
    }
    get compileTask() {
        return this.projectBuild.compileTask;
    }
    get testTask() {
        return this.projectBuild.testTask;
    }
    get preCompileTask() {
        return this.projectBuild.preCompileTask;
    }
    get postCompileTask() {
        return this.projectBuild.postCompileTask;
    }
    get packageTask() {
        return this.projectBuild.packageTask;
    }
    /**
     * Finds a file at the specified relative path within this project and all
     * its subprojects.
     *
     * @param filePath The file path. If this path is relative, it will be resolved
     * from the root of _this_ project.
     * @returns a `FileBase` or undefined if there is no file in that path
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        for (const file of this.files) {
            if (absolute === file.absolutePath) {
                return file;
            }
        }
        for (const child of this.subprojects) {
            const file = child.tryFindFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * Finds a json file by name.
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds an object file (like JsonFile, YamlFile, etc.) by name.
     * @param filePath The file path.
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds a file at the specified relative path within this project and removes
     * it.
     *
     * @param filePath The file path. If this path is relative, it will be
     * resolved from the root of _this_ project.
     * @returns a `FileBase` if the file was found and removed, or undefined if
     * the file was not found.
     */
    tryRemoveFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        const isFile = (c) => c instanceof file_1.FileBase;
        const index = this._components.findIndex((c) => isFile(c) && c.absolutePath === absolute);
        if (index !== -1) {
            return this._components.splice(index, 1)[0];
        }
        for (const child of this.subprojects) {
            const file = child.tryRemoveFile(absolute);
            if (file) {
                return file;
            }
        }
        return undefined;
    }
    /**
     * Prints a "tip" message during synthesis.
     * @param message The message
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * Exclude the matching files from pre-synth cleanup. Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `npx projen@${common_1.PROJEN_VERSION} ${task.name}`;
    }
    /**
     * Exclude these files from the bundled package. Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * Adds a .gitignore pattern.
     * @param pattern The glob pattern to ignore.
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * Consider a set of files as "generated". This method is implemented by
     * derived classes and used for example, to add git attributes to tell GitHub
     * that certain files are generated.
     *
     * @param _glob the glob pattern to match (could be a file path).
     */
    annotateGenerated(_glob) {
        // nothing to do at the abstract level
    }
    /**
     * Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all sub-projects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     */
    synth() {
        const outdir = this.outdir;
        this.logger.debug("Synthesizing project...");
        this.preSynthesize();
        for (const comp of this._components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + "/**");
        }
        // delete orphaned files before we start synthesizing new ones
        cleanup_1.cleanup(outdir, this.files.map((f) => f.path.replace(/\\/g, "/")), this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this._components) {
            comp.synthesize();
        }
        if (!util_1.isTruthy(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this._components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        if (this.ejected) {
            this.logger.debug("Ejecting project...");
            // Backup projenrc files
            const files = glob.sync(".projenrc.*", {
                cwd: this.outdir,
                dot: true,
                nodir: true,
                absolute: true,
            });
            for (const file of files) {
                fs_1.renameSync(file, `${file}.bak`);
            }
        }
        this.logger.debug("Synthesis complete");
    }
    /**
     * Whether or not the project is being ejected.
     */
    get ejected() {
        return this._ejected;
    }
    /**
     * Called before all components are synthesized.
     */
    preSynthesize() { }
    /**
     * Called after all components are synthesized. Order is *not* guaranteed.
     */
    postSynthesize() { }
    /**
     * Adds a component to the project.
     * @internal
     */
    _addComponent(component) {
        this._components.push(component);
    }
    /**
     * Adds a sub-project to this project.
     *
     * This is automatically called when a new project is created with `parent`
     * pointing to this project, so there is no real need to call this manually.
     *
     * @param sub-project The child project to add.
     * @internal
     */
    _addSubProject(subproject) {
        if (subproject.parent !== this) {
            throw new Error('"parent" of child project must be this project');
        }
        // check that `outdir` is exclusive
        for (const p of this.subprojects) {
            if (path.resolve(p.outdir) === path.resolve(subproject.outdir)) {
                throw new Error(`there is already a sub-project with "outdir": ${subproject.outdir}`);
            }
        }
        this.subprojects.push(subproject);
    }
    /**
     * Resolves the project's output directory.
     */
    determineOutdir(outdirOption) {
        if (this.parent && outdirOption && path.isAbsolute(outdirOption)) {
            throw new Error('"outdir" must be a relative path');
        }
        // if this is a subproject, it is relative to the parent
        if (this.parent) {
            if (!outdirOption) {
                throw new Error('"outdir" must be specified for subprojects');
            }
            return path.resolve(this.parent.outdir, outdirOption);
        }
        // if this is running inside a test and outdir is not explicitly set
        // use a temp directory (unless cwd is aleady under tmp)
        if (common_1.IS_TEST_RUN && !outdirOption) {
            const realCwd = fs_1.realpathSync(process.cwd());
            const realTmp = fs_1.realpathSync(os_1.tmpdir());
            if (realCwd.startsWith(realTmp)) {
                return path.resolve(realCwd, outdirOption ?? ".");
            }
            return fs_1.mkdtempSync(path.join(os_1.tmpdir(), "projen."));
        }
        return path.resolve(outdirOption ?? ".");
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.67.0" };
/**
 * The name of the default task (the task executed when `projen` is run without arguments). Normally
 * this task should synthesize the project files.
 */
Project.DEFAULT_TASK = "default";
/**
 * Which type of project this is.
 *
 * @deprecated no longer supported at the base project level
 */
var ProjectType;
(function (ProjectType) {
    /**
     * This module may be a either a library or an app.
     */
    ProjectType["UNKNOWN"] = "unknown";
    /**
     * This is a library, intended to be published to a package manager and
     * consumed by other projects.
     */
    ProjectType["LIB"] = "lib";
    /**
     * This is an app (service, tool, website, etc). Its artifacts are intended to
     * be deployed or published for end-user consumption.
     */
    ProjectType["APP"] = "app";
})(ProjectType = exports.ProjectType || (exports.ProjectType = {}));
//# sourceMappingURL=data:application/json;base64,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