"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Jest = exports.UpdateSnapshot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const semver = require("semver");
const json_1 = require("../json");
const DEFAULT_TEST_REPORTS_DIR = "test-reports";
var UpdateSnapshot;
(function (UpdateSnapshot) {
    /**
     * Always update snapshots in "test" task.
     */
    UpdateSnapshot["ALWAYS"] = "always";
    /**
     * Never update snapshots in "test" task and create a separate "test:update" task.
     */
    UpdateSnapshot["NEVER"] = "never";
})(UpdateSnapshot = exports.UpdateSnapshot || (exports.UpdateSnapshot = {}));
/**
 * Installs the following npm scripts:
 *
 * - `test`, intended for testing locally and in CI. Will update snapshots unless `updateSnapshot: UpdateSnapshot: NEVER` is set.
 * - `test:watch`, intended for automatically rerunning tests when files change.
 * - `test:update`, intended for testing locally and updating snapshots to match the latest unit under test. Only available when `updateSnapshot: UpdateSnapshot: NEVER`.
 *
 */
class Jest {
    constructor(project, options = {}) {
        this.project = project;
        // hard deprecation
        if (options.typescriptConfig) {
            throw new Error('"jestOptions.typescriptConfig" is deprecated. Use "typescriptProject.tsconfigDev" instead');
        }
        // Jest snapshot files are generated files!
        project.root.annotateGenerated("*.snap");
        this.jestVersion = options.jestVersion ? `@${options.jestVersion}` : "";
        project.addDevDeps(`jest${this.jestVersion}`);
        this.jestConfig = options.jestConfig;
        this.extraCliOptions = options.extraCliOptions ?? [];
        this.ignorePatterns = this.jestConfig?.testPathIgnorePatterns ??
            options.ignorePatterns ?? ["/node_modules/"];
        this.watchIgnorePatterns = this.jestConfig?.watchPathIgnorePatterns ?? [
            "/node_modules/",
        ];
        this.coverageReporters = this.jestConfig?.coverageReporters ?? [
            "json",
            "lcov",
            "clover",
            "cobertura",
        ];
        this.testMatch = this.jestConfig?.testMatch ?? [
            "**/__tests__/**/*.[jt]s?(x)",
            "**/?(*.)+(spec|test).[tj]s?(x)",
        ];
        const coverageDirectory = this.jestConfig?.coverageDirectory ?? "coverage";
        this.reporters = [];
        if (options.preserveDefaultReporters ?? true) {
            this.reporters.unshift("default");
        }
        this.config = {
            ...this.jestConfig,
            clearMocks: this.jestConfig?.clearMocks ?? true,
            collectCoverage: options.coverage ?? this.jestConfig?.collectCoverage ?? true,
            coverageReporters: this.coverageReporters,
            coverageDirectory: coverageDirectory,
            coveragePathIgnorePatterns: this.jestConfig?.coveragePathIgnorePatterns ?? this.ignorePatterns,
            testPathIgnorePatterns: this.ignorePatterns,
            watchPathIgnorePatterns: this.watchIgnorePatterns,
            testMatch: this.testMatch,
            reporters: this.reporters,
            snapshotResolver: (() => this._snapshotResolver),
        };
        if (options.junitReporting ?? true) {
            const reportsDir = DEFAULT_TEST_REPORTS_DIR;
            this.addReporter(["jest-junit", { outputDirectory: reportsDir }]);
            project.addDevDeps("jest-junit@^13");
            project.gitignore.exclude("# jest-junit artifacts", `/${reportsDir}/`, "junit.xml");
            project.npmignore?.exclude("# jest-junit artifacts", `/${reportsDir}/`, "junit.xml");
        }
        if (this.jestConfig?.reporters) {
            for (const reporter of this.jestConfig.reporters) {
                this.addReporter(reporter);
            }
        }
        if (this.jestConfig?.coverageThreshold) {
            this.config.coverageThreshold = {
                global: this.jestConfig?.coverageThreshold,
            };
        }
        this.configureTestCommand(options.updateSnapshot ?? UpdateSnapshot.ALWAYS);
        if (options.configFilePath) {
            this.file = new json_1.JsonFile(project, options.configFilePath, {
                obj: this.config,
            });
        }
        else {
            project.addFields({ jest: this.config });
        }
        const coverageDirectoryPath = path.posix.join("/", coverageDirectory, "/");
        project.npmignore?.exclude(coverageDirectoryPath);
        project.gitignore.exclude(coverageDirectoryPath);
        if (options.coverageText ?? true) {
            this.coverageReporters.push("text");
        }
    }
    /**
     * Adds a test match pattern.
     * @param pattern glob pattern to match for tests
     */
    addTestMatch(pattern) {
        this.testMatch.push(pattern);
    }
    /**
     * Adds a watch ignore pattern.
     * @param pattern The pattern (regular expression).
     */
    addWatchIgnorePattern(pattern) {
        this.watchIgnorePatterns.push(pattern);
    }
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    addReporter(reporter) {
        this.reporters.push(reporter);
    }
    addSnapshotResolver(file) {
        this._snapshotResolver = file;
    }
    configureTestCommand(updateSnapshot) {
        const jestOpts = ["--passWithNoTests", ...this.extraCliOptions];
        const jestConfigOpts = this.file && this.file.path != "jest.config.json"
            ? ` -c ${this.file.path}`
            : "";
        // as recommended in the jest docs, node > 14 may use native v8 coverage collection
        // https://jestjs.io/docs/en/cli#--coverageproviderprovider
        if (this.project.package.minNodeVersion &&
            semver.gte(this.project.package.minNodeVersion, "14.0.0")) {
            jestOpts.push("--coverageProvider=v8");
        }
        if (updateSnapshot === UpdateSnapshot.ALWAYS) {
            jestOpts.push("--updateSnapshot");
        }
        else {
            jestOpts.push("--ci"); // to prevent accepting new snapshots
            const testUpdate = this.project.tasks.tryFind("test:update");
            if (!testUpdate) {
                this.project.addTask("test:update", {
                    description: "Update jest snapshots",
                    exec: `jest --updateSnapshot ${jestOpts.join(" ")}${jestConfigOpts}`,
                    receiveArgs: true,
                });
            }
        }
        this.project.testTask.exec(`jest ${jestOpts.join(" ")}${jestConfigOpts}`, {
            receiveArgs: true,
        });
        const testWatch = this.project.tasks.tryFind("test:watch");
        if (!testWatch) {
            this.project.addTask("test:watch", {
                description: "Run jest in watch mode",
                exec: `jest --watch${jestConfigOpts}`,
            });
        }
    }
}
exports.Jest = Jest;
_a = JSII_RTTI_SYMBOL_1;
Jest[_a] = { fqn: "projen.javascript.Jest", version: "0.67.0" };
//# sourceMappingURL=data:application/json;base64,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