"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CiConfiguration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const yaml_1 = require("../yaml");
/**
 * CI for GitLab.
 * A CI is a configurable automated process made up of one or more stages/jobs.
 * @see https://docs.gitlab.com/ee/ci/yaml/
 */
class CiConfiguration extends component_1.Component {
    constructor(project, name, options) {
        super(project);
        /**
         * Defines default scripts that should run *after* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultAfterScript = [];
        /**
         * Defines default scripts that should run *before* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultBeforeScript = [];
        /**
         * A default list of additional Docker images to run scripts in. The service image is linked to the image specified in the  image parameter.
         */
        this.defaultServices = [];
        /**
         * Used to select a specific runner from the list of all runners that are available for the project.
         */
        this.defaultTags = [];
        /**
         * Can be `Include` or `Include[]`. Each `Include` will be a string, or an
         * object with properties for the method if including external YAML file. The external
         * content will be fetched, included and evaluated along the `.gitlab-ci.yml`.
         */
        this.include = [];
        /**
         * Groups jobs into stages. All jobs in one stage must complete before next stage is
         * executed. Defaults to ['build', 'test', 'deploy'].
         */
        this.stages = [];
        /**
         * Global variables that are passed to jobs.
         * If the job already has that variable defined, the job-level variable takes precedence.
         */
        this.variables = {};
        /**
         * The jobs in the CI configuration.
         */
        this.jobs = {};
        this.project = project;
        this.name = path.parse(name).name;
        this.path =
            this.name === "gitlab-ci"
                ? ".gitlab-ci.yml"
                : `.gitlab/ci-templates/${name.toLocaleLowerCase()}.yml`;
        this.file = new yaml_1.YamlFile(this.project, this.path, {
            obj: () => this.renderCI(),
            // GitLab needs to read the file from the repository in order to work.
            committed: true,
        });
        const defaults = options?.default;
        if (defaults) {
            this.defaultAfterScript.push(...(defaults.afterScript ?? []));
            this.defaultArtifacts = defaults.artifacts;
            defaults.beforeScript &&
                this.defaultBeforeScript.push(...defaults.beforeScript);
            this.defaultCache = defaults.cache;
            this.defaultImage = defaults.image;
            this.defaultInterruptible = defaults.interruptible;
            this.defaultRetry = defaults.retry;
            defaults.services && this.addServices(...defaults.services);
            defaults.tags && this.defaultTags.push(...defaults.tags);
            this.defaultTimeout = defaults.timeout;
        }
        this.pages = options?.pages;
        this.workflow = options?.workflow;
        if (options?.stages) {
            this.addStages(...options.stages);
        }
        if (options?.variables) {
            this.addJobs(options.variables);
        }
        if (options?.jobs) {
            this.addJobs(options.jobs);
        }
    }
    /**
     * Add additional yml/yaml files to the CI includes
     * @param includes The includes to add.
     */
    addIncludes(...includes) {
        for (const additional of includes) {
            this.assertIsValidInclude(additional);
            for (const existing of this.include) {
                if (this.areEqualIncludes(existing, additional)) {
                    throw new Error(`${this.name}: GitLab CI ${existing} already contains one or more templates specified in ${additional}.`);
                }
            }
            this.include.push(additional);
        }
    }
    /**
     * Throw an error if the provided Include is invalid.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param include the Include to validate.
     */
    assertIsValidInclude(include) {
        const combos = [
            include.local,
            include.file && include.project,
            include.remote,
            include.template,
        ];
        const len = combos.filter((x) => Boolean(x)).length;
        if (len !== 1) {
            throw new Error(`${this.name}: GitLab CI include ${include} contains ${len} property combination(s).
        A valid include configuration specifies *one* of the following property combinations.
        * local
        * file, project
        * remote
        * template  
        `);
        }
    }
    /**
     * Check if the equality of Includes.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param x First include to compare.
     * @param y Second include to compare.
     * @returns Whether the includes are equal.
     */
    areEqualIncludes(x, y) {
        if (x.local === y.local && x.local !== undefined) {
            return true;
        }
        else if (x.template === y.template && x.template !== undefined) {
            return true;
        }
        else if (x.remote === y.remote && x.remote !== undefined) {
            return true;
        }
        else if (x.project === y.project && x.ref === y.ref) {
            const xFiles = x.file ? x.file : [];
            const yFiles = y.file ? y.file : [];
            const allFiles = xFiles.concat(yFiles);
            return new Set(allFiles).size !== allFiles.length;
        }
        return false;
    }
    /**
     * Add additional services.
     * @param services The services to add.
     */
    addServices(...services) {
        for (const additional of services) {
            for (const existing of this.defaultServices) {
                if (additional.name === existing.name &&
                    additional.alias === existing.alias) {
                    throw new Error(`${this.name}: GitLab CI already contains service ${additional}.`);
                }
            }
            this.defaultServices.push(additional);
        }
    }
    /**
     * Add a globally defined variable to the CI configuration.
     * @param variables The variables to add.
     */
    addGlobalVariables(variables) {
        for (const [key, value] of Object.entries(variables)) {
            if (this.variables[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains variable ${key}.`);
            }
            this.variables[key] = value;
        }
    }
    /**
     * Add stages to the CI configuration if not already present.
     * @param stages stages to add.
     */
    addStages(...stages) {
        for (const stage of stages) {
            if (!this.stages.includes(stage)) {
                this.stages.push(stage);
            }
        }
    }
    /**
     * Add jobs and their stages to the CI configuration.
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        for (const [key, value] of Object.entries(jobs)) {
            if (this.jobs[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains job ${key}.`);
            }
            this.jobs[key] = value;
            if (value.stage) {
                this.addStages(value.stage);
            }
        }
    }
    renderCI() {
        return {
            default: this.renderDefault(),
            include: this.include.length > 0 ? snakeCaseKeys(this.include) : undefined,
            pages: snakeCaseKeys(this.pages),
            services: this.defaultServices.length > 0
                ? snakeCaseKeys(this.defaultServices)
                : undefined,
            variables: Object.entries(this.variables).length > 0 ? this.variables : undefined,
            workflow: snakeCaseKeys(this.workflow),
            stages: this.stages.length > 0 ? this.stages : undefined,
            // we do not want to change job names
            // as they can be hidden (https://docs.gitlab.com/ee/ci/jobs/index.html#hide-jobs)
            // or referenced in extends
            ...snakeCaseKeys(this.jobs, true),
        };
    }
    renderDefault() {
        const defaults = {
            afterScript: this.defaultAfterScript.length > 0
                ? this.defaultAfterScript
                : undefined,
            artifacts: this.defaultArtifacts,
            beforeScript: this.defaultBeforeScript.length > 0
                ? this.defaultBeforeScript
                : undefined,
            cache: this.defaultCache,
            image: this.defaultImage,
            interruptible: this.defaultInterruptible,
            retry: this.defaultRetry,
            services: this.defaultServices.length > 0 ? this.defaultServices : undefined,
            tags: this.defaultTags.length > 0 ? this.defaultTags : undefined,
            timeout: this.defaultTimeout,
        };
        return Object.values(defaults).filter((x) => x).length
            ? snakeCaseKeys(defaults)
            : undefined;
    }
}
exports.CiConfiguration = CiConfiguration;
_a = JSII_RTTI_SYMBOL_1;
CiConfiguration[_a] = { fqn: "projen.gitlab.CiConfiguration", version: "0.67.0" };
function snakeCaseKeys(obj, skipTopLevel = false) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map((o) => snakeCaseKeys(o));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null && k !== "variables") {
            v = snakeCaseKeys(v);
        }
        result[skipTopLevel ? k : case_1.snake(k)] = v;
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29uZmlndXJhdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9naXRsYWIvY29uZmlndXJhdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZCQUE2QjtBQUM3QiwrQkFBNkI7QUFhN0IsNENBQXlDO0FBRXpDLGtDQUFtQztBQW1DbkM7Ozs7R0FJRztBQUNILE1BQWEsZUFBZ0IsU0FBUSxxQkFBUztJQXdGNUMsWUFDRSxPQUFnQixFQUNoQixJQUFZLEVBQ1osT0FBZ0M7UUFFaEMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBNUVqQjs7V0FFRztRQUNhLHVCQUFrQixHQUFhLEVBQUUsQ0FBQztRQUtsRDs7V0FFRztRQUNhLHdCQUFtQixHQUFhLEVBQUUsQ0FBQztRQWlCbkQ7O1dBRUc7UUFDSyxvQkFBZSxHQUFjLEVBQUUsQ0FBQztRQUN4Qzs7V0FFRztRQUNNLGdCQUFXLEdBQWEsRUFBRSxDQUFDO1FBS3BDOzs7O1dBSUc7UUFDSyxZQUFPLEdBQWMsRUFBRSxDQUFDO1FBTWhDOzs7V0FHRztRQUNhLFdBQU0sR0FBYSxFQUFFLENBQUM7UUFDdEM7OztXQUdHO1FBQ2EsY0FBUyxHQUN2QixFQUFFLENBQUM7UUFLTDs7V0FFRztRQUNhLFNBQUksR0FBd0IsRUFBRSxDQUFDO1FBUTdDLElBQUksQ0FBQyxPQUFPLEdBQUcsT0FBTyxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUM7UUFDbEMsSUFBSSxDQUFDLElBQUk7WUFDUCxJQUFJLENBQUMsSUFBSSxLQUFLLFdBQVc7Z0JBQ3ZCLENBQUMsQ0FBQyxnQkFBZ0I7Z0JBQ2xCLENBQUMsQ0FBQyx3QkFBd0IsSUFBSSxDQUFDLGlCQUFpQixFQUFFLE1BQU0sQ0FBQztRQUM3RCxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksZUFBUSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLElBQUksRUFBRTtZQUNoRCxHQUFHLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUMxQixzRUFBc0U7WUFDdEUsU0FBUyxFQUFFLElBQUk7U0FDaEIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxRQUFRLEdBQUcsT0FBTyxFQUFFLE9BQU8sQ0FBQztRQUNsQyxJQUFJLFFBQVEsRUFBRTtZQUNaLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxXQUFXLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztZQUM5RCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsUUFBUSxDQUFDLFNBQVMsQ0FBQztZQUMzQyxRQUFRLENBQUMsWUFBWTtnQkFDbkIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxHQUFHLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUMxRCxJQUFJLENBQUMsWUFBWSxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUM7WUFDbkMsSUFBSSxDQUFDLFlBQVksR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDO1lBQ25DLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDO1lBQ25ELElBQUksQ0FBQyxZQUFZLEdBQUcsUUFBUSxDQUFDLEtBQUssQ0FBQztZQUNuQyxRQUFRLENBQUMsUUFBUSxJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDNUQsUUFBUSxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN6RCxJQUFJLENBQUMsY0FBYyxHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUM7U0FDeEM7UUFDRCxJQUFJLENBQUMsS0FBSyxHQUFHLE9BQU8sRUFBRSxLQUFLLENBQUM7UUFDNUIsSUFBSSxDQUFDLFFBQVEsR0FBRyxPQUFPLEVBQUUsUUFBUSxDQUFDO1FBQ2xDLElBQUksT0FBTyxFQUFFLE1BQU0sRUFBRTtZQUNuQixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ25DO1FBQ0QsSUFBSSxPQUFPLEVBQUUsU0FBUyxFQUFFO1lBQ3RCLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQ2pDO1FBQ0QsSUFBSSxPQUFPLEVBQUUsSUFBSSxFQUFFO1lBQ2pCLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQzVCO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFdBQVcsQ0FBQyxHQUFHLFFBQW1CO1FBQ3ZDLEtBQUssTUFBTSxVQUFVLElBQUksUUFBUSxFQUFFO1lBQ2pDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUN0QyxLQUFLLE1BQU0sUUFBUSxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUU7Z0JBQ25DLElBQUksSUFBSSxDQUFDLGdCQUFnQixDQUFDLFFBQVEsRUFBRSxVQUFVLENBQUMsRUFBRTtvQkFDL0MsTUFBTSxJQUFJLEtBQUssQ0FDYixHQUFHLElBQUksQ0FBQyxJQUFJLGVBQWUsUUFBUSx3REFBd0QsVUFBVSxHQUFHLENBQ3pHLENBQUM7aUJBQ0g7YUFDRjtZQUNELElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQy9CO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSyxvQkFBb0IsQ0FBQyxPQUFnQjtRQUMzQyxNQUFNLE1BQU0sR0FBRztZQUNiLE9BQU8sQ0FBQyxLQUFLO1lBQ2IsT0FBTyxDQUFDLElBQUksSUFBSSxPQUFPLENBQUMsT0FBTztZQUMvQixPQUFPLENBQUMsTUFBTTtZQUNkLE9BQU8sQ0FBQyxRQUFRO1NBQ2pCLENBQUM7UUFDRixNQUFNLEdBQUcsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFDcEQsSUFBSSxHQUFHLEtBQUssQ0FBQyxFQUFFO1lBQ2IsTUFBTSxJQUFJLEtBQUssQ0FDYixHQUFHLElBQUksQ0FBQyxJQUFJLHVCQUF1QixPQUFPLGFBQWEsR0FBRzs7Ozs7O1NBTXpELENBQ0YsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNLLGdCQUFnQixDQUFDLENBQVUsRUFBRSxDQUFVO1FBQzdDLElBQUksQ0FBQyxDQUFDLEtBQUssS0FBSyxDQUFDLENBQUMsS0FBSyxJQUFJLENBQUMsQ0FBQyxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQ2hELE9BQU8sSUFBSSxDQUFDO1NBQ2I7YUFBTSxJQUFJLENBQUMsQ0FBQyxRQUFRLEtBQUssQ0FBQyxDQUFDLFFBQVEsSUFBSSxDQUFDLENBQUMsUUFBUSxLQUFLLFNBQVMsRUFBRTtZQUNoRSxPQUFPLElBQUksQ0FBQztTQUNiO2FBQU0sSUFBSSxDQUFDLENBQUMsTUFBTSxLQUFLLENBQUMsQ0FBQyxNQUFNLElBQUksQ0FBQyxDQUFDLE1BQU0sS0FBSyxTQUFTLEVBQUU7WUFDMUQsT0FBTyxJQUFJLENBQUM7U0FDYjthQUFNLElBQUksQ0FBQyxDQUFDLE9BQU8sS0FBSyxDQUFDLENBQUMsT0FBTyxJQUFJLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxDQUFDLEdBQUcsRUFBRTtZQUNyRCxNQUFNLE1BQU0sR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7WUFDcEMsTUFBTSxNQUFNLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1lBQ3BDLE1BQU0sUUFBUSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDdkMsT0FBTyxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQyxJQUFJLEtBQUssUUFBUSxDQUFDLE1BQU0sQ0FBQztTQUNuRDtRQUNELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFdBQVcsQ0FBQyxHQUFHLFFBQW1CO1FBQ3ZDLEtBQUssTUFBTSxVQUFVLElBQUksUUFBUSxFQUFFO1lBQ2pDLEtBQUssTUFBTSxRQUFRLElBQUksSUFBSSxDQUFDLGVBQWUsRUFBRTtnQkFDM0MsSUFDRSxVQUFVLENBQUMsSUFBSSxLQUFLLFFBQVEsQ0FBQyxJQUFJO29CQUNqQyxVQUFVLENBQUMsS0FBSyxLQUFLLFFBQVEsQ0FBQyxLQUFLLEVBQ25DO29CQUNBLE1BQU0sSUFBSSxLQUFLLENBQ2IsR0FBRyxJQUFJLENBQUMsSUFBSSx3Q0FBd0MsVUFBVSxHQUFHLENBQ2xFLENBQUM7aUJBQ0g7YUFDRjtZQUNELElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQ3ZDO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGtCQUFrQixDQUFDLFNBQThCO1FBQ3RELEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxFQUFFO1lBQ3BELElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsS0FBSyxTQUFTLEVBQUU7Z0JBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQ2IsR0FBRyxJQUFJLENBQUMsSUFBSSx5Q0FBeUMsR0FBRyxHQUFHLENBQzVELENBQUM7YUFDSDtZQUNELElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO1NBQzdCO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFNBQVMsQ0FBQyxHQUFHLE1BQWdCO1FBQ2xDLEtBQUssTUFBTSxLQUFLLElBQUksTUFBTSxFQUFFO1lBQzFCLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsRUFBRTtnQkFDaEMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDekI7U0FDRjtJQUNILENBQUM7SUFFRDs7O09BR0c7SUFDSSxPQUFPLENBQUMsSUFBeUI7UUFDdEMsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDL0MsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLFNBQVMsRUFBRTtnQkFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLElBQUksQ0FBQyxJQUFJLG9DQUFvQyxHQUFHLEdBQUcsQ0FBQyxDQUFDO2FBQ3pFO1lBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7WUFDdkIsSUFBSSxLQUFLLENBQUMsS0FBSyxFQUFFO2dCQUNmLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO2FBQzdCO1NBQ0Y7SUFDSCxDQUFDO0lBRU8sUUFBUTtRQUNkLE9BQU87WUFDTCxPQUFPLEVBQUUsSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUM3QixPQUFPLEVBQ0wsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ25FLEtBQUssRUFBRSxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQztZQUNoQyxRQUFRLEVBQ04sSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQztnQkFDN0IsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDO2dCQUNyQyxDQUFDLENBQUMsU0FBUztZQUNmLFNBQVMsRUFDUCxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3hFLFFBQVEsRUFBRSxhQUFhLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUN0QyxNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3hELHFDQUFxQztZQUNyQyxrRkFBa0Y7WUFDbEYsMkJBQTJCO1lBQzNCLEdBQUcsYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDO1NBQ2xDLENBQUM7SUFDSixDQUFDO0lBRU8sYUFBYTtRQUNuQixNQUFNLFFBQVEsR0FBWTtZQUN4QixXQUFXLEVBQ1QsSUFBSSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sR0FBRyxDQUFDO2dCQUNoQyxDQUFDLENBQUMsSUFBSSxDQUFDLGtCQUFrQjtnQkFDekIsQ0FBQyxDQUFDLFNBQVM7WUFDZixTQUFTLEVBQUUsSUFBSSxDQUFDLGdCQUFnQjtZQUNoQyxZQUFZLEVBQ1YsSUFBSSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sR0FBRyxDQUFDO2dCQUNqQyxDQUFDLENBQUMsSUFBSSxDQUFDLG1CQUFtQjtnQkFDMUIsQ0FBQyxDQUFDLFNBQVM7WUFDZixLQUFLLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDeEIsS0FBSyxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQ3hCLGFBQWEsRUFBRSxJQUFJLENBQUMsb0JBQW9CO1lBQ3hDLEtBQUssRUFBRSxJQUFJLENBQUMsWUFBWTtZQUN4QixRQUFRLEVBQ04sSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3BFLElBQUksRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDaEUsT0FBTyxFQUFFLElBQUksQ0FBQyxjQUFjO1NBQzdCLENBQUM7UUFDRixPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNO1lBQ3BELENBQUMsQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDO1lBQ3pCLENBQUMsQ0FBQyxTQUFTLENBQUM7SUFDaEIsQ0FBQzs7QUFsVEgsMENBbVRDOzs7QUFFRCxTQUFTLGFBQWEsQ0FBYyxHQUFNLEVBQUUsZUFBd0IsS0FBSztJQUN2RSxJQUFJLE9BQU8sR0FBRyxLQUFLLFFBQVEsSUFBSSxHQUFHLElBQUksSUFBSSxFQUFFO1FBQzFDLE9BQU8sR0FBRyxDQUFDO0tBQ1o7SUFFRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDdEIsT0FBTyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQVEsQ0FBQztLQUNoRDtJQUVELE1BQU0sTUFBTSxHQUE0QixFQUFFLENBQUM7SUFDM0MsS0FBSyxJQUFJLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDdEMsSUFBSSxPQUFPLENBQUMsS0FBSyxRQUFRLElBQUksQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDLEtBQUssV0FBVyxFQUFFO1lBQzNELENBQUMsR0FBRyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDdEI7UUFDRCxNQUFNLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFlBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUN6QztJQUNELE9BQU8sTUFBYSxDQUFDO0FBQ3ZCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBzbmFrZSB9IGZyb20gXCJjYXNlXCI7XG5pbXBvcnQge1xuICBBcnRpZmFjdHMsXG4gIENhY2hlLFxuICBEZWZhdWx0LFxuICBJbWFnZSxcbiAgSW5jbHVkZSxcbiAgSm9iLFxuICBSZXRyeSxcbiAgU2VydmljZSxcbiAgVmFyaWFibGVDb25maWcsXG4gIFdvcmtmbG93LFxufSBmcm9tIFwiLi9jb25maWd1cmF0aW9uLW1vZGVsXCI7XG5pbXBvcnQgeyBDb21wb25lbnQgfSBmcm9tIFwiLi4vY29tcG9uZW50XCI7XG5pbXBvcnQgeyBQcm9qZWN0IH0gZnJvbSBcIi4uL3Byb2plY3RcIjtcbmltcG9ydCB7IFlhbWxGaWxlIH0gZnJvbSBcIi4uL3lhbWxcIjtcblxuLyoqXG4gKiBPcHRpb25zIGZvciBgQ2lDb25maWd1cmF0aW9uYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDaUNvbmZpZ3VyYXRpb25PcHRpb25zIHtcbiAgLyoqXG4gICAqIERlZmF1bHQgc2V0dGluZ3MgZm9yIHRoZSBDSSBDb25maWd1cmF0aW9uLiBKb2JzIHRoYXQgZG8gbm90IGRlZmluZSBvbmUgb3IgbW9yZSBvZiB0aGUgbGlzdGVkIGtleXdvcmRzIHVzZSB0aGUgdmFsdWUgZGVmaW5lZCBpbiB0aGUgZGVmYXVsdCBzZWN0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdD86IERlZmF1bHQ7XG4gIC8qKlxuICAgKiBBIHNwZWNpYWwgam9iIHVzZWQgdG8gdXBsb2FkIHN0YXRpYyBzaXRlcyB0byBHaXRsYWIgcGFnZXMuIFJlcXVpcmVzIGEgYHB1YmxpYy9gIGRpcmVjdG9yeVxuICAgKiB3aXRoIGBhcnRpZmFjdHMucGF0aGAgcG9pbnRpbmcgdG8gaXQuXG4gICAqL1xuICByZWFkb25seSBwYWdlcz86IEpvYjtcbiAgLyoqXG4gICAqIFVzZWQgdG8gY29udHJvbCBwaXBlbGluZSBiZWhhdmlvci5cbiAgICovXG4gIHJlYWRvbmx5IHdvcmtmbG93PzogV29ya2Zsb3c7XG4gIC8qKlxuICAgKiBHcm91cHMgam9icyBpbnRvIHN0YWdlcy4gQWxsIGpvYnMgaW4gb25lIHN0YWdlIG11c3QgY29tcGxldGUgYmVmb3JlIG5leHQgc3RhZ2UgaXNcbiAgICogZXhlY3V0ZWQuIElmIG5vIHN0YWdlcyBhcmUgc3BlY2lmaWVkLiBEZWZhdWx0cyB0byBbJ2J1aWxkJywgJ3Rlc3QnLCAnZGVwbG95J10uXG4gICAqL1xuICByZWFkb25seSBzdGFnZXM/OiBzdHJpbmdbXTtcbiAgLyoqXG4gICAqIEdsb2JhbCB2YXJpYWJsZXMgdGhhdCBhcmUgcGFzc2VkIHRvIGpvYnMuXG4gICAqIElmIHRoZSBqb2IgYWxyZWFkeSBoYXMgdGhhdCB2YXJpYWJsZSBkZWZpbmVkLCB0aGUgam9iLWxldmVsIHZhcmlhYmxlIHRha2VzIHByZWNlZGVuY2UuXG4gICAqL1xuICByZWFkb25seSB2YXJpYWJsZXM/OiBSZWNvcmQ8c3RyaW5nLCBhbnk+O1xuICAvKipcbiAgICogQW4gaW5pdGlhbCBzZXQgb2Ygam9icyB0byBhZGQgdG8gdGhlIGNvbmZpZ3VyYXRpb24uXG4gICAqL1xuICByZWFkb25seSBqb2JzPzogUmVjb3JkPHN0cmluZywgSm9iPjtcbn1cblxuLyoqXG4gKiBDSSBmb3IgR2l0TGFiLlxuICogQSBDSSBpcyBhIGNvbmZpZ3VyYWJsZSBhdXRvbWF0ZWQgcHJvY2VzcyBtYWRlIHVwIG9mIG9uZSBvciBtb3JlIHN0YWdlcy9qb2JzLlxuICogQHNlZSBodHRwczovL2RvY3MuZ2l0bGFiLmNvbS9lZS9jaS95YW1sL1xuICovXG5leHBvcnQgY2xhc3MgQ2lDb25maWd1cmF0aW9uIGV4dGVuZHMgQ29tcG9uZW50IHtcbiAgLyoqXG4gICAqIFRoZSBwcm9qZWN0IHRoZSBjb25maWd1cmF0aW9uIGJlbG9uZ3MgdG8uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcHJvamVjdDogUHJvamVjdDtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBjb25maWd1cmF0aW9uLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIFBhdGggdG8gQ0kgZmlsZSBnZW5lcmF0ZWQgYnkgdGhlIGNvbmZpZ3VyYXRpb24uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGF0aDogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIHdvcmtmbG93IFlBTUwgZmlsZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBmaWxlOiBZYW1sRmlsZTtcbiAgLyoqXG4gICAqIERlZmluZXMgZGVmYXVsdCBzY3JpcHRzIHRoYXQgc2hvdWxkIHJ1biAqYWZ0ZXIqIGFsbCBqb2JzLiBDYW4gYmUgb3ZlcnJpZGVuIGJ5IHRoZSBqb2IgbGV2ZWwgYGFmdGVyU2NyaXB0YC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZWZhdWx0QWZ0ZXJTY3JpcHQ6IHN0cmluZ1tdID0gW107XG4gIC8qKlxuICAgKiBEZWZhdWx0IGxpc3Qgb2YgZmlsZXMgYW5kIGRpcmVjdG9yaWVzIHRoYXQgc2hvdWxkIGJlIGF0dGFjaGVkIHRvIHRoZSBqb2IgaWYgaXQgc3VjY2VlZHMuIEFydGlmYWN0cyBhcmUgc2VudCB0byBHaXRsYWIgd2hlcmUgdGhleSBjYW4gYmUgZG93bmxvYWRlZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZWZhdWx0QXJ0aWZhY3RzPzogQXJ0aWZhY3RzO1xuICAvKipcbiAgICogRGVmaW5lcyBkZWZhdWx0IHNjcmlwdHMgdGhhdCBzaG91bGQgcnVuICpiZWZvcmUqIGFsbCBqb2JzLiBDYW4gYmUgb3ZlcnJpZGVuIGJ5IHRoZSBqb2IgbGV2ZWwgYGFmdGVyU2NyaXB0YC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZWZhdWx0QmVmb3JlU2NyaXB0OiBzdHJpbmdbXSA9IFtdO1xuICAvKipcbiAgICogQSBkZWZhdWx0IGxpc3Qgb2YgZmlsZXMgYW5kIGRpcmVjdG9yaWVzIHRvIGNhY2hlIGJldHdlZW4gam9icy4gWW91IGNhbiBvbmx5IHVzZSBwYXRocyB0aGF0IGFyZSBpbiB0aGUgbG9jYWwgd29ya2luZyBjb3B5LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRlZmF1bHRDYWNoZT86IENhY2hlO1xuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBkZWZhdWx0IGRvY2tlciBpbWFnZSB0byB1c2UgZ2xvYmFsbHkgZm9yIGFsbCBqb2JzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRlZmF1bHRJbWFnZT86IEltYWdlO1xuICAvKipcbiAgICogVGhlIGRlZmF1bHQgYmVoYXZpb3IgZm9yIHdoZXRoZXIgYSBqb2Igc2hvdWxkIGJlIGNhbmNlbGVkIHdoZW4gYSBuZXdlciBwaXBlbGluZSBzdGFydHMgYmVmb3JlIHRoZSBqb2IgY29tcGxldGVzIChEZWZhdWx0OiBmYWxzZSkuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGVmYXVsdEludGVycnVwdGlibGU/OiBib29sZWFuO1xuICAvKipcbiAgICogSG93IG1hbnkgdGltZXMgYSBqb2IgaXMgcmV0cmllZCBpZiBpdCBmYWlscy4gSWYgbm90IGRlZmluZWQsIGRlZmF1bHRzIHRvIDAgYW5kIGpvYnMgZG8gbm90IHJldHJ5LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRlZmF1bHRSZXRyeT86IFJldHJ5O1xuICAvKipcbiAgICogQSBkZWZhdWx0IGxpc3Qgb2YgYWRkaXRpb25hbCBEb2NrZXIgaW1hZ2VzIHRvIHJ1biBzY3JpcHRzIGluLiBUaGUgc2VydmljZSBpbWFnZSBpcyBsaW5rZWQgdG8gdGhlIGltYWdlIHNwZWNpZmllZCBpbiB0aGUgIGltYWdlIHBhcmFtZXRlci5cbiAgICovXG4gIHByaXZhdGUgZGVmYXVsdFNlcnZpY2VzOiBTZXJ2aWNlW10gPSBbXTtcbiAgLyoqXG4gICAqIFVzZWQgdG8gc2VsZWN0IGEgc3BlY2lmaWMgcnVubmVyIGZyb20gdGhlIGxpc3Qgb2YgYWxsIHJ1bm5lcnMgdGhhdCBhcmUgYXZhaWxhYmxlIGZvciB0aGUgcHJvamVjdC5cbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRUYWdzOiBzdHJpbmdbXSA9IFtdO1xuICAvKipcbiAgICogQSBkZWZhdWx0IHRpbWVvdXQgam9iIHdyaXR0ZW4gaW4gbmF0dXJhbCBsYW5ndWFnZSAoRXguIG9uZSBob3VyLCAzNjAwIHNlY29uZHMsIDYwIG1pbnV0ZXMpLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdFRpbWVvdXQ/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBDYW4gYmUgYEluY2x1ZGVgIG9yIGBJbmNsdWRlW11gLiBFYWNoIGBJbmNsdWRlYCB3aWxsIGJlIGEgc3RyaW5nLCBvciBhblxuICAgKiBvYmplY3Qgd2l0aCBwcm9wZXJ0aWVzIGZvciB0aGUgbWV0aG9kIGlmIGluY2x1ZGluZyBleHRlcm5hbCBZQU1MIGZpbGUuIFRoZSBleHRlcm5hbFxuICAgKiBjb250ZW50IHdpbGwgYmUgZmV0Y2hlZCwgaW5jbHVkZWQgYW5kIGV2YWx1YXRlZCBhbG9uZyB0aGUgYC5naXRsYWItY2kueW1sYC5cbiAgICovXG4gIHByaXZhdGUgaW5jbHVkZTogSW5jbHVkZVtdID0gW107XG4gIC8qKlxuICAgKiBBIHNwZWNpYWwgam9iIHVzZWQgdG8gdXBsb2FkIHN0YXRpYyBzaXRlcyB0byBHaXRsYWIgcGFnZXMuIFJlcXVpcmVzIGEgYHB1YmxpYy9gIGRpcmVjdG9yeVxuICAgKiB3aXRoIGBhcnRpZmFjdHMucGF0aGAgcG9pbnRpbmcgdG8gaXQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGFnZXM/OiBKb2I7XG4gIC8qKlxuICAgKiBHcm91cHMgam9icyBpbnRvIHN0YWdlcy4gQWxsIGpvYnMgaW4gb25lIHN0YWdlIG11c3QgY29tcGxldGUgYmVmb3JlIG5leHQgc3RhZ2UgaXNcbiAgICogZXhlY3V0ZWQuIERlZmF1bHRzIHRvIFsnYnVpbGQnLCAndGVzdCcsICdkZXBsb3knXS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzdGFnZXM6IHN0cmluZ1tdID0gW107XG4gIC8qKlxuICAgKiBHbG9iYWwgdmFyaWFibGVzIHRoYXQgYXJlIHBhc3NlZCB0byBqb2JzLlxuICAgKiBJZiB0aGUgam9iIGFscmVhZHkgaGFzIHRoYXQgdmFyaWFibGUgZGVmaW5lZCwgdGhlIGpvYi1sZXZlbCB2YXJpYWJsZSB0YWtlcyBwcmVjZWRlbmNlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZhcmlhYmxlczogUmVjb3JkPHN0cmluZywgbnVtYmVyIHwgVmFyaWFibGVDb25maWcgfCBzdHJpbmc+ID1cbiAgICB7fTtcbiAgLyoqXG4gICAqIFVzZWQgdG8gY29udHJvbCBwaXBlbGluZSBiZWhhdmlvci5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB3b3JrZmxvdz86IFdvcmtmbG93O1xuICAvKipcbiAgICogVGhlIGpvYnMgaW4gdGhlIENJIGNvbmZpZ3VyYXRpb24uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgam9iczogUmVjb3JkPHN0cmluZywgSm9iPiA9IHt9O1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIHByb2plY3Q6IFByb2plY3QsXG4gICAgbmFtZTogc3RyaW5nLFxuICAgIG9wdGlvbnM/OiBDaUNvbmZpZ3VyYXRpb25PcHRpb25zXG4gICkge1xuICAgIHN1cGVyKHByb2plY3QpO1xuICAgIHRoaXMucHJvamVjdCA9IHByb2plY3Q7XG4gICAgdGhpcy5uYW1lID0gcGF0aC5wYXJzZShuYW1lKS5uYW1lO1xuICAgIHRoaXMucGF0aCA9XG4gICAgICB0aGlzLm5hbWUgPT09IFwiZ2l0bGFiLWNpXCJcbiAgICAgICAgPyBcIi5naXRsYWItY2kueW1sXCJcbiAgICAgICAgOiBgLmdpdGxhYi9jaS10ZW1wbGF0ZXMvJHtuYW1lLnRvTG9jYWxlTG93ZXJDYXNlKCl9LnltbGA7XG4gICAgdGhpcy5maWxlID0gbmV3IFlhbWxGaWxlKHRoaXMucHJvamVjdCwgdGhpcy5wYXRoLCB7XG4gICAgICBvYmo6ICgpID0+IHRoaXMucmVuZGVyQ0koKSxcbiAgICAgIC8vIEdpdExhYiBuZWVkcyB0byByZWFkIHRoZSBmaWxlIGZyb20gdGhlIHJlcG9zaXRvcnkgaW4gb3JkZXIgdG8gd29yay5cbiAgICAgIGNvbW1pdHRlZDogdHJ1ZSxcbiAgICB9KTtcbiAgICBjb25zdCBkZWZhdWx0cyA9IG9wdGlvbnM/LmRlZmF1bHQ7XG4gICAgaWYgKGRlZmF1bHRzKSB7XG4gICAgICB0aGlzLmRlZmF1bHRBZnRlclNjcmlwdC5wdXNoKC4uLihkZWZhdWx0cy5hZnRlclNjcmlwdCA/PyBbXSkpO1xuICAgICAgdGhpcy5kZWZhdWx0QXJ0aWZhY3RzID0gZGVmYXVsdHMuYXJ0aWZhY3RzO1xuICAgICAgZGVmYXVsdHMuYmVmb3JlU2NyaXB0ICYmXG4gICAgICAgIHRoaXMuZGVmYXVsdEJlZm9yZVNjcmlwdC5wdXNoKC4uLmRlZmF1bHRzLmJlZm9yZVNjcmlwdCk7XG4gICAgICB0aGlzLmRlZmF1bHRDYWNoZSA9IGRlZmF1bHRzLmNhY2hlO1xuICAgICAgdGhpcy5kZWZhdWx0SW1hZ2UgPSBkZWZhdWx0cy5pbWFnZTtcbiAgICAgIHRoaXMuZGVmYXVsdEludGVycnVwdGlibGUgPSBkZWZhdWx0cy5pbnRlcnJ1cHRpYmxlO1xuICAgICAgdGhpcy5kZWZhdWx0UmV0cnkgPSBkZWZhdWx0cy5yZXRyeTtcbiAgICAgIGRlZmF1bHRzLnNlcnZpY2VzICYmIHRoaXMuYWRkU2VydmljZXMoLi4uZGVmYXVsdHMuc2VydmljZXMpO1xuICAgICAgZGVmYXVsdHMudGFncyAmJiB0aGlzLmRlZmF1bHRUYWdzLnB1c2goLi4uZGVmYXVsdHMudGFncyk7XG4gICAgICB0aGlzLmRlZmF1bHRUaW1lb3V0ID0gZGVmYXVsdHMudGltZW91dDtcbiAgICB9XG4gICAgdGhpcy5wYWdlcyA9IG9wdGlvbnM/LnBhZ2VzO1xuICAgIHRoaXMud29ya2Zsb3cgPSBvcHRpb25zPy53b3JrZmxvdztcbiAgICBpZiAob3B0aW9ucz8uc3RhZ2VzKSB7XG4gICAgICB0aGlzLmFkZFN0YWdlcyguLi5vcHRpb25zLnN0YWdlcyk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zPy52YXJpYWJsZXMpIHtcbiAgICAgIHRoaXMuYWRkSm9icyhvcHRpb25zLnZhcmlhYmxlcyk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zPy5qb2JzKSB7XG4gICAgICB0aGlzLmFkZEpvYnMob3B0aW9ucy5qb2JzKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQWRkIGFkZGl0aW9uYWwgeW1sL3lhbWwgZmlsZXMgdG8gdGhlIENJIGluY2x1ZGVzXG4gICAqIEBwYXJhbSBpbmNsdWRlcyBUaGUgaW5jbHVkZXMgdG8gYWRkLlxuICAgKi9cbiAgcHVibGljIGFkZEluY2x1ZGVzKC4uLmluY2x1ZGVzOiBJbmNsdWRlW10pIHtcbiAgICBmb3IgKGNvbnN0IGFkZGl0aW9uYWwgb2YgaW5jbHVkZXMpIHtcbiAgICAgIHRoaXMuYXNzZXJ0SXNWYWxpZEluY2x1ZGUoYWRkaXRpb25hbCk7XG4gICAgICBmb3IgKGNvbnN0IGV4aXN0aW5nIG9mIHRoaXMuaW5jbHVkZSkge1xuICAgICAgICBpZiAodGhpcy5hcmVFcXVhbEluY2x1ZGVzKGV4aXN0aW5nLCBhZGRpdGlvbmFsKSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAgIGAke3RoaXMubmFtZX06IEdpdExhYiBDSSAke2V4aXN0aW5nfSBhbHJlYWR5IGNvbnRhaW5zIG9uZSBvciBtb3JlIHRlbXBsYXRlcyBzcGVjaWZpZWQgaW4gJHthZGRpdGlvbmFsfS5gXG4gICAgICAgICAgKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgdGhpcy5pbmNsdWRlLnB1c2goYWRkaXRpb25hbCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFRocm93IGFuIGVycm9yIGlmIHRoZSBwcm92aWRlZCBJbmNsdWRlIGlzIGludmFsaWQuXG4gICAqIEBzZWUgaHR0cHM6Ly9naXRsYWIuY29tL2dpdGxhYi1vcmcvZ2l0bGFiLy0vYmxvYi9tYXN0ZXIvbGliL2dpdGxhYi9jaS9jb25maWcvZXh0ZXJuYWwvbWFwcGVyLnJiXG4gICAqIEBwYXJhbSBpbmNsdWRlIHRoZSBJbmNsdWRlIHRvIHZhbGlkYXRlLlxuICAgKi9cbiAgcHJpdmF0ZSBhc3NlcnRJc1ZhbGlkSW5jbHVkZShpbmNsdWRlOiBJbmNsdWRlKSB7XG4gICAgY29uc3QgY29tYm9zID0gW1xuICAgICAgaW5jbHVkZS5sb2NhbCxcbiAgICAgIGluY2x1ZGUuZmlsZSAmJiBpbmNsdWRlLnByb2plY3QsXG4gICAgICBpbmNsdWRlLnJlbW90ZSxcbiAgICAgIGluY2x1ZGUudGVtcGxhdGUsXG4gICAgXTtcbiAgICBjb25zdCBsZW4gPSBjb21ib3MuZmlsdGVyKCh4KSA9PiBCb29sZWFuKHgpKS5sZW5ndGg7XG4gICAgaWYgKGxlbiAhPT0gMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgJHt0aGlzLm5hbWV9OiBHaXRMYWIgQ0kgaW5jbHVkZSAke2luY2x1ZGV9IGNvbnRhaW5zICR7bGVufSBwcm9wZXJ0eSBjb21iaW5hdGlvbihzKS5cbiAgICAgICAgQSB2YWxpZCBpbmNsdWRlIGNvbmZpZ3VyYXRpb24gc3BlY2lmaWVzICpvbmUqIG9mIHRoZSBmb2xsb3dpbmcgcHJvcGVydHkgY29tYmluYXRpb25zLlxuICAgICAgICAqIGxvY2FsXG4gICAgICAgICogZmlsZSwgcHJvamVjdFxuICAgICAgICAqIHJlbW90ZVxuICAgICAgICAqIHRlbXBsYXRlICBcbiAgICAgICAgYFxuICAgICAgKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgaWYgdGhlIGVxdWFsaXR5IG9mIEluY2x1ZGVzLlxuICAgKiBAc2VlIGh0dHBzOi8vZ2l0bGFiLmNvbS9naXRsYWItb3JnL2dpdGxhYi8tL2Jsb2IvbWFzdGVyL2xpYi9naXRsYWIvY2kvY29uZmlnL2V4dGVybmFsL21hcHBlci5yYlxuICAgKiBAcGFyYW0geCBGaXJzdCBpbmNsdWRlIHRvIGNvbXBhcmUuXG4gICAqIEBwYXJhbSB5IFNlY29uZCBpbmNsdWRlIHRvIGNvbXBhcmUuXG4gICAqIEByZXR1cm5zIFdoZXRoZXIgdGhlIGluY2x1ZGVzIGFyZSBlcXVhbC5cbiAgICovXG4gIHByaXZhdGUgYXJlRXF1YWxJbmNsdWRlcyh4OiBJbmNsdWRlLCB5OiBJbmNsdWRlKTogYm9vbGVhbiB7XG4gICAgaWYgKHgubG9jYWwgPT09IHkubG9jYWwgJiYgeC5sb2NhbCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9IGVsc2UgaWYgKHgudGVtcGxhdGUgPT09IHkudGVtcGxhdGUgJiYgeC50ZW1wbGF0ZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9IGVsc2UgaWYgKHgucmVtb3RlID09PSB5LnJlbW90ZSAmJiB4LnJlbW90ZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9IGVsc2UgaWYgKHgucHJvamVjdCA9PT0geS5wcm9qZWN0ICYmIHgucmVmID09PSB5LnJlZikge1xuICAgICAgY29uc3QgeEZpbGVzID0geC5maWxlID8geC5maWxlIDogW107XG4gICAgICBjb25zdCB5RmlsZXMgPSB5LmZpbGUgPyB5LmZpbGUgOiBbXTtcbiAgICAgIGNvbnN0IGFsbEZpbGVzID0geEZpbGVzLmNvbmNhdCh5RmlsZXMpO1xuICAgICAgcmV0dXJuIG5ldyBTZXQoYWxsRmlsZXMpLnNpemUgIT09IGFsbEZpbGVzLmxlbmd0aDtcbiAgICB9XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhZGRpdGlvbmFsIHNlcnZpY2VzLlxuICAgKiBAcGFyYW0gc2VydmljZXMgVGhlIHNlcnZpY2VzIHRvIGFkZC5cbiAgICovXG4gIHB1YmxpYyBhZGRTZXJ2aWNlcyguLi5zZXJ2aWNlczogU2VydmljZVtdKSB7XG4gICAgZm9yIChjb25zdCBhZGRpdGlvbmFsIG9mIHNlcnZpY2VzKSB7XG4gICAgICBmb3IgKGNvbnN0IGV4aXN0aW5nIG9mIHRoaXMuZGVmYXVsdFNlcnZpY2VzKSB7XG4gICAgICAgIGlmIChcbiAgICAgICAgICBhZGRpdGlvbmFsLm5hbWUgPT09IGV4aXN0aW5nLm5hbWUgJiZcbiAgICAgICAgICBhZGRpdGlvbmFsLmFsaWFzID09PSBleGlzdGluZy5hbGlhc1xuICAgICAgICApIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgICBgJHt0aGlzLm5hbWV9OiBHaXRMYWIgQ0kgYWxyZWFkeSBjb250YWlucyBzZXJ2aWNlICR7YWRkaXRpb25hbH0uYFxuICAgICAgICAgICk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIHRoaXMuZGVmYXVsdFNlcnZpY2VzLnB1c2goYWRkaXRpb25hbCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIGdsb2JhbGx5IGRlZmluZWQgdmFyaWFibGUgdG8gdGhlIENJIGNvbmZpZ3VyYXRpb24uXG4gICAqIEBwYXJhbSB2YXJpYWJsZXMgVGhlIHZhcmlhYmxlcyB0byBhZGQuXG4gICAqL1xuICBwdWJsaWMgYWRkR2xvYmFsVmFyaWFibGVzKHZhcmlhYmxlczogUmVjb3JkPHN0cmluZywgYW55Pikge1xuICAgIGZvciAoY29uc3QgW2tleSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKHZhcmlhYmxlcykpIHtcbiAgICAgIGlmICh0aGlzLnZhcmlhYmxlc1trZXldICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIGAke3RoaXMubmFtZX06IEdpdExhYiBDSSBhbHJlYWR5IGNvbnRhaW5zIHZhcmlhYmxlICR7a2V5fS5gXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgICB0aGlzLnZhcmlhYmxlc1trZXldID0gdmFsdWU7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBzdGFnZXMgdG8gdGhlIENJIGNvbmZpZ3VyYXRpb24gaWYgbm90IGFscmVhZHkgcHJlc2VudC5cbiAgICogQHBhcmFtIHN0YWdlcyBzdGFnZXMgdG8gYWRkLlxuICAgKi9cbiAgcHVibGljIGFkZFN0YWdlcyguLi5zdGFnZXM6IHN0cmluZ1tdKSB7XG4gICAgZm9yIChjb25zdCBzdGFnZSBvZiBzdGFnZXMpIHtcbiAgICAgIGlmICghdGhpcy5zdGFnZXMuaW5jbHVkZXMoc3RhZ2UpKSB7XG4gICAgICAgIHRoaXMuc3RhZ2VzLnB1c2goc3RhZ2UpO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgam9icyBhbmQgdGhlaXIgc3RhZ2VzIHRvIHRoZSBDSSBjb25maWd1cmF0aW9uLlxuICAgKiBAcGFyYW0gam9icyBKb2JzIHRvIGFkZC5cbiAgICovXG4gIHB1YmxpYyBhZGRKb2JzKGpvYnM6IFJlY29yZDxzdHJpbmcsIEpvYj4pIHtcbiAgICBmb3IgKGNvbnN0IFtrZXksIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyhqb2JzKSkge1xuICAgICAgaWYgKHRoaXMuam9ic1trZXldICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGAke3RoaXMubmFtZX06IEdpdExhYiBDSSBhbHJlYWR5IGNvbnRhaW5zIGpvYiAke2tleX0uYCk7XG4gICAgICB9XG4gICAgICB0aGlzLmpvYnNba2V5XSA9IHZhbHVlO1xuICAgICAgaWYgKHZhbHVlLnN0YWdlKSB7XG4gICAgICAgIHRoaXMuYWRkU3RhZ2VzKHZhbHVlLnN0YWdlKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHJlbmRlckNJKCkge1xuICAgIHJldHVybiB7XG4gICAgICBkZWZhdWx0OiB0aGlzLnJlbmRlckRlZmF1bHQoKSxcbiAgICAgIGluY2x1ZGU6XG4gICAgICAgIHRoaXMuaW5jbHVkZS5sZW5ndGggPiAwID8gc25ha2VDYXNlS2V5cyh0aGlzLmluY2x1ZGUpIDogdW5kZWZpbmVkLFxuICAgICAgcGFnZXM6IHNuYWtlQ2FzZUtleXModGhpcy5wYWdlcyksXG4gICAgICBzZXJ2aWNlczpcbiAgICAgICAgdGhpcy5kZWZhdWx0U2VydmljZXMubGVuZ3RoID4gMFxuICAgICAgICAgID8gc25ha2VDYXNlS2V5cyh0aGlzLmRlZmF1bHRTZXJ2aWNlcylcbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgIHZhcmlhYmxlczpcbiAgICAgICAgT2JqZWN0LmVudHJpZXModGhpcy52YXJpYWJsZXMpLmxlbmd0aCA+IDAgPyB0aGlzLnZhcmlhYmxlcyA6IHVuZGVmaW5lZCxcbiAgICAgIHdvcmtmbG93OiBzbmFrZUNhc2VLZXlzKHRoaXMud29ya2Zsb3cpLFxuICAgICAgc3RhZ2VzOiB0aGlzLnN0YWdlcy5sZW5ndGggPiAwID8gdGhpcy5zdGFnZXMgOiB1bmRlZmluZWQsXG4gICAgICAvLyB3ZSBkbyBub3Qgd2FudCB0byBjaGFuZ2Ugam9iIG5hbWVzXG4gICAgICAvLyBhcyB0aGV5IGNhbiBiZSBoaWRkZW4gKGh0dHBzOi8vZG9jcy5naXRsYWIuY29tL2VlL2NpL2pvYnMvaW5kZXguaHRtbCNoaWRlLWpvYnMpXG4gICAgICAvLyBvciByZWZlcmVuY2VkIGluIGV4dGVuZHNcbiAgICAgIC4uLnNuYWtlQ2FzZUtleXModGhpcy5qb2JzLCB0cnVlKSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJEZWZhdWx0KCkge1xuICAgIGNvbnN0IGRlZmF1bHRzOiBEZWZhdWx0ID0ge1xuICAgICAgYWZ0ZXJTY3JpcHQ6XG4gICAgICAgIHRoaXMuZGVmYXVsdEFmdGVyU2NyaXB0Lmxlbmd0aCA+IDBcbiAgICAgICAgICA/IHRoaXMuZGVmYXVsdEFmdGVyU2NyaXB0XG4gICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBhcnRpZmFjdHM6IHRoaXMuZGVmYXVsdEFydGlmYWN0cyxcbiAgICAgIGJlZm9yZVNjcmlwdDpcbiAgICAgICAgdGhpcy5kZWZhdWx0QmVmb3JlU2NyaXB0Lmxlbmd0aCA+IDBcbiAgICAgICAgICA/IHRoaXMuZGVmYXVsdEJlZm9yZVNjcmlwdFxuICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgY2FjaGU6IHRoaXMuZGVmYXVsdENhY2hlLFxuICAgICAgaW1hZ2U6IHRoaXMuZGVmYXVsdEltYWdlLFxuICAgICAgaW50ZXJydXB0aWJsZTogdGhpcy5kZWZhdWx0SW50ZXJydXB0aWJsZSxcbiAgICAgIHJldHJ5OiB0aGlzLmRlZmF1bHRSZXRyeSxcbiAgICAgIHNlcnZpY2VzOlxuICAgICAgICB0aGlzLmRlZmF1bHRTZXJ2aWNlcy5sZW5ndGggPiAwID8gdGhpcy5kZWZhdWx0U2VydmljZXMgOiB1bmRlZmluZWQsXG4gICAgICB0YWdzOiB0aGlzLmRlZmF1bHRUYWdzLmxlbmd0aCA+IDAgPyB0aGlzLmRlZmF1bHRUYWdzIDogdW5kZWZpbmVkLFxuICAgICAgdGltZW91dDogdGhpcy5kZWZhdWx0VGltZW91dCxcbiAgICB9O1xuICAgIHJldHVybiBPYmplY3QudmFsdWVzKGRlZmF1bHRzKS5maWx0ZXIoKHgpID0+IHgpLmxlbmd0aFxuICAgICAgPyBzbmFrZUNhc2VLZXlzKGRlZmF1bHRzKVxuICAgICAgOiB1bmRlZmluZWQ7XG4gIH1cbn1cblxuZnVuY3Rpb24gc25ha2VDYXNlS2V5czxUID0gdW5rbm93bj4ob2JqOiBULCBza2lwVG9wTGV2ZWw6IGJvb2xlYW4gPSBmYWxzZSk6IFQge1xuICBpZiAodHlwZW9mIG9iaiAhPT0gXCJvYmplY3RcIiB8fCBvYmogPT0gbnVsbCkge1xuICAgIHJldHVybiBvYmo7XG4gIH1cblxuICBpZiAoQXJyYXkuaXNBcnJheShvYmopKSB7XG4gICAgcmV0dXJuIG9iai5tYXAoKG8pID0+IHNuYWtlQ2FzZUtleXMobykpIGFzIGFueTtcbiAgfVxuXG4gIGNvbnN0IHJlc3VsdDogUmVjb3JkPHN0cmluZywgdW5rbm93bj4gPSB7fTtcbiAgZm9yIChsZXQgW2ssIHZdIG9mIE9iamVjdC5lbnRyaWVzKG9iaikpIHtcbiAgICBpZiAodHlwZW9mIHYgPT09IFwib2JqZWN0XCIgJiYgdiAhPSBudWxsICYmIGsgIT09IFwidmFyaWFibGVzXCIpIHtcbiAgICAgIHYgPSBzbmFrZUNhc2VLZXlzKHYpO1xuICAgIH1cbiAgICByZXN1bHRbc2tpcFRvcExldmVsID8gayA6IHNuYWtlKGspXSA9IHY7XG4gIH1cbiAgcmV0dXJuIHJlc3VsdCBhcyBhbnk7XG59XG4iXX0=